﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Threading.Tasks;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // スケルタルアニメーションのマージボーン圧縮
    public class IfSkeletalAnimBoneMergeCompressor : IfSkeletalAnimCompressor
    {
        // コンストラクタ
        public IfSkeletalAnimBoneMergeCompressor() :
            base("IfSkeletalAnimBoneMergeCompressor_Log") { }

        // プロセス
        public override string Process
        {
            get { return "compress_bone_merge"; }
        }

        // 圧縮
        protected override void Compress()
        {
            CompressBoneMerge(this.Target, this.Streams);

            // 不要になったストリームを削除する
            StreamUtility.SortStream(this.Target, this.Streams);
        }

        internal static void CompressBoneMerge(skeletal_animType target, List<G3dStream> streams)
        {
            bone_animType[] anims = target.bone_anim_array.bone_anim;
            int animCount = anims.Length;
            BoneAnimInfo[] animInfos = new BoneAnimInfo[animCount];
            BoneAnimInfo[] parent;

            // ボーン情報の初期化
            for (int i = 0; i < animCount; i++)
            { animInfos[i] = new BoneAnimInfo(anims[i], target.skeletal_anim_info); }
            BoneAnimInfo.Setup(animInfos);

            // 削除可能なボーンを見つける
            int numFound;
            do
            {
                numFound = 0;
                for (int ianim = 1; ianim < animCount; ++ianim)
                {
                    BoneAnimInfo animInfo = animInfos[ianim];
                    if (animInfo.IsReservedToRemove)
                    {
                        continue;
                    }
                    if (CheckRemovable(animInfo))
                    {
                        // binarize_scale /    binarize_rotate / binarize_translate が false のボーンアニメーションは
                        // 圧縮できないのでエラーとする。
                        if (!GetBinarizeFlag(animInfo.BoneAnim))
                        {
                            IfStrings.Throw("SkeletalAnimCompress_Error_InvalidBinarizeFlag", animInfo.BoneAnim.name);
                        }

                        ++numFound;
                        animInfo.IsReservedToRemove = true;
                    }
                }
            } while (numFound > 0);

            // 以下の条件に当てはまる場合はルートボーンを削除する。
            // - 削除可能
            // - ルートボーンの削除されない子供の数が 1 つだけ
            // ※ Merge ボーン圧縮ではボーン名のチェックを行わない。
            //    これは NW4R 版の実装に合わせるためである。
            BoneAnimInfo rootAnimInfo = animInfos[0];
            SkeletalAnimCompressUtility.GetThinOutBoneRef(animInfos, out parent);
            BoneAnimInfo[] rootChild = Array.FindAll(
                parent, delegate(BoneAnimInfo bone) { return (bone == rootAnimInfo); });
            if (CheckRemovable(rootAnimInfo) && (rootChild.Length <= 1))
            {
                rootAnimInfo.IsReservedToRemove = true;
            }

            // ボーン圧縮後のルートボーンを見つける。
            if (rootAnimInfo.IsReservedToRemove)
            {
                int idx = Array.FindIndex(
                    parent, delegate(BoneAnimInfo bone) { return (bone == rootAnimInfo); });
                // 全てのノードが削除される場合は idx が -1 になる。
                // フォーマッタがボーンが存在していない場合に対応していないので、ルートノードのみ残す。
                if (idx >= 0)
                {
                    rootAnimInfo = animInfos[idx];
                }
                else
                {
                    rootAnimInfo.IsReservedToRemove = false;
                }
            }

            // アニメーションが付いているボーンが削除されるか確認する
            // アニメーションのマージが行われると元々アニメーションを持たないボーンに
            // カーブが設定されてしまうので正しく表示できなくなる
            // そのためこの確認はアニメーションのマージを行う前に実行する必要がある
            SkeletalAnimCompressUtility.CheckAnimationRemoved(animInfos);
            // 新しいアニメーションのインデックスを設定する
            SkeletalAnimCompressUtility.SetNewIndex(rootAnimInfo);

            // 親が削除されたら、自身のSRT値を新しい親の座標系で定義しなおす。
            var animationItems = new List<SkeletalAnimCompressUtility.SetAnimationItem>();
            for (int ianim = 1; ianim < animInfos.Length; ++ianim)
            {
                BoneAnimInfo animInfo = animInfos[ianim];
                BoneAnimInfo parentInfo = animInfo.GetIncompressibleAncestorBone();

                if (!animInfo.IsReservedToRemove && animInfo.Parent.IsReservedToRemove)
                {
                    animationItems.AddRange(
                        SkeletalAnimCompressUtility.MergeAnimation(
                            target,
                            streams,
                            parentInfo,
                            animInfo));
                    // 親ボーンとの合成後は正しく扱えないので。
                    animInfo.BoneAnim.scale_compensate = false;
                }
            }
            // SetAnimationは重いので、並列実行する
            G3dParallel.ForEach(animationItems, item =>
            {
                item.Stream = SkeletalAnimCompressUtility.SetAnimation(item.Target, item.FullValues, item.CurveTolerance, item.IsLoop, item.IsAngle);
            });

            foreach (var item in animationItems)
            {
                if (item.Stream == null)
                {
                    continue;
                }
                var streamIndex = streams.Count;
                streams.Add(item.Stream);
                Nintendo.Foundation.Contracts.Assertion.Operation.True(item.Target.Item is hermite_curveType);
                var curve = item.Target.Item as hermite_curveType;
                curve.stream_index = streamIndex;
            }

            // アニメーションの削除
            SkeletalAnimCompressUtility.RemoveBoneAnims(target, animInfos);
        }

        private static bool CheckRemovable(BoneAnimInfo animInfo)
        {
            bone_animType anim = animInfo.BoneAnim;
            if (!anim.compress_enable)
            {
                // 圧縮禁止
                return false;
            }
            // レンダリングに使用されているボーンは圧縮しない
            if (animInfo.RenderMatrix)
            {
                return false;
            }
            return true;
        }
    }
}
