﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // モデルのマージボーン圧縮
    public class IfModelBoneMergeCompressor : IfModelCompressor
    {
        // コンストラクタ
        public IfModelBoneMergeCompressor() :
            base("IfModelBoneMergeCompressor_Log") { }

        // プロセス
        public override string Process
        {
            get { return "compress_bone_merge"; }
        }

        // カウントの取得
        public override int GetCount()
        {
            return this.Target.skeleton.bone_array.bone.Length;
        }

        // 圧縮
        protected override void Compress()
        {
            CompressBoneMerge(this.Target);

            // シェイプをソートする
            ModelCompressUtility.SortShape(this.Target);

            // 不要なストリームの削除とソートを行う。
            StreamUtility.SortStream(this.Target, this.Streams);
        }

        internal static void CompressBoneMerge(modelType target)
        {
            // TODO: Unite モデル対応
            // これより下は Unite されたモデルにまだ対応していない。
            if (target.shape_array != null)
            {
                foreach (shapeType shape in target.shape_array.shape)
                {
                    if (shape.mesh_array.mesh.Length > 1)
                    {
                        IfStrings.Throw("IfModelOptimizer_Error_UnitedModelUnsupported");
                    }
                }
            }

            boneType[] bones = target.skeleton.bone_array.bone;
            int boneCount = bones.Length;
            BoneInfo[] boneInfos = new BoneInfo[boneCount];
            BoneInfo[] parent;

            // ボーンのソートチェック
            int isSorted = IfModelSortUtility.IsBoneSorted(target);
            if (isSorted != -1)
            {
                IfStrings.Throw("IfModelBoneCompressor_Error_NotSorted",
                    bones[isSorted].name, bones[isSorted + 1].name);
            }

            // ボーン情報の初期化
            Dictionary<string, BoneInfo> boneDict = new Dictionary<string, BoneInfo>();
            for (int i = 0; i < boneCount; i++)
            {
                boneInfos[i] = new BoneInfo(bones[i], target.skeleton.skeleton_info);
                boneDict.Add(boneInfos[i].Bone.name, boneInfos[i]);
            }
            BoneInfo.Setup(boneInfos);

            // 削除可能なボーンの洗い出し
            bool update = true;
            while (update)
            {
                update = false;
                for (int i = 1; i < boneCount; i++)
                {
                    BoneInfo boneInfo = boneInfos[i];
                    if (boneInfo.IsReservedToRemove) { continue; }
                    if (CheckRemovable(boneInfo))
                    {
                        boneInfo.IsReservedToRemove = true;
                        update = true;
                    }
                }
            }

            // 以下の条件に当てはまる場合はルートボーンを削除する。
            // - 削除可能
            // - ルートボーンの削除されない子供の数が 1 つだけ
            // ※ Merge ボーン圧縮ではボーン名のチェックを行わない。
            //    これは NW4R 版の実装に合わせるためである。
            BoneInfo rootBoneInfo = boneInfos[0];
            ModelCompressUtility.GetThinOutBoneRef(boneInfos, out parent);
            BoneInfo[] rootChild = Array.FindAll(
                parent, delegate (BoneInfo bone) { return (bone == rootBoneInfo); });
            if (CheckRemovable(rootBoneInfo) && (rootChild.Length == 1))
            {
                rootBoneInfo.IsReservedToRemove = true;
            }

            // ボーン圧縮後のルートボーンを見つける。
            if (rootBoneInfo.IsReservedToRemove)
            {
                int idx = Array.FindIndex(
                    parent, delegate (BoneInfo bone) { return (bone == rootBoneInfo); });
                // 全てのボーンが圧縮されてしまう場合には idx が -1 になる。
                // その時はルートボーンを削除せずに残す。
                if (idx >= 0)
                {
                    rootBoneInfo = boneInfos[idx];
                }
                else
                {
                    rootBoneInfo.IsReservedToRemove = false;
                }
            }

            // 新しいボーンに出力順のインデックスを設定する
            ModelCompressUtility.SetNewIndex(rootBoneInfo);

            // 親が削除されたら、自身のSRT値を新しい親の座標系で定義しなおす
            for (int inode = 1; inode < boneInfos.Length; ++inode)
            {
                BoneInfo boneInfo = boneInfos[inode];
                if (!boneInfo.IsReservedToRemove && boneInfo.Parent.IsReservedToRemove)
                {
                    BoneInfo newParent = boneInfo.GetIncompressibleAncestorBone();
                    Matrix44 nodeTransform =
                        ModelCompressUtility.GetBoneTransform(target, boneInfo, newParent);
                    boneType bone = boneInfo.Bone;

                    Transform trans = new Transform();
                    nodeTransform.GetTransform(trans);
                    bone.translate = trans.Translate.ToArray();
                    bone.scale = trans.Scale.ToArray();
                    if (boneInfo.SkeletonInfo.rotate_mode == skeleton_info_rotate_modeType.euler_xyz)
                    {
                        // boneType.rotate は４要素必用なので、オイラー各で回転を保持する場合は
                        // 必ず配列の [3] に 1.0 を設定しておく。
                        Vector3 euler = EulerXyz.ToDegree(trans.Rotate);
                        bone.rotate = new float[] { euler.X, euler.Y, euler.Z, 1.0f };
                    }
                    else
                    {
                        Quaternion quat = EulerXyz.ToQuaternion(trans.Rotate);
                        bone.rotate = quat.ToArray();
                    }
                    // 親ノードとの合成後は正しく扱えないので scale_compensate は無効にしておく。
                    bone.scale_compensate = false;
                }
            }

            // ボーンの削除
            ModelCompressUtility.RemoveBones(target, boneInfos);

            // <shape> の更新
            // rigid_body が true、もしくは matrix_index が 0 以上のボーンは削除されないので、
            // これらの属性を更新する必要はない。
            if (target.shape_array != null &&
                target.shape_array.shape != null)
            {
                foreach (shapeType shape in target.shape_array.shape)
                {
                    string boneName = shape.shape_info.bone_name;
                    if (!boneDict.ContainsKey(boneName))
                    {
                        // 参照しているボーンが見つからない場合はエラー
                        IfStrings.Throw("ModelCompressUtility_Error_BoneNotFound",
                            shape.name, boneName);
                    }

                    BoneInfo boneInfo = boneDict[boneName];
                    if (boneInfo.IsReservedToRemove)
                    {
                        boneType parentBone = FindFirstCompressDisabledParentBone(boneDict, shape.shape_info.bone_name);
                        if (parentBone == null)
                        {
                            shape.shape_info.bone_name = rootBoneInfo.Bone.name;
                        }
                        else
                        {
                            shape.shape_info.bone_name = parentBone.name;
                        }
                    }
                }
            }
        }

        // 削除可能かチェック
        private static bool CheckRemovable(BoneInfo boneInfo)
        {
            boneType bone = boneInfo.Bone;

            // 圧縮禁止
            if (!bone.compress_enable) { return false; }

            // 描画に利用されている
            if (boneInfo.RenderMatrix) { return false; }

            /// TODO: RigidSkin、RigidBody 対応時に要調整

            return true;
        }

        private static boneType FindFirstCompressDisabledParentBone(Dictionary<string, BoneInfo> boneDict, string parentBoneName)
        {
            if (!boneDict.ContainsKey(parentBoneName))
            {
                return null;
            }

            BoneInfo boneInfo = boneDict[parentBoneName];
            if (boneInfo.Bone.compress_enable == false)
            {
                return boneInfo.Bone;
            }

            return FindFirstCompressDisabledParentBone(boneDict, boneInfo.Bone.parent_name);
        }
    }
}
