﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.IO;

namespace nw.g3d.ifcvtr
{
    // obj パーサ
    internal class ObjParser
    {
        // パース
        internal void Parse(ObjContext context)
        {
            string[] lines = File.ReadAllLines(context.FilePath);
            for (int i = 0; i < lines.Length; i++)
            {
                string line = lines[i];
                try
                {
                    ParseLine(context, line);
                }
                catch (Exception exp)
                {
                    throw new Exception(string.Format("{0}:{1} {2}",
                        context.FilePath, i, exp.Message));
                }
            }
        }

        // ラインのパース
        private void ParseLine(ObjContext context, string line)
        {
            // 空文字列をスキップ
            if (line.Length == 0) { return; }

            // 空白文字のみの行をスキップ
            string trimLine = line.Trim();
            if (trimLine.Length == 0) { return; }

            // コメントをスキップ
            if (trimLine.StartsWith("#")) { return; }

            string[] tokens = trimLine.Split(
                (char[])null, StringSplitOptions.RemoveEmptyEntries);
            if (tokens[0] == "v")
            {
                // 位置のパース
                float[] position = new float[3];
                for (int i = 0; i < 3; i++)
                {
                    position[i] = float.Parse(tokens[i + 1]);
                }
                //Debug.WriteLine(string.Format("position[{0} {1} {2}]",
                //    position[0], position[1], position[2]));
                context.Positions.Add(position);
            }
            else if (tokens[0] == "vt")
            {
                // UV のパース
                float[] uv = new float[2];
                for (int i = 0; i < 2; i++)
                {
                    uv[i] = float.Parse(tokens[i + 1]);
                }
                //Debug.WriteLine(string.Format("uv[{0} {1}]",
                //    uv[0], uv[1]));
                context.Uvs.Add(uv);
            }
            else if (tokens[0] == "vn")
            {
                // 法線のパース
                float[] normal = new float[3];
                for (int i = 0; i < 3; i++)
                {
                    normal[i] = float.Parse(tokens[i + 1]);
                }
                //Debug.WriteLine(string.Format("normal[{0} {1} {2}]",
                //    normal[0], normal[1], normal[2]));
                context.Normals.Add(normal);
            }
            else if (tokens[0] == "mtllib")
            {
                // マテリアルライブラリ名のパース
                context.Mtllibs.Add(tokens[1]);
            }
            else if ((tokens[0] == "g") || (tokens[0] == "o"))
            {
                // グループ名、オブジェクト名のパース
                context.CurrentGroup = GetName(tokens);
            }
            else if (tokens[0] == "usemtl")
            {
                // マテリアル名のパース
                context.CurrentMaterial = tokens[1];
            }
            else if (tokens[0] == "f")
            {
                // フェイスのパース
                ParseFace(context, tokens);
            }
            else
            {
                // パースできない行は読み飛ばし
                Debug.WriteLine("Skip : " + line);
            }
        }

        //=====================================================================
        // 空白で区切られた名前を _ で連結する
        private string GetName(string[] tokens)
        {
            if (tokens.Length < 2) { return string.Empty; }
            string result = tokens[1];
            for (int i = 2; i < tokens.Length; i++)
            {
                result += "_" + tokens[i];
            }
            return result;
        }

        //=====================================================================
        // フェイスのパース
        private void ParseFace(ObjContext context, string[] tokens)
        {
            int[] triangle = new int[9];

            // トライアングルのみ対応、多角形のトライアングル化を対応するならここで
            Nintendo.Foundation.Contracts.Assertion.Operation.True(tokens.Length == 4);

            for (int i = 0; i < 3; i++)
            {
                string vertex = tokens[i + 1];
                string[] attrIndices = vertex.Split(
                    new char[] { '/' }, StringSplitOptions.None);

                // 位置インデックスのパース
                triangle[i * 3 + 0] = int.Parse(attrIndices[0]) - 1;
                if (attrIndices.Length == 1)
                {
                    // 位置のみの場合
                    triangle[i * 3 + 1] = -1;
                    triangle[i * 3 + 2] = -1;
                    continue;
                }

                // UV インデックスのパース
                if (attrIndices[1] == string.Empty)
                {
                    triangle[i * 3 + 1] = -1;
                }
                else
                {
                    triangle[i * 3 + 1] = int.Parse(attrIndices[1]) - 1;
                }
                if (attrIndices.Length == 2)
                {
                    // 位置と UV のみの場合
                    triangle[i * 3 + 2] = -1;
                    continue;
                }

                // 法線インデックスのパース
                if (attrIndices[2] == string.Empty)
                {
                    triangle[i * 3 + 2] = -1;
                }
                else
                {
                    triangle[i * 3 + 2] = int.Parse(attrIndices[2]) - 1;
                }
            }
            //Debug.WriteLine(string.Format("{9} {10}[{0}/{1}/{2}][{3}/{4}/{5}][{6}/{7}/{8}]",
            //    triangle[0], triangle[1], triangle[2],
            //    triangle[3], triangle[4], triangle[5],
            //    triangle[6], triangle[7], triangle[8],
            //    context.CurrentShape.Group, context.CurrentShape.Material));
            context.CurrentShape.Faces.Add(triangle);
        }
    }
}
