﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Diagnostics;

namespace nw.g3d.ifcvtr
{
    // mtl パーサ
    internal class MtlParser
    {
        // パース
        internal void Parse(ObjContext context, string mtllib)
        {
            string filePath = Path.Combine(Path.GetDirectoryName(context.FilePath), mtllib);
            string[] lines = File.ReadAllLines(filePath);

            for (int i = 0; i < lines.Length; i++)
            {
                string line = lines[i];
                try
                {
                    ParseLine(context, mtllib, filePath, line);
                }
                catch (Exception exp)
                {
                    throw new Exception(string.Format("{0}:{1} {2}",
                        filePath, i, exp.Message));
                }
            }
        }

        // ラインのパース
        private void ParseLine(ObjContext context, string mtllib, string filePath, string line)
        {
            // 空文字列をスキップ
            if (line.Length == 0) { return; }

            // 空白文字のみの行をスキップ
            string trimLine = line.Trim();
            if (trimLine.Length == 0) { return; }

            // コメントをスキップ
            if (trimLine.StartsWith("#")) { return; }

            string[] tokens = trimLine.Split(
                (char[])null, StringSplitOptions.RemoveEmptyEntries);

            if (tokens[0] == "newmtl")
            {
                this.CurrentMaterial = new ObjMaterial(mtllib, filePath, tokens[1]);
                context.Materials.Add(this.CurrentMaterial);
            }
            else if (this.CurrentMaterial == null)
            {
                // マテリアルが無い場合は読み進めない
                Debug.WriteLine("Material not found : " + line);
                return;
            }
            else if (tokens[0] == "Kd")
            {
                for (int i = 0; i < 3; i++)
                {
                    this.CurrentMaterial.Diffuse[i] = float.Parse(tokens[i + 1]);
                }
            }
            else if ((tokens[0] == "d") || (tokens[0] == "Tr"))
            {
                this.CurrentMaterial.Opacity = float.Parse(tokens[1]);
            }
            else if (tokens[0] == "Ka")
            {
                for (int i = 0; i < 3; i++)
                {
                    this.CurrentMaterial.Ambient[i] = float.Parse(tokens[i + 1]);
                }
            }
            else if (tokens[0] == "Ks")
            {
                for (int i = 0; i < 3; i++)
                {
                    this.CurrentMaterial.Specular[i] = float.Parse(tokens[i + 1]);
                }
            }
            else if (tokens[0] == "Ns")
            {
                // スペキュラ係数は無視する
            }
            else if ((tokens[0] == "map_Kd") || (tokens[0] == "mapKd"))
            {
                this.CurrentMaterial.AlbedoTexture = tokens[1];
            }
            else if ((tokens[0] == "map_d") || (tokens[0] == "mapd") ||
                (tokens[0] == "map_Tr") || (tokens[0] == "mapTr"))
            {
                this.CurrentMaterial.OpacityTexture = tokens[1];
            }
            else if ((tokens[0] == "map_Ka") || (tokens[0] == "mapKa"))
            {
                // アンビエントをエミッションに差し込んでおく
                this.CurrentMaterial.EmissionTexture = tokens[1];
            }
            else if ((tokens[0] == "map_bump") || (tokens[0] == "mapbump") ||
                (tokens[0] == "bump"))
            {
                // バンプマップは無視する
            }
            else if ((tokens[0] == "map_Ks") || (tokens[0] == "mapKs"))
            {
                this.CurrentMaterial.SpecularTexture = tokens[1];
            }
            else
            {
                // パースできない行は読み飛ばし
                Debug.WriteLine("Skip : " + line);
            }
        }

        private ObjMaterial CurrentMaterial;
    }
}
