﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.toollib;
using System.IO;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;
using nw.g3d.iflib;
using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Extensions;

namespace nw.g3d.ifcvtr
{
    // fbx 等の一般 3D 形式用のコーバーター
    internal class Common3dConverter : Converter
    {
        private string xsdBasePath = null;
        private string g3dToolRoot = null;
        private bool isInvertingUvVerticalEnabled = false;
        private CommandLineOptions options = null;

        public static bool ExaminesWhetherIsSupportedExtension(string extension)
        {
            return (new Assimp.AssimpContext()).IsImportFormatSupported(extension);
        }

        // コンストラクタ
        internal Common3dConverter(string sourcePath, CommandLineOptions param, string inputXsdBasePath, string g3dToolRoot)
            : base(sourcePath)
        {
            Nintendo.Foundation.Contracts.Ensure.Argument.True(!string.IsNullOrEmpty(inputXsdBasePath));
            this.xsdBasePath = inputXsdBasePath;
            this.g3dToolRoot = g3dToolRoot;

            if (!string.IsNullOrEmpty(param.Output))
            {
                if (G3dPath.IsModelPath(param.Output))
                {
                    this.OutputFilePath = System.IO.Path.GetFullPath(param.Output);
                }
                else
                {
                    this.OutputFilePath = Path.ChangeExtension(
                        param.Output, G3dPath.ModelBinaryExtension);
                }
            }
            else
            {
                this.OutputFilePath = Path.ChangeExtension(
                    this.SourceFilePath, G3dPath.ModelBinaryExtension);
            }

            this.isInvertingUvVerticalEnabled = !param.DisableInvertUv;
            this.options = param;
        }

        //=====================================================================
        // 変換
        internal override void Convert()
        {
            Assimp.AssimpContext importer = new Assimp.AssimpContext();
            Assimp.Scene scene = null;
            try
            {
                scene = importer.ImportFile(this.SourceFilePath);
            }
            catch (Exception exception)
            {
                throw new Exception(
                    string.Format(
                        $"{Resources.StringResource.Error_FailedToImport}\n{exception.Message}",
                        this.SourceFilePath));
            }

            Nintendo.Foundation.Contracts.Ensure.Operation.NotNull(scene);

            WriteNodeNames(scene.RootNode);

            // fmdb の出力
            {

                var file = new IntermediateFile(IntermediateFileKind.Model);
                var model = file.GetRootEntity<Model>();

                CreateSkeleton(model, scene);
                Assertion.Operation.True(model.Skeleton.CountBones() > 0);

                CreateMaterials(model, scene.Materials);
                CreateShapes(model, scene, this.isInvertingUvVerticalEnabled);

                string folder = System.IO.Path.GetDirectoryName(this.OutputFilePath);
                if (!System.IO.Directory.Exists(folder))
                {
                    System.IO.Directory.CreateDirectory(folder);
                }

                IfWriteUtility.WriteIntermediateFile(file, this.OutputFilePath, G3dToolUtility.GetXsdBasePath());
            }

            // ftxb の出力
            if (!this.options.DisableTexture)
            {
                IEnumerable<string> sourceTexPaths = FindUsedTexturePaths(scene);
                if (sourceTexPaths.Count() > 0)
                {
                    string outputFtxFolderPath = System.IO.Path.Combine(Path.GetDirectoryName(this.OutputFilePath), "textures");
                    if (!System.IO.Directory.Exists(outputFtxFolderPath))
                    {
                        System.IO.Directory.CreateDirectory(outputFtxFolderPath);
                    }

                    foreach (string relativeTexturePath in sourceTexPaths)
                    {
                        string inputTexturePath = ResolveTexturePath(relativeTexturePath, this.SourceFilePath);
                        if (string.IsNullOrEmpty(inputTexturePath))
                        {
                            WriteWarningMessage(Strings.Get("Error_TexturePathNotFound", relativeTexturePath));
                            continue;
                        }

                        string outputFtxFilePath = System.IO.Path.Combine(
                            outputFtxFolderPath, $"{System.IO.Path.GetFileNameWithoutExtension(relativeTexturePath)}.ftxb");

                        this.SubConverters.Add(new ImageConverter(inputTexturePath, outputFtxFilePath, this.options, this.g3dToolRoot));
                    }
                }
            }

            // magnify を適用
            if (this.options.Magnify != 1.0f)
            {
                CommandLineOptions param = new CommandLineOptions()
                {
                    Magnify = this.options.Magnify,
                    ProjectRoot = this.options.ProjectRoot,
                };
                this.SubConverters.Add(new IfModelConverter(this.OutputFilePath, param, this.xsdBasePath));
            }
        }

        private static void WriteWarningMessage(string message)
        {
            // TODO: マルチスレッドで正しくログが出るように、ログをまとめてから最後に出力するように改善する
            Console.WriteLine($"{Strings.Get("Warning")}: {message}");
        }

        private IEnumerable<string> FindUsedTexturePaths(Assimp.Scene scene)
        {
            List<string> texPaths = new List<string>();
            if (!scene.HasMaterials)
            {
                return texPaths;
            }

            foreach (var mat in scene.Materials)
            {
                if (mat.HasTextureDiffuse) { texPaths.Add(mat.TextureDiffuse.FilePath); }
                if (mat.HasTextureAmbient) { texPaths.Add(mat.TextureAmbient.FilePath); }
                if (mat.HasTextureDisplacement) { texPaths.Add(mat.TextureDisplacement.FilePath); }
                if (mat.HasTextureEmissive) { texPaths.Add(mat.TextureEmissive.FilePath); }
                if (mat.HasTextureHeight) { texPaths.Add(mat.TextureHeight.FilePath); }
                if (mat.HasTextureLightMap) { texPaths.Add(mat.TextureLightMap.FilePath); }
                if (mat.HasTextureNormal) { texPaths.Add(mat.TextureNormal.FilePath); }
                if (mat.HasTextureOpacity) { texPaths.Add(mat.TextureOpacity.FilePath); }
                if (mat.HasTextureReflection) { texPaths.Add(mat.TextureReflection.FilePath); }
                if (mat.HasTextureSpecular) { texPaths.Add(mat.TextureSpecular.FilePath); }
            }

            return texPaths.Distinct();
        }

        private string ResolveTexturePath(string relativePath, string modelFilePath)
        {
            string fbxFolderPath = System.IO.Path.GetDirectoryName(modelFilePath);
            string absTexPath = System.IO.Path.GetFullPath(
                System.IO.Path.Combine(fbxFolderPath, relativePath));
            if (System.IO.File.Exists(absTexPath))
            {
                return absTexPath;
            }

            // 相対パスに見つからなければ、モデルと同じフォルダを探索
            absTexPath = System.IO.Path.GetFullPath(
                System.IO.Path.Combine(fbxFolderPath, System.IO.Path.GetFileName(relativePath)));
            if (System.IO.File.Exists(absTexPath))
            {
                return absTexPath;
            }

            return null;
        }

        private BoneInfo[] CreateBoneInfos(modelType model)
        {
            var bones = model.skeleton.bone_array.bone;
            BoneInfo[] boneInfos = new BoneInfo[bones.Length];
            for (var boneIndex = 0; boneIndex < bones.Length; ++boneIndex)
            {
                boneInfos[boneIndex] = new BoneInfo(bones[boneIndex], model.skeleton.skeleton_info);
            }
            BoneInfo.Setup(boneInfos);
            return boneInfos;
        }

        [Conditional("DEBUG")]
        private void WriteNodeNames(Assimp.Node node, string indent = "")
        {
            WriteDebugLogLine($"{indent}{node.Name}");
            foreach (var child in node.Children)
            {
                WriteNodeNames(child, indent + "  ");
            }
        }

        [Conditional("DEBUG")]
        private static void WriteDebugLogLine(string message)
        {
            Console.WriteLine(message);
        }

        private void CreateSkeleton(Model targetModel, Assimp.Scene scene)
        {
            targetModel.Skeleton.SkeletonInfo.RotateMode = skeleton_info_rotate_modeType.quaternion;
            AddBones(targetModel, scene.RootNode, scene);
            if (targetModel.Skeleton.CountBones() == 0)
            {
                throw new Exception("Input model must has at least one bone.");
            }
        }

        private int FindUnassignedMatrixIndex(IEnumerable<Bone> bones)
        {
            int maxMatrixIndex = bones.Select(x => x.SmoothSkinningMatrixIndex).Max();
            return maxMatrixIndex + 1;
        }

        private StreamInt AddIndexStream(Model model, Assimp.Mesh sourceMesh)
        {
            var indexStream = new StreamInt();
            {
                indexStream.Column = 3;
                foreach (var face in sourceMesh.Faces)
                {
                    indexStream.Values.Add(face.Indices);
                }

                model.Streams.Add(indexStream);
            }

            return indexStream;
        }

        private void AddUvStreams(Shape shape, Assimp.Mesh sourceMesh)
        {
            Model model = shape.Parent;
            foreach (var texCoords in sourceMesh.TextureCoordinateChannels)
            {
                if (texCoords.Count == 0)
                {
                    break;
                }

                var uvStream = new StreamFloat();
                uvStream.Column = 2;

                foreach (var texCoord in texCoords)
                {
                    uvStream.Values.Add(texCoord.X);
                    if (isInvertingUvVerticalEnabled)
                    {
                        uvStream.Values.Add(1.0f - texCoord.Y);
                    }
                    else
                    {
                        uvStream.Values.Add(texCoord.Y);
                    }
                }

                model.Streams.Add(uvStream);
                shape.AddVertexAttributeUv(uvStream);
            }
        }

        private void AddNormalStream(Shape shape, Assimp.Mesh sourceMesh)
        {
            Model model = shape.Parent;
            var normalStream = new StreamFloat();
            if (sourceMesh.HasNormals)
            {
                normalStream.Column = 3;
                foreach (var normal in sourceMesh.Normals)
                {
                    normalStream.Values.Add(normal.X);
                    normalStream.Values.Add(normal.Y);
                    normalStream.Values.Add(normal.Z);
                }

                model.Streams.Add(normalStream);
                shape.AddVertexAttributeNormal(normalStream);
            }
        }

        private void AddTangentStream(Shape shape, Assimp.Mesh sourceMesh)
        {
            Model model = shape.Parent;
            var tangentStream = new StreamFloat();
            if (sourceMesh.HasTangentBasis)
            {
                tangentStream.Column = 4;
                foreach (var tangent in sourceMesh.Tangents)
                {
                    tangentStream.Values.Add(tangent.X);
                    tangentStream.Values.Add(tangent.Y);
                    tangentStream.Values.Add(tangent.Z);
                    tangentStream.Values.Add(1.0f);
                }

                model.Streams.Add(tangentStream);
                shape.AddVertexAttributeTangent(tangentStream);
            }
        }

        private void AddBinormalStream(Shape shape, Assimp.Mesh sourceMesh)
        {
            Model model = shape.Parent;
            var binormalStream = new StreamFloat();
            if (sourceMesh.HasTangentBasis)
            {
                binormalStream.Column = 4;
                foreach (var bitangent in sourceMesh.BiTangents)
                {
                    binormalStream.Values.Add(bitangent.X);
                    binormalStream.Values.Add(bitangent.Y);
                    binormalStream.Values.Add(bitangent.Z);
                    binormalStream.Values.Add(1.0f);
                }

                if (binormalStream.Values.Count > 0)
                {
                    model.Streams.Add(binormalStream);
                    shape.AddVertexAttributeBinormal(binormalStream);
                }
            }
        }

        private void AddPositionStream(
            Shape shape, Assimp.Mesh sourceMesh, Assimp.Scene scene)
        {
            Model model = shape.Parent;
            Float4x4 posTransform = new Float4x4();
            posTransform.SetToIdentity();
            if (sourceMesh.HasBones)
            {
                // Maya からエクスポートした fbx では親ノードのローカル空間で座標が格納されているので、
                // スキニングされている場合はワールド空間に座標変換
                Assimp.Node parentNode = FindAssociatedNode(sourceMesh, scene.RootNode, scene);
                Bone parentBone = model.Skeleton.EnumerateBones().FirstOrDefault(x => x.Name == parentNode.Name);
                posTransform = parentBone.CalculateTransformMatrix();
            }

            {
                var positionStream = new StreamFloat();
                positionStream.Column = 3;
                foreach (var position in sourceMesh.Vertices)
                {
                    Float3 transformedPos = posTransform * new Float3(position.X, position.Y, position.Z);
                    positionStream.Values.Add(transformedPos.X);
                    positionStream.Values.Add(transformedPos.Y);
                    positionStream.Values.Add(transformedPos.Z);
                }
                model.Streams.Add(positionStream);
                shape.AddVertexAttributePosition(positionStream);
            }
        }

        private void AddSkinningRelatedStreams(Shape shape, Assimp.Mesh sourceMesh, Assimp.Scene scene)
        {
            Model model = shape.Parent;
            int vertexCount = sourceMesh.Vertices.Count;

            // スキニング関連のストリーム追加
            // リジッドスキニングでもとりあえずスムーススキニングとして出力する
            // TODO: 最小限のストリームになるよう対応する
            int vertexSkinningCount = 0;

            if (sourceMesh.HasBones)
            {
                List<int> blendIndices = new List<int>();
                const int MaxSkinningCount = 8;
                const int NotAssigned = -1;
                blendIndices.Resize(vertexCount * MaxSkinningCount, NotAssigned);
                List<float> blendWeights = new List<float>();
                blendWeights.Resize(vertexCount * MaxSkinningCount, 0.0f);
                foreach (Assimp.Bone skinnedBone in sourceMesh.Bones)
                {
                    Assimp.Node node = FindNode(scene.RootNode, skinnedBone.Name);
                    Bone bone = model.Skeleton.EnumerateBones().FirstOrDefault(x => x.Name == skinnedBone.Name);
                    int matrixIndex;
                    if (bone.SmoothSkinningMatrixIndex != -1)
                    {
                        matrixIndex = bone.SmoothSkinningMatrixIndex;
                    }
                    else
                    {
                        // マトリックスパレット関連の設定
                        matrixIndex = FindUnassignedMatrixIndex(model.Skeleton.EnumerateBones());
                        bone.SmoothSkinningMatrixIndex = matrixIndex;
                    }

                    foreach (Assimp.VertexWeight vertexWeight in skinnedBone.VertexWeights)
                    {
                        bool isAssigned = false;
                        for (int elemIndex = 0; elemIndex < MaxSkinningCount; ++elemIndex)
                        {
                            if (blendIndices[vertexWeight.VertexID * MaxSkinningCount + elemIndex] == NotAssigned)
                            {
                                blendIndices[vertexWeight.VertexID * MaxSkinningCount + elemIndex] = matrixIndex;
                                blendWeights[vertexWeight.VertexID * MaxSkinningCount + elemIndex] = vertexWeight.Weight;
                                isAssigned = true;
                                break;
                            }
                        }

                        if (!isAssigned)
                        {
                            Strings.Throw("Error_ExceedMaxVertexSkinningCount");
                        }
                    }
                }

                // スキニング数の計算
                // スムーススキニングのスキニング数は複数バリエーションをサポートせず、4 か 8 とする
                // TODO: 最小限のスキニング数になるよう対応する
                {
                    bool isSkinningCountGreaterThan4 = false;
                    for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
                    {
                        if (blendIndices[vertexIndex * MaxSkinningCount + 4] != NotAssigned)
                        {
                            isSkinningCountGreaterThan4 = true;
                        }
                    }

                    if (isSkinningCountGreaterThan4)
                    {
                        vertexSkinningCount = 8;
                    }
                    else
                    {
                        vertexSkinningCount = 4;
                    }
                }

                // ストリームに変換
                {
                    var blendIndexStream0 = new StreamInt();
                    blendIndexStream0.Column = 4;

                    var blendWeightStream0 = new StreamFloat();
                    blendWeightStream0.Column = 4;

                    for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
                    {
                        int offset = vertexIndex * MaxSkinningCount;
                        for (int vertexElemIndex = 0; vertexElemIndex < 4; ++vertexElemIndex)
                        {
                            int blendIndex = blendIndices[offset + vertexElemIndex];
                            if (blendIndex == NotAssigned)
                            {
                                blendIndexStream0.Values.Add(0);
                                blendWeightStream0.Values.Add(0.0f);
                            }
                            else
                            {
                                blendIndexStream0.Values.Add(blendIndex);
                                blendWeightStream0.Values.Add(blendWeights[offset + vertexElemIndex]);
                            }
                        }
                    }

                    model.Streams.Add(blendIndexStream0);
                    model.Streams.Add(blendWeightStream0);

                    shape.AddVertexAttributeBlendIndex(blendIndexStream0);
                    shape.AddVertexAttributeBlendWeight(blendWeightStream0);
                }

                if (vertexSkinningCount > 4)
                {
                    var blendIndexStream1 = new StreamInt();
                    blendIndexStream1.Column = 4;

                    var blendWeightStream1 = new StreamFloat();
                    blendWeightStream1.Column = 4;

                    for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
                    {
                        int offset = vertexIndex * MaxSkinningCount + 4;
                        for (int vertexElemIndex = 0; vertexElemIndex < 4; ++vertexElemIndex)
                        {
                            int blendIndex = blendIndices[offset + vertexElemIndex];
                            if (blendIndex == NotAssigned)
                            {
                                blendIndexStream1.Values.Add(0);
                                blendWeightStream1.Values.Add(0.0f);
                            }
                            else
                            {
                                blendIndexStream1.Values.Add(blendIndex);
                                blendWeightStream1.Values.Add(blendWeights[offset + vertexElemIndex]);
                            }
                        }
                    }

                    model.Streams.Add(blendIndexStream1);
                    model.Streams.Add(blendWeightStream1);

                    shape.AddVertexAttributeBlendIndex(blendIndexStream1);
                    shape.AddVertexAttributeBlendWeight(blendWeightStream1);
                }
            }
        }

        /// <summary>
        /// シェイプを作成します。シェイプを作成する前にボーンとマテリアルを作成しておく必要があります。
        /// </summary>
        /// <param name="model">作成先のモデルです。</param>
        /// <param name="streams">作成先のストリームです。</param>
        /// <param name="scene">入力シーンデータです。</param>
        private void CreateShapes(Model model, Assimp.Scene scene, bool isInvertingUvVerticalEnabled)
        {
            foreach (Assimp.Mesh sourceMesh in scene.Meshes)
            {
                Assimp.Material mat = scene.Materials[sourceMesh.MaterialIndex];
                Assimp.PrimitiveType primType = sourceMesh.PrimitiveType;
                switch (primType)
                {
                case Assimp.PrimitiveType.Triangle:
                    break;
                case Assimp.PrimitiveType.Line:
                case Assimp.PrimitiveType.Point:
                case Assimp.PrimitiveType.Polygon:
                default:
                    Strings.Throw("Error_NotTriangulated");
                    break;
                }

                // シェイプの追加
                Assimp.Node associatedNode = FindAssociatedNode(sourceMesh, scene.RootNode, scene);
                StreamInt indexStream = AddIndexStream(model, sourceMesh);
                Shape shape = model.CreateShape(
                    model.Materials.First(x => x.Name == mat.Name),
                    model.Skeleton.EnumerateBones().First(x => x.Name == associatedNode.Name),
                    indexStream);

                AddPositionStream(shape, sourceMesh, scene);
                AddUvStreams(shape, sourceMesh);
                AddNormalStream(shape, sourceMesh);
                AddTangentStream(shape, sourceMesh);
                AddBinormalStream(shape, sourceMesh);
                AddSkinningRelatedStreams(shape, sourceMesh, scene);
            }
        }

        private wrap_uvwType ConvertWrapMode(Assimp.TextureWrapMode wrap)
        {
            switch (wrap)
            {
            case Assimp.TextureWrapMode.Clamp: return wrap_uvwType.clamp;
            case Assimp.TextureWrapMode.Mirror: return wrap_uvwType.mirror;
            case Assimp.TextureWrapMode.Wrap: return wrap_uvwType.repeat;

            // 以下は対応するものがないので clamp にしておく
            case Assimp.TextureWrapMode.Decal:
            default:
                return wrap_uvwType.clamp;
            }
        }

        private static class TextureNameHintConveter
        {
            private class TexNameHint
            {
                public TexNameHint(Assimp.TextureType type, string name, string hint)
                {
                    this.Type = type;
                    this.Name = name;
                    this.Hint = hint;
                }
                public Assimp.TextureType Type { get; }
                public string Name { get; }
                public string Hint { get; }
            }

            private static readonly string DefaultName = "sampler";
            private static readonly TexNameHint[] NameHintList = new TexNameHint[]
            {
                new TexNameHint(Assimp.TextureType.Diffuse, "_a0", "albedo0"),
                new TexNameHint(Assimp.TextureType.Opacity, "_o0", "opacity0"),
                new TexNameHint(Assimp.TextureType.Emissive, "_e0", "emission0"),
                new TexNameHint(Assimp.TextureType.Specular, "_s0", "specular0"),
                new TexNameHint(Assimp.TextureType.Reflection, "_r0", "reflection0"),
                new TexNameHint(Assimp.TextureType.Normals, "_n0", "normal0"),

                // NW 既定外は名前とヒントを Assimp の種別名で埋めておく
                new TexNameHint(Assimp.TextureType.Ambient, "ambient", "ambient"),
                new TexNameHint(Assimp.TextureType.Displacement, "displacement", "displacement"),
                new TexNameHint(Assimp.TextureType.Height, "height", "height"),
                new TexNameHint(Assimp.TextureType.Lightmap, "lightmap", "lightmap"),
                new TexNameHint(Assimp.TextureType.Shininess, "shininess", "shininess"),
                new TexNameHint(Assimp.TextureType.None, DefaultName, string.Empty),
                new TexNameHint(Assimp.TextureType.Unknown, DefaultName, string.Empty),
            };

            public static string ConvertToName(Assimp.TextureType type)
            {
                foreach (var nameHint in NameHintList)
                {
                    if (type == nameHint.Type)
                    {
                        return nameHint.Name;
                    }
                }

                return DefaultName;
            }

            public static string ConvertToHint(Assimp.TextureType type)
            {
                foreach (var nameHint in NameHintList)
                {
                    if (type == nameHint.Type)
                    {
                        return nameHint.Hint;
                    }
                }

                return string.Empty;
            }
        }

        private Sampler CreateSampler(Assimp.TextureSlot texSlot)
        {
            var sampler = new Sampler();
            sampler.Name = TextureNameHintConveter.ConvertToName(texSlot.TextureType);
            sampler.Hint = TextureNameHintConveter.ConvertToHint(texSlot.TextureType);
            sampler.TexName = System.IO.Path.GetFileNameWithoutExtension(texSlot.FilePath);
            sampler.Wrap.U = ConvertWrapMode(texSlot.WrapModeU);
            sampler.Wrap.V = ConvertWrapMode(texSlot.WrapModeV);
            return sampler;
        }

        private void CreateMaterials(
            Model targetModel, List<Assimp.Material> sourceMaterials)
        {
            foreach (var sourceMaterial in sourceMaterials)
            {
                var material = new Material()
                {
                    Name = sourceMaterial.Name,
                };

                // サンプラーの構築
                {
                    if (sourceMaterial.HasTextureDiffuse) { material.Samplers.Add(CreateSampler(sourceMaterial.TextureDiffuse)); }
                    if (sourceMaterial.HasTextureAmbient) { material.Samplers.Add(CreateSampler(sourceMaterial.TextureAmbient)); }
                    if (sourceMaterial.HasTextureDisplacement) { material.Samplers.Add(CreateSampler(sourceMaterial.TextureDisplacement)); }
                    if (sourceMaterial.HasTextureEmissive) { material.Samplers.Add(CreateSampler(sourceMaterial.TextureEmissive)); }
                    if (sourceMaterial.HasTextureHeight) { material.Samplers.Add(CreateSampler(sourceMaterial.TextureHeight)); }
                    if (sourceMaterial.HasTextureLightMap) { material.Samplers.Add(CreateSampler(sourceMaterial.TextureLightMap)); }
                    if (sourceMaterial.HasTextureNormal) { material.Samplers.Add(CreateSampler(sourceMaterial.TextureNormal)); }
                    if (sourceMaterial.HasTextureOpacity) { material.Samplers.Add(CreateSampler(sourceMaterial.TextureOpacity)); }
                    if (sourceMaterial.HasTextureReflection) { material.Samplers.Add(CreateSampler(sourceMaterial.TextureReflection)); }
                    if (sourceMaterial.HasTextureSpecular) { material.Samplers.Add(CreateSampler(sourceMaterial.TextureSpecular)); }

                }

                targetModel.Materials.Add(material);
            }
        }

        private Assimp.Node FindNode(Assimp.Node rootNode, string nodeName)
        {
            if (rootNode.Name == nodeName)
            {
                return rootNode;
            }

            if (rootNode.HasChildren)
            {
                foreach (var childNode in rootNode.Children)
                {
                    Assimp.Node foundNode = FindNode(childNode, nodeName);
                    if (foundNode != null)
                    {
                        return foundNode;
                    }
                }
            }

            return null;
        }

        private Assimp.Node FindAssociatedNode(Assimp.Mesh mesh, Assimp.Node rootNode, Assimp.Scene scene)
        {
            if (rootNode.HasMeshes)
            {
                foreach (int meshIndex in rootNode.MeshIndices)
                {
                    if (mesh == scene.Meshes[meshIndex])
                    {
                        return rootNode;
                    }
                }
            }

            if (rootNode.HasChildren)
            {
                foreach (var childNode in rootNode.Children)
                {
                    Assimp.Node foundNode = FindAssociatedNode(mesh, childNode, scene);
                    if (foundNode != null)
                    {
                        return foundNode;
                    }
                }
            }

            return null;
        }

        private bool ExaminesWhetherIsRigidBody(Assimp.Node targetNode, Assimp.Scene scene)
        {
            if (!targetNode.HasMeshes)
            {
                return false;
            }

            foreach (int meshIndex in targetNode.MeshIndices)
            {
                Assimp.Mesh mesh = scene.Meshes[meshIndex];
                if (mesh.BoneCount == 0)
                {
                    return true;
                }
            }

            return false;
        }

        private void AddBones(Model target, Assimp.Node node, Assimp.Scene scene, Bone parentBone = null)
        {
            Assimp.Vector3D scale;
            Assimp.Quaternion quaternion;
            Assimp.Vector3D translate;
            node.Transform.Decompose(out scale, out quaternion, out translate);
            var bone = new Bone()
            {
                Name = node.Name,
                RigidBody = ExaminesWhetherIsRigidBody(node, scene),
                ScaleCompensate = true,
            };
            bone.Translate.X = translate.X;
            bone.Translate.Y = translate.Y;
            bone.Translate.Z = translate.Z;
            bone.Scale.X = scale.X;
            bone.Scale.Y = scale.Y;
            bone.Scale.Z = scale.Z;
            bone.Rotate.X = quaternion.X;
            bone.Rotate.Y = quaternion.Y;
            bone.Rotate.Z = quaternion.Z;
            bone.Rotate.W = quaternion.W;
            if (parentBone == null)
            {
                target.Skeleton.RootBone = bone;
            }
            else
            {
                parentBone.ChildBones.Add(bone);
            }

            foreach (var child in node.Children)
            {
                AddBones(target, child, scene, bone);
            }
        }
    }
}
