﻿using _3dExcelToShaderAnnotation.Properties;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using Excel = Microsoft.Office.Interop.Excel;

namespace ShaderAssistExAddons.Modules.ParamCode
{
    public class ShaderDefinitionToExcelConverter
    {
        public delegate void LogAction(string message);

        public static void CreateExcelsFromShaderDefinition(string fsdbPath, LogAction logAction, LogAction errorLogAction)
        {
            nw4f_3difType file = nw.g3d.iflib.IfReadUtility.Read(fsdbPath, null);
            shader_definitionType shaderDef = file.Item as shader_definitionType;

            foreach (shading_modelType shadingModel in shaderDef.shading_model_array.Items)
            {
                using (var excelContext = new ExcelContext())
                {
                    excelContext.Create();

                    IEnumerable<pageType> pages = ExtractPages(excelContext, shadingModel);
                    IEnumerable<groupType> groups = ExtractGroups(excelContext, shadingModel);

                    var commonSheet = CreateCommonSheet(excelContext, pages, groups, shadingModel, shaderDef.shader_src_array.Items);

                    IEnumerable<object> shaderAssigns = ExtractShaderAssigns(excelContext, shadingModel);

                    foreach (pageType page in pages)
                    {
                        if (page == null)
                        {
                            excelContext.ActivateWorksheet(commonSheet.Name);
                        }
                        else
                        {
                            excelContext.ActivateWorksheet(page.name);
                        }

                        int cellRowIndex = excelContext.GetActiveSheetRow();

                        excelContext.SetCellTitleValue($"A{cellRowIndex}", "group");
                        excelContext.SetCellTitleValue($"B{cellRowIndex}", "type");
                        excelContext.SetCellTitleValue($"C{cellRowIndex}", "variable");
                        excelContext.SetCellTitleValue($"D{cellRowIndex}", "id");
                        excelContext.SetCellTitleValue($"E{cellRowIndex}", "label");
                        excelContext.SetCellTitleValue($"F{cellRowIndex}", "default");
                        excelContext.SetCellTitleValue($"G{cellRowIndex}", "other_annotation");
                        excelContext.SetCellTitleValue($"H{cellRowIndex}", "preprocess");
                        ++cellRowIndex;

                        foreach (groupType group in groups)
                        {
                            if (!ExaminesSamePage(page, group))
                            {
                                continue;
                            }

                            bool isParamFoundInGroup = false;
                            foreach (object shaderAssign in shaderAssigns)
                            {
                                option_varType optionVar = shaderAssign as option_varType;
                                if (optionVar != null)
                                {
                                    if (!ExaminesSameGroup(group, optionVar.ui_group))
                                    {
                                        continue;
                                    }

                                    if (!isParamFoundInGroup)
                                    {
                                        excelContext.SetCellValue($"A{cellRowIndex}", optionVar.ui_group != null ? optionVar.ui_group.group_name : string.Empty);
                                        isParamFoundInGroup = true;
                                    }

                                    WriteOptionVar(excelContext, optionVar, cellRowIndex);
                                    ++cellRowIndex;
                                }

                                var samplerVar = shaderAssign as sampler_varType;
                                if (samplerVar != null)
                                {
                                    if (!ExaminesSameGroup(group, samplerVar.ui_group))
                                    {
                                        continue;
                                    }

                                    if (!isParamFoundInGroup)
                                    {
                                        excelContext.SetCellValue($"A{cellRowIndex}", samplerVar.ui_group != null ? samplerVar.ui_group.group_name : string.Empty);
                                        isParamFoundInGroup = true;
                                    }

                                    WriteSamplerVar(excelContext, samplerVar, cellRowIndex);
                                    ++cellRowIndex;
                                }

                                var renderInfo = shaderAssign as render_info_slotType;
                                if (renderInfo != null)
                                {
                                    if (!ExaminesSameGroup(group, renderInfo.ui_group))
                                    {
                                        continue;
                                    }

                                    if (!isParamFoundInGroup)
                                    {
                                        excelContext.SetCellValue($"A{cellRowIndex}", renderInfo.ui_group != null ? renderInfo.ui_group.group_name : string.Empty);
                                        isParamFoundInGroup = true;
                                    }

                                    WriteRenderInfo(excelContext, renderInfo, cellRowIndex);
                                    ++cellRowIndex;
                                }

                                var uniformVar = shaderAssign as uniform_varType;
                                if (uniformVar != null)
                                {
                                    if (!ExaminesSameGroup(group, uniformVar.ui_group))
                                    {
                                        continue;
                                    }

                                    if (!isParamFoundInGroup)
                                    {
                                        excelContext.SetCellValue($"A{cellRowIndex}", uniformVar.ui_group != null ? uniformVar.ui_group.group_name : string.Empty);
                                        isParamFoundInGroup = true;
                                    }

                                    WriteUniformVar(excelContext, uniformVar, cellRowIndex);
                                    ++cellRowIndex;
                                }
                            }
                        }

                        ++cellRowIndex;
                        excelContext.SetActiveSheetRow(cellRowIndex);
                    }

                    string outputPathWithoutExtension = System.IO.Path.Combine(System.IO.Path.GetDirectoryName(fsdbPath), shadingModel.name);
                    string outputPath = $"{outputPathWithoutExtension}.xlsx";
                    try
                    {
                        excelContext.Save(outputPathWithoutExtension);
                        logAction(string.Format(Resources.Saved, outputPath));
                    }
                    catch (Exception exception)
                    {
                        errorLogAction(string.Format(Resources.ErrorSaveFileError, outputPath) + $"\n{exception.ToString()}");
                    }
                }
            }
        }

        private static string GetVariableDeclaration(shader_param_typeType type, string variableName, string converter)
        {
            switch (type)
            {
                case shader_param_typeType.@bool:
                case shader_param_typeType.@uint:
                case shader_param_typeType.@int: return $"int {variableName}";
                case shader_param_typeType.uint2:
                case shader_param_typeType.bool2:
                case shader_param_typeType.int2: return $"int {variableName}[2]";
                case shader_param_typeType.uint3:
                case shader_param_typeType.bool3:
                case shader_param_typeType.int3: return $"int {variableName}[3]";
                case shader_param_typeType.uint4:
                case shader_param_typeType.bool4:
                case shader_param_typeType.int4: return $"int {variableName}[4]";
                case shader_param_typeType.@float: return $"float {variableName}";
                case shader_param_typeType.float2: return $"vec2 {variableName}";
                case shader_param_typeType.float3: return $"vec3 {variableName}";
                case shader_param_typeType.float2x2:
                case shader_param_typeType.float4: return $"vec4 {variableName}";
                case shader_param_typeType.float2x3:
                case shader_param_typeType.float2x4:
                case shader_param_typeType.float3x2:
                case shader_param_typeType.float4x2:
                case shader_param_typeType.srt2d: return $"vec4 {variableName}[2]";
                case shader_param_typeType.texsrt:
                    {
                        if (converter == "texsrt_ex")
                        {
                            return $"vec4 {variableName}[3]";
                        }
                        else
                        {
                            return $"vec4 {variableName}[2]";
                        }
                    }
                case shader_param_typeType.float3x3: return $"mat3 {variableName}";
                case shader_param_typeType.srt3d:
                case shader_param_typeType.float4x3:
                case shader_param_typeType.float3x4: return $"vec4 {variableName}[3]";
                case shader_param_typeType.float4x4: return $"mat4 {variableName}";
                default:
                    Nintendo.ToolFoundation.Contracts.Ensure.Operation.Fail($"Invalid type {type}");
                    return null;
            }
        }


        private static string GetSamplerVariableDeclaration(sampler_varType samplerVar)
        {
            string symbolName = GetShaderParamSymbolName(samplerVar);
            switch (samplerVar.type)
            {
            case sampler_var_typeType.isampler_1d:
                return $"isampler1D {symbolName}";
            case sampler_var_typeType.isampler_1d_array:
                return $"isampler1DArray {symbolName}";
            case sampler_var_typeType.isampler_2d:
                return $"isampler2D {symbolName}";
            case sampler_var_typeType.isampler_2d_array:
                return $"isampler2DArray {symbolName}";
            case sampler_var_typeType.isampler_2dms:
                return $"isampler2DMS {symbolName}";
            case sampler_var_typeType.isampler_2dms_array:
                return $"isampler2DMSArray {symbolName}";
            case sampler_var_typeType.isampler_2drect:
                return $"isampler2DRect {symbolName}";
            case sampler_var_typeType.isampler_3d:
                return $"isampler3D {symbolName}";
            case sampler_var_typeType.isampler_buffer:
                return $"isamplerBuffer {symbolName}";
            case sampler_var_typeType.isampler_cube:
                return $"isamplerCube {symbolName}";
            case sampler_var_typeType.isampler_cube_array:
                return $"isamplerCubeArray {symbolName}";
            case sampler_var_typeType.sampler_1d:
                return $"sampler1D {symbolName}";
            case sampler_var_typeType.sampler_1d_array:
                return $"sampler1DArray {symbolName}";
            case sampler_var_typeType.sampler_1d_array_shadow:
                return $"sampler1DArrayShadow {symbolName}";
            case sampler_var_typeType.sampler_1d_shadow:
                return $"sampler1DShadow {symbolName}";
            case sampler_var_typeType.sampler_2d:
                return $"sampler2D {symbolName}";
            case sampler_var_typeType.sampler_2dms:
                return $"sampler2DMS {symbolName}";
            case sampler_var_typeType.sampler_2dms_array:
                return $"sampler2DMSArray {symbolName}";
            case sampler_var_typeType.sampler_2drect:
                return $"sampler2DRect {symbolName}";
            case sampler_var_typeType.sampler_2drect_shadow:
                return $"sampler2DRectShadow {symbolName}";
            case sampler_var_typeType.sampler_2d_array:
                return $"sampler2DArray {symbolName}";
            case sampler_var_typeType.sampler_2d_array_shadow:
                return $"sampler2DArrayShadow {symbolName}";
            case sampler_var_typeType.sampler_2d_shadow:
                return $"sampler2DShadow {symbolName}";
            case sampler_var_typeType.sampler_3d:
                return $"sampler3D {symbolName}";
            case sampler_var_typeType.sampler_buffer:
                return $"samplerBuffer {symbolName}";
            case sampler_var_typeType.sampler_cube:
                return $"samplerCube {symbolName}";
            case sampler_var_typeType.sampler_cube_array:
                return $"samplerCubeArray {symbolName}";
            case sampler_var_typeType.sampler_cube_array_shadow:
                return $"samplerCubeArrayShadow {symbolName}";
            case sampler_var_typeType.sampler_cube_shadow:
                return $"samplerCubeShadow {symbolName}";
            case sampler_var_typeType.usampler_1d:
                return $"usampler1D {symbolName}";
            case sampler_var_typeType.usampler_1d_array:
                return $"usampler1DArray {symbolName}";
            case sampler_var_typeType.usampler_2d:
                return $"usampler2D {symbolName}";
            case sampler_var_typeType.usampler_2dms:
                return $"usampler2DMS {symbolName}";
            case sampler_var_typeType.usampler_2dms_array:
                return $"usampler2DMSArray {symbolName}";
            case sampler_var_typeType.usampler_2drect:
                return $"usampler2DRect {symbolName}";
            case sampler_var_typeType.usampler_2d_array:
                return $"usampler2DArray {symbolName}";
            case sampler_var_typeType.usampler_3d:
                return $"usampler3D {symbolName}";
            case sampler_var_typeType.usampler_buffer:
                return $"usamplerBuffer {symbolName}";
            case sampler_var_typeType.usampler_cube:
                return $"usamplerCube {symbolName}";
            case sampler_var_typeType.usampler_cube_array:
                return $"usamplerCubeArray {symbolName}";
            default:
                Nintendo.ToolFoundation.Contracts.Ensure.Operation.Fail($"Invalid type {samplerVar.type}");
                return null;
            }
        }

        private static string GetShaderParamSymbolName(object shaderParam, string symbolPropertyName)
        {
            Type type = shaderParam.GetType();
            PropertyInfo propInfo = type.GetProperty(symbolPropertyName);
            vertex_symbolType vertexSymbol = propInfo.GetValue(shaderParam) as vertex_symbolType;
            if (vertexSymbol != null)
            {
                return vertexSymbol.name;
            }

            return null;
        }

        private static string GetShaderParamId(object shaderParam)
        {
            Type type = shaderParam.GetType();
            PropertyInfo propInfo = type.GetProperty("id");
            vertex_symbolType vertexSymbol = propInfo.GetValue(shaderParam) as vertex_symbolType;
            if (vertexSymbol != null)
            {
                return vertexSymbol.name;
            }

            return null;
        }

        private static string GetShaderParamSymbolName(object shaderParam)
        {
            string symbol = GetShaderParamSymbolName(shaderParam, "vertex_symbol") ??
                GetShaderParamSymbolName(shaderParam, "fragment_symbol") ??
                GetShaderParamSymbolName(shaderParam, "geometry_symbol") ??
                GetShaderParamSymbolName(shaderParam, "compute_symbol") ?? null;
            if (symbol == null)
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Operation.Fail($"Could not find symbol name of uniform variable \"{GetShaderParamId(shaderParam)}\"");
                return null;
            }

            return symbol;
        }

        private static void WriteUniformVar(ExcelContext excelContext, uniform_varType uniformVar, int cellRowIndex)
        {
            excelContext.SetCellValue($"B{cellRowIndex}", "ub_material");
            string variableDeclaration = GetVariableDeclaration(uniformVar.type, GetShaderParamSymbolName(uniformVar), uniformVar.converter);
            excelContext.SetCellValue($"C{cellRowIndex}", variableDeclaration);
            excelContext.SetCellValue($"D{cellRowIndex}", uniformVar.id);
            if (uniformVar.ui_label != null)
            {
                excelContext.SetCellValue($"E{cellRowIndex}", uniformVar.ui_label.value);
            }
            string defaultValueText = string.Empty;
            foreach (float defaultValue in uniformVar.@default)
            {
                defaultValueText += $"{defaultValue} ";
            }
            excelContext.SetCellValue($"F{cellRowIndex}", defaultValueText);

            StringBuilder otherAnnotations = new StringBuilder();
            otherAnnotations.Append($"type=\"{uniformVar.type.ToString()}\"");
            if (!string.IsNullOrEmpty(uniformVar.hint))
            {
                otherAnnotations.Append($" hint=\"{uniformVar.hint}\"");
            }
            if (!string.IsNullOrEmpty(uniformVar.depend))
            {
                otherAnnotations.Append($" depend=\"{uniformVar.depend}\"");
            }
            if (!string.IsNullOrEmpty(uniformVar.converter))
            {
                otherAnnotations.Append($" converter=\"{uniformVar.converter}\"");
            }
            if (uniformVar.ui_comment != null)
            {
                otherAnnotations.Append($" comment=\"{uniformVar.ui_comment.value}\"");
            }
            if (uniformVar.ui_visible != null)
            {
                otherAnnotations.Append($" visible=\"{uniformVar.ui_visible.value.ToString().Split('.').Last()}\"");
            }
            if (uniformVar.ui_min != null)
            {
                otherAnnotations.Append($" min=\"{uniformVar.ui_min.value}\"");
            }
            if (uniformVar.ui_max != null)
            {
                otherAnnotations.Append($" max=\"{uniformVar.ui_max.value}\"");
            }
            if (uniformVar.ui_default_min != null)
            {
                otherAnnotations.Append($" default_min=\"{uniformVar.ui_default_min.value}\"");
            }
            if (uniformVar.ui_default_max != null)
            {
                otherAnnotations.Append($" default_max=\"{uniformVar.ui_default_max.value}\"");
            }
            if (uniformVar.ui_item != null)
            {
                otherAnnotations.Append($" item=\"{uniformVar.ui_item.value.ToString().Split('.').Last()}\"");
            }
            excelContext.SetCellValue($"G{cellRowIndex}", otherAnnotations.ToString().Trim());
        }

        private static void WriteOptionVar(ExcelContext excelContext, option_varType optionVar, int cellRowIndex)
        {
            excelContext.SetCellValue($"B{cellRowIndex}", "option");
            excelContext.SetCellValue($"C{cellRowIndex}", $"{GetShaderParamSymbolName(optionVar)} ({optionVar.@default})");
            excelContext.SetCellValue($"D{cellRowIndex}", optionVar.id);
            if (optionVar.ui_label != null)
            {
                excelContext.SetCellValue($"E{cellRowIndex}", optionVar.ui_label.value);
            }
            excelContext.SetCellValue($"F{cellRowIndex}", optionVar.@default);
            StringBuilder otherAnnotations = new StringBuilder();
            otherAnnotations.Append($"type=\"{optionVar.type.ToString()}\"");
            if (!string.IsNullOrEmpty(optionVar.choice))
            {
                otherAnnotations.Append($" choice=\"{optionVar.choice}\"");
            }
            if (optionVar.ui_comment != null)
            {
                otherAnnotations.Append($" comment=\"{optionVar.ui_comment.value}\"");
            }
            if (optionVar.ui_visible != null)
            {
                otherAnnotations.Append($" visible=\"{optionVar.ui_visible.value.ToString().Split('.').Last()}\"");
            }
            excelContext.SetCellValue($"G{cellRowIndex}", otherAnnotations.ToString().Trim());
        }


        private static void WriteSamplerVar(ExcelContext excelContext, sampler_varType samplerVar, int cellRowIndex)
        {
            excelContext.SetCellValue($"B{cellRowIndex}", "sampler");
            excelContext.SetCellValue($"C{cellRowIndex}", GetSamplerVariableDeclaration(samplerVar));
            excelContext.SetCellValue($"D{cellRowIndex}", samplerVar.id);
            if (samplerVar.ui_label != null)
            {
                excelContext.SetCellValue($"E{cellRowIndex}", samplerVar.ui_label.value);
            }
            excelContext.SetCellValue($"F{cellRowIndex}", string.Empty);

            StringBuilder otherAnnotations = new StringBuilder();
            if (samplerVar.nostrip)
            {
                otherAnnotations.Append($" nostrip=\"{samplerVar.nostrip}\"");
            }
            if (!string.IsNullOrEmpty(samplerVar.alt))
            {
                otherAnnotations.Append($" alt=\"{samplerVar.alt}\"");
            }
            if (!string.IsNullOrEmpty(samplerVar.hint))
            {
                otherAnnotations.Append($" hint=\"{samplerVar.hint}\"");
            }
            if (samplerVar.ui_comment != null)
            {
                otherAnnotations.Append($" comment=\"{samplerVar.ui_comment.value}\"");
            }
            if (samplerVar.ui_visible != null)
            {
                otherAnnotations.Append($" visible=\"{samplerVar.ui_visible.value.ToString().Split('.').Last()}\"");
            }
            excelContext.SetCellValue($"G{cellRowIndex}", otherAnnotations.ToString().Trim());
        }

        private static void WriteRenderInfo(ExcelContext excelContext, render_info_slotType renderInfo, int cellRowIndex)
        {
            excelContext.SetCellValue($"B{cellRowIndex}", "renderinfo");
            excelContext.SetCellValue($"C{cellRowIndex}", string.Empty);
            excelContext.SetCellValue($"D{cellRowIndex}", renderInfo.name);
            if (renderInfo.ui_label != null)
            {
                excelContext.SetCellValue($"E{cellRowIndex}", renderInfo.ui_label.value);
            }
            excelContext.SetCellValue($"F{cellRowIndex}", renderInfo.@default);

            StringBuilder otherAnnotations = new StringBuilder();
            if (!string.IsNullOrEmpty(renderInfo.choice))
            {
                otherAnnotations.Append($" choice=\"{renderInfo.choice}\"");
            }
            if (renderInfo.count != 1)
            {
                otherAnnotations.Append($" count=\"{renderInfo.count}\"");
            }
            if (renderInfo.optional == false)
            {
                otherAnnotations.Append($" optional=\"{renderInfo.optional}\"");
            }
            if (renderInfo.ui_comment != null)
            {
                otherAnnotations.Append($" comment=\"{renderInfo.ui_comment.value}\"");
            }
            if (renderInfo.ui_item != null)
            {
                otherAnnotations.Append($" item=\"{renderInfo.ui_item.value.ToString().Split('.').Last()}\"");
            }
            excelContext.SetCellValue($"G{cellRowIndex}", otherAnnotations.ToString().Trim());
        }

        private static int GetUIOrder(object input)
        {
            {
                var assign = input as option_varType;
                if (assign != null)
                {
                    return assign.ui_order != null ? assign.ui_order.value : 0;
                }
            }
            {
                var assign = input as sampler_varType;
                if (assign != null)
                {
                    return assign.ui_order != null ? assign.ui_order.value : 0;
                }
            }
            {
                var assign = input as render_info_slotType;
                if (assign != null)
                {
                    return assign.ui_order != null ? assign.ui_order.value : 0;
                }
            }
            {
                var assign = input as uniform_varType;
                if (assign != null)
                {
                    return assign.ui_order != null ? assign.ui_order.value : 0;
                }
            }
            {
                var group = input as groupType;
                if (group != null)
                {
                    return group.ui_order != null ? group.ui_order.value : 0;
                }
            }
            {
                var page = input as pageType;
                if (page != null)
                {
                    return page.ui_order != null ? page.ui_order.value : 0;
                }
            }

            return 0;
        }

        private static IEnumerable<object> ExtractShaderAssigns(ExcelContext excelContext, shading_modelType shadingModel)
        {
            List<object> result = new List<object>();
            IEnumerable<option_varType> optionVars = ExtractOptionVars(excelContext, shadingModel);
            IEnumerable<render_info_slotType> renderInfos = ExtractRenderInfoSlots(excelContext, shadingModel);
            IEnumerable<sampler_varType> samplerVars = ExtractSamplerVars(excelContext, shadingModel);
            IEnumerable<uniform_varType> uniformVars = ExtractUniformVars(excelContext, shadingModel);
            foreach (var assign in optionVars)
            {
                result.Add(assign);
            }
            foreach (var assign in renderInfos)
            {
                result.Add(assign);
            }
            foreach (var assign in samplerVars)
            {
                result.Add(assign);
            }
            foreach (var assign in uniformVars)
            {
                result.Add(assign);
            }
            result.Sort((x, y) =>
            {
                return GetUIOrder(x) - GetUIOrder(y);
            });
            return result;
        }

        private static IEnumerable<uniform_varType> ExtractUniformVars(ExcelContext excelContex, shading_modelType shadingModel)
        {
            List<uniform_varType> result = new List<uniform_varType>();
            if (shadingModel.block_var_array == null)
            {
                return result;
            }

            foreach (var blockVar in shadingModel.block_var_array.Items)
            {
                if (blockVar.uniform_var_array == null)
                {
                    continue;
                }

                if (blockVar.type != block_var_typeType.material)
                {
                    continue;
                }

                foreach (var uniformVar in blockVar.uniform_var_array.Items)
                {
                    if (!result.Any(x => x != null && x.id == uniformVar.id))
                    {
                        result.Add(uniformVar);
                    }
                }
            }

            return result;
        }

        private static IEnumerable<render_info_slotType> ExtractRenderInfoSlots(ExcelContext excelContex, shading_modelType shadingModel)
        {
            List<render_info_slotType> result = new List<render_info_slotType>();
            if (shadingModel.render_info_slot_array == null)
            {
                return result;
            }

            foreach (var renderinfo in shadingModel.render_info_slot_array.Items)
            {
                if (!result.Any(x => x != null && x.name == renderinfo.name))
                {
                    result.Add(renderinfo);
                }
            }

            return result;
        }

        private static IEnumerable<sampler_varType> ExtractSamplerVars(ExcelContext excelContex, shading_modelType shadingModel)
        {
            List<sampler_varType> result = new List<sampler_varType>();
            if (shadingModel.sampler_var_array == null)
            {
                return result;
            }

            foreach (var samplerVar in shadingModel.sampler_var_array.Items)
            {
                if (!result.Any(x => x != null && x.id == samplerVar.id))
                {
                    result.Add(samplerVar);
                }
            }

            return result;
        }

        private static IEnumerable<option_varType> ExtractOptionVars(ExcelContext excelContex, shading_modelType shadingModel)
        {
            List<option_varType> result = new List<option_varType>();
            if (shadingModel.option_var_array == null)
            {
                return result;
            }

            foreach (var optionVar in shadingModel.option_var_array.Items)
            {
                if (!result.Any(x => x != null && x.id == optionVar.id))
                {
                    result.Add(optionVar);
                }
            }

            return result;
        }

        private static IEnumerable<groupType> ExtractGroups(ExcelContext excelContex, shading_modelType shadingModel)
        {
            List<groupType> groups = new List<groupType>();

            // グループ指定なし用に null を追加
            groups.Add(null);
            if (shadingModel.group_array == null)
            {
                return groups;
            }

            foreach (var group in shadingModel.group_array.Items)
            {
                if (!groups.Any(x => x != null && x.name == group.name))
                {
                    groups.Add(group);
                }
            }

            groups.Sort((x, y) => { return GetUIOrder(x) - GetUIOrder(y); });
            return groups;
        }

        private static IEnumerable<pageType> ExtractPages(ExcelContext excelContex, shading_modelType shadingModel)
        {
            List<pageType> pages = new List<pageType>();

            // ページ指定なし用に null を追加
            pages.Add(null);
            if (shadingModel.page_array == null)
            {
                return pages;
            }

            foreach (var page in shadingModel.page_array.page)
            {
                if (!pages.Any(x => x != null && x.name == page.name))
                {
                    pages.Add(page);
                }
            }

            pages.Sort((x, y) => { return GetUIOrder(x) - GetUIOrder(y); });
            return pages;
        }

        private static void WritePageAnnotationToExcel(ExcelContext excelContext, IEnumerable<pageType> pages)
        {
            int cellRowIndex = excelContext.GetActiveSheetRow();
            excelContext.SetCellTitleValue($"A{cellRowIndex}", "page");
            excelContext.SetCellTitleValue($"B{cellRowIndex}", "page_label");
            ++cellRowIndex;
            foreach (var page in pages)
            {
                if (page == null)
                {
                    continue;
                }

                excelContext.SetCellValue($"A{cellRowIndex}", page.name);
                excelContext.SetCellValue($"B{cellRowIndex}", page.ui_label != null ? page.ui_label.value : string.Empty);
                excelContext.AddWorksheet(page.name);
                ++cellRowIndex;
            }

            ++cellRowIndex;
            excelContext.SetActiveSheetRow(cellRowIndex);
        }

        private static void WriteGroupAnnotationToExcel(ExcelContext excelContext, IEnumerable<groupType> groups)
        {
            int cellRowIndex = excelContext.GetActiveSheetRow();
            excelContext.SetCellTitleValue($"A{cellRowIndex}", "group");
            excelContext.SetCellTitleValue($"B{cellRowIndex}", "group_label");
            excelContext.SetCellTitleValue($"C{cellRowIndex}", "group_condition");
            ++cellRowIndex;
            foreach (var group in groups)
            {
                if (group == null)
                {
                    continue;
                }

                excelContext.SetCellValue($"A{cellRowIndex}", group.name);
                excelContext.SetCellValue($"B{cellRowIndex}", group.ui_label != null ? group.ui_label.value : string.Empty);
                excelContext.SetCellValue($"C{cellRowIndex}", group.condition);
                ++cellRowIndex;
            }

            ++cellRowIndex;
            excelContext.SetActiveSheetRow(cellRowIndex);
        }

        private static string FindSourceCodeFolder(shading_modelType shadingModel, shader_srcType[] shaderSources)
        {
            if (shadingModel.vertex_stage != null)
            {
                return shaderSources[shadingModel.vertex_stage.src_index].include_path;
            }
            else if (shadingModel.fragment_stage != null)
            {
                return shaderSources[shadingModel.fragment_stage.src_index].include_path;
            }
            else if (shadingModel.geometry_stage != null)
            {
                return shaderSources[shadingModel.geometry_stage.src_index].include_path;
            }
            else if (shadingModel.compute_stage != null)
            {
                return shaderSources[shadingModel.compute_stage.src_index].include_path;
            }

            return string.Empty;
        }

        private static Excel.Worksheet CreateCommonSheet(
            ExcelContext excelContext,
            IEnumerable<pageType> pages,
            IEnumerable<groupType> groups,
            shading_modelType shadingModel,
            shader_srcType[] shaderSources)
        {
            Excel.Worksheet commonSheet = (Excel.Worksheet)excelContext.Workbook.ActiveSheet;
            commonSheet.StandardWidth = 30;
            commonSheet.Name = "General Settings";
            excelContext.SetCellTitleValue("A1", "File Name");
            excelContext.SetCellValue("B1", shadingModel.name + ".autogen.glsl");

            excelContext.SetCellTitleValue("A2", "Output Directory");
            excelContext.SetCellValue("B2", FindSourceCodeFolder(shadingModel, shaderSources));

            excelContext.SetCellTitleValue("A3", "Shader Option Uniform Block / Id");
            excelContext.SetCellValue("C3", FindBlockVarID(shadingModel, block_var_typeType.option));

            excelContext.SetCellTitleValue("A4", "Material Uniform Block / Id");
            excelContext.SetCellValue("C4", FindBlockVarID(shadingModel, block_var_typeType.material));

            excelContext.SetActiveSheetRow(6);
            if (pages.Count() > 1)
            {
                WritePageAnnotationToExcel(excelContext, pages);
            }

            if (groups.Count() > 1)
            {
                WriteGroupAnnotationToExcel(excelContext, groups);
            }

            return commonSheet;
        }

        private static bool ExaminesSameGroup(groupType group, ui_groupType ui_group)
        {
            if (group == null && ui_group == null)
            {
                return true;
            }

            if (group == null)
            {
                return false;
            }

            if (ui_group == null)
            {
                return false;
            }

            return group.name == ui_group.group_name;
        }

        private static bool ExaminesSamePage(pageType page, groupType group)
        {
            if (page == null && (group == null || string.IsNullOrEmpty(group.page_name)))
            {
                return true;
            }

            if (group == null || string.IsNullOrEmpty(group.page_name))
            {
                return false;
            }

            if (page == null)
            {
                return false;
            }

            return page.name == group.page_name;
        }

        private static string FindBlockVarID(shading_modelType shadingModel, block_var_typeType type)
        {
            if (shadingModel.block_var_array == null)
            {
                return string.Empty;
            }

            foreach (var blockVar in shadingModel.block_var_array.Items)
            {
                if (blockVar.type == type)
                {
                    return blockVar.id;
                }
            }

            return string.Empty;
        }
    }
}
