﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//=============================================================================
// include
//=============================================================================
#include "DccOutput.h"

using namespace std;

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//-----------------------------------------------------------------------------
//! @brief RVec2 型の属性を出力します。
//-----------------------------------------------------------------------------
void ROutVecAttr(
    std::ostream& os,
    const int tabCount,
    const char* name,
    const RVec2& vec
)
{
    os << RTab(tabCount) << name << "=\""
       << vec.x << " "
       << vec.y << "\"" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief RVec3 型の属性を出力します。
//-----------------------------------------------------------------------------
void ROutVecAttr(
    std::ostream& os,
    const int tabCount,
    const char* name,
    const RVec3& vec
)
{
    os << RTab(tabCount) << name << "=\""
       << vec.x << " "
       << vec.y << " "
       << vec.z << "\"" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief RVec4 型の属性を出力します。
//-----------------------------------------------------------------------------
void ROutVecAttr(
    std::ostream& os,
    const int tabCount,
    const char* name,
    const RVec4& vec
)
{
    os << RTab(tabCount) << name << "=\""
       << vec.x << " "
       << vec.y << " "
       << vec.z << " "
       << vec.w << "\"" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 変換（スケール・回転・移動）の属性を出力します。
//-----------------------------------------------------------------------------
void ROutTransform(
    std::ostream& os,
    const int tabCount,
    const RVec3& s,
    const RVec3& r,
    const RVec3& t
)
{
    const int& tc = tabCount;
    ROutVecAttr(os, tc, "scale", s);
    RVec4 outR = r; // 回転は Vec4 に変換します。
    ROutVecAttr(os, tc, "rotate", outR);
    ROutVecAttr(os, tc, "translate", t);
}

//-----------------------------------------------------------------------------
//! @brief 整数型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RIntArray& array,
    const int column
)
{
    const int valCount = static_cast<int>(array.size());
    if (valCount == 0)
    {
        return;
    }

    const int& tc = tabCount;
    int ic = 0;
    for (int ival = 0; ival < valCount; ++ival)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        os << array[ival];
        if (ival < valCount - 1)
        {
            if (ic == column - 1)
            {
                os << R_ENDL;
            }
            else
            {
                os << " ";
            }
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
    os << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 浮動小数点数型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RFloatArray& array,
    const int column
)
{
    const int valCount = static_cast<int>(array.size());
    if (valCount == 0)
    {
        return;
    }

    const int& tc = tabCount;
    int ic = 0;
    for (int ival = 0; ival < valCount; ++ival)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        os << array[ival];
        if (ival < valCount - 1)
        {
            if (ic == column - 1)
            {
                os << R_ENDL;
            }
            else
            {
                os << " ";
            }
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
    os << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief バイト型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RUCharArray& array,
    const int column
)
{
    //-----------------------------------------------------------------------------
    // check size
    const int valCount = static_cast<int>(array.size());
    if (valCount == 0)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // out
    const int& tc = tabCount;
    int ic = 0;
    for (int ival = 0; ival < valCount; ++ival)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        const uint32_t val = static_cast<uint32_t>(array[ival]);
        const uint32_t vh = val >> 4;
        const uint32_t vl = val & 0x0f;
        os << static_cast<char>((vh >= 10) ? 'A' + vh - 10 : '0' + vh)
           << static_cast<char>((vl >= 10) ? 'A' + vl - 10 : '0' + vl);
        if (ic == column - 1 || ival == valCount - 1)
        {
            os << R_ENDL;
        }
        else if (column == 12)
        {
            os << (((ic % 3) == 2) ? "  " : " ");
        }
        else
        {
            os << (((ic & 3) == 3) ? "  " : " ");
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
}

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

