﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// RAnimCurve RAnimKey

//=============================================================================
// include
//=============================================================================
#include "DccCommon.h"

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//=============================================================================
// 定数の定義です。
//=============================================================================

//-----------------------------------------------------------------------------
// 誤差の許容値の定義です。

//! @brief フレームの誤差の許容値です。
//!        アニメーションカーブ評価時、評価するフレームと特定のキーのフレームの誤差が
//!        この値未満なら、評価するフレームはそのキー上にあるとみなします。
const float R_FRAME_TOLERANCE = 0.001f;

const float R_MAKE_KEY_TOL_SHAPE_ANIM    = 0.001f; //!< シェイプアニメーションのキー作成時の許容誤差です。
const float R_MAKE_KEY_TOL_CAMERA_OTHER  = 0.01f; //!< カメラのアスペクト比のキー作成時の許容誤差です。
const float R_MAKE_KEY_TOL_LIGHT_OTHER   = 0.01f; //!< ライトの有効状態のキー作成時の許容誤差です。

//-----------------------------------------------------------------------------
//! @brief 整数サブフレームから浮動小数点数フレームを取得する関数の定義です。
//!        RAnimCurve::MakeKey の引数に使用します。
//-----------------------------------------------------------------------------
typedef float (*RFloatFrameFunc)(const int subFrame, const void* pParam);

//=============================================================================
//! @brief アニメーションキーのクラスです。
//=============================================================================
class RAnimKey
{
public:
    //! @brief 接線タイプを表す列挙型です。
    enum TangentType
    {
        HERMITE, //!< エルミート補間です。
        LINEAR,  //!< 線形補間です。
        STEP     //!< ステップ補間です。
    };

    float m_Frame; //!< フレームです。
    float m_Value; //!< 値です。
    TangentType m_Type; //!< 接線タイプです。
    float m_InSlope; //!< 左側のスロープ値です。
    float m_OtSlope; //!< 右側のスロープ値です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] frame フレームです。
    //! @param[in] value 値です。
    //! @param[in] type 接線タイプです。
    //! @param[in] inSlope 左側のスロープ値です。
    //! @param[in] otSlope 右側のスロープ値です。
    //!
    RAnimKey(
        const float frame = 0.0f,
        const float value = 0.0f,
        const TangentType type = HERMITE,
        const float inSlope = 0.0f,
        const float otSlope = 0.0f)
    : m_Frame(frame),
      m_Value(value),
      m_Type(type),
      m_InSlope(inSlope),
      m_OtSlope(otSlope)
    {
    }

    //! 0 に近い値を 0 にします。
    void SnapToZero()
    {
        m_Frame   = RSnapToZero(m_Frame);
        m_Value   = RSnapToZero(m_Value);
        m_InSlope = RSnapToZero(m_InSlope);
        m_OtSlope = RSnapToZero(m_OtSlope);
    }

    //! 出力ストリームにデバッグ出力します。
    friend std::ostream& operator<<(std::ostream& os, const RAnimKey& key)
    {
        // debug print
        std::streamsize precBak = os.precision(3);
        os << key.m_Frame << " " << key.m_Value << " "
           << key.m_InSlope << " " << key.m_OtSlope << " " << key.m_Type << R_ENDL;
        os.precision(precBak);
        return os;
    }
};

//! @brief アニメーションキー配列の定義です。
typedef std::vector<RAnimKey> RAnimKeyArray;

//=============================================================================
//! @brief アニメーションカーブのクラスです。
//=============================================================================
class RAnimCurve
{
public:
    //! @brief アニメーションカーブの範囲外のフレームの評価モードを表す列挙型です。
    enum Wrap
    {
        CLAMP,          //!< 最初または最後のキーの値を返します。
        REPEAT,         //!< カーブが単純に繰り返した値を返します。
        MIRROR,         //!< カーブが反転しつつ繰り返した値を返します。
        RELATIVE_REPEAT //!< カーブ端の値から相対的に繰り返した値を返します。
    };

    // 属性
    float m_Scale; //!< 値や傾きに乗算するスケールです。
    float m_Offset; //!< 値や傾きに加算するオフセットです。
    Wrap m_PreWrap;  //!< 最初のキーより前のフレームにおける評価モードです。
    Wrap m_PostWrap; //!< 最後のキーより後のフレームにおける評価モードです。
    bool m_Baked; //!< ベイクしたデータからアニメーションカーブデータを作成するなら true です。

    // キー配列
    RAnimKeyArray m_Keys; //!< アニメーションキー配列です。

    // 内部的なパラメータ
    RFloatArray m_FullValues; //!< ベイクしたデータ（全サブフレームにおける値の配列）です。

    bool m_ConstantFlag; //!< 値が一定なら true です。
    bool m_UseFlag; //!< アニメーションカーブを使用するなら true です。
    bool m_LoopFlag; //!< ループアニメーションなら true です。
    bool m_AngleFlag; //!< 角度のアニメーションなら true です。
    float m_Tolerance; //!< 誤差の許容値です。

    std::string m_Name; //!< アニメーションカーブ名です（デバッグ用）。

public:
    //! コンストラクタです。
    RAnimCurve()
    : m_Scale(1.0f),
      m_Offset(0.0f),
      m_PreWrap(CLAMP),
      m_PostWrap(CLAMP),
      m_Baked(false),

      m_ConstantFlag(true),
      m_UseFlag(true),
      m_LoopFlag(false),
      m_AngleFlag(false),
      m_Tolerance(R_SAME_TOLERANCE_F)
    {
    }

    //! メモリを解放します。
    void FreeMemory()
    {
        m_Keys.clear();

        m_FullValues.clear();
    }

    //! @brief アニメーションカーブの一定フラグを更新します。
    //!        最初のフレームの値と他のフレームの値との誤差が許容値未満なら一定とみなします。
    //!        呼ぶ前に m_FullValues, m_Tolerance をセットしておく必要があります。
    //!
    void UpdateConstantFlag();

    //! @brief アニメーションカーブの使用フラグを更新します。
    //!        呼ぶ前に m_FullValues, m_ConstantFlag をセットしておく必要があります。
    //!
    //! @param[in] baseValue アニメーションの初期値です。
    //!
    void UpdateUseFlag(const float baseValue);

    //! @brief ベイクしたデータからエルミート補間用のアニメーションキー配列を作成します。
    //!        呼ぶ前に m_FullValues, m_AngleFlag, m_Tolerance をセットしておく必要があります。
    //!
    //! @param[in] frameFunc 整数サブフレームから浮動小数点数フレームを取得する関数です。
    //! @param[in] pFrameParam frameFunc に渡す引数です。
    //! @param[in] continuousAngleFlag 角度が連続的になるように調整するなら true です。
    //!
    void MakeKeys(
        const RFloatFrameFunc frameFunc,
        const void* pFrameParam,
        const bool continuousAngleFlag = false
    );

    //! @brief ベイクしたデータからステップ補間用のアニメーションキー配列を作成します。
    //!        呼ぶ前に m_FullValues, m_Tolerance をセットしておく必要があります。
    //!
    //! @param[in] frameFunc 整数サブフレームから浮動小数点数フレームを取得する関数です。
    //! @param[in] pFrameParam frameFunc に渡す引数です。
    //!
    void MakeStepKeys(const RFloatFrameFunc frameFunc, const void* pFrameParam);

    //! @brief ステップ補間用のアニメーションキー配列を最適化します。
    //!        直前のキーと同じ値のキーがあれば削除します。
    //!        MakeStepKeys を使用しないで独自に m_Keys を設定した場合に利用します。
    //!        呼ぶ前に m_Keys, m_Tolerance をセットしておく必要があります。
    //!
    void OptimizeStepKeys();

    //! @brief 指定したフレームにおけるアニメーションの値を評価します。
    //!        呼ぶ前に m_Keys をセットしておく必要があります。
    //!
    //! @param[in] frame 評価するフレームです。
    //!
    //! @return アニメーションの値を返します。
    //!
    float Evaluate(const float frame) const;

    //! @brief アニメーションカーブ全体の接線タイプを取得します。
    //!        キーによって接線タイプが異なる場合は RAnimKey::HERMITE を返します。
    //!
    //! @return 接線タイプを返します。
    //!
    RAnimKey::TangentType GetTangentType() const;

    //! @brief アニメーションカーブの初期値を返します。
    float GetBaseValue() const;

    //! @brief アニメーションカーブを出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in,out] dataStreams データ列配列です。
    //! @param[in] tabCount アニメーションカーブ要素のインデントに必要なタブの数です。
    //! @param[in] isOriginal オリジナル要素用のアニメーションカーブなら true を指定します。
    //!
    void Out(
        std::ostream& os,
        RDataStreamArray& dataStreams,
        const int tabCount,
        const bool isOriginal
    ) const;

    //! @brief アニメーションターゲットとアニメーションカーブを出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in,out] dataStreams データ列配列です。
    //! @param[in] tabCount アニメーションターゲット要素のインデントに必要なタブの数です。
    //! @param[in] targetElementName アニメーションターゲット要素名です。
    //! @param[in] targetValue アニメーション対象名（target 属性の値）です。
    //! @param[in] isOriginal オリジナル要素用のアニメーションカーブなら true を指定します。
    //!
    void Out(
        std::ostream& os,
        RDataStreamArray& dataStreams,
        const int tabCount,
        const char* targetElementName,
        const char* targetValue,
        const bool isOriginal
    ) const;
};

//! @brief アニメーションカーブ配列の定義です。
typedef std::vector<RAnimCurve> RAnimCurveArray;

//=============================================================================
// 角度の調整
//=============================================================================

//! @brief 連続的な角度を取得します。角度の単位は degree です。
//!
//! @param[in] value 対象となるフレームの角度です。
//! @param[in] lastValue 1 つ前のフレームの角度です。
//!
//! @return 連続的になるように調整した角度を返します。
//!
float RGetContinuousAngle(const float value, const float lastValue);

//! @brief 角度の配列が連続的な値になるように調整します。角度の単位は degree です。
//!
//! @param[in,out] values 角度の配列です。
//!
//! @return 角度の配列を変更した場合は true を返します。
//!
bool RMakeContinuousAngleArray(RFloatArray& values);

//! @brief 角度の配列が連続的な値になるように XYZ 3 軸同時に調整します。
//!        回転順序が XYZ の場合に対応しています。角度の単位は degree です。
//!
//! @param[in,out] rxArray X 軸回転の角度の配列です。
//! @param[in,out] ryArray Y 軸回転の角度の配列です。
//! @param[in,out] rzArray Z 軸回転の角度の配列です。
//!
//! @return 角度の配列を変更した場合は true を返します。
//!
bool RMakeContinuousXyzAngleArray(
    RFloatArray& rxArray,
    RFloatArray& ryArray,
    RFloatArray& rzArray
);

//! @brief 角度の配列が連続的な値になるように XYZ 3 軸同時に調整します。
//!        回転順序が ZXY の場合に対応しています。角度の単位は degree です。
//!
//! @param[in,out] rxArray X 軸回転の角度の配列です。
//! @param[in,out] ryArray Y 軸回転の角度の配列です。
//! @param[in,out] rzArray Z 軸回転の角度の配列です。
//!
//! @return 角度の配列を変更した場合は true を返します。
//!
bool RMakeContinuousZxyAngleArray(
    RFloatArray& rxArray,
    RFloatArray& ryArray,
    RFloatArray& rzArray
);

//=============================================================================
// ベクトルのアニメーション
//=============================================================================

//! @brief ベクトルのアニメーションカーブの一定フラグを更新します。
//!        ベクトルは正規化されている必要があります。
//!        呼ぶ前に各成分のアニメーションカーブの m_FullValues をセットしておく必要があります。
//!        この関数内で各成分のアニメーションカーブの m_Tolerance に
//!        引数で指定した角度の許容誤差を設定します。
//!
//! @param[in,out] curveX X 成分のアニメーションカーブです。
//! @param[in,out] curveY Y 成分のアニメーションカーブです。
//! @param[in,out] curveZ Z 成分のアニメーションカーブです。
//! @param[in] toleranceR 角度の許容誤差（度数）です。
//!
void RUpdateVectorAnimConstantFlag(
    RAnimCurve& curveX,
    RAnimCurve& curveY,
    RAnimCurve& curveZ,
    const float toleranceR
);

//! @brief ベイクしたデータからベクトルのアニメーションカーブのアニメーションキー配列を作成します。
//!        ベクトルは正規化されている必要があります。
//!        呼ぶ前に各成分のアニメーションカーブの m_FullValues, m_Tolerance を
//!        セットしておく必要があります。
//!
//! @param[in,out] curveX X 成分のアニメーションカーブです。
//! @param[in,out] curveY Y 成分のアニメーションカーブです。
//! @param[in,out] curveZ Z 成分のアニメーションカーブです。
//! @param[in] frameFunc 整数サブフレームから浮動小数点数フレームを取得する関数です。
//! @param[in] pFrameParam frameFunc に渡す引数です。
//!
void RMakeVectorAnimKeys(
    RAnimCurve& curveX,
    RAnimCurve& curveY,
    RAnimCurve& curveZ,
    const RFloatFrameFunc frameFunc,
    const void* pFrameParam
);

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

