﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// CIntermediateFile

//=============================================================================
// include
//=============================================================================
#include "Common.h"

// DCC Library のソースを利用します。
#include <DccCommon.cpp>

using namespace std;
using namespace nn::gfx::tool::dcc;

//=============================================================================
// c2nn ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace g3dTool {
namespace c2nn {

//-----------------------------------------------------------------------------
//! @brief 文字列の末尾の整数値を取得します。
//!
//! @param[out] pNumber 整数値を格納します。文字列の末尾が数値でない場合は -1 を格納します。
//! @param[in] str 文字列です。
//!
//! @return 文字列の末尾の数値を削除した文字列を返します。
//-----------------------------------------------------------------------------
std::string GetSuffixNumberFromString(int* pNumber, const std::string& str)
{
    int digitCount = 0;
    for (size_t ic = str.size() - 1; ic >= 0; --ic)
    {
        const char c = str[ic];
        if ('0' <= c && c <= '9')
        {
            ++digitCount;
        }
        else
        {
            break;
        }
    }

    if (digitCount > 0)
    {
        const size_t prefixCount = str.size() - digitCount;
        *pNumber = atoi(str.c_str() + prefixCount);
        return (prefixCount > 0) ? str.substr(0, prefixCount) : std::string();
    }
    else
    {
        *pNumber = -1;
        return str;
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定されたモジュールを含む実行ファイルのフルパスを返します。
//-----------------------------------------------------------------------------
std::string GetModuleFilePath(HMODULE hModule)
{
    char processPathBuf[MAX_PATH];
    if (GetModuleFileNameA(hModule, processPathBuf, MAX_PATH))
    {
        return std::string(processPathBuf);
    }
    else
    {
        return std::string();
    }
}

//-----------------------------------------------------------------------------
//! @brief テキストファイルの 1 行から引数のトークンを取得します。
//!
//! @param[in,out] pTokens 取得した引数のトークンを追加する配列へのポインタです。
//! @param[in] lineStr 1 行の文字列です。
//-----------------------------------------------------------------------------
void GetArgTokensFromLine(RStringArray* pTokens, const std::string& lineStr)
{
    bool inQuat = false;
    std::string token;
    const char* str = lineStr.c_str();
    const size_t lineBytes = lineStr.size();
    for (size_t byteIdx = 0; byteIdx < lineBytes; ++byteIdx)
    {
        const char c = str[byteIdx];
        if (RIsSpace(c) && !inQuat)
        {
            if (!token.empty())
            {
                pTokens->push_back(token);
                token = "";
            }
        }
        else
        {
            if (c == '\"')
            {
                inQuat = !inQuat;
            }
            token += c;
        }
    }

    if (!token.empty())
    {
        pTokens->push_back(token);
    }
}

//-----------------------------------------------------------------------------
//! @brief テキストファイルから引数のトークンを取得します。
//!
//! @param[in,out] pTokens 取得した引数のトークンを追加する配列へのポインタです。
//! @param[in] filePath テキストファイルのパスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus GetArgTokensFromFile(RStringArray* pTokens, const std::string& filePath)
{
    //-----------------------------------------------------------------------------
    // open file
    ifstream ifs(filePath.c_str());
    if (!ifs)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + filePath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // read
    std::string lineStr;
    while (!ifs.eof())
    {
        getline(ifs, lineStr);
        if (!lineStr.empty())
        {
            lineStr = RTrimString(lineStr);
            if (lineStr.find('#') != 0)
            {
                GetArgTokensFromLine(pTokens, lineStr);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // close file
    ifs.close();

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 引数からオプション名を取得します。
//!
//! @param[out] pIsLongName ロングネームなら true、ショートネームなら false を格納します。
//! @param[in] arg 引数です。
//!
//! @return オプション名を返します。オプション引数でなければ空文字を返します。
//-----------------------------------------------------------------------------
namespace
{

std::string GetOptionNameFromArg(bool* pIsLongName, const std::string& arg)
{
    *pIsLongName = false;
    if (arg.size() >= 2 && arg[0] == '-')
    {
        *pIsLongName = (arg.size() >= 3 && arg[1] == '-');
        if (*pIsLongName)
        {
            const size_t eqIdx = arg.find('=');
            return arg.substr(2, (eqIdx != std::string::npos) ? eqIdx - 2 : std::string::npos);
        }
        else
        {
            return arg.substr(1, 1);
        }
    }
    return "";
}

} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 引数配列からオプション値を取得します。
//!
//! @param[out] pValue オプション値を格納します。
//! @param[in,out] pArgIdx 現在の引数インデックスです。次の引数がオプション値の場合は +1 した値を格納します。
//! @param[in] args 引数配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus GetOptionArgValue(
    std::string* pValue,
    int* pArgIdx,
    const RStringArray& args
)
{
    const std::string& arg = args[*pArgIdx];
    const bool isLongName = (arg.find("--") == 0);
    const size_t eqIdx = arg.find('=');

    *pValue = "";
    if (isLongName && eqIdx != std::string::npos)
    {
        *pValue = arg.substr(eqIdx + 1);
    }
    else if (!isLongName && arg.size() >= 3)
    {
        *pValue = (arg[2] == '=') ? arg.substr(3) : arg.substr(2);
    }
    else if (*pArgIdx < static_cast<int>(args.size()) - 1)
    {
        *pArgIdx += 1;
        *pValue = args[*pArgIdx];
    }
    else
    {
        return RStatus(RStatus::FAILURE, "Option has no value: " + arg); // RShowError
    }
    *pValue = RDequoteString(*pValue);
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 引数からオプション引数情報へのポインタを取得します。
//!
//! @param[out] pIsLongName ロングネームなら true、ショートネームなら false を格納します。
//! @param[out] pSuffixNum オプションの末尾の整数値を格納します。末尾が数値でない場合は -1 を格納します。
//! @param[in] optionArgInfos オプション引数情報配列です。
//! @param[in] arg 引数です。
//!
//! @return オプション引数情報へのポインタを返します。
//!         該当するオプション引数情報がなければ nullptr を返します。
//-----------------------------------------------------------------------------
const OptionArgInfo* GetOptionArgInfo(
    bool* pIsLongName,
    int* pSuffixNum,
    const OptionArgInfo* optionArgInfos,
    const std::string& arg
)
{
    const std::string optionName = GetOptionNameFromArg(pIsLongName, arg);
    if (!optionName.empty())
    {
        const std::string noSuffixName = GetSuffixNumberFromString(pSuffixNum, optionName);

        const OptionArgInfo* pInfo = optionArgInfos;
        while (pInfo->longName != nullptr)
        {
            if (*pIsLongName)
            {
                const size_t longNameSize = strlen(pInfo->longName);
                if (*pSuffixNum != -1 && pInfo->longName[longNameSize - 1] == '#')
                {
                    if (noSuffixName == std::string(pInfo->longName, longNameSize - 1))
                    {
                        return pInfo;
                    }
                }
                else
                {
                    if (optionName == pInfo->longName)
                    {
                        return pInfo;
                    }
                }
            }
            else
            {
                if (pInfo->shortName != ' ' && optionName[0] == pInfo->shortName)
                {
                    return pInfo;
                }
            }
            ++pInfo;
        }
    }
    return nullptr;
}

//-----------------------------------------------------------------------------
//! @brief 指定した引数の次の引数がオプションの値なら true を返します。
//!
//! @param[in] optionArgInfos オプション引数情報配列です。
//! @param[in] arg 引数です。
//!
//! @brief 指定した引数の次の引数がオプションの値なら true を返します。
//-----------------------------------------------------------------------------
bool IsNextArgOptionValue(const OptionArgInfo* optionArgInfos, const std::string& arg)
{
    bool isLongName;
    int suffixNum;
    const OptionArgInfo* pInfo = GetOptionArgInfo(&isLongName, &suffixNum, optionArgInfos, arg);
    if (pInfo != nullptr && pInfo->requiresValue)
    {
        if (isLongName)
        {
            return (arg.find('=') == std::string::npos);
        }
        else
        {
            return (arg.size() <= 2);
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief オプション引数を解析します。
//!
//! @param[out] ppInfo オプション引数情報へのポインタを格納します。
//! @param[out] pSuffixNum オプションの末尾の整数値を格納します。末尾が数値でない場合は -1 を格納します。
//! @param[out] pValue オプション値を格納します。
//! @param[in,out] pArgIdx 現在の引数インデックスです。次の引数がオプション値の場合は +1 した値を格納します。
//! @param[in] args 引数配列です。
//! @param[in] optionArgInfos オプション引数情報配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseOptionArg(
    const OptionArgInfo** ppInfo,
    int* pSuffixNum,
    std::string* pValue,
    int* pArgIdx,
    const RStringArray& args,
    const OptionArgInfo* optionArgInfos
)
{
    //-----------------------------------------------------------------------------
    // オプション引数情報へのポインタを取得します。
    const std::string& arg = args[*pArgIdx];
    bool isLongName;
    const OptionArgInfo* pInfo = GetOptionArgInfo(&isLongName, pSuffixNum, optionArgInfos, arg);
    *ppInfo = pInfo;
    if (pInfo == nullptr)
    {
        return RStatus(RStatus::FAILURE, "Option is wrong: " + arg); // RShowError
    }

    //-----------------------------------------------------------------------------
    // オプション値を取得します。
    const size_t eqIdx = arg.find('=');
    const bool isWithValue =
        ( isLongName && eqIdx != std::string::npos) ||
        (!isLongName && arg.size() >= 3           );
    if (pInfo->requiresValue)
    {
        if (isWithValue)
        {
            if (isLongName)
            {
                *pValue = arg.substr(eqIdx + 1);
            }
            else
            {
                *pValue = (arg[2] == '=') ? arg.substr(3) : arg.substr(2);
            }
        }
        else
        {
            if (*pArgIdx >= static_cast<int>(args.size()) - 1)
            {
                return RStatus(RStatus::FAILURE, "Option has no value: " + arg); // RShowError
            }
            *pArgIdx += 1;
            *pValue = args[*pArgIdx];
        }
        *pValue = RDequoteString(*pValue);
    }
    else // 値を必要としないオプション
    {
        if (isWithValue)
        {
            return RStatus(RStatus::FAILURE, "Option is wrong: " + arg); // RShowError
        }
        *pValue = "";
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief フォルダが存在しなければ作成します。
//!
//! @param[in] folderPath フォルダのパスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus CreateFolderIfNotExist(const std::string& folderPath)
{
    if (!RFolderExists(folderPath))
    {
        if (!::CreateDirectoryA(folderPath.c_str(), nullptr))
        {
            return RStatus(RStatus::FAILURE, "Cannot create the folder: " + folderPath); // RShowError
        }
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief メモリを確保してファイルをリードします。
//!
//! @param[out] ppBuf 確保したメモリへのポインタを格納します。
//! @param[out] pSize ファイルサイズ（バイト数）を格納します。
//! @param[in] filePath ファイルのパスを指定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ReadFileToMemory(
    uint8_t** ppBuf,
    size_t* pSize,
    const std::string& filePath
)
{
    //-----------------------------------------------------------------------------
    // open file
    std::ifstream ifs(filePath, std::ios::binary);
    if (!ifs)
    {
        // open failed
        *ppBuf = nullptr;
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + filePath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // read file
    ifs.seekg(0, std::ios::end);
    *pSize = static_cast<size_t>(ifs.tellg());
    ifs.seekg(0, std::ios::beg);
    *ppBuf = new uint8_t[*pSize];
    ifs.read(reinterpret_cast<char*>(*ppBuf), *pSize);

    //-----------------------------------------------------------------------------
    // close file
    ifs.close();

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief NW4C 中間ファイルのコンストラクタです。
//-----------------------------------------------------------------------------
CIntermediateFile::CIntermediateFile(RStatus* pStatus, const std::string& path)
:   m_Path(path),
    m_GraphicsElem(nullptr)
{
    //std::cerr << "CIntermediateFile(): " << path << std::endl;
    uint8_t* pFileBuf = nullptr;
    size_t fileSize = 0;
    RStatus status = ReadFileToMemory(&pFileBuf, &fileSize, path);
    if (status)
    {
        m_Xml.LoadDocument(nullptr, std::string(reinterpret_cast<const char*>(pFileBuf), fileSize));
        m_GraphicsElem = m_Xml.FindElement("NintendoWareIntermediateFile/GraphicsContentCtr", false);
        if (m_GraphicsElem == nullptr)
        {
            status = RStatus(RStatus::FAILURE, "GraphicsContentCtr element cannot be found: " + path); // RShowError
        }
    }
    if (pFileBuf != nullptr)
    {
        delete[] pFileBuf;
    }
    if (pStatus != nullptr)
    {
        *pStatus = status;
    }
}

//=============================================================================
// c2nn ネームスペースを終了します。
//=============================================================================
} // c2nn
} // g3dTool
} // nn

