﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinVisibilityAnim.h>

namespace nn {
namespace g3dTool {

void BinBoneVisibilityAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_bone_visibility_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_pBoneVisBoneAnimElemArray.reserve(elem.bone_vis_bone_anim_array.size());
    for (auto iter = m_pElem->bone_vis_bone_anim_array.cbegin(); iter != m_pElem->bone_vis_bone_anim_array.cend(); ++iter)
    {
        if (iter->binarize_visibility.value)
        {
            m_pBoneVisBoneAnimElemArray.push_back(&(*iter));
        }
    }

    std::vector<const nw::g3d::tool::g3dif::elem_anim_curve*> curveArray;
    for (auto iter = m_pBoneVisBoneAnimElemArray.cbegin(); iter != m_pBoneVisBoneAnimElemArray.cend(); ++iter)
    {
        pCtx->SetStr( (*iter)->bone_name.value.c_str() );

        if ((*iter)->curve)
        {
            curveArray.push_back(&(*iter)->curve.Get());
        }
    }
    m_CurveArray.resize(curveArray.size());
    SetParentBlockArray(m_CurveArray, this);
    BuildArray(pCtx, m_CurveArray, curveArray);

    int animIndex = 0;
    int curveIndex = 0;
    for (auto iter = m_pBoneVisBoneAnimElemArray.cbegin(); iter != m_pBoneVisBoneAnimElemArray.cend(); ++iter)
    {
        if ((*iter)->curve)
        {
            BinAnimCurve& curve = m_CurveArray[curveIndex];
            curve.SetTargetOffset(animIndex);
            curve.SetType(BinAnimCurve::BOOL);
            ++curveIndex;
        }
        ++animIndex;
    }

    m_DicUserData.Build(pCtx, elem.user_data_array.size());
    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr( elem.path.c_str() );
    pCtx->SetStr( elem.name.c_str() );

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinBoneVisibilityAnim::CalculateSize()
{
    auto numAnim =  m_pBoneVisBoneAnimElemArray.size();

    // ResBoneVisibilityAnimData は親から割り当てる。
    m_Chunk[ChunkType_BindIndex].size = nw::g3d::tool::util::Align(sizeof(uint16_t) * numAnim, ALIGNMENT_DEFAULT);
    m_Chunk[ChunkType_Name].size = sizeof( nn::util::BinPtrToString ) * numAnim;
    m_Chunk[ChunkType_Curve].size = sizeof(nn::g3d::ResAnimCurveData) * m_CurveArray.size();
    m_Chunk[ChunkType_BaseValue].size = static_cast<size_t>(std::ceil(numAnim / 32.0f)) * sizeof(nn::Bit32);
    m_Chunk[ ChunkType_UserDataArray ].size = sizeof( nn::gfx::ResUserDataData ) * m_UserDataArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinBoneVisibilityAnim::CalculateOffset(std::shared_ptr<Context> pCtx)
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Curve].offset;
    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResAnimCurveData);
    }

    // UserData Array のオフセット計算
    ptrdiff_t ofsUserDataArray = GetBlockOffset( Context::MemBlockType_Main ) + m_Chunk[ ChunkType_UserDataArray ].offset;
    for ( auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter )
    {
        iter->SetStructOffset( ofsUserDataArray );
        ofsUserDataArray += sizeof(nn::gfx::ResUserDataData);
    }
}

void BinBoneVisibilityAnim::Convert(std::shared_ptr<Context> pCtx)
{
    nn::g3d::ResBoneVisibilityAnimData& visibilityAnim = *GetPtr<nn::g3d::ResBoneVisibilityAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    visibilityAnim.blockHeader.signature.SetPacked( nn::g3d::ResBoneVisibilityAnim::Signature );
    pCtx->AddBinBlockHeader( &visibilityAnim.blockHeader );
    pCtx->LinkStr( &visibilityAnim.pName, m_pElem->name.c_str() );
    pCtx->LinkStr( &visibilityAnim.pPath, m_pElem->path.c_str() );

    const nw::g3d::tool::g3dif::elem_bone_visibility_anim_info& info = m_pElem->bone_visibility_anim_info;
    uint16_t flag = 0;
    if (info.loop.value)
    {
        flag |= nn::g3d::ResBoneVisibilityAnim::Flag_PlayPolicyLoop;
    }

    //flag |= nn::g3d::ResBoneVisibilityAnim::BONE_VISIBILITY;	// visibility anim は bone しか存在しなくなった。

    int numAnim = static_cast<int>(m_pBoneVisBoneAnimElemArray.size());

    visibilityAnim.flag = flag;
    visibilityAnim.frameCount = info.frame_count.value;
    visibilityAnim.animCount = static_cast<uint16_t>(numAnim);
    visibilityAnim.curveCount = static_cast<uint16_t>(m_CurveArray.size());
    visibilityAnim.bakedSize = 0;

    visibilityAnim.pBindModel.Set( nullptr );
    uint16_t* pBindIndexArray = GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BindIndex].offset);
    for (int idxAnim = 0; idxAnim < numAnim; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = nn::g3d::AnimFlag::AnimFlag_NotBound;
    }
    if (numAnim & 0x1)
    {
        // 奇数個の場合は最後にパディングを埋める。
        pBindIndexArray[numAnim] = 0;
    }
    pCtx->LinkPtr( &visibilityAnim.pBindIndexArray, GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BindIndex].offset) );

    nn::util::BinPtrToString* pName =
        GetPtr< nn::util::BinPtrToString >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Name].offset);

    nn::Bit32* pBaseValue = GetPtr<nn::Bit32>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BaseValue].offset);
    memset(pBaseValue, 0, m_Chunk[ChunkType_BaseValue].size);

    int idxAnim = 0;
    for (auto iter = m_pBoneVisBoneAnimElemArray.cbegin(); iter != m_pBoneVisBoneAnimElemArray.cend(); ++iter, ++idxAnim)
    {
        pCtx->LinkStr( &pName[idxAnim], (*iter)->bone_name.value.c_str() );
        int valueIndex = 0x1 << (idxAnim % 32);
        nn::Bit32& baseValue = pBaseValue[idxAnim / 32];
        baseValue |= ((*iter)->base_value.value != 0) ? valueIndex : 0;
    }
    pCtx->LinkPtr( &visibilityAnim.pBaseValueArray, pBaseValue );
    pCtx->LinkPtr( &visibilityAnim.pNameArray, GetPtr<nn::util::BinString>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Name].offset) );
    pCtx->LinkPtr( &visibilityAnim.pCurveArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Curve].offset));

    // UserData
    visibilityAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    void* pUserDataArray = GetPtr< void >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_UserDataArray ].offset );
    pCtx->LinkPtr( &visibilityAnim.pUserDataArray, pUserDataArray );
    m_DicUserData.ConvertData(pCtx, visibilityAnim.pUserDataDic, m_UserDataArray);
}

void BinBoneVisibilityAnim::Adjust( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResBoneVisibilityAnim* visibilityAnim = GetPtr<nn::g3d::ResBoneVisibilityAnim>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    nn::g3d::ResBoneVisibilityAnimData& visibilityAnimData = visibilityAnim->ToData();
    visibilityAnimData.pCurveArray.Relocate( pCtx->GetBasePtr() );

    size_t size = 0;
    for (int idxCurve = 0, numCurve = visibilityAnim->GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        nn::g3d::ResAnimCurve* curve = visibilityAnim->GetCurve(idxCurve);
        size += curve->CalculateBakedIntSize();
    }
    visibilityAnimData.bakedSize = static_cast<uint32_t>(size);
    visibilityAnimData.pCurveArray.Unrelocate( pCtx->GetBasePtr() );
}

}
}
