﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinTexPatternAnim.h>
#include <util/UtilError.h>

namespace nn {
namespace g3dTool {

void BinTexPatternMatAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_tex_pattern_mat_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    std::vector<const nw::g3d::tool::g3dif::elem_anim_curve*> curveArray;
    for (auto iter = elem.pattern_anim_target_array.cbegin();
        iter != elem.pattern_anim_target_array.cend(); ++iter)
    {
        if (iter->curve)
        {
            curveArray.push_back(&iter->curve.Get());
        }
    }
    m_CurveArray.resize(curveArray.size());
    SetParentBlockArray(m_CurveArray, this);
    BuildArray(pCtx, m_CurveArray, curveArray);

    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetTargetOffset(0);
        iter->SetType(BinAnimCurve::INT);
    }

    // 文字列の登録。
    pCtx->SetStr(elem.mat_name.value.c_str());

    for (auto iter = elem.pattern_anim_target_array.cbegin();
        iter != elem.pattern_anim_target_array.cend(); ++iter)
    {
        pCtx->SetStr(iter->sampler_name.value.c_str());
    }

    // 定数アニメの個数を計算します。curve を持たないターゲットは定数アニメとします。
    for (auto texPatternAnim : m_pElem->pattern_anim_target_array )
    {
        if ( !texPatternAnim.curve )
        {
            ++m_ConstantAnimCount;
        }
    }
}

void BinTexPatternMatAnim::CalculateSize()
{
    m_Chunk[TEX_INFO].size = sizeof(nn::g3d::ResTexturePatternAnimInfo) * m_pElem->pattern_anim_target_array.size();
    m_Chunk[ChunkType_Curve].size = sizeof(nn::g3d::ResAnimCurveData) * m_CurveArray.size();
    m_Chunk[ChunkType_Constant].size = sizeof(nn::g3d::ResAnimConstantData) * m_ConstantAnimCount;
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinTexPatternMatAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Curve].offset;
    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResAnimCurveData);
    }
}

void BinTexPatternMatAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResPerMaterialAnimData& matAnim = *GetPtr<nn::g3d::ResPerMaterialAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));
    matAnim.beginTexturePatternCurveIndex = static_cast< uint16_t >( m_BeginCurve );
    matAnim.beginShaderParamCurveIndex	= nn::g3d::AnimFlag_NotBound;
    matAnim.visibilityConstantIndex		= nn::g3d::AnimFlag_NotBound;
    matAnim.visibilityCurveIndex		= nn::g3d::AnimFlag_NotBound;

    pCtx->LinkStr( &matAnim.pName, m_pElem->mat_name.value.c_str() );

    matAnim.texturePatternAnimCount = static_cast<uint16_t>(m_pElem->pattern_anim_target_array.size());
    nn::g3d::ResTexturePatternAnimInfo* pPatAnimInfo = GetPtr<nn::g3d::ResTexturePatternAnimInfo>(pCtx, Context::MemBlockType_Main, m_Chunk[TEX_INFO].offset);
    pCtx->LinkPtr( &matAnim.pTexturePatternAnimInfoArray, pPatAnimInfo );

    nn::g3d::ResAnimConstantData* pResAnimConstant = GetPtr<nn::g3d::ResAnimConstantData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Constant].offset);
    if( m_ConstantAnimCount )
    {
        pCtx->LinkPtr( &matAnim.pConstantArray, pResAnimConstant);
        matAnim.constantCount = m_ConstantAnimCount;
    }

    if( m_CurveArray.size() )
    {
        pCtx->LinkPtr( &matAnim.pCurveArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Curve].offset) );
        matAnim.curveCount = static_cast<uint8_t>(m_CurveArray.size());
    }

    uint16_t idxTarget = 0;
    uint16_t idxConstAnim = 0;
    uint16_t idxCurve = 0;
    for (auto iter = m_pElem->pattern_anim_target_array.cbegin();
        iter != m_pElem->pattern_anim_target_array.cend(); ++iter, ++idxTarget)
    {
        nn::g3d::ResTexturePatternAnimInfo& patAnimInfo = pPatAnimInfo[idxTarget];
        pCtx->LinkStr( &patAnimInfo.pName, iter->sampler_name.value.c_str() );
        patAnimInfo.subbindIndex	= -1;

        if (iter->curve)
        {
            patAnimInfo.constantIndex	= nn::g3d::AnimFlag_NotBound;
            patAnimInfo.curveIndex = idxCurve;
            ++idxCurve;
        }
        else
        {
            patAnimInfo.curveIndex		= nn::g3d::AnimFlag_NotBound;
            patAnimInfo.constantIndex	= idxConstAnim;
            pResAnimConstant[idxConstAnim].iValue = static_cast<int32_t>(iter->base_value.value);	//!< uint16 からのキャストです。
            ++idxConstAnim;
        }
    }
}

void BinTexPatternAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_tex_pattern_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_TexPatternMatAnimArray.resize(elem.tex_pattern_mat_anim_array.size());
    SetParentBlockArray(m_TexPatternMatAnimArray, this);
    BuildArray(pCtx, m_TexPatternMatAnimArray, elem.tex_pattern_mat_anim_array);

    int32_t numTexture = 0;
    for (auto iter = m_TexPatternMatAnimArray.begin(); iter != m_TexPatternMatAnimArray.end(); ++iter)
    {
        iter->SetBeginTexture(numTexture);
        numTexture += iter->GetTextureCount();
    }
    m_TextureCount = numTexture;

    int32_t numCurve = 0;
    for (auto iter = m_TexPatternMatAnimArray.begin(); iter != m_TexPatternMatAnimArray.end(); ++iter)
    {
        iter->SetBeginCurve(numCurve);
        numCurve += iter->GetCurveCount();
    }
    m_CurveCount = numCurve;

    m_DicUserData.Build(pCtx, elem.user_data_array.size());
    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr(elem.path.c_str());
    pCtx->SetStr(elem.name.c_str());

    // texture 名の文字列プールへの登録。
    int idxTextureName = 0;
    for (auto iter = m_pElem->tex_pattern_array.cbegin(); iter != m_pElem->tex_pattern_array.cend(); ++iter, ++idxTextureName)
    {
        pCtx->SetStr(iter->tex_name.value.c_str());
    }

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinTexPatternAnim::CalculateSize()
{
    auto numAnim =  m_pElem->tex_pattern_mat_anim_array.size();
    m_Chunk[ChunkType_TexturePatternAnim].size = sizeof(nn::g3d::ResMaterialAnimData);
    m_Chunk[ChunkType_BindIndex].size = nw::g3d::tool::util::Align(sizeof(uint16_t) * numAnim, ALIGNMENT_DEFAULT);
    m_Chunk[ChunkType_TexturePatternMaterialAnim].size = sizeof(nn::g3d::ResPerMaterialAnimData) * numAnim;
    m_Chunk[ ChunkType_TextureDescriptorSlotArray ].size = sizeof( nn::gfx::DescriptorSlotData ) * m_pElem->tex_pattern_array.size();
    m_Chunk[ ChunkType_GfxTextureViewPtrArray ].size = sizeof( nn::util::BinTPtr<const nn::gfx::TextureView> ) * m_pElem->tex_pattern_array.size();
    m_Chunk[ ChunkType_TextureNamePtrArray ].size = sizeof( nn::util::BinPtrToString ) * m_pElem->tex_pattern_array.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinTexPatternAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_TexturePatternMaterialAnim].offset;
    for (auto iter = m_TexPatternMatAnimArray.begin(); iter != m_TexPatternMatAnimArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResPerMaterialAnimData);
    }
}

void BinTexPatternAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResMaterialAnimData& texPatternAnim = *GetPtr<nn::g3d::ResMaterialAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));

    texPatternAnim.blockHeader.signature.SetPacked(nn::g3d::ResMaterialAnim::Signature);
    pCtx->LinkStr( &texPatternAnim.pName, m_pElem->name.c_str() );
    pCtx->LinkStr( &texPatternAnim.pPath, m_pElem->path.c_str() );

    const nw::g3d::tool::g3dif::elem_tex_pattern_anim_info& info = m_pElem->tex_pattern_anim_info;
    uint16_t flag = 0;
    if (info.loop.value)
    {
        flag |= nn::g3d::ResMaterialAnim::Flag_PlayPolicyLoop;
    }

    texPatternAnim.flag = flag;
    texPatternAnim.textureCount = static_cast<uint16_t>(m_pElem->tex_pattern_array.size());
    pCtx->LinkPtr( &texPatternAnim.pTextureArray, GetPtr( pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_GfxTextureViewPtrArray].offset ) );
    texPatternAnim.frameCount = info.frame_count.value;
    texPatternAnim.perMaterialAnimCount = static_cast<uint16_t>(m_TexPatternMatAnimArray.size());

    texPatternAnim.texturePatternAnimCount= static_cast< uint16_t >( m_TextureCount );
    texPatternAnim.curveCount = static_cast< uint16_t >( m_CurveCount );
    texPatternAnim.bakedSize = 0;

    texPatternAnim.pBindModel.Set(nullptr);
    uint16_t* pBindIndexArray = GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BindIndex].offset);
    pCtx->LinkPtr( &texPatternAnim.pBindIndexArray, pBindIndexArray);

    int numAnim = static_cast<int>(m_TexPatternMatAnimArray.size());
    for (int idxAnim = 0; idxAnim < numAnim; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = nn::g3d::AnimFlag_NotBound;
    }
    if (numAnim & 0x1)
    {
        // 奇数個の場合は最後にパディングを埋める。
        pBindIndexArray[numAnim] = 0;
    }

    if( m_pElem->tex_pattern_array.size() != 0 )
    {
        nn::util::BytePtr offsetToTextureDescriptorSlotArray( GetPtr<void>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_TextureDescriptorSlotArray ].offset ) );
        if( !nn::util::is_aligned( reinterpret_cast<size_t>( offsetToTextureDescriptorSlotArray.Get() ), 8 ) )
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_INTERNAL, "Identifier_InvalidAddressAlignment", "ResMaterialAnimData pTextureSlotArray", offsetToTextureDescriptorSlotArray.Get() );
        }
        // DescriptorSlotを初期化しておく。無効値で初期化される。
        nn::gfx::DescriptorSlotData* pDescriptorSlotDataArray = offsetToTextureDescriptorSlotArray.Get<nn::gfx::DescriptorSlotData>();
        for (unsigned int idxTex = 0; idxTex < m_pElem->tex_pattern_array.size(); ++idxTex)
        {
            new(&pDescriptorSlotDataArray[idxTex]) nn::gfx::DescriptorSlot();
        }
        pCtx->LinkPtr( &texPatternAnim.pTextureSlotArray,
            GetPtr<int32_t>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_TextureDescriptorSlotArray ].offset ) );

        nn::util::BinPtrToString* pTextureNameArray =
            GetPtr<nn::util::BinPtrToString>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_TextureNamePtrArray ].offset );
        pCtx->LinkPtr( &texPatternAnim.pTextureNameArray, pTextureNameArray );

        for( auto iter = m_pElem->tex_pattern_array.cbegin(); iter != m_pElem->tex_pattern_array.cend(); ++iter, ++pTextureNameArray )
        {
            pCtx->LinkStr( pTextureNameArray, nn::util::string_view( iter->tex_name.value.data() ) );
        }
    }

    pCtx->LinkPtr( &texPatternAnim.pPerMaterialAnimArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_TexturePatternMaterialAnim].offset));

    // UserData
    texPatternAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    m_DicUserData.ConvertData(pCtx, texPatternAnim.pUserDataDic, m_UserDataArray);
}

void BinTexPatternAnim::Adjust( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResMaterialAnim* texPatternAnim = GetPtr<nn::g3d::ResMaterialAnim>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));
    nn::g3d::ResMaterialAnimData& texPatternAnimData = texPatternAnim->ToData();
    texPatternAnimData.pPerMaterialAnimArray.Relocate( pCtx->GetBasePtr() );

    size_t size = 0;
    for (int idxPerMatAnim = 0, numPerMatAnim = texPatternAnim->GetPerMaterialAnimCount(); idxPerMatAnim < numPerMatAnim; ++idxPerMatAnim)
    {
        nn::g3d::ResPerMaterialAnim* pPerMatAnim = texPatternAnim->GetPerMaterialAnim(idxPerMatAnim);
        nn::g3d::ResPerMaterialAnimData& resPerMaterialAnimData = pPerMatAnim->ToData();
        resPerMaterialAnimData.pCurveArray.Relocate( pCtx->GetBasePtr() );

        for (int idxCurve = 0, numCurve = pPerMatAnim->GetCurveCount();
            idxCurve < numCurve; ++idxCurve)
        {
            nn::g3d::ResAnimCurve* curve = pPerMatAnim->GetCurve(idxCurve);
            size += curve->CalculateBakedIntSize();
        }

        resPerMaterialAnimData.pCurveArray.Unrelocate( pCtx->GetBasePtr() );
    }

    texPatternAnimData.bakedSize = static_cast<uint32_t>(size);
    texPatternAnimData.pPerMaterialAnimArray.Unrelocate( pCtx->GetBasePtr() );
}

} // namespace g3d
} // namespace nw
