﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @file
* @brief 中間ファイルフォーマットの仕様に従った値を gfx が受け取れる値に変換するヘルパー関数群です。
*/

#pragma once

#include <g3dif/Material.h>
#include <algorithm>

#include <nn/gfx/gfx_Enum.h>
#include <nn/gfx/gfx_Types.h>
#include <nn/gfx/detail/gfx_Buffer-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Buffer-api.gx.2.h>
#include <nn/gfx/detail/gfx_Buffer-api.gl.4.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gl.4.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gx.2.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Sampler-api.gl.4.h>
#include <nn/gfx/detail/gfx_Sampler-api.gx.2.h>
#include <nn/gfx/detail/gfx_Sampler-api.nvn.8.h>


//#define GFX_VAL_FORCE_DEFAULT	//!< 強制的にデフォルト値を使用するフラグです。

namespace nn {
namespace g3dTool {
namespace {

// T に gfx の ImplData を渡すと最大サイズを返します。
template<template<class> class T> const size_t GetGfxMaxImplDataSize( void )
{
    size_t size = std::max<size_t>(
        sizeof( T<nn::gfx::ApiVariationGl4> ),
        sizeof( T<nn::gfx::ApiVariationGx2> ) );
    return std::max<size_t>( size, sizeof( T<nn::gfx::ApiVariationNvn8> ) );
}

const nn::gfx::AttributeFormat g_GfxAttributeFormatTbl[] =
{
    nn::gfx::AttributeFormat_32_Sint,
    nn::gfx::AttributeFormat_32_32_Sint,
    nn::gfx::AttributeFormat_32_32_32_Sint,
    nn::gfx::AttributeFormat_32_32_32_32_Sint,
    nn::gfx::AttributeFormat_32_Uint,
    nn::gfx::AttributeFormat_32_32_Uint,
    nn::gfx::AttributeFormat_32_32_32_Uint,
    nn::gfx::AttributeFormat_32_32_32_32_Uint,
    nn::gfx::AttributeFormat_32_Float,
    nn::gfx::AttributeFormat_32_32_Float,
    nn::gfx::AttributeFormat_32_32_32_Float,
    nn::gfx::AttributeFormat_32_32_32_32_Float
};

const nn::gfx::AttributeFormat g_GfxAttributeFormatQuantizedTbl[] =
{
    nn::gfx::AttributeFormat_8_Unorm,
    nn::gfx::AttributeFormat_8_Unorm,
    nn::gfx::AttributeFormat_8_Uint,
    nn::gfx::AttributeFormat_8_Snorm,
    nn::gfx::AttributeFormat_8_Sint,
    nn::gfx::AttributeFormat_8_UintToFloat,
    nn::gfx::AttributeFormat_8_SintToFloat,
    nn::gfx::AttributeFormat_4_4_Unorm,
    nn::gfx::AttributeFormat_16_Unorm,
    nn::gfx::AttributeFormat_16_Uint,
    nn::gfx::AttributeFormat_16_Snorm,
    nn::gfx::AttributeFormat_16_Sint,
    nn::gfx::AttributeFormat_16_Float,
    nn::gfx::AttributeFormat_16_UintToFloat,
    nn::gfx::AttributeFormat_16_SintToFloat,
    nn::gfx::AttributeFormat_8_8_Unorm,
    nn::gfx::AttributeFormat_8_8_Uint,
    nn::gfx::AttributeFormat_8_8_Snorm,
    nn::gfx::AttributeFormat_8_8_Sint,
    nn::gfx::AttributeFormat_8_8_UintToFloat,
    nn::gfx::AttributeFormat_8_8_SintToFloat,
    nn::gfx::AttributeFormat_32_Uint,
    nn::gfx::AttributeFormat_32_Sint,
    nn::gfx::AttributeFormat_32_Float,
    nn::gfx::AttributeFormat_16_16_Unorm,
    nn::gfx::AttributeFormat_16_16_Uint,
    nn::gfx::AttributeFormat_16_16_Snorm,
    nn::gfx::AttributeFormat_16_16_Sint,
    nn::gfx::AttributeFormat_16_16_Float,
    nn::gfx::AttributeFormat_16_16_UintToFloat,
    nn::gfx::AttributeFormat_16_16_SintToFloat,
    nn::gfx::AttributeFormat_Undefined,				// nn::gfx::AttributeFormat_10_11_11_Float,
    nn::gfx::AttributeFormat_8_8_8_8_Unorm,
    nn::gfx::AttributeFormat_8_8_8_8_Uint,
    nn::gfx::AttributeFormat_8_8_8_8_Snorm,
    nn::gfx::AttributeFormat_8_8_8_8_Sint,
    nn::gfx::AttributeFormat_8_8_8_8_UintToFloat,
    nn::gfx::AttributeFormat_8_8_8_8_SintToFloat,
    nn::gfx::AttributeFormat_10_10_10_2_Unorm,
    nn::gfx::AttributeFormat_10_10_10_2_Uint,
    nn::gfx::AttributeFormat_10_10_10_2_Snorm,
    nn::gfx::AttributeFormat_10_10_10_2_Sint,
    nn::gfx::AttributeFormat_32_32_Uint,
    nn::gfx::AttributeFormat_32_32_Sint,
    nn::gfx::AttributeFormat_32_32_Float,
    nn::gfx::AttributeFormat_16_16_16_16_Unorm,
    nn::gfx::AttributeFormat_16_16_16_16_Uint,
    nn::gfx::AttributeFormat_16_16_16_16_Snorm,
    nn::gfx::AttributeFormat_16_16_16_16_Sint,
    nn::gfx::AttributeFormat_16_16_16_16_Float,
    nn::gfx::AttributeFormat_16_16_16_16_UintToFloat,
    nn::gfx::AttributeFormat_16_16_16_16_SintToFloat,
    nn::gfx::AttributeFormat_32_32_32_Uint,
    nn::gfx::AttributeFormat_32_32_32_Sint,
    nn::gfx::AttributeFormat_32_32_32_Float,
    nn::gfx::AttributeFormat_32_32_32_32_Uint,
    nn::gfx::AttributeFormat_32_32_32_32_Sint,
    nn::gfx::AttributeFormat_32_32_32_32_Float
};

const uint8_t g_GfxAnisotropyTbl[] =
{
    1,		//< 0 だと GL_INVALID_ENUM になるので 1 にする。
    2,
    4,
    8,
    16
};

const nn::gfx::TextureAddressMode g_GfxTextureAddrModeTbl[] =
{
    nn::gfx::TextureAddressMode_Repeat,
    nn::gfx::TextureAddressMode_Mirror,
    nn::gfx::TextureAddressMode_ClampToEdge,
    nn::gfx::TextureAddressMode_MirrorClampToEdge
};

const Bit32 g_GfxFilterModeBitTbl[] =
{
    0, // none
    nn::gfx::detail::FilterModeBit_Point,
    nn::gfx::detail::FilterModeBit_Linear,
};

inline nn::gfx::AttributeFormat GetGfxAttributeFormat( uint32_t idx )
{
    NN_SDK_ASSERT_RANGE( idx, static_cast< uint32_t >( 0 ), sizeof( g_GfxAttributeFormatTbl ) / sizeof( nn::gfx::AttributeFormat ) );
    return g_GfxAttributeFormatTbl[ idx ];
}


inline nn::gfx::AttributeFormat GetGfxAttributeFormatQuantized( uint32_t idx )
{
    NN_SDK_ASSERT_RANGE( idx, static_cast< uint32_t >( 0 ), sizeof( g_GfxAttributeFormatQuantizedTbl ) / sizeof( nn::gfx::AttributeFormat ) );
    return g_GfxAttributeFormatQuantizedTbl[ idx ];
}

inline uint8_t GetGfxMaxAnisotropy( nw::g3d::tool::g3dif::elem_filter::enum_max_aniso maxAniso )
{
    return g_GfxAnisotropyTbl[ static_cast< uint32_t >( maxAniso ) ];
}

inline nn::gfx::TextureAddressMode GetGfxTextureAddressMode( nw::g3d::tool::g3dif::elem_wrap::enum_mode textureAddrMode )
{
    NN_SDK_ASSERT_RANGE( static_cast< int >( textureAddrMode ), static_cast< int >( 0 ),
        static_cast< int >( sizeof( g_GfxTextureAddrModeTbl ) / sizeof( nn::gfx::TextureAddressMode ) ) );

    return g_GfxTextureAddrModeTbl[ static_cast< int >( textureAddrMode ) ];
}

inline Bit32 GetGfxFilterMode( const nw::g3d::tool::g3dif::elem_filter& elemFilter )
{
    auto magFilter = elemFilter.mag.value;
    auto minFilter = elemFilter.min.value;
    auto mipFilter = elemFilter.mip.value;
    auto maxAniso  = elemFilter.max_aniso.value;

    if( maxAniso >= nw::g3d::tool::g3dif::elem_filter::aniso_2 )
    {
        return nn::gfx::FilterMode_Anisotropic;
    }

    // 範囲チェック。mag, min は none は許可していない
    NN_SDK_ASSERT_RANGE(magFilter, nw::g3d::tool::g3dif::elem_filter::point, nw::g3d::tool::g3dif::elem_filter::linear + 1);
    NN_SDK_ASSERT_RANGE(minFilter, nw::g3d::tool::g3dif::elem_filter::point, nw::g3d::tool::g3dif::elem_filter::linear + 1);
    NN_SDK_ASSERT_RANGE(mipFilter, nw::g3d::tool::g3dif::elem_filter::none, nw::g3d::tool::g3dif::elem_filter::linear + 1);

    Bit32 filterMode = 0;
    filterMode |= g_GfxFilterModeBitTbl[magFilter] << nn::gfx::detail::FilterModeBit_MagFilterShift;
    filterMode |= g_GfxFilterModeBitTbl[minFilter] << nn::gfx::detail::FilterModeBit_MinFilterShift;
    filterMode |= g_GfxFilterModeBitTbl[mipFilter] << nn::gfx::detail::FilterModeBit_MipFilterShift;

    return filterMode;
}

template< typename TAttributeType >
inline float GetGfxLodBias( TAttributeType lodBias )
{
    return lodBias.value;	// TODO: 現状 g3dif が取る値 -32.f～31.9f をそのまま返すようにする。中間ファイルフォーマットの方で gfx に合わせた値を設定すべき。
}

template< typename TAttributeType >
inline float GetGfxLodValue( TAttributeType lodValue )
{
    return lodValue.value;	//< TODO: 現状 g3dif が取る値 -32.f～31.9f をそのまま返すようにする。中間ファイルフォーマットの方で gfx に合わせた値を設定すべき。
}

//----------------------------------------------------------------------------------------------

const nn::gfx::PrimitiveTopology NotSupportedPrimitiveTopology = nn::gfx::PrimitiveTopology_End;
const nn::gfx::PrimitiveTopology g_GfxPrimitiveTopologyTbl[] =
{
    nn::gfx::PrimitiveTopology_TriangleList,		//!< triangles
    NotSupportedPrimitiveTopology,					//!< triangle_fan はサポートされていません。
    nn::gfx::PrimitiveTopology_TriangleStrip		//!< triangle_strip
};

inline nn::gfx::PrimitiveTopology GetGfxPrimitiveTopology( uint32_t idx )
{
    NN_SDK_ASSERT_RANGE( idx, static_cast< uint32_t >( 0 ), sizeof( g_GfxPrimitiveTopologyTbl ) / sizeof( nn::gfx::PrimitiveTopology ) );
    NN_SDK_ASSERT( g_GfxPrimitiveTopologyTbl[ idx ] != NotSupportedPrimitiveTopology,
        "Primitive Topology Not Supported\n" );
    return g_GfxPrimitiveTopologyTbl[ idx ];
}

const nn::gfx::PrimitiveTopology g_GfxPrimitiveTopologyAdjacencyTbl[] =
{
    nn::gfx::PrimitiveTopology_TriangleListAdjacency,	//!< triangles
    NotSupportedPrimitiveTopology,						//!< triangle_fan はサポートされていません。
    nn::gfx::PrimitiveTopology_TriangleStripAdjacency   //!< triangle_strip
};

inline nn::gfx::PrimitiveTopology GetGfxPrimitiveTopologyAdjacency( uint32_t idx )
{
    NN_SDK_ASSERT_RANGE( idx, static_cast< uint32_t >( 0 ), sizeof( g_GfxPrimitiveTopologyAdjacencyTbl ) / sizeof( nn::gfx::PrimitiveTopology ) );
    NN_SDK_ASSERT( g_GfxPrimitiveTopologyAdjacencyTbl[ idx ] != NotSupportedPrimitiveTopology,
        "Gfx Primitive Topology Adjacency Not Supported\n" );
    return g_GfxPrimitiveTopologyAdjacencyTbl[ idx ];
}

const nn::gfx::IndexFormat g_GfxIdxFmtTbl[] =
{
    nn::gfx::IndexFormat_Uint32,	//!< none
    nn::gfx::IndexFormat_Uint16,	//!< uint16
    nn::gfx::IndexFormat_Uint32		//!< uint32
};

inline nn::gfx::IndexFormat GetGfxIdxFmt( uint32_t idx )
{
    NN_SDK_ASSERT_RANGE( idx, static_cast< uint32_t >( 0 ), sizeof( g_GfxIdxFmtTbl ) / sizeof( nn::gfx::IndexFormat ) );
    NN_SDK_ASSERT( g_GfxIdxFmtTbl[ idx ] != NotSupportedPrimitiveTopology,
        "Gfx Index Not Supported\n" );
    return g_GfxIdxFmtTbl[ idx ];
}

}
}
}
