﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstdio>
#include <clocale>
#include <memory>
#include <algorithm>
#include <sstream>
#include <vector>

#include <nn/util/util_BinaryFormat.h>

#include <util/types.h>
#include <util/UtilPrint.h>
#include <util/UtilCmdArgs.h>
#include <util/UtilBuffer.h>
#include <util/UtilTime.h>
#include <util/UtilString.h>
#include <util/UtilError.h>
#include "binlib.h"

#include <nn/gfxTool/gfxTool_CommandLineParser.h>
#include "CommandLineParserWrapper.h"
#include <iostream>

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <Windows.h>
#include <crtdbg.h>

#define CVTR_DLL L"3dBinaryConverter.dll"
#define CVTR_DLL32 L"x86\\3dBinaryConverter.dll"
#define PROC_ADDRESS(handle, name, type)                                                               \
auto name = reinterpret_cast<type>(GetProcAddress(handle, #name));                                     \
    if (name == nullptr)                                                                               \
    {                                                                                                  \
        PRINT_ERROR_LOG("Not found function %s in %hs.", L#name, util::GetModuleName(handle).c_str()); \
        return EXIT_FAILURE;                                                                           \
    }                                                                                                  \

using namespace nw::g3d::tool;


void se_translator_function(unsigned int /*code*/, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; //標準C++の例外を発生させる。
}

class FindOption
{
public:
    FindOption(std::wstring name)
        : m_Name(name)
    {}

    bool operator()(const CmdOption& opt)
    {
        return opt.name == m_Name;
    }

private:
    std::wstring m_Name;
};

int Convert(CmdArgs& args, HINSTANCE hDLL, const nn::g3dTool::CommandLineParserWrapper& parserWrapper );

int wmain(int argc, const wchar_t* argv[])
{
    util::Tick startTime = util::Tick::GetSystemCurrent();

    _CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
    _CrtSetBreakAlloc(-1);

    // gfxtool によるロガーの初期化
    nn::gfxTool::Logger::SetDefaultStream();

    setlocale(LC_CTYPE, "");

    // 構造化例外の変換関数。
    _set_se_translator(se_translator_function); // このスレッドでのみ有効。

    int result = EXIT_SUCCESS;

    try
    {
        // パーサラッパーに CmdArgs を生成して貰う。
        nn::g3dTool::CommandLineParserWrapper	parserWrapper( argc, argv );
        std::shared_ptr< CmdArgs > pCmdArgs = parserWrapper.GetCmdArgs();

        CmdArgs& args = *pCmdArgs;

        HINSTANCE hDLL = nullptr;
#if defined(_M_IX86)
        if (hDLL == nullptr)
        {
            hDLL = LoadLibrary(CVTR_DLL32);
        }
#else
        if (hDLL == nullptr)
        {
            hDLL = LoadLibrary(CVTR_DLL);
        }
#endif
        if (hDLL == nullptr)
        {
            PRINT_ERROR_LOG("failed to load dll.");
            return EXIT_FAILURE;
        }
        struct Deleter
        {
            HINSTANCE* m_pHinstance;
            Deleter(HINSTANCE* pHinstance) : m_pHinstance(pHinstance) {}
            ~Deleter()
            {
                std::string moduleName = util::GetModuleName(*m_pHinstance);
                BOOL ret = FreeLibrary(*m_pHinstance);
                if (!ret)
                {
                    PRINT_ERROR_LOG("failed to release dll ( %hs ).", moduleName.c_str());
                }
                *m_pHinstance = nullptr;
            }
        } deleter(&hDLL);

        parserWrapper.LoadDllFuncs( hDLL );
        nn::g3dTool::CommandLineParserWrapper::DllFuncs dllFuncs = parserWrapper.GetDllFuncs();

        // サイレントモードの設定
        auto silentOpt = find_if(args.options.begin(), args.options.end(), FindOption(OPT_SILENT_FULL));
        if (silentOpt != args.options.end())
        {
            nw::g3d::tool::LogConfig::s_Level = nw::g3d::tool::LogConfig::Silent;
        }

        PRINT_LOG("Convert start.");

        // バージョン取得
        uint32_t version = ( *dllFuncs.pFuncGetCvtrVersion )();
        PRINT_LOG("Converter version: %d.%d.%d.%d",
                    version & 0xFF, (version >> 8) & 0xFF, (version >> 16) & 0xFF, (version >> 24) & 0xFF );

        nn::util::BinVersion binVersion;
        version = ( *dllFuncs.pFuncGetBinVersion )();
        if( version != 0 )
        {
            binVersion.SetPacked( version );
            PRINT_LOG("Binary version:    %d.%d.%d", binVersion.major, binVersion.minor, binVersion.micro );
        }

        // バージョンを表示して終了する。
        auto vOpt = find_if(args.options.begin(), args.options.end(), FindOption(OPT_VERSION_FULL));
        if (vOpt != args.options.end())
        {
            return EXIT_SUCCESS;
        }

        auto hOpt = find_if(args.options.begin(), args.options.end(), FindOption(OPT_HELP_FULL));
        if (hOpt != args.options.end())
        {
            const char* toolName = "3dBinaryConverter";
            const char* toolDescription = "This tool serializes g3d intermediate files.";
            parserWrapper.PrintHelp( toolName, toolDescription );
            return EXIT_SUCCESS;
        }

        // 入力ファイルが存在しない場合はエラーで終了する。
        if(args.inputs.size() == 0)
        {
            PRINT_ERROR_LOG("Error: No input files. Need to input one or more fmd files.\n");
            return EXIT_FAILURE;
        }
        result = Convert(args, hDLL, parserWrapper );
    }
    CATCH_ERROR_ALL(EXIT_FAILURE)

    if (result == EXIT_SUCCESS)
    {
        // 正常終了した場合のみログ出力
        util::Tick endTime = util::Tick::GetSystemCurrent();
        util::TimeSpan convertSpan = endTime - startTime;

        util::s32 sec = static_cast<util::s32>(convertSpan.GetSeconds());
        util::s32 ms = static_cast<util::s32>(convertSpan.GetMilliSeconds());

        PRINT_LOG("Convert finish: %ld.%ld sec.", sec, ms - (sec * 1000));
    }

    return result;
}

int Convert(CmdArgs& args, HINSTANCE hDLL, const nn::g3dTool::CommandLineParserWrapper& parserWrapper )
{
    NN_UNUSED( hDLL );

    bool success = true;
    // 初期化
    {
        success = parserWrapper.GetDllFuncs().pFuncInitialize();
        if (!success)
        {
            PRINT_ERROR_LOG("Init failed.");
            return EXIT_FAILURE;
        }
    }

    auto conv = [](const std::vector<CmdOption>& options) -> const wchar_t**
    {
        auto count = options.size();
        const wchar_t** optionArray = new const wchar_t*[count * 3 + 1];
        int index = 0;
        for (auto iter = options.cbegin(); iter != options.cend(); ++iter)
        {
            optionArray[index++] = iter->name.c_str();
            optionArray[index++] = iter->value.c_str();
        }
        optionArray[index] = nullptr;
        return optionArray;
    };

    // グローバルオプション
    std::wstring outpath; // 出力パス
    PRINT_LOG(LOG_LEVEL_0(L"Global Options"));
    {
        // 出力パスの取得
        int index = -1;
        index = CmdOption::FindIndex(args.options, OPT_OUTPUT_FILE_FULL);

        if (index < 0)
        {
            if (args.inputs.size() == 1)
            {
                // 入力ファイルが1つの場合に限り出力ファイルの指定を省略できる。
                // --output=inputfile.bfres を追加する。
                const CmdInput& input = args.inputs.front();
                auto lenExt = input.ext.size();
                const std::wstring& fullpath = input.fullpath;
                std::wstringstream wss;
                outpath.assign(fullpath.substr(0, fullpath.size() - lenExt));
                // --output=
                wss << OPT_OUTPUT_FILE_FULL << L"=";
                // inputfile.bfres
                wss << outpath.c_str() << OUTPUT_FILE_EXT;
                args.options.push_back(CmdOption(wss.str().c_str()));
                index = static_cast<int>(args.options.size()) - 1;
            }
            else
            {
                PRINT_ERROR_LOG("Not found output path.");
                return EXIT_FAILURE;
            }
        }

        // 拡張子抽出
        std::wstring ext;
        wchar_t nameBuf[_MAX_FNAME];
        wchar_t extBuf[_MAX_EXT];
        wchar_t fullpathBuf[_MAX_PATH];
        std::wstring outputPath;

        if (index >= 0)
        {
            if (_wsplitpath_s(args.options[index].value.c_str(), nullptr, 0, nullptr, 0, nameBuf, _MAX_FNAME, extBuf, _MAX_EXT) == 0)
            {
                ext.assign(extBuf);
            }

            outputPath = args.options[index].value;
            if (ext.empty())
            {
                // 出力先ファイルの拡張子がない場合、.bfres をつけて出力する。
                outputPath += OUTPUT_FILE_EXT;
            }

            if(_wfullpath(fullpathBuf, outputPath.c_str(), _MAX_PATH))
            {
                outpath.assign(fullpathBuf);
            }
            else
            {
                PRINT_ERROR_LOG("Output path is too long.");
                return EXIT_FAILURE;
            }
        }

        {
            std::unique_ptr<const wchar_t*[]> options(conv(args.options));
            // オプションの登録。
            success = parserWrapper.GetDllFuncs().pFuncSetOptions( options.get() );
            if (!success)
            {
                PRINT_ERROR_LOG("SetOptions() failed.");
                return EXIT_FAILURE;
            }
        }
    }

    // 各入力ファイル。
    PRINT_LOG(LOG_LEVEL_0(L"Input Files"));
    for (auto iter = args.inputs.cbegin(); iter != args.inputs.cend(); ++iter)
    {
        util::Buffer file;
        {
            success = file.LoadFile(iter->fullpath.c_str());
            if (!success)
            {
                PRINT_ERROR_LOG("Load file failed.");
                PRINT_ERROR_LOG("File: %s", iter->fullpath.c_str());
                return EXIT_FAILURE;
            }
        }

        const wchar_t* path[3];
        path[0] = iter->fullpath.c_str();
        path[1] = iter->fname.c_str();
        path[2] = iter->ext.c_str();
        std::unique_ptr<const wchar_t*[]> options(conv(iter->options));

        {
            success = parserWrapper.GetDllFuncs().pFuncAddFile(file.Get(), file.Size(), path, options.get());
            if (!success)
            {
                PRINT_ERROR_LOG("AddFile() failed.");
                PRINT_ERROR_LOG("File: %s", iter->fullpath.c_str());
                return EXIT_FAILURE;
            }
        }
    }

    // テストモードが有効な場合は中間ファイルのチェックが終わった段階で終了する。
    if (CmdOption::FindIndex(args.options, OPT_VERIFY_FULL) >= 0)
    {
        return EXIT_SUCCESS;
    }

    // 出力ファイルサイズの計算
    size_t size = parserWrapper.GetDllFuncs().pFuncCalculateSize();
    if (size == 0)
    {
        PRINT_ERROR_LOG("CalcSize() failed.");
        return EXIT_FAILURE;
    }
    size_t alignment = parserWrapper.GetDllFuncs().pFuncGetAlignmentSize();
    if (alignment == 0)
    {
        PRINT_ERROR_LOG("GetAlignmentSize() failed.");
        return EXIT_FAILURE;
    }
    util::Buffer outfile;
    outfile.Alloc(size, alignment);

    if (outfile.Get() == nullptr)
    {
        PRINT_ERROR_LOG("Alloc() failed.");
        return EXIT_FAILURE;
    }

    // 変換処理
    {
        success = parserWrapper.GetDllFuncs().pFuncConvert(outfile.Get(), outfile.Size());
        if (!success)
        {
            PRINT_ERROR_LOG("Convert() failed.");
            return EXIT_FAILURE;
        }
    }

    // バインド
    {
        success = parserWrapper.GetDllFuncs().pFuncBind(outfile.Get(), outfile.Size());
        if (!success)
        {
            PRINT_ERROR_LOG("Bind() failed.");
            return EXIT_FAILURE;
        }
    }

    // エンディアンスワップ
    {
        success = parserWrapper.GetDllFuncs().pFuncSwapEndian(outfile.Get(), outfile.Size());
        if (!success)
        {
            PRINT_ERROR_LOG("EndianSwap() failed.");
            return EXIT_FAILURE;
        }
    }

    // 出力ファイルの書き出し
    {
        success = outfile.SaveFile(outpath.c_str());
        if (!success)
        {
            PRINT_ERROR_LOG("Output file failed.");
            PRINT_ERROR_LOG("File: %s", outpath.c_str());
            return EXIT_FAILURE;
        }
    }

    // 初期化
    {
        success = parserWrapper.GetDllFuncs().pFuncShutdown();
        if (!success)
        {
            PRINT_ERROR_LOG("Shutdown failed.");
            return EXIT_FAILURE;
        }
    }

    return EXIT_SUCCESS;
}
