﻿namespace G3dCore.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Converters;
    using G3dCore.Entities;
    using G3dCore.IO;

    /// <summary>
    /// シェーダコンバータのユーティリティクラスです。
    /// </summary>
    public static class ShaderConverterUtility
    {
        /// <summary>
        /// シェーダコンバータでシェーダ定義を生成し、バリエーションに関する情報を取得します。
        /// </summary>
        /// <param name="baseShaderConfig"></param>
        /// <param name="targetShader"></param>
        /// <param name="shaderConfigDir"></param>
        /// <param name="shaderDefinition"></param>
        /// <returns></returns>
        public static ShaderConverterResult GenerateShaderDefinitionFromShaderConfig(
            ShaderConfig baseShaderConfig, Shader targetShader, string shaderConfigDir)
        {
            Debug.Assert(baseShaderConfig != null);
            Debug.Assert(targetShader != null);
            Debug.Assert(!string.IsNullOrWhiteSpace(shaderConfigDir));

            ShaderConverterResult resultForId = null;
            ShaderConverterResult resultForValue = null;

            string temporaryDir = TemporaryFileUtility.GetTemporaryDir();
            string temporaryShaderDefName = TemporaryFileUtility.MakeTemporaryName();

            // (1) バリエーションがない fsc を作成し、シェーダコンバータで fsd に変換する。
            //     -> fsd から バリエーションの id、branch を取得。（この時のエラーログの warning は出力しない。）
            {
                G3dFile shaderConfigFile = new G3dFile(G3dKind.ShaderConfig);
                ShaderConfig shaderConfig = shaderConfigFile.GetRootEntity<ShaderConfig>();
                shaderConfig.CreateEmptyShaderConfig(baseShaderConfig);

                Shader shader = targetShader.Clone() as Shader;
                shader.Variations.Clear();  // バリエーションなし。
                shaderConfig.Shaders.Add(shader);

                string fscaFilePath = shaderConfigDir + @"\temp.fsca";
                string fsdbFilePath = Path.Combine(temporaryDir, temporaryShaderDefName + "_0.fsdb");

                resultForId = ShaderConverterManager.ConvertToShaderDefinition(
                    shaderConfigFile.CreateBinaryData(),
                    fscaFilePath,
                    fsdbFilePath,
                    new string[] { "--auto_extract", null, "--unified_annotation", null });

                if (resultForId.HasError)
                {
                    return resultForId;
                }
            }

            // (2) choice="*"、default="" のバリエーションで fsc を作成し、シェーダコンバータで fsd に変換する。
            //     -> fsd から choice、default の値を取得。
            {
                ShaderDefinition shaderDefinitionForId = resultForId.GetShaderDefinition();

                ShadingModel shadingModel = shaderDefinitionForId.ShadingModels.FirstOrDefault();
                Debug.Assert(shadingModel != null);

                if (shadingModel.OptionVars.FirstOrDefault() != null)
                {
                    G3dFile shaderConfigFile = new G3dFile(G3dKind.ShaderConfig);
                    ShaderConfig shaderConfig = shaderConfigFile.GetRootEntity<ShaderConfig>();
                    shaderConfig.CreateEmptyShaderConfig(baseShaderConfig);

                    Shader shader = targetShader.Clone() as Shader;
                    shader.Variations.Clear();
                    shaderConfig.Shaders.Add(shader);

                    int index = 0;
                    foreach (var optionVar in shadingModel.OptionVars)
                    {
                        Variation variation = new Variation();
                        variation.Index = index;
                        variation.Id = optionVar.Id;
                        variation.Choice = "*";
                        variation.Default = "";
                        variation.Branch = optionVar.Branch;
                        variation.IsOutputable = true;
                        shader.Variations.Add(variation);

                        ++index;
                    }

                    string fscaFilePath = shaderConfigDir + @"\temp.fsca";
                    string fsdbFilePath = Path.Combine(temporaryDir, temporaryShaderDefName + "_1.fsdb");

                    resultForValue = ShaderConverterManager.ConvertToShaderDefinition(
                        shaderConfigFile.CreateBinaryData(),
                        fscaFilePath,
                        fsdbFilePath,
                        new string[] { "--auto_extract", null, "--unified_annotation", null }
                     );
                }
                else
                {
                    resultForValue = resultForId;
                }
            }

            return resultForValue;
        }

        /// <summary>
        /// シェーダ設定からシェーダコンバートが行えるかどうか試します。
        /// </summary>
        /// <param name="shaderConfigFile">コンバートするシェーダ設定ファイルです。</param>
        /// <param name="shaderConfigDir">シェーダ設定ファイルのあるディレクトリです。</param>
        /// <param name="result">コンバートの結果を出力します。</param>
        /// <returns>シェーダコンバートが行えた場合、 true を返します。</returns>
        public static bool TryShaderConvert(
            G3dFile shaderConfigFile, string shaderConfigDir, out ShaderConverterResult result)
        {
            Debug.Assert(shaderConfigFile != null);
            Debug.Assert(shaderConfigFile.Kind == G3dKind.ShaderConfig);
            Debug.Assert(!string.IsNullOrWhiteSpace(shaderConfigDir));

            var shaderConfig = shaderConfigFile.GetRootEntity<ShaderConfig>();
            if (shaderConfig.Shaders.FirstOrDefault() == null)
            {
                result = null;
                return true;
            }

            string temporaryDir = TemporaryFileUtility.GetTemporaryDir();
            string temporaryShaderDefName = TemporaryFileUtility.MakeTemporaryName();
            string fscaFilePath = shaderConfigDir + @"\temp.fsca";
            string fsdbFilePath = Path.Combine(temporaryDir, temporaryShaderDefName + ".fsdb");

            result = ShaderConverterManager.ConvertToShaderDefinition(
                shaderConfigFile.CreateBinaryData(),
                fscaFilePath,
                fsdbFilePath,
                new string[] { "--auto_extract", null, "--unified_annotation", null }
             );

            return result.GetShaderDefinition() != null;
        }
    }
}
