﻿namespace G3dCore.Entities
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using nw.g3d.nw4f_3dif;
    using Opal.Security.Cryptography;

    /// <summary>
    /// シェーダ設定のエンティティクラスです。
    /// </summary>
    public class ShaderConfig : RootEntity<shader_configType>
    {
        private readonly ObservableCollection<IncludePath> includePaths = new ObservableCollection<IncludePath>();
        private readonly ObservableCollection<ForceIncludeFile> forceIncludeFiles = new ObservableCollection<ForceIncludeFile>();
        private readonly ObservableCollection<Shader> shaders = new ObservableCollection<Shader>();
        private ShaderConfigInfo shaderConfigInfo = new ShaderConfigInfo();
        private G3dFile file = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderConfig()
        {
            this.includePaths.CollectionChanged += this.OnCollectionChanged<IncludePath>;
            this.forceIncludeFiles.CollectionChanged += this.OnCollectionChanged<ForceIncludeFile>;
            this.shaders.CollectionChanged += this.OnCollectionChanged<Shader>;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="data">設定するデータです。</param>
        /// <param name="file">データをもつ中間ファイルです。</param>
        public ShaderConfig(shader_configType data, G3dFile file)
            : this()
        {
            Debug.Assert(data != null);
            Debug.Assert(file != null);

            this.file = file;

            if (data.shader_config_info != null)
            {
                this.shaderConfigInfo = new ShaderConfigInfo(data.shader_config_info);
            }

            if (data.include_path_array != null)
            {
                this.AddIncludePaths(data.include_path_array.include_path);
            }

            if (data.force_include_file_array != null)
            {
                this.AddForceIncludeFiles(data.force_include_file_array.force_include_file);
            }

            if (data.shader_array != null)
            {
                this.AddShaders(data.shader_array.shader);
            }
        }

        /// <summary>
        /// データをもつ中間ファイルを取得します。
        /// </summary>
        public G3dFile File
        {
            get
            {
                return this.file;
            }
        }

        /// <summary>
        /// シェーダ設定情報を取得します。
        /// </summary>
        public ShaderConfigInfo ShaderConfigInfo
        {
            get
            {
                return this.shaderConfigInfo;
            }
        }

        /// <summary>
        /// インクルードパスを取得します。
        /// </summary>
        public ObservableCollection<IncludePath> IncludePaths
        {
            get
            {
                return this.includePaths;
            }
        }

        /// <summary>
        /// 強制インクルードファイルを取得します。
        /// </summary>
        public ObservableCollection<ForceIncludeFile> ForceIncludeFiles
        {
            get
            {
                return this.forceIncludeFiles;
            }
        }

        /// <summary>
        /// シェーダ設定を取得します。
        /// </summary>
        public ObservableCollection<Shader> Shaders
        {
            get
            {
                return this.shaders;
            }
        }

        /// <summary>
        /// 設定情報のみコピーした空の ShaderConfig を生成します。
        /// </summary>
        /// <param name="original">元となる設定情報のある ShaderConfig です。</param>
        /// <returns></returns>
        public void CreateEmptyShaderConfig(ShaderConfig original)
        {
            Debug.Assert(original != null);

            if (original.shaderConfigInfo != null)
            {
                this.shaderConfigInfo = original.shaderConfigInfo.Clone() as ShaderConfigInfo;
            }

            this.includePaths.Clear();
            foreach (var includePath in original.includePaths)
            {
                this.includePaths.Add(includePath.Clone() as IncludePath);
            }

            this.forceIncludeFiles.Clear();
            foreach (var forceIncludeFile in original.forceIncludeFiles)
            {
                this.forceIncludeFiles.Add(forceIncludeFile.Clone() as ForceIncludeFile);
            }

            this.shaders.Clear();
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>現在のインスタンスのコピーである新しいオブジェクトを返します。</returns>
        public override IEntity Clone()
        {
            var instance = new ShaderConfig();
            instance.shaderConfigInfo = this.shaderConfigInfo.Clone() as ShaderConfigInfo;

            foreach (var includePath in this.includePaths)
            {
                instance.includePaths.Add(includePath.Clone() as IncludePath);
            }

            foreach (var forceIncludeFile in this.forceIncludeFiles)
            {
                instance.forceIncludeFiles.Add(forceIncludeFile.Clone() as ForceIncludeFile);
            }

            foreach (var shader in this.shaders)
            {
                instance.shaders.Add(shader.Clone() as Shader);
            }

            return instance;
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_configType CreateWriteData()
        {
            var instance = new shader_configType();
            instance.shader_config_info = this.shaderConfigInfo.CreateWriteData();
            instance.include_path_array = this.CreateIncludePathArray();
            instance.force_include_file_array = this.CreateForceIncludeFileArray();
            instance.shader_array = this.CreateShaderArray();

            return instance;
        }

        /// <summary>
        /// エンティティのCRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRCの値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.GetHashCode())));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.shaderConfigInfo.GetHashCode())));

            foreach (var includePath in this.includePaths)
            {
                buffers.AddRange(BitConverter.GetBytes(includePath.HashValue));
            }

            foreach (var forceIncludeFile in this.forceIncludeFiles)
            {
                buffers.AddRange(BitConverter.GetBytes(forceIncludeFile.HashValue));
            }

            foreach (var shader in this.shaders)
            {
                buffers.AddRange(BitConverter.GetBytes(shader.HashValue));
            }

            var hashValue = crc.ComputeHashUInt32(buffers.ToArray());
            return hashValue;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var includePath in this.includePaths)
            {
                includePath.Reset();
            }

            foreach (var forceIncludeFile in this.forceIncludeFiles)
            {
                forceIncludeFile.Reset();
            }

            foreach (var shader in this.shaders)
            {
                shader.Reset();
            }

            if (this.shaderConfigInfo != null)
            {
                this.shaderConfigInfo.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var includePath in this.includePaths)
            {
                includePath.Refresh();
            }

            foreach (var forceIncludeFile in this.forceIncludeFiles)
            {
                forceIncludeFile.Refresh();
            }

            foreach (var shader in this.shaders)
            {
                shader.Refresh();
            }

            if (this.shaderConfigInfo != null)
            {
                this.shaderConfigInfo.Refresh();
            }
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var includePath in this.includePaths)
            {
                includePath.AutoCalc = this.AutoCalc;
            }

            foreach (var forceIncludeFile in this.forceIncludeFiles)
            {
                forceIncludeFile.AutoCalc = this.AutoCalc;
            }

            foreach (var shader in this.shaders)
            {
                shader.AutoCalc = this.AutoCalc;
            }

            if (this.shaderConfigInfo != null)
            {
                this.shaderConfigInfo.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// インクルードパス配列のデータを作成します。
        /// </summary>
        /// <returns>データのインスタンスを返します。</returns>
        protected include_path_arrayType CreateIncludePathArray()
        {
            if (this.includePaths.FirstOrDefault() == null)
            {
                return null;
            }

            var includePathArray = new include_path_arrayType();
            List<include_pathType> includePathTypes = new List<include_pathType>();

            int index = 0;
            foreach (var includePath in this.includePaths)
            {
                includePath.Index = index;
                ++index;
                includePathTypes.Add(includePath.CreateWriteData());
            }

            includePathArray.include_path = includePathTypes.ToArray();
            includePathArray.length = includePathTypes.Count;
            return includePathArray;
        }

        /// <summary>
        /// 強制インクルードファイル配列のデータを作成します。
        /// </summary>
        /// <returns>データのインスタンスを返します。</returns>
        protected force_include_file_arrayType CreateForceIncludeFileArray()
        {
            if (this.forceIncludeFiles.FirstOrDefault() == null)
            {
                return null;
            }

            var forceIncludeFileArray = new force_include_file_arrayType();
            List<force_include_fileType> forceIncludeFileTypes = new List<force_include_fileType>();

            int index = 0;
            foreach (var forceIncludeFile in this.forceIncludeFiles)
            {
                forceIncludeFile.Index = index;
                ++index;
                forceIncludeFileTypes.Add(forceIncludeFile.CreateWriteData());
            }

            forceIncludeFileArray.force_include_file = forceIncludeFileTypes.ToArray();
            forceIncludeFileArray.length = forceIncludeFileTypes.Count;
            return forceIncludeFileArray;
        }

        /// <summary>
        /// シェーダ設定配列のデータを作成します。
        /// </summary>
        /// <returns>データのインスタンスを返します。</returns>
        protected shader_arrayType CreateShaderArray()
        {
            if (this.shaders.FirstOrDefault() == null)
            {
                return null;
            }

            var shaderArray = new shader_arrayType();
            List<shaderType> shaderTypes = new List<shaderType>();

            int index = 0;
            foreach (var shader in this.shaders)
            {
                shader.Index = index;
                ++index;
                shaderTypes.Add(shader.CreateWriteData());
            }

            shaderArray.shader = shaderTypes.ToArray();
            shaderArray.length = shaderTypes.Count;
            return shaderArray;
        }

        /// <summary>
        /// インクルードパスを追加します。
        /// </summary>
        /// <param name="data">追加するデータです。</param>
        protected void AddIncludePaths(include_pathType[] data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var value in data)
            {
                IncludePath includePath = new IncludePath(value);
                this.includePaths.Add(includePath);
            }
        }

        /// <summary>
        /// 強制インクルードファイルを追加します。
        /// </summary>
        /// <param name="data">追加するデータです。</param>
        protected void AddForceIncludeFiles(force_include_fileType[] data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var value in data)
            {
                ForceIncludeFile forceIncludeFile = new ForceIncludeFile(value);
                this.forceIncludeFiles.Add(forceIncludeFile);
            }
        }

        /// <summary>
        /// シェーダ設定定義を追加します。
        /// </summary>
        /// <param name="data">追加するデータです。</param>
        protected void AddShaders(shaderType[] data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var value in data)
            {
                Shader shader = new Shader(value);
                this.shaders.Add(shader);
            }
        }
    }
}
