﻿namespace G3dCore.Configurations
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Xml;
    using System.Xml.Serialization;
    using G3dCore.Utilities;
    using Opal.Configurations;

    /// <summary>
    /// SubConfig を任意に追加可能なコンフィグクラスです。
    /// </summary>
    public abstract class ExtensibleConfig : Config
    {
        /// <summary>
        /// 管理している SubConfig です。
        /// </summary>
        public SubConfigs Configs
        {
            get;
            set;
        }

        /// <summary>
        /// 管理している SubConfig の数です。
        /// </summary>
        public int SubConfigsCount
        {
            get
            {
                if (this.Configs != null && this.Configs.AnyConfig != null)
                {
                    return this.Configs.AnyConfig.Count;
                }

                return 0;
            }
        }

        /// <summary>
        /// SubConfig のインスタンスを取得します。
        /// </summary>
        /// <typeparam name="T">取得する SubConfig の型です。</typeparam>
        /// <returns>取得に成功した場合、SubConfig のインスタンスを返します。</returns>
        public T GetSubConfig<T>() where T : SubConfig
        {
            if (this.Configs != null)
            {
                if (this.Configs.ConfigInstances != null)
                {
                    // 管理しているインスタンスが存在する場合にはそれを返す。
                    var subConfig = this.Configs.ConfigInstances.FirstOrDefault(x => x is T);
                    if (subConfig != null)
                    {
                        return (T)subConfig;
                    }
                }

                if (this.Configs.AnyConfig != null)
                {
                    // 管理しているインスタンスがなく、取得しようとする SubConfig がある場合には
                    // XmlElement から SubConfig のインスタンスを生成する。
                    var element = this.Configs.AnyConfig.FirstOrDefault(x => x.Name == typeof(T).Name);
                    if (element != null)
                    {
                        var subConfig = XmlUtility.ContertXmlElementToObject<T>(element);
                        this.Configs.ConfigInstances.Add(subConfig);

                        return subConfig;
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// SubConfig のインスタンスを追加します。
        /// </summary>
        /// <typeparam name="T">追加する SubConfig の型です。</typeparam>
        /// <param name="pluginConfig">追加する SubConfig です。</param>
        /// <returns>追加に成功した場合、true を返します。</returns>
        public bool AddSubConfig<T>(T subConfig) where T : SubConfig
        {
            Debug.Assert(subConfig != null);

            if (this.Configs == null)
            {
                this.Configs = new SubConfigs();
            }

            var existingSubConfig = this.Configs.ConfigInstances.FirstOrDefault(x => x is T);
            if (existingSubConfig != null)
            {
                // 同型のプラグインコンフィグは複数追加できません。
                return false;
            }

            this.Configs.ConfigInstances.Add(subConfig);

            return this.WriteSubConfig<T>();
        }

        /// <summary>
        /// SubConfig を書き込みます。
        /// ファイルへの書き込みは Save() を使用します。
        /// </summary>
        /// <typeparam name="T">書き込む SubConfig の型です。</typeparam>
        /// <returns>書き込みに成功した場合、true を返します。</returns>
        public bool WriteSubConfig<T>() where T : SubConfig
        {
            if (this.Configs == null)
            {
                return false;
            }

            T subConfig = (T)this.Configs.ConfigInstances.FirstOrDefault(x => x is T);
            if (subConfig != null)
            {
                if (this.Configs.AnyConfig == null)
                {
                    this.Configs.AnyConfig = new List<XmlElement>();
                }

                var element = this.Configs.AnyConfig.FirstOrDefault(x => x.Name == subConfig.GetType().Name);
                if (element != null)
                {
                    this.Configs.AnyConfig.Remove(element);
                }

                this.Configs.AnyConfig.Add(XmlUtility.ConvertObjectToXmlElement(subConfig, 1));

                return true;
            }

            return false;
        }

        /// <summary>
        /// SubConfig を削除します。
        /// </summary>
        /// <typeparam name="T">削除する SubConfig の型です。</typeparam>
        /// <returns>削除に成功した場合、true を返します。</returns>
        public bool RemoveSubConfig<T>() where T : SubConfig
        {
            bool result = false;

            if (this.Configs != null)
            {
                {
                    var subConfig = this.Configs.ConfigInstances.FirstOrDefault(x => x is T);
                    if (subConfig != null)
                    {
                        result |= this.Configs.ConfigInstances.Remove(subConfig);
                    }
                }

                if (this.Configs.AnyConfig != null)
                {
                    var element = this.Configs.AnyConfig.FirstOrDefault(x => x.Name == typeof(T).Name);
                    if (element != null)
                    {
                        result |= this.Configs.AnyConfig.Remove(element);
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// すべての SubConfig を書き込みます。
        /// </summary>
        internal void WriteSubConfigs()
        {
            if (this.Configs == null)
            {
                return;
            }

            foreach (var subConfig in this.Configs.ConfigInstances)
            {
                if (this.Configs.AnyConfig == null)
                {
                    this.Configs.AnyConfig = new List<XmlElement>();
                }

                var element = this.Configs.AnyConfig.FirstOrDefault(x => x.Name == subConfig.GetType().Name);
                if (element != null)
                {
                    this.Configs.AnyConfig.Remove(element);
                }

                this.Configs.AnyConfig.Add(XmlUtility.ConvertObjectToXmlElement(subConfig));
            }
        }

#if false
        /// <summary>
        /// すべての SubConfig のすべてのプロパティ変更を通知します。
        /// </summary>
        internal void RaiseAllPropertyChanged()
        {
            foreach (var config in this.Configs.ConfigInstances)
            {
                config.RaiseAllPropertyChanged();
            }
        }

        /// <summary>
        /// SubConfig のインスタンスを最後に書き込んだ状態にリセットします。
        /// </summary>
        internal void ResetSubConfigs()
        {
            foreach (var config in this.Configs.ConfigInstances)
            {
                Type type = config.GetType();
                string targetTypeName = type.Name;
                var element = this.Configs.AnyConfig.FirstOrDefault(x => x.Name == targetTypeName);
                if (element == null)
                {
                    continue;
                }

                var subConfig = IfDomUtility.ConvertXmlElementToObject(element, type) as SubConfig;
                subConfig.CopyAllPropertiesTo(config);
            }
        }
#endif

        /// <summary>
        /// SubConfig を管理するクラスです。
        /// </summary>
        public sealed class SubConfigs
        {
            private readonly List<SubConfig> configInstances = new List<SubConfig>();

            /// <summary>
            /// SubConfig のエレメントのリストです。
            /// </summary>
            [XmlAnyElementAttribute]
            public List<XmlElement> AnyConfig { get; set; }

            /// <summary>
            /// SubConfig のインスタンスのリストです。
            /// </summary>
            internal List<SubConfig> ConfigInstances
            {
                get
                {
                    return this.configInstances;
                }
            }
        }
    }
}
