﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Data;
    using ShaderAssistAddons.Modules.ShaderConfig.Commands;
    using ShaderAssistAddons.Modules.ShaderConfig.Utilities;
    using G3dCore.Entities;
    using G3dCore.Configurations.TeamConfigs;
    using G3dCore.Converters;
    using G3dCore.Modules.ErrorView;
    using G3dCore.Resources;
    using G3dCore.Utilities;
    using G3dCore.ViewModels;
    using G3dCore.Windows.Utilities;
    using Opal.App;
    using Opal.Operations;
    using Opal.ViewModels;

    /// <summary>
    /// シェーダ設定定義のビューモデルクラスです。
    /// </summary>
    public class ShaderViewModel : ViewModel
    {
        private readonly ObservableCollection<VariationViewModel> variationViewModels = new ObservableCollection<VariationViewModel>();
        private WeakReference<Shader> shaderData;
        private ShaderConfigViewModel parentShaderConfigViewModel;
        private string name = string.Empty;
        private ShaderDefinition shaderDefinition = null;
        private SortDescriptionType[] variationSortDescriptions = new SortDescriptionType[(int)VariationColumnTypes.MaxCount];
        private SortDescriptionType[] macroSortDescriptions = new SortDescriptionType[(int)MacroColumnTypes.MaxCount];

        /// <summary>
        /// バリエーションの列のタイプです。
        /// </summary>
        public enum VariationColumnTypes
        {
            /// <summary>
            /// 出力
            /// </summary>
            Output,

            /// <summary>
            /// ID
            /// </summary>
            Id,

            /// <summary>
            /// choice
            /// </summary>
            Choice,

            /// <summary>
            /// シェーダ定義の choice
            /// </summary>
            ChoiceShaderDefinition,

            /// <summary>
            /// デフォルト
            /// </summary>
            Default,

            /// <summary>
            /// シェーダ定義のデフォルト
            /// </summary>
            DefaultShaderDefinition,

            /// <summary>
            /// 静的分岐
            /// </summary>
            Branch,

            /// <summary>
            /// 選択オプション
            /// </summary>
            SelectedOption,

            /// <summary>
            /// 最大値
            /// </summary>
            MaxCount
        }

        /// <summary>
        /// マクロの列のタイプです。
        /// </summary>
        public enum MacroColumnTypes
        {
            /// <summary>
            /// 名前
            /// </summary>
            Name,

            /// <summary>
            /// 値
            /// </summary>
            Value,

            /// <summary>
            /// 最大値
            /// </summary>
            MaxCount
        }

        private enum SortDescriptionType
        {
            Default,
            Ascending,
            Descending,
            MaxCount
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="dataModel">データモデルです。</param>
        /// <param name="parentShaderConfigViewModel">親のシェーダ設定ビューモデルです。</param>
        public ShaderViewModel(Shader dataModel, ShaderConfigViewModel parentShaderConfigViewModel)
        {
            Debug.Assert(dataModel != null);

            BindingOperations.EnableCollectionSynchronization(this.variationViewModels, this.SyncRoot);

            this.shaderData = new WeakReference<Shader>(dataModel);
            dataModel.PropertyChanged += this.ShaderPropertyChanged;

            this.parentShaderConfigViewModel = parentShaderConfigViewModel;
            this.SetShaderData(dataModel);
        }

        /// <summary>
        /// バリエーション選択変更時コマンドを取得します。
        /// </summary>
        public DelegateCommand VariationSelectionChangedCommand
        {
            get
            {
                return new DelegateCommand(this.VariationSelectionChangedExecute);
            }
        }

        /// <summary>
        /// 出力チェックボックス変更時コマンドを取得します。
        /// </summary>
        public DelegateCommand IsOutputableCheckedCommand
        {
            get
            {
                return new DelegateCommand(this.IsOutputableCheckedExecute);
            }
        }

        /// <summary>
        /// シェーダ設定定義のデータを取得します。
        /// </summary>
        public Shader ShaderData
        {
            get
            {
                Shader data;
                this.shaderData.TryGetTarget(out data);
                Debug.Assert(data != null);
                return data;
            }
        }

        /// <summary>
        /// バリエーションビューモデルを取得します。
        /// </summary>
        public ObservableCollection<VariationViewModel> VariationViewModels
        {
            get
            {
                return this.variationViewModels;
            }
        }

        /// <summary>
        /// 名前を取得設定します。
        /// </summary>
        public string Name
        {
            get
            {
                return this.name;
            }

            set
            {
                this.SetProperty(ref this.name, value);
            }
        }

        /// <summary>
        /// 抽出済みシェーダ定義を取得します。
        /// </summary>
        public ShaderDefinition ShaderDefinitionData
        {
            get
            {
                return this.shaderDefinition;
            }
        }

        /// <summary>
        /// シェーダコンバートによりシェーダ定義から OptionVar を抽出し、Variation を設定します。
        /// </summary>
        /// <returns>シェーダコンバータのログを返します。</returns>
        public ErrorViewLog ExtractVariationFromShaderDefinition()
        {
            ThreadUtility.BeginInvokeOnUIThread(
                () => { AppManager.WriteInfo(string.Format(Messages.ExtractVariations, this.name)); });

            ShaderConverterResult result = ShaderConverterUtility.GenerateShaderDefinitionFromShaderConfig(
                this.parentShaderConfigViewModel.ShaderConfigData,
                this.ShaderData,
                Path.GetDirectoryName(this.parentShaderConfigViewModel.FileViewModel.FilePath));

            ErrorViewLog log = result.CreateErrorViewLog();
            this.shaderDefinition = result.GetShaderDefinition();
            if (this.shaderDefinition != null && this.shaderDefinition.ShadingModels.FirstOrDefault() != null)
            {
                this.AddVariationsFromShaderDefinition(this.shaderDefinition);
            }
            else
            {
                this.ShaderData.Variations.Clear();
            }

            return log;
        }

        /// <summary>
        /// シェーダ定義を元に Variation を追加します。
        /// </summary>
        /// <param name="shaderDefinition">シェーダ定義です。</param>
        public void AddVariationsFromShaderDefinition(ShaderDefinition shaderDefinition)
        {
            if (shaderDefinition == null) throw new ArgumentNullException();
            this.shaderDefinition = shaderDefinition;

            IList<Variation> variations = this.ShaderData.Variations;
            int idx = 0;
            foreach (OptionVar optionVar in this.shaderDefinition.ShadingModels[0].OptionVars)
            {
                Variation variation = variations.FirstOrDefault(x => x.Id == optionVar.Id);
                if (variation == null)
                {
                    // シェーダ設定にバリエーション情報が無いので生成する。
                    variation = CreateVariationFromOptionVar(optionVar);
                    variations.Insert(idx, variation);
                }
                else if (!this.ShaderData.IsVariationsUpdated)
                {
                    variation.IsOutputable = true;
                }

                ++idx;
            }
        }

        /// <summary>
        /// Variation を必要に応じて更新して追加します。
        /// </summary>
        public void UpdateAndAddVariations()
        {
            if (this.shaderDefinition != null && this.shaderDefinition.ShadingModels.FirstOrDefault() != null)
            {
                if (!this.ShaderData.IsVariationsUpdated)
                {
                    this.UpdateVariations();
                }

                this.AddVariations();
            }
            else
            {
                this.ShaderData.Variations.Clear();
            }
        }

        /// <summary>
        /// 次のソート方法に設定します。
        /// </summary>
        /// <param name="column">設定する列です。</param>
        public void SetNextSortDescription(VariationColumnTypes column)
        {
            this.variationSortDescriptions[(int)column] =
                (SortDescriptionType)(((int)this.variationSortDescriptions[(int)column] + 1) % (int)SortDescriptionType.MaxCount);
        }

        /// <summary>
        /// 次のソート方法に設定します。
        /// </summary>
        /// <param name="column">設定する列です。</param>
        public void SetNextSortDescription(MacroColumnTypes column)
        {
            this.macroSortDescriptions[(int)column] =
                (SortDescriptionType)(((int)this.macroSortDescriptions[(int)column] + 1) % (int)SortDescriptionType.MaxCount);
        }

        /// <summary>
        /// ソート方法を設定します。
        /// </summary>
        public void SetSortDescriptions()
        {
            {
                var collectionView = (CollectionView)CollectionViewSource.GetDefaultView(this.variationViewModels);
                collectionView.SortDescriptions.Clear();

                for (int idx = 0; idx < (int)VariationColumnTypes.MaxCount; ++idx)
                {
                    string propertyName = string.Empty;
                    switch (idx)
                    {
                        case (int)VariationColumnTypes.Output:
                            propertyName = "IsOutputable";
                            break;
                        case (int)VariationColumnTypes.Id:
                            propertyName = "Id";
                            break;
                        case (int)VariationColumnTypes.Choice:
                            propertyName = "ShaderConfigChoice";
                            break;
                        case (int)VariationColumnTypes.ChoiceShaderDefinition:
                            propertyName = "ShaderDefinitionChoice";
                            break;
                        case (int)VariationColumnTypes.Default:
                            propertyName = "ShaderConfigDefault";
                            break;
                        case (int)VariationColumnTypes.DefaultShaderDefinition:
                            propertyName = "ShaderDefinitionDefault";
                            break;
                        case (int)VariationColumnTypes.Branch:
                            propertyName = "Branch";
                            break;
                        case (int)VariationColumnTypes.SelectedOption:
                            propertyName = "SelectedOption";
                            break;
                    }

                    if (this.variationSortDescriptions[idx] == SortDescriptionType.Ascending)
                    {
                        collectionView.SortDescriptions.Add(new SortDescription(propertyName, ListSortDirection.Ascending));
                    }
                    else if (this.variationSortDescriptions[idx] == SortDescriptionType.Descending)
                    {
                        collectionView.SortDescriptions.Add(new SortDescription(propertyName, ListSortDirection.Descending));
                    }
                }

                collectionView.Refresh();
            }

            {
                var collectionView = (CollectionView)CollectionViewSource.GetDefaultView(this.ShaderData.Macros);
                collectionView.SortDescriptions.Clear();

                for (int idx = 0; idx < (int)MacroColumnTypes.MaxCount; ++idx)
                {
                    string propertyName = string.Empty;
                    switch (idx)
                    {
                        case (int)MacroColumnTypes.Name:
                            propertyName = "Name";
                            break;
                        case (int)MacroColumnTypes.Value:
                            propertyName = "Value";
                            break;
                    }

                    if (this.macroSortDescriptions[idx] == SortDescriptionType.Ascending)
                    {
                        collectionView.SortDescriptions.Add(new SortDescription(propertyName, ListSortDirection.Ascending));
                    }
                    else if (this.macroSortDescriptions[idx] == SortDescriptionType.Descending)
                    {
                        collectionView.SortDescriptions.Add(new SortDescription(propertyName, ListSortDirection.Descending));
                    }
                }

                collectionView.Refresh();
            }
        }

        private void UpdateVariations()
        {
            IList<Variation> variations = this.ShaderData.Variations;
            int idx = 0;
            foreach (OptionVar optionVar in this.shaderDefinition.ShadingModels[0].OptionVars)
            {
                Variation variation = variations.FirstOrDefault(x => x.Id == optionVar.Id);
                SetVariationFromOptionVar(variation, optionVar);

                variation.Index = idx;

                int currentIndex = variations.IndexOf(variation);
                if (currentIndex != idx)
                {
                    variations.Remove(variation);
                    variations.Insert(idx, variation);
                }

                ++idx;
            }

            this.ShaderData.IsVariationsUpdated = true;
        }

        private void AddVariations()
        {
            IList<Variation> variations = this.ShaderData.Variations;

            foreach (OptionVar optionVar in this.shaderDefinition.ShadingModels[0].OptionVars)
            {
                Variation variation = variations.FirstOrDefault(x => x.Id == optionVar.Id);
                Debug.Assert(variation != null);

                this.AddVariationViewModel(variation, optionVar);
            }
        }

        private void AddVariationViewModel(Variation variation, OptionVar optionVar)
        {
            var viewModel = new VariationViewModel(variation, optionVar);
            this.variationViewModels.Add(viewModel);
        }

        private void VariationSelectionChangedExecute()
        {
            var editShaderConfigTool = AppManager.GetToolViewModel<EditShaderConfigViewModel>();
            if (editShaderConfigTool != null)
            {
                editShaderConfigTool.ClearTargets();

                var selectedVariations = this.variationViewModels.Where(x => x.IsSelected);
                editShaderConfigTool.AddTargets(selectedVariations.ToArray());
                editShaderConfigTool.Update();
            }
        }

        private void IsOutputableCheckedExecute()
        {
            var changedVariation = this.variationViewModels.FirstOrDefault(x => x.IsOutputableChanged);
            if (changedVariation == null)
            {
                return;
            }

            var selectedVariations = new List<VariationViewModel>();

            if (changedVariation.IsSelected)
            {
                selectedVariations.AddRange(this.variationViewModels.Where(x => x.IsSelected));
            }

            if (!selectedVariations.Contains(changedVariation))
            {
                selectedVariations.Add(changedVariation);
            }

            foreach (var variation in this.variationViewModels)
            {
                variation.ResetIsOutputableChanged();
            }

            OperationSet operations = new OperationSet();
            operations.Description = Labels.Output;

            foreach (var variation in selectedVariations)
            {
                Variation data = variation.VariationData;
                if (data != null)
                {
                    operations.Add(new PropertyEditOperation<Variation, bool>(data, "IsOutputable", changedVariation.IsOutputable));
                }
            }

            if (operations.Count > 0)
            {
                var document = AppManager.GetActiveDocument() as G3dDocumentViewModel;
                if (document != null)
                {
                    document.FileViewModel.ExecuteOperation(operations);
                }
            }
        }

        private void SetShaderData(Shader shader)
        {
            if (shader != null)
            {
                this.Name = shader.Name;
            }
        }

        private void ShaderPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            Shader shader = this.ShaderData;
            if (shader != null)
            {
                this.SetShaderData(shader);
            }
        }

        private static void SetVariationFromOptionVar(Variation variation, OptionVar optionVar)
        {
            switch (ShaderConfigUtility.GetAppMode())
            {
                case AppMode.AglCompatible:
                    break;

                default:    // AppMode.Default
                    if (variation.Choice == string.Empty)
                    {
                        variation.Choice = ShaderConfigUtility.GetChoiceWithoutLabels(optionVar.Choice);
                    }

                    if (variation.Default == string.Empty)
                    {
                        variation.Default = optionVar.Default;
                    }

                    break;
            }
        }

        private static Variation CreateVariationFromOptionVar(OptionVar optionVar)
        {
            var variation = new Variation();

            switch (ShaderConfigUtility.GetAppMode())
            {
                case AppMode.AglCompatible:
                    variation.Id = optionVar.Id;
                    variation.Choice = "*";             // デフォルトでは "*" を指定しておく。
                    variation.Default = string.Empty;   // option_var から情報をコピーしない。
                    variation.Branch = false;           // デフォルトで false にしておく。
                    variation.IsOutputable = false;
                    break;

                default:    // AppMode.Default
                    variation.Id = optionVar.Id;
                    variation.Choice = ShaderConfigUtility.GetChoiceWithoutLabels(optionVar.Choice);
                    variation.Default = optionVar.Default;
                    variation.Branch = optionVar.Branch;    // option block に variation と一致する unform ある場合には true になる。
                    variation.IsOutputable = true;
                    break;
            }

            return variation;
        }
    }
}
