﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Entities;
    using G3dCore.Extensions;
    using G3dCore.ViewModels;
    using Opal.App;
    using Opal.Operations;
    using Opal.ViewModels;
    using ShaderAssistAddons.Modules.ShaderConfig.Commands;
    using ShaderAssistAddons.Modules.ShaderConfig.Utilities;
    using ShaderAssistAddons.ViewModels;

    /// <summary>
    /// シェーダ設定編集のビューモデルクラスです。
    /// </summary>
    public class EditShaderConfigViewModel : ToolViewModel
    {
        private readonly ObservableCollection<CheckedListItemViewModel> choiceListItems = new ObservableCollection<CheckedListItemViewModel>();
        private readonly ObservableCollection<CheckedListItemViewModel> defaultListItems = new ObservableCollection<CheckedListItemViewModel>();
        private readonly List<VariationViewModel> targetVariations = new List<VariationViewModel>();
        private bool? useShaderDefinitionChoice = false;
        private bool isChoiceListBoxEnabled = false;
        private ShaderConfigUtility.ChoiceType choiceType = ShaderConfigUtility.ChoiceType.Bool;
        private string[] expandedChoices = null;
        private string shaderDefinitionDefault = string.Empty;
        private CheckedListItemViewModel selectedDefault = null;
        private bool? useBranch = false;

        /// <summary>
        /// choice のリストアイテムを取得します。
        /// </summary>
        public ObservableCollection<CheckedListItemViewModel> ChoiceListItems
        {
            get
            {
                return this.choiceListItems;
            }
        }

        /// <summary>
        /// default のリストアイテムを取得します。
        /// </summary>
        public ObservableCollection<CheckedListItemViewModel> DefaultListItems
        {
            get
            {
                return this.defaultListItems;
            }
        }

        /// <summary>
        /// choice のリストボックスの有効無効を取得設定します。
        /// </summary>
        public bool IsChoiceListBoxEnabled
        {
            get
            {
                return this.isChoiceListBoxEnabled;
            }

            set
            {
                if (value != this.isChoiceListBoxEnabled)
                {
                    this.isChoiceListBoxEnabled = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// シェーダ定義の choice、default を用いる取得設定します。
        /// </summary>
        public bool? UseShaderDefinitionChoice
        {
            get
            {
                return this.useShaderDefinitionChoice;
            }

            set
            {
                if (value != this.useShaderDefinitionChoice)
                {
                    this.useShaderDefinitionChoice = value;
                    this.RaisePropertyChanged();

                    this.IsChoiceListBoxEnabled = value == false;
                }
            }
        }

        /// <summary>
        /// シェーダ定義の default を取得設定します。
        /// </summary>
        public string ShaderDefinitionDefault
        {
            get
            {
                return this.shaderDefinitionDefault;
            }

            set
            {
                if (value != this.shaderDefinitionDefault)
                {
                    this.shaderDefinitionDefault = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// 選択中の default を取得設定します。
        /// </summary>
        public CheckedListItemViewModel SelectedDefault
        {
            get
            {
                return this.selectedDefault;
            }

            set
            {
                if (value != this.selectedDefault)
                {
                    this.selectedDefault = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// 静的分岐を使用するを取得設定します。
        /// </summary>
        public bool? UseBranch
        {
            get
            {
                return this.useBranch;
            }

            set
            {
                if (value != this.useBranch)
                {
                    this.useBranch = value;
                    this.RaisePropertyChanged();
                }
            }
        }

#region DelegateCommands

        /// <summary>
        /// シェーダ定義の choice、default を用いる変更時コマンドを取得します。
        /// </summary>
        public DelegateCommand UseShaderDefinitionChoiceCheckedCommand
        {
            get
            {
                return new DelegateCommand(this.UseShaderDefinitionChoiceCheckedExecute);
            }
        }

        /// <summary>
        /// シェーダ設定の choice 変更時コマンドを取得します。
        /// </summary>
        public DelegateCommand ShaderConfigChoiceCheckedCommand
        {
            get
            {
                return new DelegateCommand(this.ShaderConfigChoiceCheckedExecute);
            }
        }

        /// <summary>
        /// シェーダ設定の default 変更時コマンドを取得します。
        /// </summary>
        public DelegateCommand ShaderConfigDefaultSelectionChangedCommand
        {
            get
            {
                return new DelegateCommand(this.ShaderConfigDefaultSelectionChangedExecute);
            }
        }

        /// <summary>
        /// 静的分岐を使用する変更時コマンドを取得します。
        /// </summary>
        public DelegateCommand UseBranchCheckedCommand
        {
            get
            {
                return new DelegateCommand(this.UseBranchCheckedExecute);
            }
        }

#endregion

        /// <summary>
        /// 編集バリエーションを追加します。
        /// </summary>
        /// <param name="variations">追加するバリエーションです。</param>
        public void AddTargets(params VariationViewModel[] variations)
        {
            this.targetVariations.AddRange(variations);
        }

        /// <summary>
        /// 編集バリエーションをクリアします。
        /// </summary>
        public void ClearTargets()
        {
            this.targetVariations.Clear();
        }

        /// <summary>
        /// 更新処理を行います。
        /// </summary>
        public void Update()
        {
            this.UpdateUseShaderDefinitionChoice();
            this.UpdateChoicesListBox();
            this.UpdateShaderDefinitionDefault();
            this.UpdateShaderConfigDefault();
            this.UpdateUseBranch();
        }

        private void UpdateUseShaderDefinitionChoice()
        {
            // シェーダ定義の choice、default を用いるの更新。

            bool checkOff = false;
            bool checkOn = false;

            foreach (var variation in this.targetVariations)
            {
                if (variation.ShaderConfigChoice == "*")
                {
                    checkOn = true;
                }
                else
                {
                    checkOff = true;
                }

                if (checkOn && checkOff)
                {
                    break;
                }
            }

            if (!checkOff && checkOn)
            {
                this.UseShaderDefinitionChoice = true;
                this.IsChoiceListBoxEnabled = false;
            }
            else if (!checkOn)
            {
                this.UseShaderDefinitionChoice = false;
                this.IsChoiceListBoxEnabled = true;
            }
            else
            {
                this.UseShaderDefinitionChoice = null;
                this.IsChoiceListBoxEnabled = true;
            }
        }

        private void UpdateChoicesListBox()
        {
            // choice リストボックスの更新。
            var newListItems = new List<CheckedListItemViewModel>();

            this.UpdateShaderDefinitionChoices(newListItems);
            this.UpdateShaderConfigChoices(newListItems);

            bool isRebuildNeeded = this.choiceListItems.Count != newListItems.Count;
            for (int i = 0; i < this.choiceListItems.Count && !isRebuildNeeded; ++i)
            {
                if (this.choiceListItems[i].Tag.ToString() != newListItems[i].Tag.ToString())
                {
                    isRebuildNeeded = true;
                }
            }

            if (isRebuildNeeded)
            {
                this.choiceListItems.Clear();
                newListItems.ForEach(x => this.choiceListItems.Add(x));
            }
            else
            {
                for (int i = 0; i < this.choiceListItems.Count; ++i)
                {
                    if (this.choiceListItems[i].IsChecked != newListItems[i].IsChecked)
                    {
                        this.choiceListItems[i].IsChecked = newListItems[i].IsChecked;
                    }
                }
            }
        }

        private void UpdateShaderDefinitionChoices(IList<CheckedListItemViewModel> newListItems)
        {
            // シェーダ定義の choice （リスト）の更新。
            this.expandedChoices = null;

            bool firstTime = true;
            foreach (var variation in this.targetVariations)
            {
                ShaderConfigUtility.ChoiceType optionVarChoiceType =
                   ShaderConfigUtility.ExpandChoice(variation.ShaderDefinitionChoice, out this.expandedChoices);
                string[] choiceValues = ShaderConfigUtility.GetValuesOrLabelsFromChoice(this.expandedChoices, false);
                string[] choiceLabels = ShaderConfigUtility.GetValuesOrLabelsFromChoice(this.expandedChoices, true);
                Debug.Assert(choiceValues.Length == choiceLabels.Length);

                if (firstTime)
                {
                    this.choiceType = optionVarChoiceType;

                    for (int idx = 0; idx < choiceValues.Length; ++idx)
                    {
                        newListItems.Add(new CheckedListItemViewModel()
                        {
                            Label = choiceLabels[idx],  // choice のエイリアス
                            Tag = choiceValues[idx]     // choice の値
                        });
                    }

                    firstTime = false;
                }
                else
                {
                    // 複数選択したバリエーションの choice がそれぞれ異なる場合には複数一括編集はできない。

                    bool mismatch = false;
                    if (choiceLabels.Length > 0 && newListItems.Count == choiceLabels.Length)
                    {
                        for (int i = 0; i < choiceLabels.Length; ++i)
                        {
                            if (choiceLabels[i] != newListItems[i].Label)
                            {
                                mismatch = true;
                                break;
                            }
                        }
                    }
                    else
                    {
                        mismatch = true;
                    }

                    if (mismatch)
                    {
                        newListItems.Clear();
                        break;
                    }
                }
            }
        }

        private void UpdateShaderConfigChoices(IList<CheckedListItemViewModel> newListItems)
        {
            // シェーダ設定の choice （チェックボックス）の更新。
            if (newListItems.FirstOrDefault() == null)
            {
                return;
            }

            bool firstTime = true;
            foreach (var variation in this.targetVariations)
            {
                // シェーダ設定 choice のラベル配列。
                string[] choiceLabels;
                {
                    ShaderConfigUtility.ExpandChoice(variation.ShaderConfigChoice, out choiceLabels);
                    choiceLabels = ShaderConfigUtility.GetValuesOrLabelsFromChoice(choiceLabels, false);
                    for (int i = 0; i < choiceLabels.Length; ++i)
                    {
                        choiceLabels[i] = ShaderConfigUtility.FindLabelFromChoices(choiceLabels[i], this.expandedChoices);
                    }
                }

                int idxEnum = 0;
                for (int idxList = 0; idxList < newListItems.Count; ++idxList)
                {
                    CheckedListItemViewModel listItem = newListItems[idxList];
                    if (variation.ShaderConfigChoice == "*")
                    {
                        listItem.IsChecked = true;
                        continue;
                    }

                    if (firstTime)
                    {
                        listItem.IsChecked = false;
                    }

                    if (listItem.Label == string.Empty)
                    {
                        continue;
                    }

                    if (idxEnum < choiceLabels.Length && listItem.Label == choiceLabels[idxEnum])
                    {
                        if (firstTime)
                        {
                            listItem.IsChecked = true;
                        }
                        else if (listItem.IsChecked == false)
                        {
                            listItem.IsChecked = null;
                        }

                        ++idxEnum;
                    }
                    else if (!firstTime)
                    {
                        if (listItem.IsChecked == true)
                        {
                            listItem.IsChecked = null;
                        }
                    }
                }

                firstTime = false;
            }
        }

        private void UpdateShaderDefinitionDefault()
        {
            // シェーダ定義の default の更新。
            string @default = null;
            foreach (var variation in this.targetVariations)
            {
                string label = ShaderConfigUtility.FindLabelFromChoices(variation.ShaderDefinitionDefault, this.expandedChoices);

                if (@default == null)
                {
                    @default = label;
                }
                else
                {
                    if (@default != label)
                    {
                        @default = string.Empty;
                        break;
                    }
                }
            }

            this.ShaderDefinitionDefault = @default ?? string.Empty;
        }

        private void UpdateShaderConfigDefault()
        {
            // シェーダ設定の default （コンボボックス）の更新。
            this.defaultListItems.Clear();

            if (this.UseShaderDefinitionChoice == false)
            {
                // default で"空白"が選択できるように空アイテムを１つ追加しておきます。
                this.defaultListItems.Add(new CheckedListItemViewModel());

                foreach (var item in this.choiceListItems)
                {
                    if (item.IsChecked == true)
                    {
                        this.defaultListItems.Add(item);
                    }
                }
            }

            CheckedListItemViewModel selectedItem = null;
            foreach (var variation in this.targetVariations)
            {
                string label = ShaderConfigUtility.FindLabelFromChoices(variation.ShaderConfigDefault, expandedChoices);
                var defaultItem = this.defaultListItems.FirstOrDefault(x => x.Label == label);

                if (selectedItem == null)
                {
                    selectedItem = defaultItem;
                }
                else
                {
                    if (selectedItem.Label != label)
                    {
                        selectedItem = null;
                        break;
                    }
                }
            }

            this.SelectedDefault = selectedItem;
        }

        private void UpdateUseBranch()
        {
            // 静的分岐を使用するの更新。
            this.UseBranch = false;

            bool checkOff = false;
            bool checkOn = false;

            foreach (var variation in this.targetVariations)
            {
                if (variation.Branch)
                {
                    checkOn = true;
                }
                else
                {
                    checkOff = true;
                }

                if (checkOn && checkOff)
                {
                    break;
                }
            }

            if (!checkOff && checkOn)
            {
                this.UseBranch = true;
            }
            else if (!checkOn)
            {
                this.UseBranch = false;
            }
            else
            {
                this.UseBranch = null;
            }
        }

        private void UseShaderDefinitionChoiceCheckedExecute()
        {
            var variationChoices = this.GetShaderDefinitionVariationChoices();

            OperationSet operations = new OperationSet();
            operations.Description = "シェーダ定義の choice、default を用いる";

            foreach (var variationChoice in variationChoices)
            {
                Variation data = variationChoice.Item1.VariationData;
                if (data != null)
                {
                    operations.Add(data.CreateEditChoiceOperation(variationChoice.Item2));

                    if (this.UseShaderDefinitionChoice == true)
                    {
                        operations.Add(data.CreateEditDefaultOperation(string.Empty));
                    }
                }
            }

            this.ExecuteOperation(operations);
        }

        private void ShaderConfigChoiceCheckedExecute()
        {
            var variationChoices = this.GetShaderConfigVariationChoices();

            OperationSet operations = new OperationSet();
            operations.Description = "choice";

            foreach (var variationChoice in variationChoices)
            {
                Variation data = variationChoice.Item1.VariationData;
                if (data != null)
                {
                    operations.Add(data.CreateEditChoiceOperation(variationChoice.Item2));

                    if (variationChoice.Item3)
                    {
                        // default、選択オプションで設定していた choice が無くなった場合には "空白" にする。
                        operations.Add(data.CreateEditDefaultOperation(string.Empty));
                    }
                }
            }

            this.ExecuteOperation(operations);
        }

        private void ShaderConfigDefaultSelectionChangedExecute()
        {
            if (this.SelectedDefault == null || this.targetVariations.FirstOrDefault() == null)
            {
                return;
            }

            string variationDefault =
                this.SelectedDefault.Label != string.Empty ? this.SelectedDefault.Tag.ToString() : string.Empty;
            var changedVariations = this.targetVariations.Where(x => x.ShaderConfigDefault != variationDefault);

            OperationSet operations = new OperationSet();
            operations.Description = "default";

            foreach (var variation in changedVariations)
            {
                Variation data = variation.VariationData;
                if (data != null)
                {
                    operations.Add(data.CreateEditDefaultOperation(variationDefault));
                }
            }

            this.ExecuteOperation(operations);
        }

        private void UseBranchCheckedExecute()
        {
            OperationSet operations = new OperationSet();
            operations.Description = "静的分岐を使用する";

            foreach (var variation in this.targetVariations)
            {
                Variation data = variation.VariationData;
                if (data != null)
                {
                    operations.Add(data.CreateEditBranchOperation(this.UseBranch == true));
                }
            }

            this.ExecuteOperation(operations);
        }

        private void ExecuteOperation(OperationSet operations)
        {
            if (operations.Count > 0)
            {
                var document = AppManager.GetActiveDocument() as G3dDocumentViewModel;
                if (document != null)
                {
                    document.FileViewModel.ExecuteOperation(operations);
                }
            }
        }

        private ReadOnlyCollection<Tuple<VariationViewModel, string>> GetShaderDefinitionVariationChoices()
        {
            var variationChoices = new List<Tuple<VariationViewModel, string>>();

            foreach (var variation in this.targetVariations)
            {
                if (this.UseShaderDefinitionChoice == true)
                {
                    variationChoices.Add(new Tuple<VariationViewModel, string>(variation, "*"));
                }
                else
                {
                    string choice = variation.ShaderConfigChoice;

                    if (choice == "*")
                    {
                        string[] shaderDefinitionChoices = null;
                        ShaderConfigUtility.ChoiceType optionVarChoiceType =
                            ShaderConfigUtility.ExpandChoice(variation.ShaderDefinitionChoice, out shaderDefinitionChoices);

                        string[] choices = ShaderConfigUtility.GetValuesOrLabelsFromChoice(shaderDefinitionChoices, false);
                        choice = ShaderConfigUtility.GetVariationChoiceFromArray(this.choiceType, choices);
                    }

                    variationChoices.Add(new Tuple<VariationViewModel, string>(variation, choice));
                }
            }

            return new ReadOnlyCollection<Tuple<VariationViewModel, string>>(variationChoices);
        }

        private ReadOnlyCollection<Tuple<VariationViewModel, string, bool>> GetShaderConfigVariationChoices()
        {
            var variationChoices = new List<Tuple<VariationViewModel, string, bool>>();

            foreach (var variation in this.targetVariations)
            {
                string[] currentVariationChoices = null;
                ShaderConfigUtility.ExpandChoice(variation.ShaderConfigChoice, out currentVariationChoices);
                currentVariationChoices = ShaderConfigUtility.GetValuesOrLabelsFromChoice(currentVariationChoices, false);

                var newVariationChoices = new List<string>();

                foreach (var item in this.choiceListItems)
                {
                    if (item.IsChecked == true ||
                        (item.IsChecked == null && currentVariationChoices.Contains(item.Tag.ToString())))
                    {
                        newVariationChoices.Add(item.Tag.ToString());
                    }
                }

                bool isDefaultLost = !newVariationChoices.Contains(variation.ShaderConfigDefault);

                string choice = ShaderConfigUtility.GetVariationChoiceFromArray(this.choiceType, newVariationChoices.ToArray());
                variationChoices.Add(new Tuple<VariationViewModel, string, bool>(variation, choice, isDefaultLost));
            }

            return new ReadOnlyCollection<Tuple<VariationViewModel, string, bool>>(variationChoices);
        }
    }
}
