﻿namespace ShaderAssistAddons.Modules.ShaderConfig.Commands.Generic
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows;
    using System.Windows.Input;
    using System.Windows.Threading;

    /// <summary>
    /// デリゲートを受け取るICommandの実装（パラメータ付き）
    /// </summary>
    public class DelegateCommand<T> : ICommand
    {
        readonly Action<T> _execute;
        readonly Func<T, bool> _canExecute;
        Dispatcher _dispatcher;

        static readonly bool IsValueType;

        static DelegateCommand()
        {
            IsValueType = typeof(T).IsValueType;
        }

        /// <summary>
        /// コマンドのExecuteメソッドで実行する処理を指定してDelegateCommandのインスタンスを作成します。
        /// </summary>
        public DelegateCommand(Action<T> execute)
            : this(execute, o => true)
        {
        }

        /// <summary>
        /// コマンドのExecuteメソッドで実行する処理とCanExecuteメソッドで実行する処理を指定して
        /// DelegateCommandのインスタンスを作成します。
        /// </summary>
        public DelegateCommand(Action<T> execute, Func<T, bool> canExecute)
        {
            if (execute == null)
            {
                throw new ArgumentNullException("execute");
            }

            if (canExecute == null)
            {
                throw new ArgumentNullException("canExecute");
            }

            this._execute = execute;
            this._canExecute = canExecute;
            this._dispatcher = Application.Current.Dispatcher;
        }

        /// <summary>
        /// コマンドが実行可能な状態化どうか問い合わせます。
        /// </summary>
        public bool CanExecute(T parameter)
        {
            if (System.Threading.Thread.CurrentThread != this._dispatcher.Thread)
            {
                return (bool)this._dispatcher.Invoke(_canExecute, parameter);
            }
            return this._canExecute(parameter);
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        public void Execute(T parameter)
        {
            if (System.Threading.Thread.CurrentThread != this._dispatcher.Thread)
            {
                this._dispatcher.Invoke(_execute, parameter);
            }
            else
            {
                this._execute(parameter);
            }
        }

        /// <summary>
        /// ICommand.CanExecuteの明示的な実装。
        /// </summary>
        bool ICommand.CanExecute(object parameter)
        {
            return this.CanExecute(Cast(parameter));
        }

        /// <summary>
        /// ICommand.Executeの明示的な実装。
        /// </summary>
        void ICommand.Execute(object parameter)
        {
            this.Execute(Cast(parameter));
        }

        /// <summary>
        /// CanExecuteの結果に変更があったことを通知するイベント。
        /// </summary>
        public event EventHandler CanExecuteChanged
        {
            add { CommandManager.RequerySuggested += value; }
            remove { CommandManager.RequerySuggested -= value; }
        }

        /// <summary>
        /// コマンドの実行可否状態が変更された事を知らせるイベントを発生させます。
        /// </summary>
        public void RaiseCanExecuteChanged()
        {
            if (System.Threading.Thread.CurrentThread != this._dispatcher.Thread)
            {
                this._dispatcher.Invoke(CommandManager.InvalidateRequerySuggested);
            }
            else
            {
                CommandManager.InvalidateRequerySuggested();
            }
        }

        private T Cast(object parameter)
        {
            if (parameter == null && IsValueType)
            {
                return default(T);
            }
            return (T)parameter;
        }
    }
}
