﻿namespace Chalkboard
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Input;
    using ChalkboardCore.Modules.About;
    using G3dCore.Storages;
    using G3dCore.ViewModels;
    using Opal.App;
    using Opal.Commands;
    using Opal.Menus;
    using Opal.ViewModels;

    /// <summary>
    /// メインビューモデルクラスです。
    /// </summary>
    public class MainViewModel : MainWindowViewModel
    {
        private readonly ICommand aboutCommand;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MainViewModel()
        {
            this.aboutCommand = new RelayCommand(param => this.ExecuteAbout(), parma => { return true; });
        }

        /// <summary>
        /// バージョン情報表示コマンドを取得します。
        /// </summary>
        public ICommand AboutCommand
        {
            get
            {
                return this.aboutCommand;
            }
        }

        /// <summary>
        /// ウィンドウが閉じられるときに呼び出されます。
        /// </summary>
        /// <returns>閉じる処理をキャンセルする場合に true を返します。</returns>
        protected override bool ClosingWindowInternal()
        {
            var weakStorage = AppManager.GetStorage<G3dStorage>();
            Debug.Assert(weakStorage != null);

            G3dStorage storage = null;
            if (weakStorage.TryGetTarget(out storage))
            {
                if (!storage.ConfirmAndCloseDocumentsAll())
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// Undoを実行します。
        /// </summary>
        /// <param name="parameter">パラメータです。</param>
        protected override void ExecuteUndo(object parameter)
        {
            G3dFileViewModel fileViewModel = this.GetActiveFileViewModel();
            Debug.Assert(fileViewModel != null);

            fileViewModel.Undo();
        }

        /// <summary>
        /// Undoが実行可能かどうかを返します。
        /// </summary>
        /// <param name="parameter">パラメータです。</param>
        /// <returns>結果を返します。</returns>
        protected override bool CanExecuteUndo(object parameter)
        {
            G3dFileViewModel fileViewModel = this.GetActiveFileViewModel();
            if (fileViewModel == null)
            {
                return false;
            }

            return fileViewModel.CanUndo();
        }

        /// <summary>
        /// Redoを実行します。
        /// </summary>
        /// <param name="parameter">パラメータです。</param>
        protected override void ExecuteRedo(object parameter)
        {
            G3dFileViewModel fileViewModel = this.GetActiveFileViewModel();
            Debug.Assert(fileViewModel != null);

            fileViewModel.Redo();
        }

        /// <summary>
        /// Redoが実行可能かどうかを返します。
        /// </summary>
        /// <param name="parameter">パラメータです。</param>
        /// <returns>結果を返します。</returns>
        protected override bool CanExecuteRedo(object parameter)
        {
            G3dFileViewModel fileViewModel = this.GetActiveFileViewModel();
            if (fileViewModel == null)
            {
                return false;
            }

            return fileViewModel.CanRedo();
        }

        private void ExecuteAbout()
        {
            AppManager.ShowWindow<AboutWindow>();
        }

        private G3dFileViewModel GetActiveFileViewModel()
        {
            G3dDocumentViewModel activeDocument = AppManager.GetActiveDocument() as G3dDocumentViewModel;
            if (activeDocument == null)
            {
                return null;
            }

            return activeDocument.FileViewModel;
        }
    }
}
