﻿namespace Chalkboard
{
    using System;
    using System.Collections.Generic;
    using System.Configuration;
    using System.Data;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Runtime;
    using System.Threading.Tasks;
    using System.Windows;
    using System.Windows.Threading;
    using ChalkboardCore;
    using G3dCore.Configurations;
    using Opal.App;
    using Opal.Configurations;
    using Opal.Logs;
    using Opal.Menus;
    using Opal.Modules.Output;
    using Opal.Operations;
    using Opal.Panes;
    using Opal.Plugins;
    using Opal.Resources;
    using Opal.Storages;
    using Opal.Utilities;
    using Opal.Windows;
    using Opal.Windows.Documents;
    using Opal.Windows.Tools;

    /// <summary>
    /// App.xaml の相互作用ロジック
    /// </summary>
    public partial class App : Application
    {
        private readonly ChalkboardManager appManager = new ChalkboardManager();

        /// <summary>
        /// スタートアップ処理です。
        /// </summary>
        /// <param name="e">イベント引数です。</param>
        protected override async void OnStartup(StartupEventArgs e)
        {
            ProfileOptimization.SetProfileRoot(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location));
            ProfileOptimization.StartProfile("Chalkboard.JIT.Profile");

            if (!Environment.Is64BitOperatingSystem || !Environment.Is64BitProcess)
            {
                MessageBox.Show(Errors.NotSupport32BitOS, Labels.Error, MessageBoxButton.OK);
                Environment.Exit(-1);
            }

#if !DEBUG// デバッグビルド時例外を補足しない。
            this.DispatcherUnhandledException +=
                this.Application_DispatcherUnhandledException;

            AppDomain.CurrentDomain.UnhandledException +=
                new UnhandledExceptionEventHandler(this.CurrentDomain_UnhandledException);
#endif

            // DocumentManager 登録
            {
                DocumentManager documentManager = new DocumentManager();
                this.appManager.AddDocumentManager(documentManager);
            }

            // ToolManager 登録
            {
                ToolManager toolManager = new ToolManager();
                this.appManager.AddToolManager(toolManager);
            }

            // WindowManager 登録
            {
                WindowManager windowManager = new WindowManager();
                this.appManager.AddWindowManager(windowManager);
            }

            // StorageManager 登録
            {
                var storageManager = new StorageManager();
                this.appManager.AddStorageManager(storageManager);

                G3dCore.Storages.G3dStorage g3dStroage = new G3dCore.Storages.G3dStorage();
                g3dStroage.Initialize();
                AppManager.AddStorage(g3dStroage);
            }

            // LogManager 登録
            {
                var conductor = new OutputConductor();
                LogManager logManager = new LogManager(conductor);
                this.appManager.AddLoggerManager(logManager);
            }

            // OperationManager 登録
            {
                OperationManager operationManager = new OperationManager();
                this.appManager.AddOperationManager(operationManager);
            }

            // MenuManager 登録
            {
                MenuManager menuManager = new MenuManager();
                this.appManager.AddMenuManager(menuManager);
            }

            // PaneManager 登録
            {
                PaneManager paneManager = new PaneManager();
                this.appManager.AddPaneManager(paneManager);
            }

            // ConfigManager 登録
            {
                ConfigManager configManager = new ConfigManager();
                this.appManager.AddConfigManager(configManager);

                {
                    string path = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), @".\settings.config");
                    ChalkboardConfig chalkboardConfig = ChalkboardConfig.Load(path);
                    AppManager.AddConfig<ChalkboardConfig>(chalkboardConfig);
                }

                {
                    string teamConfigFileName = "team.config";
                    string teamConfigDir = Environment.ExpandEnvironmentVariables("%CHALKBOARD_TEAM_CONFIG%");
                    string teamConfigFilePath = Path.Combine(teamConfigDir, teamConfigFileName);
                    if (string.IsNullOrEmpty(teamConfigDir) || !File.Exists(teamConfigFilePath))
                    {
                        teamConfigDir = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
                        teamConfigFilePath = Path.Combine(teamConfigDir, teamConfigFileName);
                    }

                    var teamConfig = TeamConfig.Load(teamConfigFilePath);
                    AppManager.AddConfig<TeamConfig>(teamConfig);
                }

                {
                    string userConfigFileName = "user.config";
                    string userConfigDir = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
                    string userConfigFilePath = Path.Combine(userConfigDir, userConfigFileName);

                    var userConfig = UserConfig.Load(userConfigFilePath);
                    AppManager.AddConfig<UserConfig>(userConfig);
                }
            }

            var window = new MainWindow();
            var viewModel = new MainViewModel();
            AppManager.SetOwnerWindow(window);
            AppManager.SetOwnerViewModel(viewModel);

            window.DataContext = viewModel;

            window.Show();
            window.IsEnabled = false;

            // プラグインの登録
            {
                Assembly assemblyCore =
                    AppDomain.CurrentDomain.GetAssemblies().FirstOrDefault(x => x.GetName().Name == "3dShaderMakerCore");

                List<string> paths = new List<string>();

                // addons フォルダも検索対象に含める。
                {
                    paths.Add(Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "addons"));
                }

                PluginManager pluginManager = new PluginManager(assemblyCore, paths.ToArray());

                await pluginManager.InitializePlugins();

                this.RegisterPlugins(pluginManager);
            }

            // アプリケーション初期化処理
            AppManager.Initialize();

            if (LayoutUtility.HasAvalonDockLayout())
            {
                window.LoadLayout();
            }
            else
            {
                AppManager.MakeDefaultTools();
            }

            window.IsEnabled = true;
        }

        private void RegisterPlugins(PluginManager pluginManager)
        {
            // ドキュメントマネージャにドキュメントメーカーを追加
            {
                DocumentManager documentManager = AppManager.GetDocumentManager() as DocumentManager;
                foreach (DocumentMaker documentMaker in pluginManager.GetPlugins<DocumentMakerPlugin>())
                {
                    documentManager.AddDocumentMaker(documentMaker.Key, documentMaker);
                }
            }

            // ツールマネージャにツールメーカーを追加
            {
                ToolManager toolManager = AppManager.GetToolManager() as ToolManager;
                foreach (ToolMaker toolMaker in pluginManager.GetPlugins<ToolMakerPlugin>())
                {
                    toolManager.AddToolMaker(toolMaker.Key, toolMaker);
                }
            }

            // メニューマネージャーにプラグインメニュを追加
            {
                MenuManager menuManager = AppManager.GetMenuManager() as MenuManager;
                IEnumerable<MenuPlugin> menuPlugins = pluginManager.GetPlugins<MenuPlugin>();
                menuPlugins = menuPlugins.OrderBy(menu => menu.Order);
                foreach (Menu menu in menuPlugins)
                {
                    menuManager.AddMenu(menu.Category, menu);
                }
            }

            // ウィンドウマネージャーにプラグインウィンドウを追加
            {
                WindowManager windowManager = AppManager.GetWindowManager() as WindowManager;
                foreach (WindowMaker maker in pluginManager.GetPlugins<WindowMakerPlugin>())
                {
                    windowManager.AddWindowMaker(maker);
                }
            }
        }

        private void Application_DispatcherUnhandledException(object sender, DispatcherUnhandledExceptionEventArgs e)
        {
            MessageBox.Show(e.Exception.ToString());
            Environment.Exit(-1);
        }

        private void CurrentDomain_UnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            MessageBox.Show(((Exception)e.ExceptionObject).ToString());
            Environment.Exit(-1);
        }
    }
}
