﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System;
using App;
using App.ConfigData;
using App.Data;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace Viewer
{
    public static class ViewerUtility
    {
        public static bool AllFowarding()
        {
            // Viewerが切断されていたら、ここで終了
            if (Viewer.Manager.Instance.IsConnected == false)
            {
                return true;
            }

            DebugConsole.WriteLine("Viewer.ViewerUtility.AllFowarding() ---------------------------------------------");

            using (var vdsb = new ViewerDrawSuppressBlock())
            {
                // アニメーションの送信
                SendAnimations();

                // モデルの送信(アニメーションセットも)
                SendModels();

                // プレビューシーンの送信
                SendPreviewSceneAnimSet();

                // 選択オブジェクトの送信
                Viewer.ViewerUtility.SendSelectedObjects(App.AppContext.SelectedTarget, ApplicationConfig.Setting.Preview.IsPreviewCameraFocus);
            }

            return true;
        }

        /// <summary>
        /// ランタイム連携を切断します。
        /// </summary>
        static public void Disconnect(string warning, bool showReconnectOption, bool showReconnectDialog)
        {
            if (Viewer.Manager.Instance.IsConnected)
            {
                // ランタイム連携中にモデルやアニメーションが外部から変更された場合には、
                // ランタイム連携を切断します。

                // メッセージ表示する。
                if (showReconnectDialog)
                {
                    var dialog = new App.Controls.DisconnectViewerDialog(warning, showReconnectOption);
                    dialog.ShowDialog();
                }
                else
                {
                    //Strings.Document_ReloadOnFileUpdating
                    MessageLog.Write(MessageLog.LogType.Information, warning);
                }

                // 切断
                App.TheApp.MainFrame.ConnectToHio(false);
            }
        }

        public static void RenameDocument(string srcDocName, Document dstDoc, List<Model> animationOwners)
        {
            // Viewerが切断されていたら、ここで終了
            if (Viewer.Manager.Instance.IsConnected == false)
            {
                return;
            }

            if (dstDoc is Model)
            {
                var dstModel = dstDoc as Model;
                using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                {
                    // 対象モデルをバインドしているモデルのボーンバインドを解除する情報を転送。
                    // ビュワー側での不正参照を回避するために、対象モデルを再転送する前に解除する。
                    var srcModelName = srcDocName.Substring(0, srcDocName.Length - (srcDocName.EndsWith(dstModel.FileDotExt) ? dstModel.FileDotExt.Length : 0));
                    var unbindModels = DocumentManager.Models.Where(m => (m.PreviewInfo.BindModelName == srcModelName));
                    foreach (var model in unbindModels.OfType<Model>())
                    {
                        model.SendEditBoneBind();
                    }

                    // 一旦モデルを閉じるメッセージを送信する。
                    Viewer.Close.Send(dstDoc);
                    // モデルを再転送
                    Viewer.LoadOrReloadModel.Send(dstModel);
                    // アニメーションバインド情報を転送
                    Viewer.ViewerUtility.SendAnimationSet(dstModel);
                    // モデルのプレビュー情報を転送
                    dstModel.SendEditBoneBind();
                    dstModel.SendEditModelLayout(true);

                    // 対象モデルをバインドしてるモデルのボーンバインド情報を転送する
                    var bindModels = DocumentManager.Models.Where(m => (m.PreviewInfo.BindModelName == dstModel.Name));
                    foreach (var model in bindModels.OfType<Model>())
                    {
                        model.SendEditBoneBind();
                    }

                    // 選択オブジェクトが再送したモデルに含まれていれば選択オブジェクト全体を転送。
                    // 「ファイルを開き直す」機能の挙動に合わせて、選択オブジェクトの抽出は行っていない。
                    if (!App.AppContext.SelectedTarget.IsEmpty)
                    {
                        var objects = new HashSet<GuiObject>(dstModel.ContentObjects);
                        if (App.AppContext.SelectedTarget.Objects.Any(o => objects.Contains(o)))
                        {
                            Viewer.ViewerUtility.SendSelectedObjects(App.AppContext.SelectedTarget, ApplicationConfig.Setting.Preview.IsPreviewCameraFocus);
                        }
                    }
                }
            }
            else if (dstDoc is AnimationDocument)
            {
                var dstAnim = dstDoc as AnimationDocument;
                using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                {
                    // 不正アニメーションチェック
                    dstAnim.CheckAndDisConnect();

                    // 一旦アニメーションを閉じるメッセージを送信する。
                    Viewer.Close.Send(dstAnim);

                    // アニメーションを再転送
                    Viewer.LoadOrReloadAnimation.Send(dstAnim);

                    foreach (Model model in animationOwners)
                    {
                        SendAnimationSet(model);
                    }
                }
            }
        }

        public static void SendModels()
        {
            foreach (var model in DocumentManager.Models)
            {
                Viewer.LoadOrReloadModel.Send(model);
                SendAnimationSet(model);
            }
        }

        public static void SendPreviewSceneAnimSet()
        {
            //Viewer.UnbindSceneAnimations.Send();
            DocumentManager.SendBindSceneAnimations();
        }

        public static void SendAnimations()
        {
            foreach (var animation in DocumentManager.Animations)
            {
                //if(animation.ObjectID == GuiObjectID.ShapeAnimation) continue;
                // 不正アニメーションチェック
                animation.CheckAndDisConnect();
                Viewer.LoadOrReloadAnimation.Send(animation);
            }
        }


        public static void SendAnimationSet(Model model)
        {
            // アニメーションセットを再設定することで、
            // アニメーションのアンバインド・バインドを行う。
            //Viewer.UnbindAnimations.Send(model);
            model.SendBindAnimations();
        }

        public static void SendSelectedObjects(GuiObjectGroup objects, bool focusMsg = false)
        {
#if true
            var groups = from obj in objects.Objects
                         group obj by obj.ObjectID;
            foreach (var group in groups)
            {
                Viewer.EditSelectTarget.Send(new ArrayList(group.ToList()), group.Key);
            }
#else
            var guiObjects = new ArrayList();
//          GuiObjectID objType = GuiObjectID.Model;
            var objType = GuiObjectID.DummyObject;
            if (objects.Active != null)
            {
                objType = objects.Active.ObjectID;
            }

            foreach (GuiObject guiObject in objects.Objects)
            {
                // 先頭(もしくは、送信できるタイプ)のオブジェクトと同じ型のものだけ送信します。
                if (guiObject.ObjectID == objType)
                {
                    guiObjects.Add(guiObject);
                }
            }

            // 選択メッセージの送信
            Viewer.EditSelectTarget.Send(guiObjects, objType);
#endif
        }

        public static void SendMaterialShaderOption(ArrayList targets, object[] _data, shader_optionType option)
        {
            // Viewerが切断されていたら、ここで終了
            if (Viewer.Manager.Instance.IsConnected == false)
            {
                return;
            }

            var sendData = new List<object>();

            foreach (var value in _data)
            {
                var data = new SetMaterialShaderOption.PacketData(option.id, (string)value);
                sendData.Add(data);
            }

            using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
            {
                SetMaterialShaderOption.Send(targets, sendData.ToArray(), 0xFFFFFFFF, 0xFFFFFFFF);
            }
        }
        public static void SendMaterialShaderOption(GuiObjectGroup targets, App.PropertyEdit.ShaderParamControls.OptionValueChangedEventArgs e, shader_optionType option)
        {
            // Viewerが切断されていたら、ここで終了
            if (Viewer.Manager.Instance.IsConnected == false)
            {
                return;
            }

            var data = new SetMaterialShaderOption.PacketData(option.id, (string)e.ParamValue);
            using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
            {
                SetMaterialShaderOption.Send(targets, data, 0xFFFFFFFF, 0xFFFFFFFF);
            }
        }
    }
}
