﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using App;
using App.Utility;
using App.ConfigData;

namespace Viewer
{
    /// <summary>
    /// ビューアマネージャー。
    /// </summary>
    public sealed class Manager
    {
        private static readonly Manager	instance_	= new Manager();

        private readonly MessageQueue	messageQueue_	= new MessageQueue();
        private Connecter		connecter_		= null;

        public object PollPingLockObject
        {
            get
            {
                return (connecter_ == null) ? null : connecter_.PollPingLockObject;
            }
        }

        /// <summary>
        /// シングルトンインスタンス
        /// </summary>
        public static Manager Instance{
            get{	return instance_;	}
        }

         // suppress beforefieldinit
        static Manager()
        {
        }

        /// <summary>
        /// 接続しているかどうか？
        /// </summary>
        public bool IsConnected{
            get{
                return (connecter_ != null) && connecter_.IsConnected;
            }
        }

        public static bool IsDeviceTargeted
        {
            get
            {
                return TargetDeviceType == ConfigCommon.HioTarget.Device;
            }
        }

        /// <summary>
        ///接続先の種類
        /// </summary>
        /// <remarks>ビューアとの接続開始を試みた時のみ値が変更される</remarks>
        private static ConfigCommon.HioTarget TargetDeviceType;

        public bool IsExistMessages
        {
            get{
                return messageQueue_.IsExist;
            }
        }

        private bool isExecuting;
        public bool IsExecuting
        {
            get
            {
                lock (this)
                {
                    return isExecuting;
                }
            }
            set
            {
                lock (this)
                {
                    isExecuting = value;
                }
            }
        }

        // メッセージ更新間隔
        private int? messageInterval_ = null;

        /// <summary>
        /// メッセージ更新間隔を設定する。
        /// </summary>
        public void SetMessageInterval(int interval)
        {
            lock (this)
            {
                messageInterval_ = interval;
                if (connecter_ != null)
                {
                    connecter_.IntervalExistMessage = messageInterval_.Value;
                }
            }
        }

        private Manager()
        {
        }

        public void Initialize()
        {
            // connecter_ に設定するメッセージ更新間隔。
            int messageInterval;
            lock (this)
            {
                messageInterval = messageInterval_.HasValue ? messageInterval_.Value : ApplicationConfig.Preview.MessageInterval;
            }

            connecter_ = new Connecter(this)
            {
                IntervalExistMessage = messageInterval
            };

            AddChangeConnectedEvent(
                () =>
                {
                    Viewer.EnableRuntimeDebugMessage.Send(ApplicationConfig.UserSetting.ExternalProgram.EnableRuntimeDebugLog);
                    Viewer.SetFrameCount.Send(TheApp.MainFrame.FrameCount);
                    Viewer.SetPlayPolicy.Send(TheApp.MainFrame.PlayPolicy, (float)TheApp.MainFrame.CurrentFrame);
                    Viewer.SetFrameStep.Send((float)TheApp.MainFrame.FrameScale);
                }
            );
        }

        public void Destroy()
        {
            if (connecter_ != null)
            {
                connecter_.Destroy();
            }
        }

        public void Close()
        {
            Debug.Assert(connecter_ != null);
            connecter_.Close();

            ChangeConnectedEvent();
        }

        public void Connect(TeamConfig.PlatformPreset platform, ConfigCommon.HioTarget target)
        {
            Debug.Assert(connecter_ != null);
            TargetDeviceType = target;
            connecter_.Connect(platform, target);

            ChangeConnectedEvent();
        }

        public void PushMessage(BaseMessage message)
        {
            if(IsConnected)
            {
                if (!ViewerDrawSuppressBlock.Blocking)
                {
                    // ブロッキングされていない場合は、キューに書き込む
                    messageQueue_.PushMessage(message);
                }
                else
                {
                    // ブロッキングされている場合は、ブロックされたメッセージとしてため込む
                    ViewerDrawSuppressBlock.AddMessage(message);
                }
            }
        }

        public void PushMessages(IEnumerable<BaseMessage> messages)
        {
            if(IsConnected)
            {
                messageQueue_.PushMessages(messages);
            }
        }

        public BaseMessage PopMessage()
        {
            return messageQueue_.PopMessage();
        }

        public BaseMessage[] PopMessages()
        {
            return messageQueue_.PopMessages();
        }

        public void ClearMessages()
        {
            messageQueue_.ClearMessages();
        }

        public void StartCanPush()
        {
            messageQueue_.StartCanPush();
        }

        public void StartCanPush(BaseMessage message)
        {
            messageQueue_.StartCanPush(message);
        }

        public void StartCanPush(IEnumerable<BaseMessage> messages)
        {
            messageQueue_.StartCanPush(messages);
        }

        public void StopCanPush()
        {
            messageQueue_.StopCanPush();
        }

        public void AddSelectedEvent()
        {
            if (IsConnected)
            {
                connecter_.AddSelectedEvent();
            }
        }

        public void RemoveSelectedEvent()
        {
            if (IsConnected)
            {
                connecter_.RemoveSelectedEvent();
            }
        }

        /// <summary>接続状態変更時のデリゲート用関数宣言</summary>
        public delegate void ChangeConnected();

        /// <summary>接続状態変更時のデリゲート関数インスタンス</summary>
        private ChangeConnected callbackfunc = null;

        /// <summary>デリゲート関数追加</summary>
        public void AddChangeConnectedEvent(ChangeConnected func)
        {
            callbackfunc += func;
        }

        /// <summary>デリゲート関数設定</summary>
        public void DeleteChangeConnectedEvent(ChangeConnected func)
        {
            callbackfunc -= func;
        }

        /// <summary>デリゲート関数を実行</summary>
        private void ChangeConnectedEvent()
        {
            if (callbackfunc != null)
            {
                callbackfunc();
            }
        }

        public void AddFileReloadedAction(Action action)
        {
            connecter_.AddFileReloadedAction(action);
        }

        public void RemoveFileReloadedAction(Action action)
        {
            connecter_.RemoveFileReloadedAction(action);
        }

        public void AddFileLoadedAction(Action action)
        {
            connecter_.AddFileLoadedAction(action);
        }

        public void RemoveFileLoadedAction(Action action)
        {
            connecter_.RemoveFileLoadedAction(action);
        }

    }
}
