﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace App.Utility
{
    /// <summary>
    /// 文字列ユーティリティ。
    /// </summary>
    public static class StringUtility
    {
        /// <summary>
        /// "", "_copy", "_copy_copy", ... を列挙します。
        /// </summary>
        public static IEnumerable<string> AppendCopy
        {
            get
            {
                StringBuilder builder = new StringBuilder("");
                while (true)
                {
                    yield return builder.ToString();
                    builder.Append("_copy");
                }
            }
        }

        /// <summary>
        /// 正規表現チェックタイプ。
        /// </summary>
        public enum ValidNameRegExpType
        {
            NumberAlphabetMoreThan0,
            NumberAlphabetMoreThan1,
        };

        /// <summary>
        /// 正規表現チェックタイプと正規表現クラスの変換テーブル。
        /// </summary>
        public static Dictionary<ValidNameRegExpType, Regex> ValidNameRegExpDict = new Dictionary<ValidNameRegExpType, Regex>()
        {
            {ValidNameRegExpType.NumberAlphabetMoreThan0, new Regex("[0-9A-Za-z\\-\\._]*")},
            {ValidNameRegExpType.NumberAlphabetMoreThan1, new Regex("[0-9A-Za-z\\-\\._]+")},
        };

        /// <summary>
        /// 正規表現に一致するかを確認。
        /// </summary>
        public static bool CheckValidName(string input, ValidNameRegExpType type)
        {
            Match match = ValidNameRegExpDict[type].Match(input);
            return match.Success && match.Length == input.Length;
        }

        /// <summary>
        /// original と appendices から ngNames に一致しない名前を生成します。
        /// </summary>
        public static string UniqueName(string original, IEnumerable<string> ngNames, IEnumerable<string> appendices = null)
        {
            if (appendices == null)
            {
                appendices = AppendCopy;
            }

            foreach (var appendix in appendices)
            {
                if (!ngNames.Contains(original + appendix))
                {
                    return original + appendix;
                }
            }

            return null;
        }

        static readonly Regex envVarRegex_ = new Regex("%(.*?)%", RegexOptions.Compiled);

        // パス文字列から、環境変数列を求める
        public static IEnumerable<string> MakeEnvironmentVariablesFromPathString(string src)
        {
            var m = envVarRegex_.Match(src);

            while (m.Success)
            {
                yield return m.Groups[1].Value;
                m = m.NextMatch();
            }
        }

        public static bool TrySafeIntParse(string text, out int value)
        {
            long lval;
            if (!Int64.TryParse(text, out lval))
            {
                value = 0;
                return false;
            }

            if (lval < Int32.MinValue)
            {
                value = Int32.MinValue;
            }
            else if (lval > Int32.MaxValue)
            {
                value = Int32.MaxValue;
            }
            else
            {
                value = (int)lval;
            }
            return true;
        }

        public static bool TrySafeUintParse(string text, out uint value)
        {
            long lval;
            if (!long.TryParse(text, out lval))
            {
                value = 0;
                return false;
            }

            if (lval < uint.MinValue)
            {
                value = uint.MinValue;
            }
            else if (lval > uint.MaxValue)
            {
                value = uint.MaxValue;
            }
            else
            {
                value = (uint)lval;
            }
            return true;
        }

        public static bool TrySafeFloatParse(string text, out float value)
        {
            double lval;
            if (!double.TryParse(text, out lval))
            {
                value = 0.0f;
                return false;
            }

            if (lval < float.MinValue)
            {
                value = float.MinValue;
            }
            else if (lval > float.MaxValue)
            {
                value = float.MaxValue;
            }
            else
            {
                value = (float)lval;
            }
            return true;
        }

        public static int[] ParseVersion(string version)
        {
            return version.Split('.').Select(x =>
            {
                int v;
                if (!int.TryParse(x, out v))
                {
                    v = -1;
                }
                return v;
            }).ToArray();
        }
    }
}
