﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using App.Data;
using System.Diagnostics;

namespace App.Utility
{
    /// <summary>
    /// 数学ユーティリティクラス。
    /// </summary>
    public static class MathUtility
    {
        /// <summary>
        /// 補間計算。
        /// </summary>
        public static float Interpolate(float min1, float max1, float val1, float min2, float max2)
        {
            return (max2 - min2) * ((val1 - min1) / (max1 - min1)) + min2;
        }

        /// <summary>
        /// 補間計算。
        /// ただしmin2 と max2 の間を小数点以下 decimals で丸める
        /// </summary>
        public static double InterpolateRound(double min1, double max1, double val1, double min2, double max2, int decimals)
        {
            double value = (max2 - min2) * ((val1 - min1) / (max1 - min1)) + min2;
            if (decimals >= 0)
            {
                return Math.Round(value, decimals);
            }
            else
            {
                double pow = Math.Pow(10, decimals);
                return Math.Round(value * pow)/pow;
            }
        }
        /// <summary>
        /// 補間計算。
        /// </summary>
        public static double Interpolate(double min1, double max1, double val1, double min2, double max2)
        {
            return (max2 - min2) * ((val1 - min1) / (max1 - min1)) + min2;
        }

        /// <summary>
        /// 入れ替え。
        /// </summary>
        public static void Swap(ref int val1, ref int val2)
        {
            int temp = val1;
            val1 = val2;
            val2 = temp;
        }

        /// <summary>
        /// 入れ替え。
        /// </summary>
        public static void Swap(ref float val1, ref float val2)
        {
            float temp = val1;
            val1 = val2;
            val2 = temp;
        }

        /// <summary>
        /// 入れ替え。
        /// </summary>
        public static void Swap(ref double val1, ref double val2)
        {
            double temp = val1;
            val1 = val2;
            val2 = temp;
        }

        /// <summary>
        /// 切り詰め
        /// </summary>
        public static void Clamp(ref int val, int min, int max)
        {
            if (val < min)
            {
                val = min;
            }
            else if (val > max)
            {
                val = max;
            }
        }

        /// <summary>
        /// 切り詰め
        /// </summary>
        public static void Clamp(ref float val, float min, float max)
        {
            if (val < min)
            {
                val = min;
            }
            else if (val > max)
            {
                val = max;
            }
        }

        public static float ClampedValue(float val, float min, float max)
        {
            if (val < min)
            {
                return min;
            }

            if (val > max)
            {
                return max;
            }

            return val;
        }

        /// <summary>
        /// 2のべき乗か？
        /// </summary>
        public static bool IsPowerOf2(uint number)
        {
            return (number & (number - 1)) == 0;
        }

        public static float ToDegree(float radian)
        {
            return (float)(radian * 180.0 / Math.PI);
        }

        public static double ToDegree(double radian)
        {
            return radian * 180.0 / Math.PI;
        }

        public static float ToRadian(float degree)
        {
            return (float)(degree * Math.PI / 180.0);
        }

        public static double ToRadian(double degree)
        {
            return degree * Math.PI / 180.0;
        }

        static public float LinearInterpolation(float frame, KeyFrame lhs, KeyFrame rhs)
        {
            float length = rhs.Frame - lhs.Frame;
            if (length == 0f)
            {
                return rhs.Value;
            }

            float d = length;
            float p = frame - lhs.Frame;
            float invd = 1f / d;
            float s = p * invd;

            return lhs.Value + (rhs.Value - lhs.Value) * s;
        }

        static public float HermiteInterpolation(float frame, KeyFrame lhs, KeyFrame rhs, float slopeScale = 1.0f)
        {
            float length = rhs.Frame - lhs.Frame;
            if (length == 0f)
            {
                return rhs.Value;
            }

            // NW4R g3d_resanm.cpp HermiteInterpolation() に準拠した実装
            float v0 = lhs.Value;
            float v1 = rhs.Value;
            float t0 = lhs.OutSlope * slopeScale;
            float t1 = rhs.InSlope * slopeScale;
            float p = frame - lhs.Frame;
            float d = length;

            float invd = 1f / d;
            float s = p * invd;
            //Assertion.Operation.True((s >= 0f) && (s <= 1f));
            float s1 = s - 1f;

            return v0 + (((v0 - v1) * ((2f * s) - 3f)) * s * s) + (p * s1 * ((s1 * t0) + (s * t1)));
        }

        /// <summary>
        /// エルミート曲線の最大最小値を求めます。
        /// min, max について、Key : 最小または最大となるフレーム, Value : 最小値または最大値
        /// </summary>
        static public void HermiteMinMax(KeyFrame lhs, KeyFrame rhs, out KeyValuePair<float, float> min, out KeyValuePair<float, float> max)
        {
            if (lhs.Value <= rhs.Value)
            {
                min = new KeyValuePair<float, float>(lhs.Frame, lhs.Value);
                max = new KeyValuePair<float, float>(rhs.Frame, rhs.Value);
            }
            else
            {
                min = new KeyValuePair<float, float>(rhs.Frame, rhs.Value);
                max = new KeyValuePair<float, float>(lhs.Frame, lhs.Value);
            }

            float length = rhs.Frame - lhs.Frame;
            if (length == 0f)
            {
                return;
            }

            float v0 = lhs.Value;
            float v1 = rhs.Value;
            float t0 = lhs.OutSlope;
            float t1 = rhs.InSlope;
            float d = length;

            // a, b, c は s をパラメーターとするエルミット曲線の微分の係数です。a * s^2 + b * s + c
            double a = 6 * (v0 - v1) + 3 * d * (t1 + t0);
            double b = 6 * (v1 - v0) - 2 * d * t1 - 4 * d * t0;
            double c = d * t0;

            // 解
            float[] roots;

            // 解の個数
            if (Math.Abs(a) < 1.0e-6)
            {
                // 一時式
                roots = new float[1] { (float)(-c / b) };
            }
            else
            {
                // 判別式
                double D = b * b - 4 * a * c;
                if (D > 0)
                {
                    double sqrtD = Math.Sqrt(D);
                    roots = new float[2] { (float)((-b + sqrtD) / (2 * a)), (float)((-b - sqrtD) / (2 * a)) };
                }
                else if (D == 0)
                {
                    roots = new float[1] { (float)(-b / (2 * a)) };
                }
                else
                {
                    roots = new float[0] { };
                }
            }

            foreach (var x in roots)
            {
                if (0 < x && x < 1)
                {
                    float y = v0 + (((v0 - v1) * ((2 * x) - 3)) * x * x) + (d * x * (x - 1) * (((x - 1) * t0) + (x * t1)));
                    if (y < min.Value)
                    {
                        min = new KeyValuePair<float, float>((1 - x) * lhs.Frame + x * rhs.Frame, y);
                    }
                    else if (y > max.Value)
                    {
                        max = new KeyValuePair<float, float>((1 - x) * lhs.Frame + x * rhs.Frame, y);
                    }
                }
            }
        }

        /// <summary>
        /// エルミート曲線の最大最小値を求めます。
        /// </summary>
        static public void HermiteMinMax(KeyFrame lhs, KeyFrame rhs, out float min, out float max, float xstart, float xend)
        {
            KeyValuePair<float, float> min1;
            KeyValuePair<float, float> max1;
            HermiteMinMax(lhs, rhs, out min1, out max1, xstart, xend);
            min = min1.Value;
            max = max1.Value;
        }

        /// <summary>
        /// リニア曲線の最大最小値を求めます。
        /// </summary>
        static public void LinearMinMax(KeyFrame lhs, KeyFrame rhs, out float min, out float max, float xstart, float xend)
        {
            min = float.MaxValue;
            max = float.MinValue;
            if (xstart <= lhs.Frame && lhs.Frame <= xend)
            {
                min = Math.Min(lhs.Value, min);
                max = Math.Max(lhs.Value, max);
            }
            else if (lhs.Frame < xstart && xstart < rhs.Frame)
            {
                var v = LinearInterpolation(xstart, lhs, rhs);
                min = Math.Min(v, min);
                max = Math.Max(v, max);
            }

            if (xstart <= rhs.Frame && rhs.Frame <= xend)
            {
                min = Math.Min(rhs.Value, min);
                max = Math.Max(rhs.Value, max);
            }
            else if (lhs.Frame < xend && xend < rhs.Frame)
            {
                var v = LinearInterpolation(xend, lhs, rhs);
                min = Math.Min(v, min);
                max = Math.Max(v, max);
            }
        }

        /// <summary>
        /// ステップ曲線の最大最小値を求めます。
        /// </summary>
        static public void StepMinMax(KeyFrame lhs, KeyFrame rhs, out float min, out float max, float xstart, float xend)
        {
            min = float.MaxValue;
            max = float.MinValue;
            if ((xstart <= lhs.Frame && lhs.Frame <= xend) ||
                (lhs.Frame < xstart && xstart < rhs.Frame))
            {
                min = Math.Min(lhs.Value, min);
                max = Math.Max(lhs.Value, max);
            }

            if (xstart <= rhs.Frame && rhs.Frame <= xend)
            {
                min = Math.Min(rhs.Value, min);
                max = Math.Max(rhs.Value, max);
            }
        }

        /// <summary>
        /// エルミート曲線の最大最小値を求めます。
        /// min, max について、Key : 最小または最大となるフレーム, Value : 最小値または最大値
        /// </summary>
        static public void HermiteMinMax(KeyFrame lhs, KeyFrame rhs, out KeyValuePair<float, float> min, out KeyValuePair<float, float> max, float xstart, float xend)
        {
            float length = rhs.Frame - lhs.Frame;
            if (length == 0f)
            {
                Debug.Assert(xstart <= rhs.Frame && lhs.Frame <= xend);
                if (lhs.Value <= rhs.Value)
                {
                    min = new KeyValuePair<float, float>(lhs.Frame, lhs.Value);
                    max = new KeyValuePair<float, float>(rhs.Frame, rhs.Value);
                }
                else
                {
                    min = new KeyValuePair<float, float>(rhs.Frame, rhs.Value);
                    max = new KeyValuePair<float, float>(lhs.Frame, lhs.Value);
                }
                return;
            }

            min = new KeyValuePair<float, float>(0, float.MaxValue);
            max = new KeyValuePair<float, float>(0, float.MinValue);
            float v0 = lhs.Value;
            float v1 = rhs.Value;
            float t0 = lhs.OutSlope;
            float t1 = rhs.InSlope;
            float d = length;

            // a, b, c は s をパラメーターとするエルミット曲線の微分の係数です。a * s^2 + b * s + c
            double a = 6 * (v0 - v1) + 3 * d * (t1 + t0);
            double b = 6 * (v1 - v0) - 2 * d * t1 - 4 * d * t0;
            double c = d * t0;

            // 解
            List<float> roots;

            // 解の個数
            if (Math.Abs(a) < 1.0e-6)
            {
                // 一時式
                roots = new List<float> { (float)(-c / b) };
            }
            else
            {
                // 判別式
                double D = b * b - 4 * a * c;
                if (D > 0)
                {
                    double sqrtD = Math.Sqrt(D);
                    roots = new List<float> { (float)((-b + sqrtD) / (2 * a)), (float)((-b - sqrtD) / (2 * a)) };
                }
                else if (D == 0)
                {
                    roots = new List<float> { (float)(-b / (2 * a)) };
                }
                else
                {
                    roots = new List<float> { };
                }
            }

            float low = 0;
            if (xstart <= lhs.Frame)
            {
                roots.Add(0);
            }
            else
            {
                low = (xstart - lhs.Frame) / length;
                roots.Add(low);
            }

            float hi = 1;
            if (rhs.Frame <= xend)
            {
                roots.Add(1);
            }
            else
            {
                hi = (xend - lhs.Frame) / length;
                roots.Add(hi);
            }

            foreach (var x in roots)
            {
                if (low <= x && x <= hi)
                {
                    float y = v0 + (((v0 - v1) * ((2 * x) - 3)) * x * x) + (d * x * (x - 1) * (((x - 1) * t0) + (x * t1)));
                    if (y < min.Value)
                    {
                        min = new KeyValuePair<float, float>((1 - x) * lhs.Frame + x * rhs.Frame, y);
                    }

                    if (y > max.Value)
                    {
                        max = new KeyValuePair<float, float>((1 - x) * lhs.Frame + x * rhs.Frame, y);
                    }
                }
            }
        }

        /// <summary>
        /// エルミート曲線の最大最小値を求めます。
        /// </summary>
        static public void HermiteMinMax(KeyFrame lhs, KeyFrame rhs, out float min, out float max)
        {
            KeyValuePair<float, float> minPair;
            KeyValuePair<float, float> maxPair;
            HermiteMinMax(lhs, rhs, out minPair, out maxPair);
            min = minPair.Value;
            max = maxPair.Value;
        }

        static public float HermiteSlope(float frame, KeyFrame lhs, KeyFrame rhs)
        {
            double length = rhs.Frame - (double)lhs.Frame;
            if (length == 0f)
            {
                Debug.Assert(false);
                return 0;
            }

            double v0 = lhs.Value;
            double v1 = rhs.Value;
            double t0 = lhs.OutSlope;
            double t1 = rhs.InSlope;
            double d = length;
            // a, b, c は s をパラメーターとするエルミット曲線の微分の係数です。a * s^2 + b * s + c
            double a = 6 * (v0 - v1) + 3 * d * (t1 + t0);
            double b = 6 * (v1 - v0) - 2 * d * t1 - 4 * d * t0;
            double c = d * t0;
            double s = (frame - (double)lhs.Frame) / d;
            return (float)(((a * s + b) * s + c)/d);
        }
    }
}
