﻿using App.Command;
using App.Data;
using App.PropertyEdit;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace App.Utility
{
    /// <summary>
    /// ライトアニメーションを管理するユーティリティです。
    /// </summary>
    public static class LightAnimationAssignUtility
    {
        /// <summary>
        /// ライトをロードしたときに必要な修正をカーブに施します。
        /// </summary>
        /// <param name="lightAnims">ライトアニメーション</param>
        /// <returns>カーブ修正コマンド。</returns>
        public static EditCommandSet ExecuteFixLightAnimations(LightAnimation[] lightAnims)
        {
            DebugConsole.WriteLine("ExecuteFixLightAnimations");

            var commandSet = new EditCommandSet();

            foreach (var lightAnim in lightAnims)
            {
                var target = new GuiObjectGroup(lightAnim);

                var lightType = lightAnim.Data.type;

                var lightAnimTargets = lightAnim.LightAnimTargets;
                var lightAnimTargetTypes = LightAnimation.GetTargetTypes(lightType);

                var defaultBase = LightAnimation.GetDefaultBases(lightType);

                // Executeするまではコマンドが実行されないので、コマンドで順次変更を加えていくOldLightAnimTargetsは
                // ここで作成したクローンに対して変更情報をためていく
                var oldLightAnimTargets = new Dictionary<light_anim_target_targetType, List<KeyFrame>>(lightAnim.OldLightAnimTargets);

                // プリセットに設定されていないターゲットを削除する
                foreach (var lightAnimTarget in lightAnimTargets)
                {
                    if (lightAnimTargetTypes.Contains(lightAnimTarget.targetType)) continue;
                    if (lightAnimTarget.KeyFrames.Count == 0) continue;

                    var targetType = lightAnimTarget.targetType;

                    var animCurve = new LightAnimationCurveTreeNodeInfo(lightAnim, targetType);

                    // 現在のデータを退避してカーブを削除する
                    {
                        // 退避用にカーブのクローンを作成
                        var oldKeys = ObjectUtility.Clone(animCurve.KeyFrames);

                        // 新しくカーブを追加して、復元用のカーブリストのクローンを作成
                        oldLightAnimTargets[targetType] = oldKeys;
                        var oldLightAnimTargetsClone = new Dictionary<light_anim_target_targetType, List<KeyFrame>>(oldLightAnimTargets);

                        // 復元用のカーブリストにカーブを退避するコマンドを作成
                        commandSet.Add(CreateEditCommand_OldLightAnimTargets(target, oldLightAnimTargetsClone));

                        // カーブを削除する
                        commandSet.Add(CreateEditCommand_LightAnimTargets(target, lightAnim, targetType, null));
                    }
                }

                // プリセットに設定されていて割り当てられていないターゲットを追加する
                foreach (var lightAnimTargetType in lightAnimTargetTypes)
                {
                    var lightAnimTarget = lightAnimTargets.FirstOrDefault(x => x.targetType == lightAnimTargetType);
                    if (lightAnimTarget.KeyFrames.Count == 0)
                    {
                        commandSet.Add(CreateEditCommand_LightAnimTargets(target, lightAnim, lightAnimTargetType, defaultBase[lightAnimTargetType]));
                    }
                }
            }

            var commandSet2 = new EditCommandSet();

            // コマンドの処理が全て終わってからランタイムに情報を送るようにする
            {
                commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                commandSet2.Add(commandSet);
                var lightOwners = lightAnims.Select(x => x.Owner).Distinct().ToArray();
                commandSet2.OnPostEdit += (s, e1) =>
                {
                    foreach (var lightOwner in lightOwners)
                    {
                        Viewer.LoadOrReloadAnimation.Send(lightOwner);
                    }
                };
            }

            return commandSet2;
        }

        /// <summary>
        /// ライトタイプを変更したときに必要な修正をカーブに施します。
        /// </summary>
        /// <param name="lightAnims">ライトアニメーション</param>
        /// <param name="lightType">ライトタイプ</param>
        /// <returns>カーブ修正コマンド。</returns>
        public static EditCommandSet ExecuteFixLightAnimations(LightAnimation[] lightAnims, string lightType)
        {
            DebugConsole.WriteLine("ExecuteFixLightAnimations");

            var commandSet = new EditCommandSet();

            // ライトアニメーションごとの処理
            foreach (var lightAnim in lightAnims)
            {
                if (lightAnim.Data.type == lightType) continue;

                var target = new GuiObjectGroup(lightAnim);

                var oldLightAnimTargetTypes = LightAnimation.GetTargetTypes(lightAnim.Data.type);
                var lightAnimTargetTypes = LightAnimation.GetTargetTypes(lightType);

                var oldDefaultBase = LightAnimation.GetDefaultBases(lightAnim.Data.type);
                var defaultBase = LightAnimation.GetDefaultBases(lightType);

                // Executeするまではコマンドが実行されないので、コマンドで順次変更を加えていくOldLightAnimTargetsは
                // ここで作成したクローンに対して変更情報をためていく
                var oldLightAnimTargets = new Dictionary<light_anim_target_targetType, List<KeyFrame>>(lightAnim.OldLightAnimTargets);

                // カーブごとの処理
                foreach (var lightAnimTarget in lightAnim.LightAnimTargets)
                {
                    var targetType = lightAnimTarget.targetType;

                    bool isOldTargetType = oldLightAnimTargetTypes.Any(x => x == targetType);
                    bool isTargetType = lightAnimTargetTypes.Any(x => x == targetType);

                    // ターゲット -> ターゲット外
                    if (isOldTargetType && !isTargetType)
                    {
                        if (lightAnimTarget.KeyFrames.Count == 0) continue;

                        var animCurve = new LightAnimationCurveTreeNodeInfo(lightAnim, targetType);

                        // 現在のデータを退避してカーブ削除する
                        {
                            // 退避用にカーブのクローンを作成
                            var oldKeys = ObjectUtility.Clone(animCurve.KeyFrames);

                            // 新しくカーブを追加して、復元用のカーブリストのクローンを作成
                            oldLightAnimTargets[targetType] = oldKeys;
                            var oldLightAnimTargetsClone = new Dictionary<light_anim_target_targetType, List<KeyFrame>>(oldLightAnimTargets);

                            // 復元用のカーブリストにカーブを退避するコマンドを作成
                            commandSet.Add(CreateEditCommand_OldLightAnimTargets(target, oldLightAnimTargetsClone));

                            // カーブを削除する
                            commandSet.Add(CreateEditCommand_LightAnimTargets(target, lightAnim, targetType, null));
                        }
                    }
                    // ターゲット外 -> ターゲット
                    else if (!isOldTargetType && isTargetType)
                    {
                        List<KeyFrame> oldKeyFrames = lightAnim.OldLightAnimTargets[targetType];

                        // 復元用のカーブデータがあればそれを復元する
                        if (oldKeyFrames != null)
                        {
                            // 復元用のカーブデータをクローンしてそれを適用する
                            var animCurve = new LightAnimationCurveTreeNodeInfo(lightAnim, targetType);
                            var newKeys = ObjectUtility.Clone(oldKeyFrames);
                            commandSet.Add(AnimationCurveEditCommand.Create(target, GuiObjectID.LightAnimation, animCurve, newKeys));

                            // 復元したカーブをクリアして、復元用のカーブリストのクローンを作成
                            oldLightAnimTargets[targetType] = null;
                            var oldLightAnimTargetsClone = new Dictionary<light_anim_target_targetType, List<KeyFrame>>(oldLightAnimTargets);

                            // 復元用のカーブをクリアするコマンドを作成
                            commandSet.Add(CreateEditCommand_OldLightAnimTargets(target, oldLightAnimTargetsClone));
                        }
                        // カーブデータがなければデフォルト値のキーを追加する
                        else
                        {
                            commandSet.Add(CreateEditCommand_LightAnimTargets(target, lightAnim, targetType, defaultBase[targetType]));
                        }
                    }
                }
            }

            var commandSet2 = new EditCommandSet();

            // コマンドの処理が全て終わってからランタイムに情報を送るようにする
            {
                commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                commandSet2.Add(commandSet);
                var lightOwners = lightAnims.Select(x => x.Owner).Distinct().ToArray();
                commandSet2.OnPostEdit += (s, e1) =>
                {
                    foreach (var lightOwner in lightOwners)
                    {
                        Viewer.LoadOrReloadAnimation.Send(lightOwner);
                    }
                };
            }

            return commandSet2;
        }

        private static ICommand CreateEditCommand_LightAnimTargets(GuiObjectGroup targets, LightAnimation lightAnim, light_anim_target_targetType targetType, float? defaultValue)
        {
            var animCurve = new LightAnimationCurveTreeNodeInfo(lightAnim, targetType);

            var newKeys = new List<KeyFrame>();
            if (defaultValue != null)
            {
                newKeys.Add(
                    new KeyFrame()
                    {
                        Frame = 0,
                        Value = (float)defaultValue
                    }
                );
            }

            return AnimationCurveEditCommand.Create(targets, GuiObjectID.LightAnimation, animCurve, newKeys);
        }

        private static GroupEditCommand CreateEditCommand_OldLightAnimTargets(GuiObjectGroup targets, Dictionary<light_anim_target_targetType, List<KeyFrame>> oldLightAnimTargets)
        {
            return
                new GeneralGroupReferenceEditCommand<Dictionary<light_anim_target_targetType, List<KeyFrame>>>(
                    targets,
                    GuiObjectID.LightAnimation,
                    ObjectUtility.MultipleClone(oldLightAnimTargets, targets.Objects.Count),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        LightAnimation lightAnimation = target as LightAnimation;
                        var oldTargets = (Dictionary<light_anim_target_targetType, List<KeyFrame>>)data;

                        swap = lightAnimation.OldLightAnimTargets;
                        lightAnimation.OldLightAnimTargets = oldTargets;
                    }
                );
        }
    }
}
