﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;
using System.Xml.XPath;
using nw.g3d.nw4f_3dif;

namespace App.Utility
{
    public static class IfUtility
    {
        // メモ：MakeArrayStringの逆の操作をやる関数は
        //       G3dTool\nw4f_3dif\src\Common\G3dDataParser.cs
        //       に実装されています。
        public static string MakeArrayString(IEnumerable<bool> values)
        {
            if (!values.Any()) return null;
            var sb = new StringBuilder();
            {
                foreach (var v in values)
                {
                    sb.Append(string.Format("{0}\t", v? 1: 0));
                }
            }
            return sb.ToString();
        }

        public static string MakeArrayString(IEnumerable<float> values)
        {
            if (!values.Any()) return null;
            var sb = new StringBuilder();
            {
                foreach (var v in values)
                {
                    sb.Append(string.Format("{0:R}\t", v));
                }
            }
            return sb.ToString();
        }

        public static string MakeArrayString<T>(IEnumerable<T> values)
        {
            if (!values.Any()) return null;
            var sb = new StringBuilder();
            {
                foreach (T v in values)
                {
                    sb.Append(string.Format("{0}\t", v));
                }
            }
            return sb.ToString();
        }
#if false
        /// <summary>
        /// T をXmlElement に変換します。level のタブインデントをXmlElement 内部に設定します。
        /// </summary>
        public static XmlElement ConvertToXmlElement<T>(T obj, int level)
        {
            // インデントのレベルは1以上
            Debug.Assert(level > 0);

            // タワーに変換
            var root = new TTower<T>();
            var leaf = root;
            for (int i = 1; i < level; i++)
            {
                leaf.Tower = new TTower<T>();
                leaf = leaf.Tower;
            }
            leaf.Data = obj;

            // シリアライズ
            var stream = new MemoryStream();
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            settings.IndentChars = "\t";
            using (XmlWriter writer = XmlWriter.Create(stream, settings))
            {
                var serializer = new XmlSerializer(typeof(TTower<T>));
                serializer.Serialize(writer, root);
            }
            stream.Seek(0, SeekOrigin.Begin);

            // 変換
            var doc = XDocument.Load(stream, LoadOptions.PreserveWhitespace);
            doc.Root.Name = typeof(XmlElementTower).Name;
            var element = doc.XPathSelectElement("//Data");
            element.Name = typeof(T).Name;

            // デシリアライズ
            XmlElementTower node;
            using (XmlReader reader = doc.CreateReader())
            {
                var deserializer = new XmlSerializer(typeof(XmlElementTower));
                node = deserializer.Deserialize(reader) as XmlElementTower;
            }

            // タワーから要素を取り出します。
            while (node.Tower != null)
            {
                node = node.Tower;
            }
            return node.Elements[0];
        }


        /// <summary>
        /// XmlElement をT に変換します。
        /// </summary>
        public static T ConvertToEditData<T>(XmlElement elem) where T : class
        {
            // シリアライズ
            var doc = new XDocument();
            using (XmlWriter writer = doc.CreateWriter())
            {
                var serializer = new XmlSerializer(typeof(XmlElement));
                serializer.Serialize(writer, elem);
            }

            // 変換
            doc.Root.Name = typeof(T).Name;

            // デシリアライズ
            T t;
            using (XmlReader reader = doc.CreateReader())
            {
                var deserializer = new XmlSerializer(typeof(T));
                t = deserializer.Deserialize(reader) as T;
            }

            return t;
        }
#endif
        public static IList GetIList(this G3dStream stream)
        {
            switch (stream.type)
            {
                case stream_typeType.@byte:
                    return stream.ByteData;
                case stream_typeType.@float:
                    return stream.FloatData;
                case stream_typeType.@int:
                    return stream.IntData;
            }

            return null;
        }

        public enum ShaderParamPrimitiveType
        {
            @bool,
            @int,
            @uint,
            @float,
        };

        public class ShaderParamTypeInfo
        {
            public ShaderParamPrimitiveType[]	PrimitiveType	{ get; set; }
            public shader_param_typeType	Type			{ get; set; }
        };

        public static ShaderParamTypeInfo GetShaderParamTypeInfo(shader_param_typeType type)
        {
            ShaderParamTypeInfo	info = null;

            switch(type)
            {
                case shader_param_typeType.@bool:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@bool, 1).ToArray() };		break;
                case shader_param_typeType.bool2:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@bool, 2).ToArray() };		break;
                case shader_param_typeType.bool3:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@bool, 3).ToArray() };		break;
                case shader_param_typeType.bool4:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@bool, 4).ToArray() };		break;

                case shader_param_typeType.@int:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@int, 1).ToArray() };		break;
                case shader_param_typeType.int2:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@int, 2).ToArray() };		break;
                case shader_param_typeType.int3:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@int, 3).ToArray() };		break;
                case shader_param_typeType.int4:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@int, 4).ToArray() };		break;

                case shader_param_typeType.@uint:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@uint, 1).ToArray() };		break;
                case shader_param_typeType.uint2:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@uint, 2).ToArray() };		break;
                case shader_param_typeType.uint3:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@uint, 3).ToArray() };		break;
                case shader_param_typeType.uint4:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@uint, 4).ToArray() };		break;

                case shader_param_typeType.@float:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 1).ToArray() };		break;
                case shader_param_typeType.float2:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 2).ToArray() };		break;
                case shader_param_typeType.float3:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 3).ToArray() };		break;
                case shader_param_typeType.float4:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 4).ToArray() };		break;
                case shader_param_typeType.float2x2:			info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 2*2).ToArray() };		break;
                case shader_param_typeType.float2x3:			info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 3*2).ToArray() };		break;
                case shader_param_typeType.float2x4:			info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 4*2).ToArray() };		break;
                case shader_param_typeType.float3x2:			info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 2*3).ToArray() };		break;
                case shader_param_typeType.float3x3:			info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 3*3).ToArray() };		break;
                case shader_param_typeType.float3x4:			info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 4*3).ToArray() };		break;
                case shader_param_typeType.float4x2:			info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 2*4).ToArray() };		break;
                case shader_param_typeType.float4x3:			info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 3*4).ToArray() };		break;
                case shader_param_typeType.float4x4:			info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 4*4).ToArray() };		break;

                case shader_param_typeType.srt2d:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 5).ToArray() };		break;
                case shader_param_typeType.srt3d:				info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@float, 3*3).ToArray() };		break;

                case shader_param_typeType.texsrt:			info = new ShaderParamTypeInfo(){ PrimitiveType = Enumerable.Repeat(ShaderParamPrimitiveType.@uint, 1).Concat(Enumerable.Repeat(ShaderParamPrimitiveType.@float,  2+1+2)).ToArray() };	break;
            }

            Debug.Assert(info != null);

            info.Type = type;

            return info;
        }

        private static readonly Dictionary<curve_frame_typeType, int> curveFrameTypeDict = new Dictionary<curve_frame_typeType, int>()
        {
            {curve_frame_typeType.none,		4},
            {curve_frame_typeType.frame8,	1},
            {curve_frame_typeType.frame16,	2},
            {curve_frame_typeType.frame32,	4},
        };

        private static readonly Dictionary<curve_key_typeType, int> curveKeyTypeDict = new Dictionary<curve_key_typeType, int>()
        {
            {curve_key_typeType.none,		4},
            {curve_key_typeType.key8,		1},
            {curve_key_typeType.key16,		2},
            {curve_key_typeType.key32,		4},
        };

        public static int GetByteSizeFromCurveFrameType(curve_frame_typeType type)
        {
            return curveFrameTypeDict[type];
        }

        public static int GetByteSizeFromCurveKeyType(curve_key_typeType type)
        {
            return curveKeyTypeDict[type];
        }

        /// <summary>
        /// G3dArrayElement の要素を取得
        /// </summary>
        internal static IEnumerable<T> GetItems<T>(this G3dArrayElement<T> array) where T : IG3dIndexHintElement
        {
            if (array == null)
            {
                return Enumerable.Empty<T>();
            }

            return array.Items;
        }

        /// <summary>
        /// 要素の列をG3dArrayElement に変換
        /// </summary>
        internal static S ToG3dArrayElement<T, S>(this IEnumerable<T> items)
            where T : IG3dIndexHintElement
            where S : G3dArrayElement<T>, new()
        {
            if (items == null)
            {
                return null;
            }

            T[] array = items.ToArray();

            if (array.Length == 0)
            {
                return null;
            }

            return new S { Items = array };
        }

        public static bool IsSameStream(G3dStream lhs, G3dStream rhs)
        {
            if (lhs == rhs)
            {
                return true;
            }

            if (lhs.type != rhs.type)
            {
                return false;
            }

            switch (lhs.type)
            {
                case stream_typeType.@byte:
                    return IsSameList(lhs.ByteData, rhs.ByteData);
                case stream_typeType.@float:
                    return IsSameList(lhs.FloatData, rhs.FloatData);
                case stream_typeType.@int:
                    return IsSameList(lhs.IntData, rhs.IntData);
                case stream_typeType.@string:
                case stream_typeType.wstring:
                    return lhs.StringData == rhs.StringData;
            }

            return false;
        }

        private static bool IsSameList<T>(List<T> lhs, List<T> rhs) where T : struct
        {
            return lhs.Count == rhs.Count && lhs.SequenceEqual(rhs);
        }
    }

    /// <summary>
    /// stream の比較結果をキャッシュする
    /// </summary>
    public class G3dStreamCachedComparer
    {
        public G3dStreamCachedComparer(G3dStream s)
        {
            stream = s;
        }

        private G3dStream stream;
        private List<WeakReference<G3dStream>> sames = new List<WeakReference<G3dStream>>();
        private List<WeakReference<G3dStream>> notsames = new List<WeakReference<G3dStream>>();
        public bool IsSame(G3dStream s)
        {
            // 参照として同じか
            if (s == stream)
            {
                return true;
            }

            // 型が同じか
            if (s.type != stream.type)
            {
                return false;
            }

            // 要素数が同じか
            switch (stream.type)
            {
                case stream_typeType.@byte:
                    if (stream.ByteData.Count != s.ByteData.Count)
                    {
                        return false;
                    }
                    break;
                case stream_typeType.@float:
                    if (stream.FloatData.Count != s.FloatData.Count)
                    {
                        return false;
                    }
                    break;
                case stream_typeType.@int:
                    if (stream.IntData.Count != s.IntData.Count)
                    {
                        return false;
                    }
                    break;
                case stream_typeType.@string:
                case stream_typeType.wstring:
                    if (stream.StringData.Length != s.StringData.Length)
                    {
                        return false;
                    }
                    break;
            }

            // 等しいもののリストから探す
            var items = sames.ToArray();
            sames.Clear();
            bool found = false;
            foreach (var item in items)
            {
                G3dStream target;
                if (item.TryGetTarget(out target))
                {
                    sames.Add(item);
                    if (target == s)
                    {
                        found = true;
                    }

                }
            }

            if (found)
            {
                return true;
            }

            // 等しくないもののリストから探す
            items = notsames.ToArray();
            notsames.Clear();
            foreach (var item in items)
            {
                G3dStream target;
                if (item.TryGetTarget(out target))
                {
                    notsames.Add(item);
                    if (target == s)
                    {
                        found = true;
                    }
                }
            }

            if (found)
            {
                return false;
            }

            // 比較
            if (IfUtility.IsSameStream(s, stream))
            {
                sames.Add(new WeakReference<G3dStream>(s));
                return true;
            }
            else
            {
                notsames.Add(new WeakReference<G3dStream>(s));
                return false;
            }
        }
    }

#if false
    /// <summary>
    /// 変換用ヘルパクラス
    /// </summary>
    public class TTower<T>
    {
        [XmlElement]
        public TTower<T> Tower;

        public T Data;
    }

    /// <summary>
    /// 変換用ヘルパクラス
    /// </summary>
    public class XmlElementTower
    {
        [XmlElement]
        public XmlElementTower Tower;

        [XmlAnyElement]
        public XmlElement[] Elements;
    }
#endif
}
