﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using System.Diagnostics;
namespace App.Utility
{
    #region PointUtility
    /// <summary>
    /// ポイントユーティリティクラス。
    /// </summary>
    public static class PointUtility
    {
        /// <summary>
        /// ２点間の距離を取得。
        /// </summary>
        public static float GetDistance(PointF pt1, PointF pt2)
        {
            return (float)Math.Sqrt((pt2.X - pt1.X) * (pt2.X - pt1.X) + (pt2.Y - pt1.Y) * (pt2.Y - pt1.Y));
        }
    }
    #endregion

    #region RectangleUtility
    /// <summary>
    /// レクタングルユーティリティクラス。
    /// </summary>
    public static class RectangleUtility
    {
        /// <summary>
        /// 左位置を設定。
        /// </summary>
        public static void SetLeft(ref Rectangle rect, int value)
        {
            if (value > rect.Left) { rect.Width -= value - rect.Left; }
            else { rect.Width += rect.Left - value; }
            rect.X = value;
        }

        /// <summary>
        /// 左位置を設定。
        /// </summary>
        public static void SetLeft(ref RectangleF rect, float value)
        {
            if (value > rect.Left) { rect.Width -= value - rect.Left; }
            else { rect.Width += rect.Left - value; }
            rect.X = value;
        }

        /// <summary>
        /// 上位置を設定。
        /// </summary>
        public static void SetTop(ref Rectangle rect, int value)
        {
            if (value > rect.Top) { rect.Height -= value - rect.Top; }
            else { rect.Height += rect.Top - value; }
            rect.Y = value;
        }

        /// <summary>
        /// 上位置を設定。
        /// </summary>
        public static void SetTop(ref RectangleF rect, float value)
        {
            if (value > rect.Top) { rect.Height -= value - rect.Top; }
            else { rect.Height += rect.Top - value; }
            rect.Y = value;
        }

        /// <summary>
        /// 右位置を設定。
        /// </summary>
        public static void SetRight(ref Rectangle rect, int value)
        {
            if (value > rect.Right) { rect.Width += value - rect.Right; }
            else { rect.Width -= rect.Right - value; }
        }

        /// <summary>
        /// 右位置を設定。
        /// </summary>
        public static void SetRight(ref RectangleF rect, float value)
        {
            if (value > rect.Right) { rect.Width += value - rect.Right; }
            else { rect.Width -= rect.Right - value; }
        }

        /// <summary>
        /// 下位置を設定。
        /// </summary>
        public static void SetBottom(ref Rectangle rect, int value)
        {
            if (value > rect.Bottom) { rect.Height += value - rect.Bottom; }
            else { rect.Height -= rect.Bottom - value; }
        }

        /// <summary>
        /// 下位置を設定。
        /// </summary>
        public static void SetBottom(ref RectangleF rect, float value)
        {
            if (value > rect.Bottom) { rect.Height += value - rect.Bottom; }
            else { rect.Height -= rect.Bottom - value; }
        }

        /// <summary>
        /// 左位置を移動。
        /// </summary>
        public static void OffsetLeft(ref Rectangle rect, int delta)
        {
            rect.X += delta;
            rect.Width -= delta;
        }

        /// <summary>
        /// 左位置を移動。
        /// </summary>
        public static void OffsetLeft(ref RectangleF rect, float delta)
        {
            rect.X += delta;
            rect.Width -= delta;
        }

        /// <summary>
        /// 上位置を移動。
        /// </summary>
        public static void OffsetTop(ref Rectangle rect, int delta)
        {
            rect.Y += delta;
            rect.Height -= delta;
        }

        /// <summary>
        /// 上位置を移動。
        /// </summary>
        public static void OffsetTop(ref RectangleF rect, float delta)
        {
            rect.Y += delta;
            rect.Height -= delta;
        }

        /// <summary>
        /// 右位置を移動。
        /// </summary>
        public static void OffsetRight(ref Rectangle rect, int delta)
        {
            rect.Width += delta;
        }

        /// <summary>
        /// 右位置を移動。
        /// </summary>
        public static void OffsetRight(ref RectangleF rect, float delta)
        {
            rect.Width += delta;
        }

        /// <summary>
        /// 下位置を移動。
        /// </summary>
        public static void OffsetBottom(ref Rectangle rect, int delta)
        {
            rect.Height += delta;
        }

        /// <summary>
        /// 下位置を移動。
        /// </summary>
        public static void OffsetBottom(ref RectangleF rect, float delta)
        {
            rect.Height += delta;
        }

        /// <summary>
        /// 中心位置を取得。
        /// </summary>
        public static Point GetCenter(Rectangle rect)
        {
            return new Point(
                (rect.Left + rect.Right) / 2,
                (rect.Top + rect.Bottom) / 2
            );
        }

        /// <summary>
        /// 中心位置を取得。
        /// </summary>
        public static PointF GetCenter(RectangleF rect)
        {
            return new PointF(
                (rect.Left + rect.Right) / 2.0f,
                (rect.Top + rect.Bottom) / 2.0f
            );
        }

        /// <summary>
        /// 左上位置を取得。
        /// </summary>
        public static Point GetTopLeft(Rectangle rect)
        {
            return new Point(rect.Left, rect.Top);
        }

        /// <summary>
        /// 左上位置を取得。
        /// </summary>
        public static PointF GetTopLeft(RectangleF rect)
        {
            return new PointF(rect.Left, rect.Top);
        }

        /// <summary>
        /// 右上位置を取得。
        /// </summary>
        public static Point GetTopRight(Rectangle rect)
        {
            return new Point(rect.Right, rect.Top);
        }

        /// <summary>
        /// 右上位置を取得。
        /// </summary>
        public static PointF GetTopRight(RectangleF rect)
        {
            return new PointF(rect.Right, rect.Top);
        }

        /// <summary>
        /// 左下位置を取得。
        /// </summary>
        public static Point GetBottomLeft(Rectangle rect)
        {
            return new Point(rect.Left, rect.Bottom);
        }

        /// <summary>
        /// 左下位置を取得。
        /// </summary>
        public static PointF GetBottomLeft(RectangleF rect)
        {
            return new PointF(rect.Left, rect.Bottom);
        }

        /// <summary>
        /// 右下位置を取得。
        /// </summary>
        public static Point GetBottomRight(Rectangle rect)
        {
            return new Point(rect.Right, rect.Bottom);
        }

        /// <summary>
        /// 右下位置を取得。
        /// </summary>
        public static PointF GetBottomRight(RectangleF rect)
        {
            return new PointF(rect.Right, rect.Bottom);
        }

        /// <summary>
        /// ２点から作成。
        /// </summary>
        public static Rectangle FromPoints(Point pt1, Point pt2)
        {
            return new Rectangle(
                Math.Min(pt1.X, pt2.X),
                Math.Min(pt1.Y, pt2.Y),
                Math.Abs(pt2.X - pt1.X),
                Math.Abs(pt2.Y - pt1.Y)
            );
        }

        /// <summary>
        /// ２点から作成。
        /// </summary>
        public static RectangleF FromPoints(PointF pt1, PointF pt2)
        {
            return new RectangleF(
                Math.Min(pt1.X, pt2.X),
                Math.Min(pt1.Y, pt2.Y),
                Math.Abs(pt2.X - pt1.X),
                Math.Abs(pt2.Y - pt1.Y)
            );
        }

        /// <summary>
        /// RECT型に変換。
        /// </summary>
        public static Win32.RECT ToRECT(Rectangle rect)
        {
            Win32.RECT result = new Win32.RECT();
            result.left = rect.Left;
            result.top = rect.Top;
            result.right = rect.Right;
            result.bottom = rect.Bottom;
            return result;
        }
    }
    #endregion

    #region GraphicsUtility
    /// <summary>
    /// 描画ユーティリティクラス。
    /// </summary>
    public static class GraphicsUtility
    {
        //---------------------------------------------------------------------
        // 四角形
        //---------------------------------------------------------------------
        /// <summary>
        /// 四角形を描画。
        /// </summary>
        public static void DrawRectangle(Graphics g, Pen pen, Rectangle rect)
        {
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// 四角形を描画。
        /// </summary>
        public static void DrawRectangle(Graphics g, Pen pen, RectangleF rect)
        {
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// 四角形を描画。
        /// </summary>
        public static void DrawCrossRectangle(Graphics g, Pen pen, Rectangle rect)
        {
            g.DrawLine(pen, rect.X, rect.Y, rect.Right - 1, rect.Bottom - 1);
            g.DrawLine(pen, rect.X, rect.Bottom - 1, rect.Right - 1, rect.Y);
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// 四角形を描画。
        /// </summary>
        public static void DrawCrossRectangle(Graphics g, Pen pen, RectangleF rect)
        {
            g.DrawLine(pen, rect.X, rect.Y, rect.Right - 1, rect.Bottom - 1);
            g.DrawLine(pen, rect.X, rect.Bottom - 1, rect.Right - 1, rect.Y);
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// 丸みのある四角形を描画。
        /// </summary>
        public static void DrawRoundishRectangle(Graphics g, Pen pen, Rectangle rect)
        {
            rect.Width -= 1;
            rect.Height -= 1;

            g.DrawLine(pen, rect.Left + 1, rect.Top, rect.Right - 1, rect.Top);
            g.DrawLine(pen, rect.Right, rect.Top + 1, rect.Right, rect.Bottom - 1);
            g.DrawLine(pen, rect.Left, rect.Top + 1, rect.Left, rect.Bottom - 1);
            g.DrawLine(pen, rect.Left + 1, rect.Bottom, rect.Right - 1, rect.Bottom);
        }

        /// <summary>
        /// 丸みのある四角形を描画。
        /// </summary>
        public static void DrawRoundishRectangle(Graphics g, Pen pen, RectangleF rect)
        {
            rect.Width -= 1;
            rect.Height -= 1;

            g.DrawLine(pen, rect.Left + 1, rect.Top, rect.Right - 1, rect.Top);
            g.DrawLine(pen, rect.Right, rect.Top + 1, rect.Right, rect.Bottom - 1);
            g.DrawLine(pen, rect.Left, rect.Top + 1, rect.Left, rect.Bottom - 1);
            g.DrawLine(pen, rect.Left + 1, rect.Bottom, rect.Right - 1, rect.Bottom);
        }

        /// <summary>
        /// ３Ｄ四角形を描画。
        /// </summary>
        public static void Draw3DRectangle(Graphics g, Pen penLT, Pen penRB, RectangleF rect)
        {
            rect.Width -= 1;
            rect.Height -= 1;

            g.DrawLine(penLT, rect.Left, rect.Top, rect.Right - 1, rect.Top);
            g.DrawLine(penLT, rect.Left, rect.Top, rect.Left, rect.Bottom - 1);
            g.DrawLine(penRB, rect.Right, rect.Top, rect.Right, rect.Bottom);
            g.DrawLine(penRB, rect.Left, rect.Bottom, rect.Right, rect.Bottom);
        }

        /// <summary>
        /// フォーカス四角形を描画。
        /// </summary>
        public static void DrawFocusRectangle(Graphics g, Rectangle rect)
        {
            IntPtr hdc = g.GetHdc();
            Win32.RECT rc = RectangleUtility.ToRECT(rect);
            Win32.NativeMethods.DrawFocusRect(hdc, ref rc);
            g.ReleaseHdc(hdc);
        }

        //---------------------------------------------------------------------
        // テキスト
        //---------------------------------------------------------------------
        /// <summary>
        /// 矩形領域にテキストを描画。
        /// </summary>
        public static void DrawText(Graphics g, string text, Font font, Color color, Rectangle rect, HorizontalAlignment alignment)
        {
            DrawText(g, text, font, color, rect, alignment, null);
        }

        /// <summary>
        /// 矩形領域にテキスト（＋イメージ）を描画。
        /// </summary>
        public static void DrawText(Graphics g, string text, Font font, Color color, Rectangle rect, HorizontalAlignment alignment, Image image)
        {
            // イメージ
            if (image != null)
            {
                var rcImageSrc = new Rectangle(0, 0, image.Width, image.Height);
                var rcImageDst = rect;
                RectangleUtility.OffsetLeft(ref rcImageDst, 2);
                rcImageDst.Y += (rect.Height - image.Height) / 2;
                rcImageDst.Height = image.Height;
                if (rcImageDst.Width > image.Width) rcImageDst.Width = image.Width;

                g.DrawImageUnscaledAndClipped(image, rcImageDst);

                // テキスト描画用にオフセット
                RectangleUtility.OffsetLeft(ref rect, rcImageDst.Right - rect.Left + 1);
            }

            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1814#8
            // 文字がダブって表示される問題の対策
            if (rect.Width <= 0) return;

            // テキスト
            using (StringFormat sf = new StringFormat())
            {
                sf.LineAlignment = StringAlignment.Center;
                sf.Trimming = StringTrimming.None;
                sf.FormatFlags |= StringFormatFlags.NoWrap;

                switch (alignment)
                {
                    case HorizontalAlignment.Left: sf.Alignment = StringAlignment.Near; break;
                    case HorizontalAlignment.Right: sf.Alignment = StringAlignment.Far; break;
                    case HorizontalAlignment.Center: sf.Alignment = StringAlignment.Center; break;
                    default: break;
                }

                using (Brush brush = new SolidBrush(color))
                {
                    g.DrawString(text, font, brush, rect, sf);
                }
            }
        }

        //---------------------------------------------------------------------
        // カラー
        //---------------------------------------------------------------------
        /// <summary>
        /// カラーボックスを描画。
        /// </summary>
        public static void DrawColorBox(Graphics g, Rectangle bounds, Color color, bool enableAlpha)
        {
            // 枠線色
            Color lineColor = Color.Black;
            if (color.R == 0 && color.G == 0 && color.B == 0)
            {
                lineColor = Color.DimGray;
            }

            // カラー部
            using (Brush brush = new SolidBrush(Color.FromArgb(255, color)))
            {
                g.FillRectangle(brush, bounds);
            }

            // アルファ部
            if (enableAlpha)
            {
                Rectangle rcAlpha = bounds;
                RectangleUtility.OffsetLeft(ref rcAlpha, bounds.Width / 4 * 3);

                using (Brush brush = new HatchBrush(HatchStyle.LargeCheckerBoard, Color.LightGray, Color.Black))
                {
                    g.FillRectangle(brush, rcAlpha);
                }
                using (Brush brush = new SolidBrush(Color.FromArgb(color.A, Color.White)))
                {
                    g.FillRectangle(brush, rcAlpha);
                }

                // 罫線
                if (color.A == 0)
                {
                    lineColor = Color.DimGray;
                }
                using (Pen pen = new Pen(lineColor))
                {
                    g.DrawLine(pen, rcAlpha.Left, rcAlpha.Top, rcAlpha.Left, rcAlpha.Bottom - 1);
                }
            }

            // 枠線
            using (Pen pen = new Pen(lineColor))
            {
                GraphicsUtility.DrawRectangle(g, pen, bounds);
            }
        }

        //---------------------------------------------------------------------
        // 特殊項目
        //---------------------------------------------------------------------
        /// <summary>
        /// サイズ変更グリップを描画。
        /// </summary>
        public static void DrawSizeGrip(Graphics g, Control control)
        {
            // Form.OnPaint()で実装されているグリップ描画と同じ方法
            Size szClient = control.ClientSize;
            ControlPaint.DrawSizeGrip(g, control.BackColor, szClient.Width - 0x10, szClient.Height - 0x10, 0x10, 0x10);
        }

        public static int DrawTextureImage(Texture texture, Graphics graphics, Rectangle bounds)
        {
            Rectangle rcImage = bounds;
            rcImage.X += 2;
            rcImage.Y += 2;
            rcImage.Width  = Texture.ThumbnailSize;
            rcImage.Height = Texture.ThumbnailSize;

            texture.DrawColorThumbnail(graphics, rcImage, true, true);

            Rectangle rcFrame = rcImage;
            rcFrame.X -= 1;
            rcFrame.Y -= 1;

            GraphicsUtility.DrawRectangle(graphics, Pens.Black, rcFrame);

            // アルファイメージ
            rcImage.X += (rcImage.Width + 1);
            if (texture.HasAlpha)
            {
                texture.DrawAlphaThumbnail(graphics, rcImage, true, true);

                rcFrame = rcImage;
                rcFrame.X -= 1;
                rcFrame.Y -= 1;

                GraphicsUtility.DrawRectangle(graphics, Pens.Black, rcFrame);
            }

            rcImage.X += rcImage.Width + 1;

            return rcImage.X;
        }

        public static void DrawImageSafe(this Graphics graphics, Bitmap bitmap, float x, float y)
        {
            var minX = -bitmap.Width;
            var minY = -bitmap.Height;
            var maxX = graphics.ClipBounds.Width;
            var maxY = graphics.ClipBounds.Height;

            if ((x >= minX) && (x <= maxX) &&
                (y >= minY) && (y <= maxY))
            {
                graphics.DrawImage(bitmap, x, y);
            }
        }

        public static void DrawImageSafe(this Graphics graphics, Bitmap bitmap, float x, float y, float width, float height)
        {
            var minX = -width;
            var minY = -height;
            var maxX = graphics.ClipBounds.Width;
            var maxY = graphics.ClipBounds.Height;

            if ((x >= minX) && (x <= maxX) &&
                (y >= minY) && (y <= maxY))
            {
                graphics.DrawImage(bitmap, x, y, width, height);
            }
        }

        public static void DrawGrayImage(
            Graphics graphics,
            Image image,
            int srcX, int srcY, int srcW, int srcH,
            int dstX, int dstY, int dstW, int dstH
        )
        {
            const float r = 0.298912F;
            const float g = 0.586611F;
            const float b = 0.114478F;

            float[][] matrixElement =
                      {new []{r,    r,    r,    0.0f, 0.0f},
                       new []{g,    g,    g,    0.0f, 0.0f},
                       new []{b,    b,    b,    0.0f, 0.0f},
                       new []{0.0f, 0.0f, 0.0f, 1.0f, 0.0f},
                       new []{0.0f, 0.0f, 0.0f, 0.0f, 1.0f}};
            var matrix = new ColorMatrix(matrixElement);
            var attr = new ImageAttributes();
            attr.SetColorMatrix(matrix);

            graphics.DrawImage(image,
                new Rectangle(dstX, dstY, dstW, dstH),
                srcX, srcY, srcW, srcH,
                GraphicsUnit.Pixel,
                attr
            );
        }

        public static void DrawGrayImageSafe(
            Graphics graphics,
            Image image,
            int srcX, int srcY, int srcW, int srcH,
            float dstX, float dstY, float dstW, float dstH
        )
        {
            const float r = 0.298912F;
            const float g = 0.586611F;
            const float b = 0.114478F;

            float[][] matrixElement =
                      {new []{r,    r,    r,    0.0f, 0.0f},
                       new []{g,    g,    g,    0.0f, 0.0f},
                       new []{b,    b,    b,    0.0f, 0.0f},
                       new []{0.0f, 0.0f, 0.0f, 1.0f, 0.0f},
                       new []{0.0f, 0.0f, 0.0f, 0.0f, 1.0f}};
            var matrix = new ColorMatrix(matrixElement);
            var attr = new ImageAttributes();
            attr.SetColorMatrix(matrix);

            var minX = -dstW;
            var minY = -dstH;
            var maxX = graphics.ClipBounds.Width;
            var maxY = graphics.ClipBounds.Height;

            if ((dstX >= minX) && (dstX <= maxX) &&
                (dstY >= minY) && (dstY <= maxY))
            {
                graphics.DrawImage(image,
                    new Rectangle((int)dstX, (int)dstY, (int)dstW, (int)dstH),
                    srcX, srcY, srcW, srcH,
                    GraphicsUnit.Pixel,
                    attr
                );
            }
        }

        /// <summary>
        /// Grpahics により文字列を描画したときの幅
        /// </summary>
        public static int MeasureString(string str)
        {
            using (var g = Graphics.FromHwnd(IntPtr.Zero))
            {
                return (int)g.MeasureString(str, TheApp.GuiFont).Width;
            }
        }

        public static void SafeFillRectangle(this Graphics g, Brush brush, double x, double y, double width, double height)
        {
            int X = (int)Math.Max(g.ClipBounds.X - 1, x);
            int Y = (int)Math.Max(g.ClipBounds.Y - 1, y);
            int Right = (int)Math.Min(g.ClipBounds.Right + 1, x + width);
            int Bottom = (int)Math.Min(g.ClipBounds.Bottom + 1, y + height);
            if (Right - X > 0 && Bottom - Y > 0)
            {
                g.FillRectangle(brush, X, Y, Right - X, Bottom - Y);
            }
        }

        public static void SafeDrawRectangle(this Graphics g, Pen pen, double x, double y, double width, double height)
        {
            int X = (int)Math.Max(g.ClipBounds.X - 1, x);
            int Y = (int)Math.Max(g.ClipBounds.Y - 1, y);
            int Right = (int)Math.Min(g.ClipBounds.Right + 1, x + width);
            int Bottom = (int)Math.Min(g.ClipBounds.Bottom + 1, y + height);
            if (Right - X > 0 && Bottom - Y > 0)
            {
                g.DrawRectangle(pen, X, Y, Right - X, Bottom - Y);
            }
        }

        public static void SafeDrawLine(this Graphics g, Pen pen, double x1, double y1, double x2, double y2)
        {
            if (ClipLineWithRectangleF(g.ClipBounds, ref x1, ref y1, ref x2, ref y2))
            {
                g.DrawLine(pen, (int)x1, (int)y1, (int)x2, (int)y2);
            }
        }

        public static void SafeDrawLines(this Graphics g, Pen pen, List<PointF> points)
        {
            var clipedPoints = new List<PointF>();
            var rect = g.ClipBounds;
            //rect.Offset(10, 10);
            PointF last = new PointF(float.NaN, float.NaN);
            for (int i = 1; i < points.Count; i++)
            {
                double x1 = points[i-1].X;
                double y1 = points[i-1].Y;
                double x2 = points[i].X;
                double y2 = points[i].Y;
                if (ClipLineWithRectangleF(rect, ref x1, ref y1, ref x2, ref y2))
                {
                    PointF p1 = new PointF((float)x1, (float)y1);
                    PointF p2 = new PointF((float)x2, (float)y2);
                    if (last != p1)
                    {
                        clipedPoints.Add(p1);
                        last = p1;
                    }
                    if (last != p2)
                    {
                        clipedPoints.Add(p2);
                        last = p2;
                    }
                }
            }

            if (clipedPoints.Count >= 2)
            {
                g.DrawLines(pen, clipedPoints.ToArray());
            }
        }


        public static void SafeDrawBezier(this Graphics g, Pen pen, PointF p0, PointF p1, PointF p2, PointF p3)
        {
            var points = new PointF[] {p0, p1, p2, p3};

            // オーバーフロー例外を防ぐために、適当な値でクランプする
            for (var i = 0; i < points.Count(); i++)
            {
                points[i].X = MathUtility.ClampedValue(points[i].X, Int16.MinValue, Int16.MaxValue);
                points[i].Y = MathUtility.ClampedValue(points[i].Y, Int16.MinValue, Int16.MaxValue);
            }

            try
            {
                g.DrawBeziers(pen, points);
            }
            catch (Exception e)
            {
                Debug.Print("DrawBeziers Exception: {0}", e.Message);
            }
        }


        public static bool ClipLineWithRectangleF(RectangleF rect, ref double x1, ref double y1, ref double x2, ref double y2)
        {
            double top = rect.Top;
            double bottom = rect.Bottom;
            double left = rect.Left;
            double right = rect.Right;
            if (x1 < left)
            {
                if (x2 < left)
                {
                    return false;
                }

                y1 = MathUtility.Interpolate(x1, x2, left, y1, y2);
                x1 = left;
            }
            else if (x2 < left)
            {
                y2 = MathUtility.Interpolate(x1, x2, left, y1, y2);
                x2 = left;
            }

            if (right < x1)
            {
                if (right < x2)
                {
                    return false;
                }

                y1 = MathUtility.Interpolate(x1, x2, right, y1, y2);
                x1 = right;
            }
            else if (right < x2)
            {
                y2 = MathUtility.Interpolate(x1, x2, right, y1, y2);
                x2 = right;
            }

            if (y1 < top)
            {
                if (y2 < top)
                {
                    return false;
                }

                x1 = MathUtility.Interpolate(y1, y2, top, x1, x2);
                y1 = top;
            }
            else if (y2 < top)
            {
                x2 = MathUtility.Interpolate(y1, y2, top, x1, x2);
                y2 = top;
            }

            if (bottom < y1)
            {
                if (bottom < y2)
                {
                    return false;
                }

                x1 = MathUtility.Interpolate(y1, y2, bottom, x1, x2);
                y1 = bottom;
            }
            else if (bottom < y2)
            {
                x2 = MathUtility.Interpolate(y1, y2, bottom, x1, x2);
                y2 = bottom;
            }

            Debug.Assert(left - 10 < x1 && x1 < right + 10 && left - 10 < x2 && x2 < right + 10 &&
                top - 10 < y1 && y1 < bottom + 10 && top - 10 < y2 && y2 < bottom + 10);
            Debug.Assert(x1 != double.NaN && y1 != double.NaN && x2 != double.NaN && y2 != double.NaN);

            return true;
        }
    }
    #endregion
}

