﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Properties;
using App.res;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class AnimationObjectPropertyPanel : ObjectPropertyPanel
    {
        public event EventHandler AllShowNodeChanged = null;
        public event EventHandler ShaderParamAnimGroupCondShowChanged = null;
        public event EventHandler OnlyTargetShowNodeChanged = null;
        public event EventHandler ShowKeyedMaterialChanged = null;
        public event EventHandler SortByAlphabeticalChanged = null;
        public event EventHandler ResetNodeExpandClick = null;
        public event EventHandler OpenNodeExpandClick = null;
        public event EventHandler VisibleAllNodeClick = null;
        public event EventHandler InvisibleAllNodeClick = null;
        public event EventHandler SearchTextChanged = null;

        public bool IsAllShowNode{ get; private set; }
        public bool IsShaderParamAnimGroupCondShow{ get; private set; }
        public bool IsSortByAlphabetical{ get; private set; }
        public bool IsShowKeyedMaterial { get; private set; }

        private bool isOnlyTargetShowNode_;
        public bool IsOnlyTargetShowNode
        {
            get
            {
                return isOnlyTargetShowNode_;
            }

            private set
            {
                isOnlyTargetShowNode_ = value;

                // 押しピンを有効/無効も同時に切り替える
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=968#7
                Owner.TargetFixed = IsOnlyTargetShowNode;
            }
        }

        public List<string> SearchTexts { get; set; }
        public event EventHandler PreWrapChanged = null;
        public event EventHandler PostWrapChanged = null;
        public event EventHandler CompressEnableChanged = null;
        public event EventHandler BinarizeVisibilityChanged = null;

        // シェーダーパラメーターアニメーションのコンディションごとの表示・非表示
        protected bool IsShaderParamAnimGroupCondShowVisible
        {
            get { return tsmShaderParamAnimGroupCondShow.Visible;	}
            set { tsmShaderParamAnimGroupCondShow.Visible = value;	}
        }

        protected bool IsOnlyTargetShowNodeVisible
        {
            get { return tsmOnlyTargetShowNode.Visible;	}
            set { tsmOnlyTargetShowNode.Visible = value;	}
        }

        protected bool IsSortByAlphabeticalVisible
        {
            get { return tsmSortNode.Visible; }
            set
            {
                tsmSortNode.Visible = value;
                tssSortNode.Visible = value;
            }
        }

        protected bool IsShowKeyedMaterialVisible
        {
            get { return tsmShowKeyedMaterial.Visible; }
            set { tsmShowKeyedMaterial.Visible = value; }
        }

        const int EM_SETCUEBANNER = 0x1500 + 1;

        public AnimationObjectPropertyPanel()
        {

            InitializeComponent();

            SearchTexts = new List<string>();

            // ツールチップの設定
            var hintToolTip = new ToolTip();
            hintToolTip.SetToolTip(tbxSearch, res.Strings.CurveTreeView_InputSearchWords);

            // ウォーターマークテキストの設定
            var setCueBanner = (Action)(() =>
            {
                var ptr = Marshal.StringToBSTR(res.Strings.CurveTreeView_Search);
                Win32.NativeMethods.SendMessage(tbxSearch.Handle, EM_SETCUEBANNER, IntPtr.Zero, ptr);
            });

            // 不要かもしれないが、念のためにイベントハンドラに登録しておく
            tbxSearch.HandleCreated += (s, e) => setCueBanner();

            if (tbxSearch.IsHandleCreated)
            {
                setCueBanner();
            }
        }

        protected override void InitializeFormInternal()
        {
            base.InitializeFormInternal();
            SetCategoryMinimumSize();

            IsAllShowNode = ApplicationConfig.Setting.CurveEditor.IsAllShowNode;
            SetAllShowNode(IsAllShowNode);

            IsShaderParamAnimGroupCondShow = ApplicationConfig.Setting.CurveEditor.IsShaderParamAnimGroupCondShow;
            SetShaderParamAnimGroupCondShow(IsShaderParamAnimGroupCondShow);

            IsOnlyTargetShowNode = ApplicationConfig.Setting.CurveEditor.IsOnlyTargetShowNode;
            SetOnlyTargetShowNode(IsOnlyTargetShowNode);

            IsSortByAlphabetical = ApplicationConfig.Setting.CurveEditor.IsSortByAlphabetical;
            SetSortByAlphabetical(IsSortByAlphabetical);

            IsShowKeyedMaterial = ApplicationConfig.Setting.CurveEditor.IsShowKeyedMaterial;
            SetShowKeyedMaterial(IsShowKeyedMaterial);

            //SetEnumWrapTypeToComboBox(cmbPreWrap);
            //SetEnumWrapTypeToComboBox(cmbPostWrap);
        }

        /// <summary>
        /// カーブ要素のパネルの表示/非表示の設定。
        /// </summary>
        public void VisibleCurveElementPanel(bool visible)
        {
            panel1.Visible = visible;
        }

        private void tsmAllShowNode_Click(object sender, EventArgs e)
        {
            IsAllShowNode = ! IsAllShowNode;
            ApplicationConfig.Setting.CurveEditor.IsAllShowNode = IsAllShowNode;

            SetAllShowNode(IsAllShowNode);
        }

        private void tsmSortNode_Click(object sender, EventArgs e)
        {
            IsSortByAlphabetical = !IsSortByAlphabetical;
            ApplicationConfig.Setting.CurveEditor.IsSortByAlphabetical = IsSortByAlphabetical;

            SetSortByAlphabetical(IsSortByAlphabetical);
        }

        private void tsmShaderParamAnimGroupCondShow_Click(object sender, EventArgs e)
        {
            IsShaderParamAnimGroupCondShow = ! IsShaderParamAnimGroupCondShow;
            ApplicationConfig.Setting.CurveEditor.IsShaderParamAnimGroupCondShow = IsShaderParamAnimGroupCondShow;

            SetShaderParamAnimGroupCondShow(IsShaderParamAnimGroupCondShow);
        }

        private void tsmOnlyTargetShowNode_Click(object sender, EventArgs e)
        {
            IsOnlyTargetShowNode = ! IsOnlyTargetShowNode;
            ApplicationConfig.Setting.CurveEditor.IsOnlyTargetShowNode = IsOnlyTargetShowNode;

            SetOnlyTargetShowNode(IsOnlyTargetShowNode);
        }

        private void tsmShowKeyedMaterial_Click(object sender, EventArgs e)
        {
            IsShowKeyedMaterial = !IsShowKeyedMaterial;
            ApplicationConfig.Setting.CurveEditor.IsShowKeyedMaterial = IsShowKeyedMaterial;

            SetShowKeyedMaterial(IsShowKeyedMaterial);
        }

        private void tsmResetNodeExpand_Click(object sender, EventArgs e)
        {
            if (ResetNodeExpandClick != null)
            {
                ResetNodeExpandClick(sender, e);
            }
        }

        private void tsmOpenNodeExpand_Click(object sender, EventArgs e)
        {
            if (OpenNodeExpandClick != null)
            {
                OpenNodeExpandClick(sender, e);
            }
        }

        private void tsmVisibleAllNode_Click(object sender, EventArgs e)
        {
            if (VisibleAllNodeClick != null)
            {
                VisibleAllNodeClick(sender, e);
            }
        }

        private void tsmInvisibleAllNode_Click(object sender, EventArgs e)
        {
            if (InvisibleAllNodeClick != null)
            {
                InvisibleAllNodeClick(sender, e);
            }
        }

        protected override void OnCategoryAfterSelect()
        {
            spcLeft.Panel2Collapsed = (ActivePage is IAnimationEditPage) == false;
        }

        public void SetAllShowNode(bool isAllShowNode)
        {
            IsAllShowNode = isAllShowNode;

            UpdateIcon();

            if (AllShowNodeChanged != null)
            {
                AllShowNodeChanged(this, EventArgs.Empty);
            }
        }

        public void SetSortByAlphabetical(bool isSortByAlphabetical)
        {
            IsSortByAlphabetical = isSortByAlphabetical;

            UpdateIcon();

            if (SortByAlphabeticalChanged != null)
            {
                SortByAlphabeticalChanged(this, EventArgs.Empty);
            }
        }


        public void SetShaderParamAnimGroupCondShow(bool isShaderParamAnimGroupCondShow)
        {
            IsShaderParamAnimGroupCondShow = isShaderParamAnimGroupCondShow;

            UpdateIcon();

            if (ShaderParamAnimGroupCondShowChanged != null)
            {
                ShaderParamAnimGroupCondShowChanged(this, EventArgs.Empty);
            }
        }

        public void SetOnlyTargetShowNode(bool isOnlyTargetShowNode)
        {
            IsOnlyTargetShowNode = isOnlyTargetShowNode;

            UpdateIcon();

            if (OnlyTargetShowNodeChanged != null)
            {
                OnlyTargetShowNodeChanged(this, EventArgs.Empty);
            }
        }

        public void SetShowKeyedMaterial(bool isShowKeyedMaterial)
        {
            IsShowKeyedMaterial = isShowKeyedMaterial;

            UpdateIcon();

            if (ShowKeyedMaterialChanged != null)
            {
                ShowKeyedMaterialChanged(this, EventArgs.Empty);
            }
        }

        private UIListControlItem[] floatWarpItems = new UIListControlItem[] {
            new UIListControlItem(Strings.Enum_Curve_Element_Wrap_Clamp, curve_wrapType.clamp),
            new UIListControlItem(Strings.Enum_Curve_Element_Wrap_Repeat, curve_wrapType.repeat),
            new UIListControlItem(Strings.Enum_Curve_Element_Wrap_Mirror, curve_wrapType.mirror),
            new UIListControlItem(Strings.Enum_Curve_Element_Wrap_RelativeRepeat, curve_wrapType.relative_repeat),
        };

        private UIListControlItem[] intWarpItems = new UIListControlItem[] {
            new UIListControlItem(Strings.Enum_Curve_Element_Wrap_Clamp, curve_wrapType.clamp),
            new UIListControlItem(Strings.Enum_Curve_Element_Wrap_Repeat, curve_wrapType.repeat),
            new UIListControlItem(Strings.Enum_Curve_Element_Wrap_Mirror, curve_wrapType.mirror),
            new UIListControlItem(Strings.Enum_Curve_Element_Wrap_RelativeRepeat, curve_wrapType.relative_repeat),
        };

        private void SetEnumWrapTypeToComboBox(UIComboBox cmb, bool floatItems)
        {
            var items = floatItems ? floatWarpItems : intWarpItems;
            if (cmb.Items.Count == items.Length)
            {
                return;
            }
            using(var ub = new UpdateBlock(cmb))
            {
                cmb.Items.Clear();
                cmb.Items.AddRange(items);
            }
        }

        private void UpdateIcon()
        {
            tsmAllShowNode.Image =
                IsAllShowNode ?
                    Resources.AnimationEditor_Curve_Visible :
                    Resources.AnimationEditor_Curve_Invisible;

            tsmShaderParamAnimGroupCondShow.Image =
                IsShaderParamAnimGroupCondShow ?
                    Resources.AnimationEditor_ShaderParamAnimGroupCondShow_Visible :
                    Resources.AnimationEditor_ShaderParamAnimGroupCondShow_Invisible;

            tsmOnlyTargetShowNode.Image =
                IsOnlyTargetShowNode ?
                    Resources.AnimationEditor_OnlyTargetShowNode_Invisible :
                    Resources.AnimationEditor_OnlyTargetShowNode_Visible;

            tsmSortNode.Image =
                IsSortByAlphabetical ?
                    Resources.AnimationEditor_SortByAlphabetical :
                    Resources.AnimationEditor_SortByIndex;

            tsmShowKeyedMaterial.Image =
                IsShowKeyedMaterial ?
                    Resources.AnimationEditor_ShowKeyedMaterial_Visible :
                    Resources.AnimationEditor_ShowKeyedMaterial_Invisible;
        }

        private void cmbPreWrap_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (PreWrapChanged != null && grpCurveElement.Enabled)
            {
                PreWrapChanged(sender, e);
            }
        }

        private void cmbPostWrap_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (PostWrapChanged != null && grpCurveElement.Enabled)
            {
                PostWrapChanged(sender, e);
            }
        }

        private void chkCompressEnable_CheckedChanged(object sender, EventArgs e)
        {
            if (CompressEnableChanged != null && grpCurveElement.Enabled)
            {
                CompressEnableChanged(sender, e);
            }
        }

        private void chkBinarizeVisibility_CheckedChanged(object sender, EventArgs e)
        {
            if (BinarizeVisibilityChanged != null && grpCurveElement.Enabled)
            {
                BinarizeVisibilityChanged(sender, e);
            }
        }

        private GuiObjectID CurvePanelID = GuiObjectID.ShaderParameterAnimation;
        public void SetPanelGuiObjectID(GuiObjectID id)
        {
            CurvePanelID = id;

            // 位置や Visible を調整する対象
            Control[][] panelControls = new Control[][] {
                new Control[] { chkCompressEnable },
                new Control[] { chkBinarizeVisibility },
                new Control[] { lblRenderMatrix, lwtRenderMatrix },
                new Control[] { lblRigidBody, lwtRigidBody },
                new Control[] { lblBaseName, ltbBaseName },

                new Control[] { lblFrameType, ltbFrameType },
                new Control[] { lblKeyType, ltbKeyType },
                new Control[] { lblScale, ltbScale },
                new Control[] { lblOffset, iepOffset },
                new Control[] { lblPreWrap, cmbPreWrap },
                new Control[] { lblPostWrap, cmbPostWrap },
                new Control[] { lblBaked, ltbBaked }
            };

            // 共通部分に表示する
            Control[] common = new Control[] {
                ltbFrameType,
                ltbKeyType,
                ltbScale,
                iepOffset,
                cmbPreWrap,
                cmbPostWrap,
                ltbBaked
            };

            // タイプ固有のコントロール
            Control[] specific = new Control[] { };
            switch (CurvePanelID)
            {
                case GuiObjectID.ShapeAnimation:
                    specific = new Control[] {
                        ltbBaseName
                    };
                    break;
                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                case GuiObjectID.CameraAnimation:
                case GuiObjectID.FogAnimation:
                case GuiObjectID.LightAnimation:
                    specific = new Control[] {
                    };
                    break;
                case GuiObjectID.MaterialVisibilityAnimation:
                    specific = new Control[] {
                    };
                    break;
                case GuiObjectID.BoneVisibilityAnimation:
                    specific = new Control[] {
                        chkCompressEnable,
                        chkBinarizeVisibility,
                        lwtRenderMatrix,
                        lwtRigidBody
                    };
                    break;
                case GuiObjectID.TexturePatternAnimation:
                    specific = new Control[] {
                    };
                    break;
            }

            // 位置等を調整
            HashSet<Control> visibleControls = new HashSet<Control>(common.Concat(specific));
            HashSet<Control> settedControls = new HashSet<Control>();
            int Y = 16;
            int margin = 2;
            foreach (var items in panelControls)
            {
                if (items.Any(x => visibleControls.Contains(x)))
                {
                    int height = items.Max(x => x.Height);
                    foreach (var item in items)
                    {
                        item.Location = new Point(item.Location.X, Y + (height - item.Height) / 2);
                        settedControls.Add(item);
                    }
                    Y += height + margin;
                }
            }

            foreach (var item in panelControls.SelectMany(x => x))
            {
                item.Visible = settedControls.Contains(item);
            }

            grpCurveElement.Height = Y + 3;
            panel1.Height = grpCurveElement.Height + 5;

        }

        protected override void UpdateFormInternal()
        {
            switch (CurvePanelID)
            {
                case GuiObjectID.ShapeAnimation:
                    //tsmCurveCreate.Enabled = DocumentManager.Models.Any(
                    //	x => x.AllAnimations.Contains(Targets.Active) &&
                    //	x.Shapes.Any(y => y.Data.key_shape_array != null &&
                    //	!((ShapeAnimation)Targets.Active).VertexShapeAnims.Any(z => z.shape_name == y.Name)));
                    break;
            }
        }

        /// <summary>
        /// カーブ要素のパネルのコントロールに選択されたカーブの情報を設定します。
        /// </summary>
        public void UpdateCurveElement(object sender, EventArgs e)
        {
            CurveEditorPanel panel = sender as CurveEditorPanel;

            UpdateCurveElement(panel);
        }

        /// <summary>
        /// カーブ要素のパネルのコントロールに選択されたカーブの情報を設定します。
        /// </summary>
        private void UpdateCurveElement(CurveEditorPanel panel)
        {
            if (panel.TargetGroup.Active is BoneVisibilityAnimation)
            {
                SetCurveElementValue_BoneVisibilityAnimation(panel);
            }
            else
            {
                SetCurveElementValue_Animation(panel);
            }
        }

        private void SetEnable(bool setValue)
        {
            if (setValue == false)
            {
                grpCurveElement.Enabled = false;
            }
            else
            {
                grpCurveElement.Enabled = true;
            }
        }

        private void SetEnableAllCurveElementControl(bool enableflg)
        {
            ltbFrameType.Enabled = enableflg;
            lblFrameType.Enabled = enableflg;
            ltbKeyType.Enabled = enableflg;
            lblKeyType.Enabled = enableflg;
            ltbScale.Enabled = enableflg;
            lblScale.Enabled = enableflg;
            iepOffset.Enabled = enableflg;
            lblOffset.Enabled = enableflg;
            cmbPreWrap.Enabled = enableflg;
            lblPreWrap.Enabled = enableflg;
            cmbPostWrap.Enabled = enableflg;
            lblPostWrap.Enabled = enableflg;
            lwtRenderMatrix.Enabled = enableflg;
            lblRenderMatrix.Enabled = enableflg;
            lwtRigidBody.Enabled = enableflg;
            lblRigidBody.Enabled = enableflg;
            chkCompressEnable.Enabled = enableflg;
            lblBaked.Enabled = enableflg;
            ltbBaked.Enabled = enableflg;
            lblBaseName.Enabled = enableflg;
            ltbBaseName.Enabled = enableflg;
        }

        private void SetCurveElementValue_Animation(CurveEditorPanel panel)
        {
            // 選択されているカーブからコントロールへ情報を設定します。
            var setValue = false;
            var isEditable = true;
            curve_frame_typeType? frameType = null;
            curve_key_typeType? keyType = null;
            float? scale = null;
            float? offset = null;
            curve_wrapType? preWrap = null;
            curve_wrapType? postWrap = null;
            bool? baked = null;
            var allFloat = true;
            var anyConstant = false;

            foreach (var curve in panel.SelectedCurves)
            {
                isEditable = isEditable && curve.IsEditable;
                var paramAnimTarget = curve.GetAnimTarget(panel.TargetGroup.Active);
                //if (paramAnimTarget != null && paramAnimTarget.KeyFrames.Any() && !paramAnimTarget.KeyFrames.IsConstantCurve(out dummy))
                if (paramAnimTarget != null && paramAnimTarget.KeyFrames.Any())
                {
                    float dummy;
                    allFloat &= curve.CurvePrimitiveType == PrimitiveTypeKind.Float;

                    if (paramAnimTarget.KeyFrames.IsConstantCurve(
                        paramAnimTarget.CurveInterpolationType, out dummy))
                    {
                        frameType = null;
                        keyType = null;
                        scale = null;
                        offset = null;
                        preWrap = null;
                        postWrap = null;
                        baked = null;
                        setValue = true;
                        anyConstant = true;
                        break;
                    }

                    if (!setValue)
                    {
                        frameType = paramAnimTarget.QuantizationInfo.frame_type;
                        keyType = paramAnimTarget.QuantizationInfo.key_type;
                        scale = paramAnimTarget.QuantizationInfo.scale;
                        offset = paramAnimTarget.QuantizationInfo.offset;
                        preWrap = paramAnimTarget.pre_wrap;
                        postWrap = paramAnimTarget.post_wrap;
                        baked = paramAnimTarget.baked;
                        setValue = true;
                    }
                    else
                    {
                        if (frameType != paramAnimTarget.QuantizationInfo.frame_type)
                        {
                            frameType = null;
                        }
                        if (keyType != paramAnimTarget.QuantizationInfo.key_type)
                        {
                            keyType = null;
                        }
                        if (scale != paramAnimTarget.QuantizationInfo.scale)
                        {
                            scale = null;
                        }
                        if (offset != paramAnimTarget.QuantizationInfo.offset)
                        {
                            offset = null;
                        }
                        if (preWrap != paramAnimTarget.pre_wrap)
                        {
                            preWrap = null;
                        }
                        if (postWrap != paramAnimTarget.post_wrap)
                        {
                            postWrap = null;
                        }
                        if (baked != paramAnimTarget.baked)
                        {
                            baked = null;
                        }
                    }
                }
            }

            ltbFrameType.Text = frameType.HasValue ? UIText.EnumValue(frameType.Value) : "";
            ltbKeyType.Text = keyType.HasValue ? UIText.EnumValue(keyType.Value) : "";
            ltbScale.Text = scale.HasValue ? scale.Value.ToString() : "";
            iepOffset.Text = offset.HasValue ? offset.Value.ToString() : "";
            if (setValue && !anyConstant)
            {
                SetEnumWrapTypeToComboBox(cmbPreWrap, allFloat);
                SetEnumWrapTypeToComboBox(cmbPostWrap, allFloat);
            }
            cmbPreWrap.SetSelectedItemData(preWrap.HasValue ? (object)preWrap.Value : null);
            cmbPostWrap.SetSelectedItemData(postWrap.HasValue ? (object)postWrap.Value : null);
            ltbBaked.Text = baked.HasValue ? UIText.FlagYesNo(baked.Value) : "";

            SetEnable(setValue);
            cmbPreWrap.Enabled = setValue && !anyConstant && isEditable;
            cmbPostWrap.Enabled = setValue && !anyConstant && isEditable;
        }

        private void SetCurveElementValue_BoneVisibilityAnimation(CurveEditorPanel panel)
        {
            // 選択されているカーブからコントロールへ情報を設定します。
            bool setValue = false;
            curve_frame_typeType? frameType = null;
            curve_key_typeType? keyType = null;
            float? scale = null;
            float? offset = null;
            curve_wrapType? preWrap = null;
            curve_wrapType? postWrap = null;
            bool setSupplementValue = false;
            bool? baked = null;
            bool? renderMatrix = null;
            bool? rigidBody = null;
            bool? compressEnable = null;
            bool? binarizeVisibility = null;
            foreach (IAnimationCurve curve in panel.SelectedCurves)
            {
                string boneName = curve.ParentName;

                BoneVisibilityAnimation ParamAnim = panel.TargetGroup.Active as BoneVisibilityAnimation;
                if (ParamAnim == null)
                {
                    break;
                }

                BoneVisibilityAnimation.BoneVisibilityBoneAnim boneAnim = ParamAnim.GetParamAnimFromId(boneName);
                if (boneAnim != null)
                {
                    float dummy;
                    bool hasCurve = boneAnim.KeyFrames.Any() && !boneAnim.KeyFrames.IsConstantCurve(out dummy);
                    if (hasCurve)
                    {
                        if (!setValue)
                        {
                            frameType = boneAnim.QuantizationInfo.frame_type;
                            keyType = boneAnim.QuantizationInfo.key_type;
                            scale = boneAnim.QuantizationInfo.scale;
                            offset = boneAnim.QuantizationInfo.offset;
                            preWrap = boneAnim.pre_wrap;
                            postWrap = boneAnim.post_wrap;
                            baked = boneAnim.baked;
                            setValue = true;
                        }
                        else
                        {
                            if (frameType != boneAnim.QuantizationInfo.frame_type)
                            {
                                frameType = null;
                            }
                            if (keyType != boneAnim.QuantizationInfo.key_type)
                            {
                                keyType = null;
                            }
                            if (scale != boneAnim.QuantizationInfo.scale)
                            {
                                scale = null;
                            }
                            if (offset != boneAnim.QuantizationInfo.offset)
                            {
                                offset = null;
                            }
                            if (preWrap != boneAnim.pre_wrap)
                            {
                                preWrap = null;
                            }
                            if (postWrap != boneAnim.post_wrap)
                            {
                                postWrap = null;
                            }
                            if (baked != boneAnim.baked)
                            {
                                baked = null;
                            }
                        }
                    }

                    if (!setSupplementValue)
                    {
                        renderMatrix = boneAnim.render_matrix;
                        rigidBody = boneAnim.rigid_body;
                        compressEnable = boneAnim.compress_enable;
                        binarizeVisibility = boneAnim.binarize_visibility;
                        setSupplementValue = true;

                        // バイナリ出力がOFFの場合は圧縮可能のチェックボックスは編集できない
                        SetChkCompressEnable(boneAnim.binarize_visibility);
                    }
                    else
                    {
                        if (renderMatrix != boneAnim.render_matrix)
                        {
                            renderMatrix = null;
                        }
                        if (rigidBody != boneAnim.rigid_body)
                        {
                            rigidBody = null;
                        }
                        if (compressEnable != boneAnim.compress_enable)
                        {
                            compressEnable = null;
                        }
                        if (binarizeVisibility != boneAnim.binarize_visibility)
                        {
                            binarizeVisibility = null;
                        }
                    }
                }
            }

            ltbFrameType.Text = frameType.HasValue ? UIText.EnumValue(frameType.Value) : "";
            ltbKeyType.Text = keyType.HasValue ? UIText.EnumValue(keyType.Value) : "";
            ltbScale.Text = scale.HasValue ? scale.Value.ToString() : "";
            iepOffset.Text = offset.HasValue ? offset.Value.ToString() : "";
            if (setValue)
            {
                SetEnumWrapTypeToComboBox(cmbPreWrap, false);
                SetEnumWrapTypeToComboBox(cmbPostWrap, false);
            }
            cmbPreWrap.SetSelectedItemData(preWrap.HasValue ? (object)preWrap.Value : null);
            cmbPostWrap.SetSelectedItemData(postWrap.HasValue ? (object)postWrap.Value : null);
            ltbBaked.Text = baked.HasValue ? UIText.FlagYesNo(baked.Value) : "";

            lwtRenderMatrix.Text = renderMatrix.HasValue ? UIText.FlagYesNo(renderMatrix.Value) : "";
            lwtRigidBody.Text = rigidBody.HasValue ? UIText.FlagYesNo(rigidBody.Value) : "";
            if (setSupplementValue)
            {
                chkCompressEnable.CheckState = compressEnable.HasValue ? (compressEnable.Value ? CheckState.Checked : CheckState.Unchecked) : CheckState.Indeterminate;
                chkBinarizeVisibility.CheckState = binarizeVisibility.HasValue ? (binarizeVisibility.Value ? CheckState.Checked : CheckState.Unchecked) : CheckState.Indeterminate;
            }
            else
            {
                chkCompressEnable.CheckState = CheckState.Unchecked;
                chkBinarizeVisibility.CheckState = CheckState.Unchecked;
            }

            SetEnable(setValue || setSupplementValue);
            cmbPreWrap.Enabled = setValue;
            cmbPostWrap.Enabled = setValue;

            if (setValue)
            {
                // バイナリ出力がOFFの場合はラップ設定のコンボボックスは編集できない
                UpdateCmbWrap();
            }
        }

        public void FitCurveTreeViewWidth()
        {
            int width = 0;
            {
                using(var g = CurveTreeView.CreateGraphics())
                {
                    foreach (TreeNode node in CurveTreeView.Nodes)
                    {
                        width = Math.Max(width, MakeMaxNodeWidth(node, 0, g, CurveTreeView.Font));
                    }
                }
            }
            width += 4 + Win32.NativeMethods.GetSystemMetrics(Win32.NativeMethods.SystemMetric.SM_CXVSCROLL);

            if (this.IsHandleCreated)
            {
                // 最小化されていると例外が出るので BeginInvoke で後回し
                BeginInvoke((Action)(() =>
                {
                    Owner.Width += width - LeftWidth;
                    LeftWidth = width;
                }));
            }
            else
            {
                Owner.Width += width - LeftWidth;
                LeftWidth = width;
            }
        }

        private int MakeMaxNodeWidth(TreeNode node, int depth, Graphics g, Font font)
        {
            const int xBase			= 54;	// 開始値
            const int xWidth		= 19;	// depthごとの幅
            const int xLeafWidth	= 12;	// 末端(箱)

            int width = 0;
            {
                Debug.Assert(node is TreeNode);

                var cNode = node as TreeNode;

                int textWidth    = TextRenderer.MeasureText(g, cNode.Text, font).Width;
                int currentWidth = textWidth + xBase + xWidth * depth + ((cNode.Nodes.Count == 0) ? xLeafWidth : 0);

                width = Math.Max(width, currentWidth);
            }

            foreach (TreeNode childWode in node.Nodes)
            {
                width = Math.Max(width, MakeMaxNodeWidth(childWode, depth + 1, g, font));
            }

            return width;
        }

        // 空白にマッチ
        Regex regex = new Regex(@"\s");
        private void tbxSearch_TextChanged(object sender, EventArgs e)
        {
            // 空白で区切る
            SearchTexts.Clear();
            int last = 0;
            var text = tbxSearch.Text ?? string.Empty;
            foreach (Match match in regex.Matches(text))
            {
                if (last != match.Index)
                {
                    SearchTexts.Add(text.Substring(last, match.Index - last));
                }
                last = match.Index + match.Length;
            }

            if (text.Length != last)
            {
                SearchTexts.Add(text.Substring(last, text.Length - last));
            }

            if (SearchTextChanged != null)
            {
                SearchTextChanged(sender, e);
            }
        }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            if (base.ProcessCmdKey(ref msg, keyData))
            {
                return true;
            }

            if (keyData == (Keys.Control | Keys.F))
            {
                if (!tbxSearch.Focused)
                {
                    tbxSearch.Focus();
                    tbxSearch.SelectAll();
                    return true;
                }
            }

            return false;
        }

        public void ShowNoNodeWarning(bool show)
        {
            uilTreeViewWarning.Visible = show;
        }

        private void spcLeft_Panel2_Layout(object sender, LayoutEventArgs e)
        {
            using (var g = uilTreeViewWarning.CreateGraphics())
            {
                uilTreeViewWarning.Height = (int)g.MeasureString(uilTreeViewWarning.Text, uilTreeViewWarning.Font, spcLeft.Width - 32).Height + 6;
            }

            base.OnLayout(e);
        }

        public void SetChkCompressEnable(bool enableflg)
        {
            chkCompressEnable.Enabled = enableflg;
        }

        public void UpdateCmbWrap()
        {
            cmbPostWrap.Enabled = chkCompressEnable.Enabled;
            cmbPreWrap.Enabled = chkCompressEnable.Enabled;
        }
    }
}
