﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.res;

namespace App.PropertyEdit
{
    public partial class ShapeAnimationCreateDialog : OkCancelDialog
    {
        public string ShapeName
        {
            get { return cmbShapeName.Text; }
        }

        public string BaseName
        {
            get { return cmbBaseName.Text; }
        }

        private readonly ShapeAnimation ShapeAnimation;

        public ShapeAnimationCreateDialog(ShapeAnimation shapeAnimation)
        {
            InitializeComponent();

            ShapeAnimation = shapeAnimation;

            using(var ub = new UpdateBlock(cmbModel))
            {
                foreach(var model in ParentModel)
                {
                    cmbModel.AddItem(model.Name, model);
                }
            }

            if (cmbModel.Items.Count > 0)
            {
                cmbModel.SelectedIndex = 0;
            }

            UpdateShapeNames();
        }

        private void UpdateShapeNames()
        {
            using(var ub = new UpdateBlock(cmbShapeName))
            {
                cmbShapeName.Items.Clear();

                Model model = cmbModel.SelectedItemData as Model;
                if (model != null)
                {
                    foreach(var shape in model.Shapes.Where(x => x.Data.key_shape_array != null))
                    {
                        cmbShapeName.AddItem(shape.Name, shape);
                    }
                }

                if (cmbShapeName.Items.Count > 0)
                {
                    cmbShapeName.SelectedIndex = 0;
                }
            }

            UpdateBaseNames();
        }

        private void UpdateBaseNames()
        {
            using (var ub = new UpdateBlock(cmbBaseName))
            {
                cmbBaseName.Items.Clear();

                Shape shape = cmbShapeName.SelectedItemData as Shape;
                if (shape != null)
                {
                    foreach (var key_shape in shape.Data.key_shape_array.key_shape)
                    {
                        cmbBaseName.AddItem(key_shape.name, key_shape);
                    }
                }

                if (cmbBaseName.Items.Count > 0)
                {
                    cmbBaseName.SelectedIndex = 0;
                }
            }
        }

        private IEnumerable<Model> ParentModel
        {
            get
            {
                return
                    from model in DocumentManager.Models
                    from anim in model.AllAnimations
                    where anim == ShapeAnimation
                    select model;
            }
        }

        private void ShapeAnimationCreateDialog_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {
                // 空文字列でないか
                {
                    if (ShapeName == string.Empty)
                    {
                        UIMessageBox.Warning(Strings.MaterialSelectDialog_EmptyName);
                        e.Cancel = true;
                        return;
                    }
                }

                // 使えない文字が入っていないか
                {
                    if (Material.IsValidateNameString(ShapeName) == false)
                    {
                        UIMessageBox.Warning(Strings.MaterialSelectDialog_InvalidateCharacter);
                        e.Cancel = true;
                        return;
                    }
                }

                // 重複チェック
                {
                    if (ShapeAnimation.VertexShapeAnims.Any(x => x.shape_name == ShapeName))
                    {
                        UIMessageBox.Warning(Strings.MaterialSelectDialog_AlreadyExistMaterial);
                        e.Cancel = true;
                        return;
                    }
                }
            }
        }

        private void cmbModel_SelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateShapeNames();
        }

        private void cmbShapeName_SelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateBaseNames();
        }
    }
}
