﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using App.Properties;
using App.Utility;
using ConfigCommon;

namespace App.PropertyEdit
{
    public partial class CurveViewPainter
    {
        private static readonly Bitmap	TexturePatternKeyImage			= Resources.AnimationEditor_TexturePatternKey;
        private static readonly int		texturePatternKeyImageWidth_	= TexturePatternKeyImage.Width;
        private static readonly int		texturePatternKeyImageHeight_	= TexturePatternKeyImage.Height;

        public static int TexturePatternKeyImageWidth{	get{ return texturePatternKeyImageWidth_;  } }
        public static int TexturePatternKeyImageHeight{	get{ return texturePatternKeyImageHeight_; } }

        // 矢印先から左上までのオフセット
        public static int TexturePatternKeyLeftOffset{	get{ return -TexturePatternKeyImageWidth / 2; } }
        public static int TexturePatternKeyTopOffset{	get{ return -TexturePatternKeyImageHeight;    } }

        // テクスチャパターンカーブ編集用
        private bool DrawHRulerTexturePatternBar(PaintEventArgs e, GuiObject target)
        {
            if (!(target is IHasTexturePatternAnimation))
            {
                return false;
            }

            var curves = view_.SelectedTexturePatternCurves;

            if (curves.Any(x => x.KeyFrames.Any()))
            {
                DrawTexturePatternBar(e, curves, target);

                using (var hqg = new HighQualityGraphics(e.Graphics, isInterpolationMode:false, isSmoothingMode:false))
                {
                    DrawTexturePatternKeyCursors(e, curves, target);
                }

                return true;
            }

            if (curves.Any())
            {
                DrawEmptyBar(e);
                return true;
            }

            return false;
        }

        private void DrawTexturePatternBar(PaintEventArgs e, List<IAnimationCurve> curves, GuiObject target)
        {
            Debug.Assert(target is IHasTexturePatternAnimation);

            var rect = new Rectangle(CurveBgLeft, HRulerBgTop - CurveEditorConst.ColorBarSize, CurveBgWidth, CurveEditorConst.ColorBarSize);

            if (curves.Count >= 2)
            {
                // 複数あれば単色のグレー
                e.Graphics.FillRectangle(Brushes.Gray, rect);
            }
            else
            {
                var texPattern = target as IHasTexturePatternAnimation;
                var curve = curves.First();
                var animTarget = curve.GetAnimTarget(target);

                if (curve.KeyFrames.Count >= 2 && curve.KeyFrames[curve.KeyFrames.Count - 1].Frame - curve.KeyFrames[0].Frame < State.ScaleX &&
                    (animTarget.pre_wrap != nw.g3d.nw4f_3dif.curve_wrapType.clamp || animTarget.post_wrap != nw.g3d.nw4f_3dif.curve_wrapType.clamp))
                {
                    // 幅が狭いときはグレー
                    e.Graphics.FillRectangle(Brushes.Gray, rect);
                }
                else
                {
                    // 下地
                    e.Graphics.FillRectangle(Brushes.White, rect);
                    foreach(var i in MakeEditBarIntervals(curve, target))
                    {
                        DrawEditBarInterval(e, i, texPattern, curve.IsEditable == false);
                    }
                }
            }
        }

        private void DrawEditBarInterval(
            PaintEventArgs e,
            EditBarInterval editBarInterval,
            IHasTexturePatternAnimation texPattern,
            bool isGray
        )
        {
            var height	= CurveEditorConst.ColorBarSize;
            var y		= HRulerBgTop - CurveEditorConst.ColorBarSize;

            var texture = texPattern.GetTextureFromPatternIndex((int)editBarInterval.Value);
            if (texture == null)
            {
                e.Graphics.SafeFillRectangle(Brushes.White, editBarInterval.LeftX, y, editBarInterval.Width, height);
            }
            else
            {
                var x = editBarInterval.LeftX;

                while (x <= editBarInterval.RightX)
                {
                    if (texture.HasColor == false)
                    {
                        e.Graphics.SafeFillRectangle(Brushes.White, x, y, CurveEditorConst.ColorBarSize, height);
                    }
                    else
                    {
                        var rect = new Rectangle((int)x, y, CurveEditorConst.ColorBarSize, height);
                        texture.DrawColorThumbnailFit(e.Graphics, rect, isGray);
                    }

                    x += CurveEditorConst.ColorBarSize;
                }
            }
        }

        private void DrawTexturePatternKeyCursors(PaintEventArgs e, List<IAnimationCurve> curves, GuiObject target)
        {
            Debug.Assert(target is IHasTexturePatternAnimation);

            if (curves.Count == 1)
            {
                var texPattern = target as IHasTexturePatternAnimation;

                var curve = curves.First();
                if (texPattern == null) return;

                foreach (var keyFrame in curve.KeyFrames)
                {
                    var texture			= texPattern.GetTextureFromPatternIndex((int)keyFrame.Value);
                    var frameScreenX	= (float)MakeCurveViewScreenPosX(keyFrame.Frame);

                    DrawTexturePatternKeyCursor(e.Graphics, frameScreenX, HRulerBgTop, texture, curve.IsEditable == false);
                }
            }
        }

        // x, y は矢印の先の座標を指定します。
        private void DrawTexturePatternKeyCursor(Graphics g, float x, float y, Texture texture, bool isGray)
        {
            x += TexturePatternKeyLeftOffset;
            y += TexturePatternKeyTopOffset - CurveEditorConst.ColorBarSize;

            g.DrawImageSafe(TexturePatternKeyImage, x, y);

            if (texture != null && texture.HasColor)
            {
                var size = TexturePatternKeyImage.Width - 2;
                var rect = new Rectangle((int) x + 1, (int) y + 1, size, size);

                texture.DrawColorThumbnailFit(g, rect, isGray);
            }
        }
    }
}
