﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using App.Properties;
using App.Utility;

namespace App.PropertyEdit
{
    public partial class CurveViewPainter
    {
        private static readonly Bitmap	colorKeyImage_			= Resources.AnimationEditor_ColorKey;
        private static readonly int		colorKeyImageWidth_		= colorKeyImage_.Width;
        private static readonly int		colorKeyImageHeight_	= colorKeyImage_.Height;

        public static int ColorKeyImageWidth{	get{ return colorKeyImageWidth_;  } }
        public static int ColorKeyImageHeight{	get{ return colorKeyImageHeight_; } }

        // 矢印先から左上までのオフセット
        public static int ColorKeyLeftOffset{	get{ return -ColorKeyImageWidth / 2; } }
        public static int ColorKeyTopOffset{	get{ return -ColorKeyImageHeight;    } }

        private const int colorCellWidth = 2;

        // カラーカーブ編集用
        private bool DrawHRulerColorBar(PaintEventArgs e, GuiObject target)
        {
            // 色カーブが選択されていなければ何もしない
            if (view_.IsSelectedColorCurve == false)
            {
                return false;
            }

            var curves = view_.SelectedColorCurves;

            if (curves.Any(x => x.KeyFrames.Any()))
            {
                var isAnyColorSet	= view_.IsAnyColorSet(curves);
                var colorCurveGroup	= MakeColorCurveGroup(true);

                bool hdr = false;
                foreach (var tuple in colorCurveGroup.SelectMany(x => x.Value))
                {
                    if (tuple.KeyFrames.Any())
                    {
                        var animTarget = tuple.GetAnimTarget(target);
                        float min;
                        float max;
                        AnimationCurveExtensions.GetMinMax(
                            tuple.KeyFrames,
                            tuple.CurveInterpolationType,
                            FrameCount,
                            animTarget.pre_wrap,
                            animTarget.post_wrap,
                            out min,
                            out max);

                        // 少し余裕を持たせる
                        if (min < -1e-6 || 1+1e-6 < max)
                        {
                            hdr = true;
                            break;
                        }
                    }
                }

                if (hdr)
                {
                    DrawHDRBar(e);
                }
                else
                {
                    DrawColorBar(e, curves, colorCurveGroup, isAnyColorSet);
                }

                using (var hqg = new HighQualityGraphics(e.Graphics, isInterpolationMode:false, isSmoothingMode:false))
                {
                    DrawColorKeyCursors(e, curves, colorCurveGroup, isAnyColorSet, target, true);
                }

                return true;
            }
            else
            {
                DrawEmptyBar(e);
                return true;
            }
        }

        public bool IsHDR(GuiObject target)
        {
            var colorCurveGroup = MakeColorCurveGroup(true);
            foreach (var tuple in colorCurveGroup.SelectMany(x => x.Value))
            {
                if (tuple.KeyFrames.Any())
                {
                    var animTarget = tuple.GetAnimTarget(target);
                    float min;
                    float max;
                    AnimationCurveExtensions.GetMinMax(
                        tuple.KeyFrames,
                        tuple.CurveInterpolationType,
                        FrameCount,
                        animTarget.pre_wrap,
                        animTarget.post_wrap,
                        out min,
                        out max);

                    // 少し余裕を持たせる
                    if (min < -1e-6 || 1 + 1e-6 < max)
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        public Dictionary<string, List<IAnimationCurve>>	MakeColorCurveGroup(bool isOnlySelectedCurve)
        {
            var dict = new Dictionary<string, List<IAnimationCurve>>();

            IEnumerable<IAnimationCurve> curves = view_.MovableCurves;
            if (isOnlySelectedCurve)
            {
                HashSet<IAnimationCurve> selectedCurves = new HashSet<IAnimationCurve>(view_.SelectedCurves);
                curves = curves.Where(x => selectedCurves.Contains(x));
            }

            foreach (var info in curves)
            {
                if (info.IsColorCurve)
                {
                    var key = info.ParentName + " " + info.Name;
                    if (!dict.ContainsKey(key))
                    {
                        dict.Add(key, new List<IAnimationCurve>());
                    }
                }
            }

            foreach (var c in view_.AllCurves)
            {
                if (c.IsColorCurve)
                {
                    var key = c.ParentName + " " + c.Name;
                    if (dict.ContainsKey(key))
                    {
                        dict[key].Add(c);
                    }
                }
            }

            return dict;
        }

        private void DrawColorBar(
            PaintEventArgs e,
            List<IAnimationCurve> curves,
            Dictionary<string, List<IAnimationCurve>> colorCurveGroup,
            bool isAnyColorSet)
        {
            var rect = new Rectangle(CurveBgLeft, HRulerBgTop + 1 - 1 - CurveEditorConst.ColorBarSize, CurveBgWidth, CurveEditorConst.ColorBarSize);

            using (var brush = new HatchBrush(HatchStyle.SmallCheckerBoard, Color.LightGray, Color.Black))
            {
                e.Graphics.FillRectangle(brush, rect);
            }

            using(var brush = new LinearGradientBrush(rect, Color.Red, Color.Blue, LinearGradientMode.Horizontal))
            {
                brush.InterpolationColors = MakeColorBlend(curves, colorCurveGroup, isAnyColorSet, curves.Any(x => x.IsEditable == false));
                e.Graphics.FillRectangle(brush, rect);
            }
        }

        private ColorBlend MakeColorBlend(
            List<IAnimationCurve> curves,
            Dictionary<string, List<IAnimationCurve>> colorCurveGroup,
            bool isAnyColorSet,
            bool isGray
        )
        {
            var front = curves.First();

            if (isAnyColorSet || (colorCurveGroup.ContainsKey(front.ParentName + " " + front.Name) == false))
            {
                // 複数あれば単色のグレー
                return
                    new ColorBlend()
                    {
                        Positions	= new [] {0.0f, 1.0f},
                        Colors		= new [] {Color.Gray, Color.Gray},
                    };
            }
            else
            {
                // 複数なければカーブの色
                var positions = new List<float>();
                var colors    = new List<Color>();
                {
                    var interval = Math.Max(1, (int)(CurveBgWidth / colorCellWidth));

                    var rgba = new float[4];

                    var isGammaGammaCorrection = ConfigData.ApplicationConfig.Color.GammaCorrection;

                    for(var i = 0;i <= interval;++ i)
                    {
                        var pos = (float)i / (float)interval;
                        positions.Add(pos);

                        rgba[0] = 0.0f;
                        rgba[1] = 0.0f;
                        rgba[2] = 0.0f;
                        rgba[3] = 1.0f;
                        {
                            var frame = (float)view_.GetXInCurvePlane(CurveBgLeft + (int)(pos * CurveBgWidth));

//							var front = curves.First();
                            Debug.Assert(colorCurveGroup.ContainsKey(front.ParentName + " " + front.Name));

                            foreach(var curveInfo in colorCurveGroup[front.ParentName + " " + front.Name])
                            {
                                var curve  = curveInfo;
                                var target = curve.GetAnimTarget(view_.TargetGroup.Active);

                                // カーブを評価する
                                // アルファチャンネルでキーが１つもない場合は1.0fとして扱う
                                rgba[curve.ColorComponentIndex] =
                                    ((curve.ColorComponentIndex == 3) && (curve.KeyFrames.Any() == false)) ?
                                        1.0f :
                                        Math.Max(0.0f, Math.Min(1.0f, curve.GetValue(frame, target.pre_wrap, target.post_wrap)));
                            }
                        }

                        var color =
                            Color.FromArgb(
                                (byte)(rgba[3] * 255.0f),
                                (byte)(rgba[0] * 255.0f),
                                (byte)(rgba[1] * 255.0f),
                                (byte)(rgba[2] * 255.0f)
                            );

                        if (isGray)
                        {
                            color = ColorUtility.ToGray(color);
                        }

                        colors.Add(isGammaGammaCorrection ? ColorUtility.GammaCorrect22(color) : color);
                    }
                }

                return
                    new ColorBlend()
                    {
                        Positions	= positions.ToArray(),
                        Colors		= colors.ToArray()
                    };
            }
        }

        private void DrawColorKeyCursors(
            PaintEventArgs e,
            List<IAnimationCurve> curves,
            Dictionary<string, List<IAnimationCurve>> colorCurveGroup,
            bool isAnyColorSet,
            GuiObject target,
            bool colorBarDrawn)
        {
            // 複数あればなにもしない
            if (isAnyColorSet == false)
            {
                var isGammaGammaCorrection = ConfigData.ApplicationConfig.Color.GammaCorrection;

                foreach(var colorSet in view_.MakeSameFrameColorSet(curves, view_.TargetGroup.Active, colorCurveGroup))
                {
                    var frameScreenX = (float)MakeCurveViewScreenPosX(colorSet.Frame);
                    DrawColorKeyCursor(
                        e.Graphics,
                        frameScreenX,
                        HRulerBgTop,
                        isGammaGammaCorrection ?
                            ColorUtility.GammaCorrect22(colorSet.Color.ToColor()) :
                            colorSet.Color.ToColor(),
                        colorSet.Curves.Any(x => x.IsEditable == false),
                        colorBarDrawn
                    );
                }
            }
        }

        // x, y は矢印の先の座標を指定します。
        private void DrawColorKeyCursor(Graphics g, float x, float y, Color color, bool isGray, bool isColorBarDrawn)
        {
            x += ColorKeyLeftOffset;
            y += ColorKeyTopOffset;
            if (isColorBarDrawn)
            {
                y -= 1 + CurveEditorConst.ColorBarSize;
            }

            g.DrawImageSafe(colorKeyImage_, x, y);

            using(var brush = new SolidBrush(isGray ? ColorUtility.ToGray(color) : color))
            {
                var size = colorKeyImage_.Width - 2;
                g.SafeFillRectangle(brush, x + 1, y + 1, size, size);
            }
        }
    }
}
