﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Utility;
using App.res;
using ConfigCommon;
using System.Collections.Generic;
using nw.g3d.iflib.nw3de;

namespace App.PropertyEdit
{
    public partial class SearchPathPage : ObjectPropertyPage
    {
        public SearchPathPage() :
            base(PropertyPageID.ModelSearchPath)
        {
            InitializeComponent();
        }

        // TODO:
#if false
        public override string HelpPath
        {
            get
            {
                return "html/Graphics/3DEditor/html/ui/common/user_data/user_data.html";
            }
        }
#endif

        public static SearchPathPage CreateInstance(object arg)
        {
            return new SearchPathPage();
        }

        protected override void InitializeFormInternal()
        {
            ;
        }

        internal IntermediateFileDocument ActiveDocument
        {
            get
            {
                return
                    ((Targets?.Active is Material) && (Targets.Active?.OwnerDocument is SeparateMaterial)) ?
                    (SeparateMaterial)Targets.Active.OwnerDocument : (Targets.Active as IntermediateFileDocument);
            }
        }

        private bool isFirstUpdate = true;

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            {
                uiModifiedMarkLabel1.IsModified = ActiveDocument.IsSearchPathsChanged();
                int selectIndex = -1;
                if (lvwSearchPath.SelectedIndices.Count > 0)
                {
                    selectIndex = lvwSearchPath.SelectedIndices[0];
                }

                lvwSearchPath.BeginUpdate();
                {
                    lvwSearchPath.SetItemCount(ActiveDocument.SearchPaths.Count);
                    for (int i = 0; i < ActiveDocument.SearchPaths.Count; i++)
                    {
                        ListViewItem item = lvwSearchPath.Items[i];
                        item.SubItems[clhPath.Index].Text = ActiveDocument.SearchPaths[i].Path;
                        item.SubItems[clhReccursive.Index].Text = UIText.FlagYesNo(ActiveDocument.SearchPaths[i].Recursive);
                    }
                }
                lvwSearchPath.EndUpdate();

                if (lvwSearchPath.Items.Count > 0)
                {
                    if ((selectIndex != -1) && (selectIndex < lvwSearchPath.Items.Count))
                    {
                        lvwSearchPath.Items[selectIndex].Selected = true;
                        lvwSearchPath.Items[selectIndex].Focused = true;
                    }
                    else
                    {
                        btnAdd.Select();
                    }
                }
                else
                {
                    lvwSearchPath_SelectedIndexChanged(null, null);
                }
            }

            // ページを開いた直後は最後を選択状態にする
            if (isFirstUpdate)
            {
                if (lvwSearchPath.Items.Count > 0)
                {
                    var lastIndex = lvwSearchPath.Items.Count - 1;

                    lvwSearchPath.Items[lastIndex].Selected = true;
                    lvwSearchPath.Items[lastIndex].Focused = true;
                }

                isFirstUpdate = false;
            }

            UpdateState();
        }

        public static bool IsModified(IntermediateFileDocument activeTarget)
        {
            return activeTarget != null && activeTarget.IsSearchPathsChanged();
        }

        private void lvwSearchPath_SelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateState();
        }

        private void lvwSearchPath_SelectionChanged(object sender, EventArgs e)
        {
            UpdateState();
        }

        private void UpdateState()
        {
            bool isSelected = lvwSearchPath.SelectedIndex != -1;

            btnEdit.Enabled		= isSelected;
            btnDelete.Enabled	= isSelected;
            btnDown.Enabled		= isSelected && (lvwSearchPath.SelectedIndex >= 0) && (lvwSearchPath.SelectedIndex <= lvwSearchPath.Items.Count - 2);
            btnUp.Enabled		= isSelected && (lvwSearchPath.SelectedIndex > 0);
        }

        #region イベント
        private void btnUp_Click(object sender, EventArgs e)
        {
            if (lvwSearchPath.SelectedItems.Count == 0) { return; }
            Debug.Assert(lvwSearchPath.SelectedItems.Count == 1);

            // リストビューの変更
            ListViewItem item = lvwSearchPath.SelectedItems[0];
            int index = lvwSearchPath.Items.IndexOf(item);
            if (index == 0) { return; }

            // コマンド発行
            var list = ActiveDocument.SearchPaths.ToList();
            var tmp = list[index-1];
            list[index - 1] = list[index];
            list[index] = tmp;
            TheApp.CommandManager.Execute(CreateEditCommand_SearchPath(Targets, list));
            lvwSearchPath.Items[index - 1].Selected = true;
            btnUp.Focus();
        }

        private void btnDown_Click(object sender, EventArgs e)
        {
            if (lvwSearchPath.SelectedItems.Count == 0) { return; }
            Debug.Assert(lvwSearchPath.SelectedItems.Count == 1);

            // リストビューの変更
            ListViewItem item = lvwSearchPath.SelectedItems[0];
            int index = lvwSearchPath.Items.IndexOf(item);
            if ((index + 1) == lvwSearchPath.Items.Count) { return; }

            // コマンド発行
            var list = ActiveDocument.SearchPaths.ToList();
            var tmp = list[index + 1];
            list[index + 1] = list[index];
            list[index] = tmp;
            TheApp.CommandManager.Execute(CreateEditCommand_SearchPath(Targets, list));
            lvwSearchPath.Items[index + 1].Selected = true;
            btnDown.Focus();
        }

        private void btnAdd_Click(object sender, EventArgs e)
        {
            using (var dialog = new SearchPathDialog(
                string.Empty,
                ActiveDocument.FilePath,
                false,
                ActiveDocument.SearchPaths.Select(x => x.Path).ToList()))
            {
                var result = dialog.ShowDialog(Owner.Owner);

                if (result == DialogResult.OK)
                {
                    int index;
                    if (lvwSearchPath.SelectedIndices.Count == 0)
                    {
                        index = lvwSearchPath.Items.Count;
                    }
                    else
                    {
                        Debug.Assert(lvwSearchPath.SelectedIndices.Count == 1);
                        index = lvwSearchPath.SelectedIndices[0];
                    }

                    // カーソル位置の次の位置
                    // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1627
                    ++index;
                    index = Math.Min(index, lvwSearchPath.Items.Count);

                    var list = ActiveDocument.SearchPaths.ToList();
                    list.Insert(index, new nw3de_SearchPath.SearchPath()
                    {
                        Path = dialog.FolderPath,
                        Recursive = dialog.Recursive,
                    });

                    // コマンド発行
                    TheApp.CommandManager.Execute(CreateEditCommand_SearchPath(Targets, list));
                    //item.Selected = selected;
                }
            }
        }

        public static EditCommand CreateEditCommand_SearchPath(GuiObjectGroup targets, List<nw3de_SearchPath.SearchPath> searchPaths)
        {
            var editable =
                targets.Objects
                .Where(
                    x =>
                    x.ObjectID == GuiObjectID.Model || x.ObjectID == GuiObjectID.MaterialAnimation ||
                    x.ObjectID == GuiObjectID.TexturePatternAnimation ||
                    ((x.ObjectID == GuiObjectID.Material) && (x.OwnerDocument is SeparateMaterial)))
                .Select(
                    x =>
                    ((x.ObjectID == GuiObjectID.Material) && (x.OwnerDocument is SeparateMaterial)) ? x.OwnerDocument : x).ToArray();

            return new GeneralGroupReferenceEditCommand<List<nw3de_SearchPath.SearchPath>>(
                new GuiObjectGroup(editable),
                null,
                ObjectUtility.MultipleClone(searchPaths, editable.Count()).ToArray(),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var document = (IntermediateFileDocument)target;
                    swap = document.SearchPaths;
                    document.SearchPaths = (List<nw3de_SearchPath.SearchPath>)data;
                }
                );
        }

        private void btnDelete_Click(object sender, EventArgs e)
        {
            if (lvwSearchPath.SelectedItems.Count == 0) { return; }
            Debug.Assert(lvwSearchPath.SelectedItems.Count == 1);

            // リストビューの変更
            ListViewItem item = lvwSearchPath.SelectedItems[0];
            int index = lvwSearchPath.Items.IndexOf(item);


            // コマンド発行
            var list = ActiveDocument.SearchPaths.ToList();
            list.RemoveAt(index);
            TheApp.CommandManager.Execute(CreateEditCommand_SearchPath(Targets, list));
        }

        private void btnEdit_Click(object sender, EventArgs e)
        {
            Debug.Assert(lvwSearchPath.SelectedIndices.Count == 1);
            ListViewItem item = lvwSearchPath.SelectedItems[0];

            int index = lvwSearchPath.SelectedIndices[0];

            using (var dialog = new SearchPathDialog(
                ActiveDocument.SearchPaths[index].Path,
                ActiveDocument.FilePath,
                ActiveDocument.SearchPaths[index].Recursive,
                ActiveDocument.SearchPaths.Select(x => x.Path).ToList()))
            {
                var result = dialog.ShowDialog(Owner.Owner);

                if (result == DialogResult.OK)
                {
                    var list = ActiveDocument.SearchPaths.ToList();
                    list[index] = new nw3de_SearchPath.SearchPath()
                    {
                        Path = dialog.FolderPath,
                        Recursive = dialog.Recursive,
                    };

                    // コマンド発行
                    TheApp.CommandManager.Execute(CreateEditCommand_SearchPath(Targets, list));
                }
            }
        }

        private void lvwSearchPath_DoubleClick(object sender, EventArgs e)
        {
            if (lvwSearchPath.SelectedItems.Count > 0)
            {
                if (lvwSearchPath.SelectedItems.Count == 1)
                {
                    btnEdit_Click(null, null);
                }
            }
        }
        #endregion

        #region コピー＆ペースト
        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        private class CopyData
        {
            public List<nw3de_SearchPath.SearchPath> SearchPath { get; set; }
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveDocument);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(GuiObject target)
        {
            var doc = (IntermediateFileDocument)target;

            return
                new CopyData()
                {
                    SearchPath = ObjectUtility.Clone(doc.SearchPaths),
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            var editables = targets.Objects.Where(x => x.Editable);
            var commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;
                commandSet.Add(CreateEditCommand_SearchPath(targets, copyData.SearchPath));
            }

            return commandSet.Execute();
        }
        #endregion
    }
}
