﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Data;

namespace App.PropertyEdit
{
    public class TexturePatternAnimationCurveTreeNodeInfo : IAnimationCurve
    {
        public TexturePatternAnimation	target_;
        public string					materialName_;
        public string					sampler_name_;

        public bool									IsEditable		{ get; set; }
        public AnimationDocument.NonEditableKind	NonEditableKind	{ get; set; }
        public object								NonEditableKindDisplayAux	{ get; set; }
        public bool IsColorCurve { get { return false; } }
        public int ColorComponentIndex { get { return -1; } }

        public TexturePatternAnimationCurveTreeNodeInfo(TexturePatternAnimation target, string materialName, string sampler_name)
        {
            target_			= target;
            materialName_	= materialName;
            sampler_name_			= sampler_name;
        }

        public AnimTarget GetAnimTarget(GuiObject editTarget)
        {
            var animation = (TexturePatternAnimation)editTarget;
            var texPatternMatAnim = animation.TexPatternMatAnims.FirstOrDefault(x => x.mat_name == materialName_);
            if (texPatternMatAnim == null)
            {
                return null;
            }

            return texPatternMatAnim.PatternAnimTargets.FirstOrDefault(x => x.sampler_name == sampler_name_);
        }

        public AnimTarget CreateAnimTarget(GuiObject editTarget)
        {
            var animation = (TexturePatternAnimation)editTarget;
            var patternAnimTarget = new TexturePatternAnimation.PatternAnimTarget()
            {
                sampler_name = sampler_name_,
                hint = ""
            };

            var texPatternMatAnim = animation.TexPatternMatAnims.FirstOrDefault(x => x.mat_name == materialName_);
            if (texPatternMatAnim == null)
            {
                return null;
            }

            texPatternMatAnim.PatternAnimTargets.Add(patternAnimTarget);

            return patternAnimTarget;
        }

        public void UpdateIsModified(GuiObject editTarget)
        {
            var animation = (TexturePatternAnimation)editTarget;
            var animTarget = GetAnimTarget(animation);
            var savedTexPatternMatAnim = animation.savedTexPatternMatAnims.FirstOrDefault(x => x.mat_name == materialName_);
            var savedTarget = savedTexPatternMatAnim != null ? savedTexPatternMatAnim.PatternAnimTargets.FirstOrDefault(x => x.sampler_name == sampler_name_) : null;
            animation.UpdateIsModifiedAnimTarget(
                (TexturePatternAnimation.PatternAnimTarget)animTarget,
                savedTarget);
        }

        public AnimTarget CreateTemporaryTarget(GuiObject editTarget)
        {
            var source = (TexturePatternAnimation.PatternAnimTarget)GetAnimTarget(editTarget);
            return new TexturePatternAnimation.PatternAnimTarget()
            {
                sampler_name = source.sampler_name,
                hint = source.hint,
            };
        }

        public void SetAnimTarget(GuiObject editTarget, AnimTarget animTarget)
        {
            var animation = (TexturePatternAnimation)editTarget;

            var texPatternMatAnim = animation.TexPatternMatAnims.FirstOrDefault(x => x.mat_name == materialName_);
            if (texPatternMatAnim == null)
            {
                Debug.Assert(false);
                return;
            }

            var index = texPatternMatAnim.PatternAnimTargets.FindIndex(x => x.sampler_name == sampler_name_);
            if (index == -1)
            {
                texPatternMatAnim.PatternAnimTargets.Add((TexturePatternAnimation.PatternAnimTarget)animTarget);
            }
            else
            {
                texPatternMatAnim.PatternAnimTargets[index] = (TexturePatternAnimation.PatternAnimTarget)animTarget;
            }
        }

        public float GetDefaultValue(GuiObject targetOwner)
        {
            // バインドされたマテリアル
            var materials = from model in DocumentManager.Models
                            where model.AllAnimations.Contains(targetOwner)
                            from material in model.Materials
                            where material.Name == materialName_
                            select material;

            var values = from material in materials
                         from param in material.ResolvedSamplers
                         where param.name == sampler_name_
                         let index = ((TexturePatternAnimation)targetOwner).TexPatterns.FindIndex(x => x.tex_name == param.tex_name)
                         where index >= 0
                         select (float)index;
            return values.Concat(Enumerable.Range(0,1).Select(x => GetAnimTarget(targetOwner).GetBaseValue())).First();
        }

        public void IntermediateFileIndices(TexturePatternAnimation target, out int materialIndex, out int paramAnimIndex)
        {
            var exportedMaterials = (from texPatternMatAnim in target.TexPatternMatAnims
                                    where texPatternMatAnim.PatternAnimTargets.Any(x => x.ExportType != CurveExportType.Ignored)
                                    select texPatternMatAnim).ToList();
            materialIndex = exportedMaterials.FindIndex(x => x.mat_name == materialName_);
            Debug.Assert(materialIndex >= 0);

            var paramAnims = exportedMaterials[materialIndex].PatternAnimTargets.Where(x => x.ExportType != CurveExportType.Ignored).ToList();
            paramAnimIndex = paramAnims.FindIndex(x => x.sampler_name == sampler_name_);
        }

        public List<KeyFrame> KeyFrames
        {
            get
            {
                var paramAnim = target_.GetParamAnimFromId(materialName_, sampler_name_);
                return
                    (paramAnim != null) ?
                        paramAnim.KeyFrames :
                        new List<KeyFrame>();
            }
        }

        public InterpolationType CurveInterpolationType
        {
            get
            {
                return InterpolationType.Step;
            }

            set
            {
                ;   // 無視
            }
        }

        public string ParentName{		get { return materialName_;		} }
        public string Name{				get { return sampler_name_;		} }
        public string FullPath{			get { return string.Format("{0} / {1} / {2}", target_.Name, materialName_, Name);	} }

        public int ComponentIndex{		get { return 0;					} }
        public bool IsRotate{			get { return false;				} }
        public float? MinClampValue{	get { return 0.0f;				} }
        public float? MaxClampValue{	get { return Math.Max(target_.TexPatterns.Count - 1, 0);				} }
        public float? MinFitValue{		get { return 0.0f;				} }
        public float? MaxFitValue{		get { return Math.Max(target_.TexPatterns.Count - 1, 0);				} }
        public PrimitiveTypeKind CurvePrimitiveType { get { return PrimitiveTypeKind.Uint; } }

        public Color CurveColor { get { return CurveView.MakeColorFromString(target_.Name + ParentName + Name); } }

        public bool IsSame(IAnimationCurve curve)
        {
            var shapeCurve = (TexturePatternAnimationCurveTreeNodeInfo)curve;
            return shapeCurve.target_ == target_ &&
                shapeCurve.materialName_ == materialName_ &&
                shapeCurve.sampler_name_ == sampler_name_;
        }
    }

    public class TexturePatternAnimationCurveTreeNodeInfo2 : IAnimationCurve
    {
        public MaterialAnimation target_;
        public string materialName_;
        public string sampler_name_;

        public bool IsEditable { get; set; }
        public AnimationDocument.NonEditableKind NonEditableKind { get; set; }
        public object NonEditableKindDisplayAux { get; set; }
        public bool IsColorCurve { get { return false; } }
        public int ColorComponentIndex { get { return -1; } }

        public TexturePatternAnimationCurveTreeNodeInfo2(MaterialAnimation target, string materialName, string sampler_name)
        {
            target_ = target;
            materialName_ = materialName;
            sampler_name_ = sampler_name;
        }

        public AnimTarget GetAnimTarget(GuiObject editTarget)
        {
            var animation = (MaterialAnimation)editTarget;
            var texPatternMatAnim = animation.PerMaterialAnims[materialName_];
            if (texPatternMatAnim == null)
            {
                return null;
            }

            return texPatternMatAnim.PatternAnimTargets.FirstOrDefault(x => x.sampler_name == sampler_name_);
        }

        public AnimTarget CreateAnimTarget(GuiObject editTarget)
        {
            var animation = (MaterialAnimation)editTarget;
            var patternAnimTarget = new TexturePatternAnimation.PatternAnimTarget()
            {
                sampler_name = sampler_name_,
                hint = ""
            };

            var texPatternMatAnim = animation.PerMaterialAnims[materialName_];
            if (texPatternMatAnim == null)
            {
                return null;
            }

            texPatternMatAnim.PatternAnimTargets.Add(patternAnimTarget);

            return patternAnimTarget;
        }

        public void UpdateIsModified(GuiObject editTarget)
        {
            var animation = (MaterialAnimation)editTarget;
            var animTarget = GetAnimTarget(animation);
            var savedTexPatternMatAnim = animation.savedPerMaterialAnims[materialName_];
            var savedTarget = savedTexPatternMatAnim != null ? savedTexPatternMatAnim.PatternAnimTargets.FirstOrDefault(x => x.sampler_name == sampler_name_) : null;
            animation.UpdateIsModifiedAnimTarget(
                (TexturePatternAnimation.PatternAnimTarget)animTarget,
                savedTarget);
        }

        public AnimTarget CreateTemporaryTarget(GuiObject editTarget)
        {
            var source = (TexturePatternAnimation.PatternAnimTarget)GetAnimTarget(editTarget);
            return new TexturePatternAnimation.PatternAnimTarget()
            {
                sampler_name = source.sampler_name,
                hint = source.hint,
            };
        }

        public void SetAnimTarget(GuiObject editTarget, AnimTarget animTarget)
        {
            var animation = (MaterialAnimation)editTarget;

            var texPatternMatAnim = animation.PerMaterialAnims[materialName_];
            if (texPatternMatAnim == null)
            {
                Debug.Assert(false);
                return;
            }

            var index = texPatternMatAnim.PatternAnimTargets.FindIndex(x => x.sampler_name == sampler_name_);
            if (index == -1)
            {
                texPatternMatAnim.PatternAnimTargets.Add((TexturePatternAnimation.PatternAnimTarget)animTarget);
            }
            else
            {
                texPatternMatAnim.PatternAnimTargets[index] = (TexturePatternAnimation.PatternAnimTarget)animTarget;
            }
        }

        public float GetDefaultValue(GuiObject targetOwner)
        {
            // バインドされたマテリアル
            var materials = from model in DocumentManager.Models
                            where model.AllAnimations.Contains(targetOwner)
                            from material in model.Materials
                            where material.Name == materialName_
                            select material;

            var values = from material in materials
                         from param in material.ResolvedSamplers
                         where param.name == sampler_name_
                         let index = ((MaterialAnimation)targetOwner).TexPatterns.FindIndex(x => x.tex_name == param.tex_name)
                         where index >= 0
                         select (float)index;
            return values.Concat(Enumerable.Range(0, 1).Select(x => GetAnimTarget(targetOwner).GetBaseValue())).First();
        }

        public void IntermediateFileIndices(MaterialAnimation target, out int materialIndex, out int paramAnimIndex)
        {
            materialIndex = 0;
            paramAnimIndex = 0;
            foreach (var perMaterialAnim in target.PerMaterialAnims)
            {
                if (perMaterialAnim.mat_name == materialName_)
                {
                    paramAnimIndex += ShaderParameterAnimationCurveTreeNodeInfo2.CurvesInBinary(perMaterialAnim).Count();
                    foreach (var animTarget in perMaterialAnim.PatternAnimTargets)
                    {
                        if (animTarget.sampler_name == sampler_name_)
                        {
                            return;
                        }

                        if (animTarget.ExportType != CurveExportType.Ignored)
                        {
                            paramAnimIndex++;
                        }
                    }
                }
                else
                {
                    var curveCount = ShaderParameterAnimationCurveTreeNodeInfo2.CurvesInBinary(perMaterialAnim).Count() +
                        TexturePatternAnimationCurveTreeNodeInfo2.CurvesInBinary(perMaterialAnim).Count() +
                        (MaterialVisibilityAnimationCurveTreeNodeInfo2.AnyCurveInBinary(perMaterialAnim) ? 1 : 0);
                    if (curveCount > 0)
                    {
                        paramAnimIndex += curveCount;
                        materialIndex++;
                    }
                }
            }
        }

        public static IEnumerable<AnimTarget> CurvesInBinary(MaterialAnimation.PerMaterialAnim perMaterialAnim)
        {
            return perMaterialAnim.PatternAnimTargets.Where(x => x.ExportType != CurveExportType.Ignored);
        }

        public List<KeyFrame> KeyFrames
        {
            get
            {
                var paramAnim = target_.GetPatternAnimFromId(materialName_, sampler_name_);
                return
                    (paramAnim != null) ?
                        paramAnim.KeyFrames :
                        new List<KeyFrame>();
            }
        }

        public InterpolationType CurveInterpolationType
        {
            get
            {
                return InterpolationType.Step;
            }

            set
            {
                ;   // 無視
            }
        }

        public string ParentName { get { return materialName_; } }
        public string Name { get { return sampler_name_; } }
        public string FullPath { get; set; }

        public int ComponentIndex { get { return 0; } }
        public bool IsRotate { get { return false; } }
        public float? MinClampValue { get { return 0.0f; } }
        public float? MaxClampValue { get { return Math.Max(target_.TexPatterns.Count - 1, 0); } }
        public float? MinFitValue { get { return 0.0f; } }
        public float? MaxFitValue { get { return Math.Max(target_.TexPatterns.Count - 1, 0); } }
        public PrimitiveTypeKind CurvePrimitiveType { get { return PrimitiveTypeKind.Uint; } }

        public Color CurveColor { get { return CurveView.MakeColorFromString(target_.Name + ParentName + Name); } }

        public bool IsSame(IAnimationCurve curve)
        {
            var shapeCurve = curve as TexturePatternAnimationCurveTreeNodeInfo2;
            return shapeCurve != null &&
                shapeCurve.target_ == target_ &&
                shapeCurve.materialName_ == materialName_ &&
                shapeCurve.sampler_name_ == sampler_name_;
        }
    }
}
