﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Utility;
using App.Win32;

namespace App.PropertyEdit
{
    public partial class ShaderDefinitionSourceCodeSubPage : ShaderDefinitionPropertyPage
    {
        public ShaderDefinitionSourceCodeSubPage(int index) :
            base(PropertyPageID.ShaderDefinitionSourceCodeSub)
        {
            index_ = index;

            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_shader_definition_property_window_source_code_page;
            }
        }

        private readonly int index_;

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new ShaderDefinitionSourceCodeSubPage((int)arg);
        }

        protected override void InitializeFormInternal()
        {
            InitializeSearch();
        }

        private string SourceCode
        {
            get
            {
                int sourceCodeStreamIndex = ActiveTarget.Data.shader_src_array.shader_src[index_].stream_index;
                var stream = ActiveTarget.BinaryStreams;

                return
                    ((sourceCodeStreamIndex >= 0) && (sourceCodeStreamIndex < stream.Count)) ?
                        stream[sourceCodeStreamIndex].StringData :
                        string.Empty;
            }
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            // ソースを出力しておく
            ShaderDifinitionUtility.OutputGLSLFileFromFsd(ActiveTarget);

            var path = ActiveTarget.Data.shader_src_array.shader_src[index_].path;
            lblInclude.IsModified = ActiveTarget.IsDisplayIncludePathModified(path);
            ltpInclude.Text		= ActiveTarget.DisplayIncludePaths[ActiveTarget.Data.shader_src_array.shader_src[index_].path];
            ltbPath.Text		= ActiveTarget.Data.shader_src_array.shader_src[index_].path;

            bool binarizeError = ActiveTarget.FailedToBinarize;
            pbxBinirizeError.Visible = binarizeError;
            lblBinarizeError.Visible = binarizeError;
            lblBinarizeError.Text = Viewer.Manager.IsDeviceTargeted ?
                res.Strings.ShaderDefinitionSourceCodeSubPage_BinarizeErrorCATDEV:
                res.Strings.ShaderDefinitionSourceCodeSubPage_BinarizeErrorPC;

//			if (tbxSourceCode.Text == SourceCode)
            {
                // スクロール位置をなるべく保つ
                var scrollPos = tbxSourceCode.ScrollPos;
                tbxSourceCode.Text	= SourceCode;
                tbxSourceCode.ScrollPos = scrollPos;
            }
//			else
//			{
//				tbxSourceCode.Text	= SourceCode;
//			}

            var shadingModels = ActiveTarget.ShadingModelIndices(Path.GetFileName(ActiveTarget.Data.shader_src_array.shader_src[index_].path))
                                .Select(x => ActiveTarget.Data.shading_model_array.shading_model[x].name).Distinct().ToArray();
            var materials = from model in DocumentManager.Models
                            from material in model.Materials
                            where material.MaterialShaderAssign.ShaderDefinitionFileName == ActiveTarget.Name &&
                            shadingModels.Contains(material.MaterialShaderAssign.ShaderName)
                            select material;

            btnEdit.Enabled = (Viewer.Manager.Instance.IsConnected &&
                (ActiveTarget.IsAttached ||materials.Any() || ActiveTarget.FailedToBinarize)) ||
                (ConfigData.ApplicationConfig.Setting.Debug != null && ConfigData.ApplicationConfig.Setting.Debug.EditFsd);

            lblSourceCode.IsModified = ActiveTarget.ModifiedSources.Contains(ActiveTarget.Data.shader_src_array.shader_src[index_].path) && ActiveTarget.savedData.shader_src_array.shader_src.Any(x => x.path == path);

            ltbForceInclude.Text = UIText.FlagYesNo(ActiveTarget.Data.force_include_array.GetItems().Any(x => x.src_index == index_));

            CancelOldSearchResult();

            var isCaseSensitive	= tsbCaseSensitive.Checked;
            var isRegex			= tsbRegex.Checked;

            UpdateFormSearchWord(isCaseSensitive, isRegex);
        }

        public static bool IsModified(ShaderDefinition activeTarget, string path, int index)
        {
            return activeTarget != null &&
                (activeTarget.IsDisplayIncludePathModified(path) ||
                activeTarget.ModifiedSources.Contains(path) ||
                activeTarget.IsForceIncludeModified(path));
        }

        private void btnEdit_Click(object sender, EventArgs e)
        {
            EditSourceCode();
        }

        private void EditSourceCode()
        {
            // glslファイルをテンポラリに出力する。
            // streamにあるglslファイルの内容をテンポラリディレクトリに出力する。
            var filenames = ShaderDifinitionUtility.OutputGLSLFileFromFsd(ActiveTarget);
            var filename = filenames[index_];

            string TextEditor = ApplicationConfig.UserSetting.ExternalProgram.TextEditor;
            if (string.IsNullOrEmpty(TextEditor))
            {
                // Textエディタを指定せず、glslファイルを読み込ませる！
                try
                {
                    Process.Start(filename);
                }
                catch (Exception e)
                {
                    UIMessageBox.Error(e.Message);
                }
            }
            else
            {
                // テキストエディタが存在するか確認
                if (System.IO.File.Exists(TextEditor))
                {
                    // 対象ファイルをテキストエディタに渡して、エディタ起動。
                    try
                    {
                        var process = new Process();
                        process.StartInfo.FileName = TextEditor;
                        process.StartInfo.Arguments = ApplicationConfig.UserSetting.ExternalProgram.TextEditorFileOpenOption.Replace("%1", filename);
                        process.Start();
                    }
                    catch (Exception e)
                    {
                        UIMessageBox.Error(e.Message);
                    }
                }
                else
                {
                    // 指定されたテキストエディタが存在しない
                    UIMessageBox.Error(string.Format(res.Strings.IO_TextEditorNotFound, TextEditor));
                }
            }
        }

        private void tbxSourceCode_Enter(object sender, EventArgs e)
        {
            var pos = tbxSourceCode.ScrollPos;

            OneShotIdleProcess.Execute(() => {
                if (!tbxSourceCode.IsDisposed)
                {
                    tbxSourceCode.ScrollPos = pos;
                }
            });
        }
    }
}
