﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Command;
using App.Controls;
using App.Data;
using App.Utility;
using ConfigCommon;

namespace App.PropertyEdit
{
    public partial class MaterialVisibilityAnimationCurveEditPage : MaterialVisibilityAnimationPropertyPage, IAnimationEditPage
    {
        public MaterialVisibilityAnimationCurveEditPage() :
            base(PropertyPageID.MaterialVisibilityAnimationCurveEdit)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_curve_editor;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new MaterialVisibilityAnimationCurveEditPage();
        }

        protected override void InitializeFormInternal()
        {
            AnimationObjectPropertyPanel animationObjectPropertyPanel = Owner as AnimationObjectPropertyPanel;
            Debug.Assert(animationObjectPropertyPanel != null);

            CurveEditorPanel.Initialize(animationObjectPropertyPanel, ActiveTarget.ObjectID);

            CurveEditorPanel.UpdateTreeView       += (s, e) => UpdateTreeView();
            CurveEditorPanel.ChangeSelectedCurves += (s, e) => UpdateSelected();
            CurveEditorPanel.ChangeVisibledCurves += (s, e) => UpdateVisibled();
            CurveEditorPanel.GetFrameCount        = () => ActiveTarget.Data.mat_visibility_anim_info.frame_count;
            CurveEditorPanel.SetInterpolationTypeButtons(false, false, false);

            // 選択オブジェクトが変更になったら更新する
            {
                EventHandler SelectedTargetChanged = (s, e) =>
                {
                    if (Targets.Active is MaterialVisibilityAnimation && (Owner as AnimationObjectPropertyPanel).IsOnlyTargetShowNode)
                    {
                        UpdateFormInternal(new UpdateFormInfo()
                        {
                            IsPageCreated = false,
                            TargetOrPageChanged = false,
                        });
                    }
                };

                App.AppContext.SelectedTargetChanged += SelectedTargetChanged;
                Disposed += (s, a) => App.AppContext.SelectedTargetChanged -= SelectedTargetChanged;
            }
        }

        public override Size DefaultPageSize
        {
            get
            {
                return new Size(
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width),
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height));
            }
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            if ((Owner != null) && (Owner.ActivePage == this) && !ObjectPropertyDialog.InternallyChangingSize.IsChanging &&
                Owner.Owner.WindowState == System.Windows.Forms.FormWindowState.Normal)
            {
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width = Width;
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height = Height;
            }

            base.OnSizeChanged(e);
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            CurveEditorPanel.TargetGroup = Targets;

            CurveEditorPanel.UpdateForm(updateFormInfo);
        }

        protected override void UpdateFormOnPageCreatedInternal()
        {
            CurveEditorPanel.UpdateFormOnPageCreatedInternal();
        }

        public override void BeforePageDeactivated()
        {
            CurveEditorPanel.BeforePageDeactivated();
        }

        public override void AfterPageActiveted(ObjectPropertyPage oldPage)
        {
            CurveEditorPanel.AfterPageActivated();
        }

        private void UpdateSelected()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private void UpdateVisibled()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private readonly Dictionary<string, bool> visibledNodes_ = new Dictionary<string, bool>();
        private Dictionary<string, bool> isExpandedNodes_ = new Dictionary<string, bool>();

        private void UpdateTreeView()
        {
            using (var sb = new UIControlEventSuppressBlock())
            {
                // ノードの状態を保存しておく
                if (CurveEditorPanel.CurveTreeView.Nodes.Count > 0)
                {
                    var root = (CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0];
                    foreach (var tuple in CurveTreeView.NodesAndFullPath(root, root.id))
                    {
                        // チェック状態
                        visibledNodes_[tuple.Item2] = tuple.Item1.Checked;

                        // 開閉状態
                        isExpandedNodes_[tuple.Item2] = tuple.Item1.Nodes.Count == 0 || tuple.Item1.IsExpanded;
                    }
                }

                bool isAllShowNode			= (Owner as AnimationObjectPropertyPanel).IsAllShowNode;
                bool isOnlyTargetShowNode	= (Owner as AnimationObjectPropertyPanel).IsOnlyTargetShowNode;

                // 現在選択されているモデル/マテリアル名
                var selectedMaterials = App.AppContext.SelectedTarget.GetObjects(GuiObjectID.Material).Select(x => x as Material);

                var materials = new HashSet<string>((from model in DocumentManager.Models
                                                     where model.AllAnimations.Contains(ActiveTarget)
                                                     from material in model.Materials
                                                     select material.Name).Distinct());

                // つくり直す
                {
                    var rootNode = new CurveTreeInfo()
                    {
                        Text = ActiveTarget.Name,
                        Id = ActiveTarget.Name,
                        ShowEmptyNode = true,
                    };

                    foreach (var materialVisibilityMatAnim in ActiveTarget.MaterialVisibilityMatAnims)
                    {
                        var materialName = materialVisibilityMatAnim.mat_name;

                        // 未選択されているマテリアルは無視する
                        if (isOnlyTargetShowNode && selectedMaterials.Any(x => (x.Name == materialName) && x.Referrers.Any(y => y.AllAnimations.Any(z => z.Name == ActiveTarget.Name))) == false)
                        {
                            continue;
                        }

                        var materialNode = new CurveTreeInfo()
                        {
                            Text			= materialName,
                            Id				= materialName,
                            AnimationCurve	= new MaterialVisibilityAnimationCurveTreeNodeInfo(ActiveTarget, materialName),
                            IsBound			= materials.Contains(materialName),
                            NonEditableKind	= materials.Contains(materialName) ? AnimationDocument.NonEditableKind.Editable :
                                                                                 AnimationDocument.NonEditableKind.MaterialVisibility_NotFoundMaterial,
                            ShowEmptyNode	= isAllShowNode,
                            IsModified		= materialVisibilityMatAnim.IsModified,
                        };

                        rootNode.Nodes.Add(materialNode);
                    }

                    rootNode.TrimInvisibleNodes();

                    // 文字列によるフィルター
                    CurveTreeInfo.FilterCurveTreeInfoRoot(rootNode, (Owner as AnimationObjectPropertyPanel).SearchTexts);

                    var nodes = rootNode.ConvertToTreeNode();
                    if (nodes.IsSameStructure(CurveEditorPanel.CurveTreeView.Nodes) == false)
                    {
                        if ((CurveEditorPanel.UpdateFormInfo == null) || (CurveEditorPanel.UpdateFormInfo.TargetOrPageChanged == false))
                        {
                            // つくり直す


                            foreach (var tuple in CurveTreeView.NodesAndFullPath((CurveTreeNode)nodes, nodes.id))
                            {
                                // チェック状態を設定する
                                bool isChecked = false;
                                if (visibledNodes_.TryGetValue(tuple.Item2, out isChecked))
                                {
                                    if (isChecked != tuple.Item1.Checked)
                                    {
                                        tuple.Item1.Checked = isChecked;
                                    }
                                }

                                bool isExpanded = false;
                                if (isExpandedNodes_.TryGetValue(tuple.Item2, out isExpanded))
                                {
                                    if (isExpanded)
                                    {
                                        if (!tuple.Item1.IsExpanded)
                                        {
                                            tuple.Item1.Expand();
                                        }
                                    }
                                    else
                                    {
                                        if (tuple.Item1.IsExpanded)
                                        {
                                            tuple.Item1.Collapse();
                                        }
                                    }
                                }
                            }

                            CurveEditorPanel.CurveTreeView.Nodes.Clear();
                            CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);
                        }
                        else
                        {
                            nodes.ExpandAll();

                            // つくり直す
                            CurveEditorPanel.CurveTreeView.Nodes.Clear();
                            CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);

                            // 新規に開いたときは全表示にする
                            CurveEditorPanel.VisibleAllNode();
                        }
                    }
                    else
                    {
                        ((CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0]).CopyInfo(nodes);
                    }
                }
            }
        }

        #region コピー＆ペースト
        private class CopyData
        {
            public int frame_count { get; set; }
            public bool loop { get; set; }
            public List<MaterialVisibilityAnimation.MaterialVisibilityMatAnim> MaterialVisibilityMatAnims { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(MaterialVisibilityAnimation target)
        {
            return
                new CopyData()
                {
                    frame_count = target.Data.mat_visibility_anim_info.frame_count,
                    loop = target.Data.mat_visibility_anim_info.loop,
                    MaterialVisibilityMatAnims = ObjectUtility.Clone(target.MaterialVisibilityMatAnims)
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            EditCommandSet commandSet = new EditCommandSet();
            commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
            using (var block = new Viewer.ViewerDrawSuppressBlock(AnimationCurveEditCommand.AnimationMessageFilter))
            {
                var copyData = (CopyData)pasteObject;
                commandSet.Add(MaterialVisibilityAnimationGeneralPage.CreateEditCommand_frame_count(targets, copyData.frame_count).Execute());
                commandSet.Add(MaterialVisibilityAnimationGeneralPage.CreateEditCommand_loop(targets, copyData.loop).Execute());
                commandSet.Add(
                    new GeneralGroupReferenceEditCommand<List<MaterialVisibilityAnimation.MaterialVisibilityMatAnim>>(
                        targets,
                        GuiObjectID.MaterialVisibilityAnimation,
                        ObjectUtility.MultipleClone(copyData.MaterialVisibilityMatAnims, targets.Objects.Count),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            Debug.Assert(target is MaterialVisibilityAnimation);
                            var materialVisAnim = target as MaterialVisibilityAnimation;

                            swap = ObjectUtility.Clone(materialVisAnim.MaterialVisibilityMatAnims);
                            materialVisAnim.MaterialVisibilityMatAnims.Clear();
                            materialVisAnim.MaterialVisibilityMatAnims.AddRange(data as List<MaterialVisibilityAnimation.MaterialVisibilityMatAnim>);

                            materialVisAnim.UpdateIsModifiedAnimTargetAll();
                            Viewer.LoadOrReloadAnimation.Send(materialVisAnim);
                        }
                    ).Execute()
                );

                var active = (MaterialVisibilityAnimation)targets.Active;
                var command = active.CreateUpdateBindCommand();
                if (command != null)
                {
                    commandSet.Add(command.Execute());
                }
            }

            commandSet.Reverse();
            return commandSet;
        }
        #endregion
    }
}
