﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.res;
using App.Utility;
using System.Text.RegularExpressions;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public class ShaderGroupBox : UITitleBarGroupBox, IGroupName
    {
        LinkButton btnLink;
        public event LinkClickedEventHandler LinkClicked;

        public LabelHelper labelHelper = new LabelHelper();
        public virtual string ParamName
        {
            get;
            set;
        }

        public string ParamLabel
        {
            get;
            set;
        }

        /// <summary>
        /// 親グループ
        /// </summary>
        public string GroupName
        {
            get;
            set;
        }

        public void UpdateLabel()
        {
            Text = labelHelper.GetLabelString();
            //lblParameter.Text = labelHelper.GetLabelString();
            if (labelHelper.linkTarget != null)
            {
                if (btnLink == null || btnLink.Parent != this)
                {
                    btnLink = LabelHelper.CreateLinkButton();
                    btnLink.Click += (s, e) =>
                    {
                        if (LinkClicked != null)
                        {
                            LinkClicked(this, labelHelper.ClickArgs());
                        }
                    };
                    btnLink.Top = 1;
                    Controls.Add(btnLink);
                }

                btnLink.Left = ModifiedMarkRight();
                btnLink.Enabled = !labelHelper.LinkError;
                UpdateLinkToolTip(false);
            }

            UpdateLabelColor();
        }

        public void UpdateLabelColor()
        {
            switch (State)
            {
                case GroupState.Enabled:
                    TitleTextColor = labelHelper.color ?? DefaultTitleTextColor;
                    TitleBarColor = labelHelper.backGroundColor ?? DefaultTitleBarColor;
                    break;
                case GroupState.Disabled:
                    TitleTextColor = DefaultTitleTextColor;
                    TitleBarColor = Color.Gray;
                    break;
                case GroupState.ParentDisabled:
                    TitleTextColor = SystemColors.ControlText;
                    TitleBarColor = Color.LightGray;
                    break;
            }
        }

        public enum GroupState
        {
            Enabled,
            Disabled,
            ParentDisabled,
        }

        public GroupState _groupState;
        public GroupState State
        {
            get
            {
                return _groupState;
            }
            set
            {
                if (_groupState != value)
                {
                    _groupState = value;
                    UpdateLabelColor();
                }
            }
        }

        public bool SetLabel(
            string label,
            CustomUI customUI,
            Definition.ShadingModelTable table,
            Predicate<string> visibleGroups,
            HashSet<string> visiblePages,
            bool showId,
            bool showOriginalLabel,
            bool isParentCustomLabel)
        {
            var update = labelHelper.SetLabel(ParamName, ParamLabel, label, customUI, table, visibleGroups, visiblePages, showId, showOriginalLabel, isParentCustomLabel);
            if (update)
            {
                UpdateLabel();
            }
            return update;
        }

        public void UpdateLinkToolTip(bool force)
        {
            try
            {
                if (btnLink != null && parentHint != null)
                {
                    Debug.Assert(!btnLink.Disposing);
                    var toolTip = labelHelper.GetLinkLabelString();
                    var current = parentHint.GetToolTip(btnLink);
                    if (toolTip != current || force)
                    {
                        parentHint.SetToolTip(btnLink, toolTip);
                    }
                }
            }
            catch
            {
                Debug.Assert(false);
            }
        }

        private HintToolTip parentHint;
        public void SetHint(HintToolTip hint, string message)
        {
            parentHint = hint;

            if (!string.IsNullOrEmpty(message))
            {
                try
                {
                    hint.SetToolTip(this, message);
                }
                catch
                {
                    Debug.Assert(false);
                }
            }

            UpdateLinkToolTip(true);
        }

        #region ツールチップヒント関係
        // ShaderParamControl とほぼ同じ
        private bool showHint = false;

        protected override void OnMouseMove(MouseEventArgs e)
        {
            // ボタンが無効のときにツールチップが表示されないので自前で表示する
            var child = GetChildAtPoint(e.Location);
            if (child is LinkButton &&
                parentHint != null &&
                !showHint)
            {
                ShaderParamControl.toolTipTarget = child;
                StartToolTip();
            }
            else if (showHint && ShaderParamControl.toolTipTarget != child)
            {
                StopToolTip();

            }

            base.OnMouseMove(e);
        }

        protected override void OnMouseLeave(EventArgs e)
        {
            if (showHint)
            {
                StopToolTip();
            }
            base.OnMouseLeave(e);
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                var child = GetChildAtPoint(e.Location);
                if (child?.ContextMenuStrip != null && !child.Enabled && child is UILabel)
                    child.ContextMenuStrip.Show(this, e.Location);
            }
            base.OnMouseUp(e);
        }

        private void StartToolTip()
        {
            ShaderParamControl.showToolTip = (Action)(() =>
            {
                // TODO: プロパティウィンドウにフォーカスがないと表示されない
                // 座標も指定しないと無効なコントロール上で一瞬で消える
                if (ShaderParamControl.toolTipTarget != null)
                {
                    var pt = ShaderParamControl.toolTipTarget.PointToClient(Cursor.Position);
                    var text = parentHint.GetToolTip(ShaderParamControl.toolTipTarget);
                    if (!string.IsNullOrEmpty(text))
                    {
                        parentHint.Show(text, ShaderParamControl.toolTipTarget, pt.X, pt.Y + ShaderParamControl.toolTipTarget.Height);
                    }
                }
            });
            ShaderParamControl.timer.Start();
            showHint = true;
        }

        private void StopToolTip()
        {
            ShaderParamControl.timer.Stop();
            if (ShaderParamControl.toolTipTarget != null)
            {
                parentHint.Hide(ShaderParamControl.toolTipTarget);
            }
            showHint = false;
            ShaderParamControl.toolTipTarget = null;
        }
        #endregion
    }
}
