﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.Utility;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditSamplerAssign : ShaderParamControl
    {
        public string Value { get; set; }
        private Material Material;
        public string ParamHint { get; set; }

        public EditSamplerAssign()
        {
            InitializeComponent();

            cbxValue.Tag = (uint)(1<<0);
        }

        public const int minTextWidth = 100;
        public const int textMargin = 30;
        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            Value = value;
            Material = material;
            var items = cbxValue.Items.OfType<UIListControlItem>().ToArray();
            cbxValue.BeginUpdate();
            var resolvedSamplers = material.ResolvedSamplers.ToArray();
            if (!items.Select(x => (string)x.Data).SequenceEqual(
                resolvedSamplers.Select(x => x.name).Concat(Enumerable.Repeat(value, 1)).Concat(Enumerable.Repeat("", 1)).Distinct()))
            {
                {
                    cbxValue.Items.Clear();

                    HashSet<string> names = new HashSet<string>();
                    foreach (var sampler in resolvedSamplers)
                    {
                        names.Add(sampler.name);
                        cbxValue.AddItem(sampler.name + " : " + sampler.tex_name, sampler.name);
                    }

                    if (Value != string.Empty && !names.Contains(Value))
                    {
                        names.Add(Value);
                        cbxValue.AddItem(Value, Value);
                    }

                    cbxValue.AddItem(res.Strings.ShaderParamControl_NotAssigned, string.Empty);
                }
            }
            else
            {
                // 表示文字のみ更新
                foreach (var item in items)
                {
                    if ((string)item.Data != string.Empty)
                    {
                        var sampler = resolvedSamplers.FirstOrDefault(x => x.name == (string)item.Data);
                        if (sampler == null)
                        {
                            item.Text = (string)item.Data;
                        }
                        else
                        {
                            item.Text = sampler.name + " : " + sampler.tex_name;
                        }
                    }
                }
            }
            cbxValue.SelectedItemData = string.Empty;
            cbxValue.SelectedItemData = Value;
            cbxValue.EndUpdate();

            using (var g = CreateGraphics())
            {
                int width = Math.Max(
                    minTextWidth,
                    cbxValue.Items.OfType<UIListControlItem>().Max(x => (int)g.MeasureString(x.Text, cbxValue.Font).Width) + textMargin + (Texture.ThumbnailSize + 2) * 2);
                if (cbxValue.Width != width)
                {
                    cbxValue.Width = width;
                    return true;
                }
            }
            return false;
        }

        private void cbxValue_SelectedIndexChanged(object sender, EventArgs e)
        {
            SamplerValueChangedEventArgs args = new SamplerValueChangedEventArgs();
            {
                args.ParamName	= ParamName;
                args.ParamValue = cbxValue.SelectedItemData;
            }

            InvokeValueChanged(this, args);
        }

        private void cbxValue_CustomDrawItem(object sender, CustomDrawListControlItemEventArgs e)
        {
            if (e.Item != null)
            {
                samplerType sampler = Material.ResolvedSamplers.FirstOrDefault(x => x.name == (string)e.Item.Data);
                string texturePath = null;
                if (sampler != null)
                {
                    Material.Owner.ReferenceTexturePaths.TryGetValue(sampler.tex_name, out texturePath);
                }
                App.Data.Texture texture = sampler != null ?
                    DocumentManager.Textures.FirstOrDefault(x => x.Name == sampler.tex_name && string.Compare(x.FilePath, texturePath, true) == 0):
                    null;

                e.ForeColor = (string)e.Item.Data == string.Empty || sampler != null ? SystemColors.ControlText : Color.Red;
                if (texture == null)
                {
                    e.DoDefaultProcess = true;
                }
                else
                {
                    //                Debug.Assert(e.Item.Data is App.Data.Texture);

                    //                var texture = e.Item.Data as App.Data.Texture;

                    e.DrawBackground();

                    int fontHeight = (int)e.Graphics.MeasureString(e.Item.Text, e.Font, e.Bounds.Width, new StringFormat()).Height;

                    int left = GraphicsUtility.DrawTextureImage(texture, e.Graphics, e.Bounds);
                    int top = e.Bounds.Y + (e.Bounds.Height - fontHeight) / 2;

                    Control cmb = sender as Control;
                    Debug.Assert(cmb != null);
                    Brush brush = cmb != null && cmb.Enabled ? SystemBrushes.ControlText : SystemBrushes.GrayText;
                    e.Graphics.DrawString(e.Item.Text, e.Font, brush, left, top);

                    e.DrawFocusRectangle();
                    e.DoDefaultProcess = false;
                }
            }
            else
            {
                e.DoDefaultProcess = true;
            }
        }

        public override ParamType ParamType
        {
            get { return ShaderParamControls.ParamType.sampler_var; }
        }
    }
}
