﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.Utility;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditRenderInfoIntSliders : EditRenderInfoBase
    {
        public int[] Value
        {
            get
            {
                return intUpDowns.Take(ItemCount).Select(x => x.Value).ToArray();
            }
            set
            {
                ItemCount = 0;
                for (int i= 0; i< Math.Min(value.Length, Row); i++)
                {
                    intTrackBars[i].Value = intUpDowns[i].Value = value[i];
                    ItemCount++;
                }
            }
        }

        int ItemCount = 0;
        readonly UILabel lblMinimum;
        readonly UILabel lblMaximum;
        readonly Controls.IntUpDown[] intUpDowns;
        readonly Controls.IntTrackBar[] intTrackBars;

        public int Row { get; private set; }
        public const int MarginX = 4;
        public const int MarginY = 20;
        public int subControlHeight = 24;
        public int marginBetweenColumn = 6;
        public render_info_slotType type;
        public const int minItemWidth = 100;
        public const int textMargin = 30;
        public EditRenderInfoIntSliders(int row, render_info_slotType type)
        {
            InitializeComponent();
            Row = row;
            this.type = type;
            intUpDowns = new IntUpDown[Row];
            intTrackBars = new IntTrackBar[Row];
            int tabIndex = 0;
            for (int i = 0; i < Row; i++)
            {
                var control = new IntUpDown()
                {
                    Width = minItemWidth,
                };
                control.Tag = i;
                control.SequentialValueChanged += cbxValue_ValueChanged;
                control.TabIndex = tabIndex;
                control.Width = 74;
                tabIndex++;
                Controls.Add(control);
                intUpDowns[i] = control;

                var bar = new IntTrackBar();
                bar.Width = (type != null && type.optional)? 186:240;
                intTrackBars[i] = bar;
                Controls.Add(bar);
                bar.Tag = i;
                bar.SequentialValueChanged += bar_SequentialValueChanged;
                bar.TabIndex = tabIndex;
                tabIndex++;
                Controls.Add(control);
            }
            if (type != null && type.optional)
            {
                btnIncrement.TabIndex = tabIndex;
                tabIndex++;
                btnDecrement.TabIndex = tabIndex;
            }
            else
            {
                btnIncrement.Visible = false;
                btnDecrement.Visible = false;
            }

            lblMinimum = new UILabel()
            {
                AutoSize = false,
                TextAlign = ContentAlignment.TopLeft,
            };
            lblMaximum = new UILabel()
            {
                AutoSize = false,
                TextAlign = ContentAlignment.TopRight,
            };
            lblMinimum.Height = lblMaximum.Height = 13;
            Controls.Add(lblMinimum);
            Controls.Add(lblMaximum);
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);
        }
        public override void Align()
        {
            int width = DefaultWidth - MarginX + marginBetweenColumn;
            for (int i = 0; i < Row; i++)
            {
                Control control = intUpDowns[i];
                control.Location = new Point(MarginX, MarginY + subControlHeight * i);

                var bar = intTrackBars[i];
                bar.Location = new Point(MarginX + control.Size.Width, MarginY + subControlHeight * i - 1);
            }

            lblMinimum.Location = new Point(intTrackBars[0].Location.X + 3, MarginY + subControlHeight * Row);
            lblMinimum.Width = (intTrackBars[0].Width - 6) / 2;
            lblMaximum.Location = new Point(lblMinimum.Right, MarginY + subControlHeight * Row);
            lblMaximum.Width = lblMinimum.Width;

            if (type != null && type.optional)
            {
                btnIncrement.Location = new Point(intTrackBars[0].Right + 4, MarginY);
                btnDecrement.Location = new Point(btnIncrement.Right + 4, btnIncrement.Top);
            }
            else
            {
                // Align の都合上左上に寄せておく
                btnIncrement.Location = new Point(0, 0);
                btnDecrement.Location = new Point(0, 0);
            }
        }

        public override bool SetValues(Material material, List<string> values, bool candidateModified, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel, Material.ValueResolvedState valueResolvedState)
        {
            Value = values.Select(x => { int result; int.TryParse(x, out result); return result; }).ToArray();

            bool rangeSet = false;
            int min = 0;
            int max = 0;
            int[] defaults = new int[0];
            bool packFromHio = false;

            if (Viewer.Manager.Instance.IsConnected)
            {
                var candidate = material.RenderInfoPackFromHio.IntItems.FirstOrDefault(x => x.name == ParamName);
                if (candidate != null)
                {
                    if (candidate.hasMinMax)
                    {
                        min = candidate.min;
                        max = candidate.max;
                        rangeSet = true;
                    }
                    defaults = candidate.defaults.ToArray();
                    packFromHio = true;
                }
            }

            if (type != null && !packFromHio)
            {
                if (IfShaderAssignUtility.TryParseIntRange(type.choice, out min, out max))
                {
                    rangeSet = true;
                }

                defaults = G3dDataParser.Tokenize(type.Default()).Select(x => { int result; int.TryParse(x, out result); return result; }).ToArray();
            }

            lblParameter.BackColor = GetBgColor(candidateModified, packFromHio, rangeSet);

            // デフォルト値を設定
            for (int i = ItemCount; i < Math.Min(intUpDowns.Length, defaults.Length); i++)
            {
                intTrackBars[i].Value = intUpDowns[i].Value = defaults[i];
            }

            bool validDefault = defaults.All(x => min <= x && x <= max);

            for (int i = 0; i < intUpDowns.Length; i++)
            {
                var control = intUpDowns[i];
                if (!rangeSet)
                {
                    intTrackBars[i].Enabled = control.Enabled = false;
                }
                else
                {
                    intTrackBars[i].Enabled = control.Enabled = i < ItemCount && type != null && validDefault;
                    intTrackBars[i].Minimum = control.Minimum = min;
                    intTrackBars[i].Maximum = control.Maximum = max;

                }
            }

            if (rangeSet)
            {
                lblMinimum.Text = min.ToString();
                lblMaximum.Text = max.ToString();
            }
            else
            {
                lblMinimum.Text = string.Empty;
                lblMaximum.Text = string.Empty;
            }

            btnIncrement.Enabled = ItemCount < intUpDowns.Length && rangeSet && type != null && validDefault;
            btnDecrement.Enabled = 0 < ItemCount && rangeSet && type != null && validDefault;

            updateThumColor();
            return false;
        }

        void updateThumColor()
        {
            for (int i = 0; i < Row; i++)
            {
                var value = intTrackBars[i].Value;
                intTrackBars[i].DrawRedThumb = value < intTrackBars[i].Minimum || intTrackBars[i].Maximum < value;
            }
        }

        private void bar_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            var bar = (IntTrackBar)sender;
            using (new UIControlEventSuppressBlock())
            {
                intUpDowns[(int)bar.Tag].Value = bar.Value;
                updateThumColor();
            }

            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                ParamValue = Value.Select(x => x.ToString()).ToList(),
                SequentialValueChangedEventArgs = e,
                type = render_info_typeType.@int,
            };
            InvokeValueChanged(this, args);
        }

        private void cbxValue_ValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                ParamValue = Value.Select(x => x.ToString()).ToList(),
                SequentialValueChangedEventArgs = e,
                type = render_info_typeType.@int,
            };
            InvokeValueChanged(this, args);
        }

        private void btnIncrement_Click(object sender, EventArgs e)
        {
            ItemCount++;
            Debug.Assert(ItemCount <= intUpDowns.Length);
            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                ParamValue = Value.Select(x => x.ToString()).ToList(),
                SequentialValueChangedEventArgs = null,
                type = render_info_typeType.@int,
            };
            InvokeValueChanged(this, args);
        }

        private void btnDecrement_Click(object sender, EventArgs e)
        {
            ItemCount--;
            Debug.Assert(ItemCount >= 0);
            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                ParamValue = Value.Select(x => x.ToString()).ToList(),
                SequentialValueChangedEventArgs = null,
                type = render_info_typeType.@int,
            };
            InvokeValueChanged(this, args);
        }
    }
}
