﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Controls;
using App.Data;
using App.res;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class ColorFloat : ShaderParamControl
    {
        // カラーの表示切替
        public event ColorControlChangedEventHandler ColorControlChanged = null;

        protected override void labelContextMenuStrip_Opening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            // コンテキストメニューの構築
            base.labelContextMenuStrip_Opening(sender, e);
            if (ColorControlChanged != null)
            {
                var cmiColorControl = new System.Windows.Forms.ToolStripMenuItem();
                cmiColorControl.Size = new System.Drawing.Size(152, 22);
                cmiColorControl.Text = Strings.ShaderParamControl_ShowAsNumber;
                cmiColorControl.Click += (s, args) =>
                {
                    // フォーカスを外す
                    ActiveControl = null;

                    ColorControlChanged(s, new ColorControlChangedEventArgs()
                    {
                        ParamName = ParamName,
                        ShowColor = false,
                    });
                };
                labelContextMenuStrip.Items.Add(cmiColorControl);
            }
        }

        public RgbaColor Value
        {
            get{	return cepValue.Color;	}
            set{	cepValue.Color = value;	}
        }

        public ColorEditMode EditMode
        {
            get{	return cepValue.EditMode;	}
            set{
                cepValue.EditMode = value;

                if (cepValue.EditMode == ColorEditMode.RGBA)
                {
                    cepValue.Height = 112;
                    NumComponentCurveEditorOpeningButtons = 4;
                }
                else
                if (cepValue.EditMode == ColorEditMode.RGB)
                {
                    cepValue.Height = 90;
                    NumComponentCurveEditorOpeningButtons = 3;
                }
                else
                {
                    Debug.Assert(false);
                }
                foreach (var button in ComponentCurveEditorOpeningButtons)
                {
                    Controls.Remove(button);
                }
            }
        }

        public float? minimumValue_ = null;
        public float? MinimumValue
        {
            get
            {
                return minimumValue_;
            }

            set
            {
                minimumValue_ = value;
            }
        }

        public float? maximumValue_ = null;
        public float? MaximumValue
        {
            get
            {
                return maximumValue_;
            }

            set
            {
                maximumValue_ = value;
                cepValue.IsFreeRange = (ConfigData.ApplicationConfig.DefaultValue.UIRange.Clamp == false) || (maximumValue_.HasValue == false);
                cepValue.MaximumValue = maximumValue_.HasValue ? maximumValue_.Value : 1.0f;
            }
        }

        private System.Drawing.Rectangle? initialBounds_ = null;

        public void SetDefaultMax(float value)
        {
            cepValue.SetDefaultMax(value);
        }

        public override ParamType ParamType
        {
            get { return ParamType.uniform_var; }
        }

        public ColorFloat()
        {
            InitializeComponent();

            cepValue.ColorPickerText = Strings.ColorPicker;

            cepValue.IsDefaultLinear = ConfigData.ApplicationConfig.Color.GammaCorrection;
            cepValue.Tag = (uint)(1<<0);

            NumComponentCurveEditorOpeningButtons =
                (EditMode == ColorEditMode.RGBA) ? 4 :
                (EditMode == ColorEditMode.RGB) ? 3 :
                (EditMode == ColorEditMode.A) ? 1 :
                0;
        }

        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginallabel)
        {
            float[] paramArray = G3dDataParser.ParseFloatArray(value);

            if (paramArray == null || paramArray.Length < 3 || paramArray.Length > 4)
            {
                Enabled = false;
                return false;
            }

            Value =
                (paramArray.Length == 3) ?
                    new RgbaColor(paramArray[0], paramArray[1], paramArray[2], 1.0f) :
                    new RgbaColor(paramArray[0], paramArray[1], paramArray[2], paramArray[3]);

            return false;
        }

        public void FitRange()
        {
            cepValue.FitRange();
        }

        private void cepValue_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            ShaderParamValueChangedEventArgs args = new ShaderParamValueChangedEventArgs(shader_param_typeType.@float);
            {
                args.ParamName							= ParamName;
                args.ParamValue = Value.ToArray(cepValue.EditMode == ColorEditMode.RGBA);
                args.ElementBits						= e.ElementBits;
                args.SequentialValueChangedEventArgs	= e;
            }

            InvokeValueChanged(this, args);
        }

        public override IEnumerable<IHasShaderParameterAnimation> FilterAnimation(IEnumerable<IHasShaderParameterAnimation> animations)
        {
            return animations.Where(x => x.IsColorType());
        }

        public override void OnBeforeDisposedOrCached()
        {
            cepValue.EndColorPickerConnection();
            base.OnBeforeDisposedOrCached();
        }

        public override void Align()
        {
            // スライダーバーの右横に「カーブエディタを開く」ボタンを配置。
            bool showButtons = false;
            do
            {
                var chanelKeys =
                    (EditMode == ColorEditMode.RGBA) ? new string[] { "ctlR", "ctlG", "ctlB", "ctlA" } :
                    (EditMode == ColorEditMode.RGB) ? new string[] { "ctlR", "ctlG", "ctlB" } :
                    (EditMode == ColorEditMode.A) ? new string[] { "ctlA" } :
                    new string[] { };

                int numChannels = chanelKeys.Length;
                if ((numChannels <= 1) || (numChannels != ComponentCurveEditorOpeningButtons.Count))
                {
                    break;
                }

                var channelCtrls = new System.Windows.Forms.Control[numChannels];
                for (int i = 0; i < numChannels; i++)
                {
                    channelCtrls[i] = cepValue.Controls.Find(chanelKeys[i], false).FirstOrDefault();
                }
                if (!channelCtrls.All(x => x != null))
                {
                    break;
                }

                for (int i = 0; i < numChannels; i++)
                {
                    ComponentCurveEditorOpeningButtons[i].Location = new System.Drawing.Point(cepValue.Right + 4, cepValue.Top + channelCtrls[i].Top);
                }

                // コントロールサイズ拡大の妨げになるアンカーを一時的に外しておく。
                var oColorEditPanelAnchors = new System.Windows.Forms.AnchorStyles[Controls.Count];
                for (int i = 0; i < Controls.Count; i++)
                {
                    oColorEditPanelAnchors[i] = Controls[i].Anchor;
                    Controls[i].Anchor &= ~(System.Windows.Forms.AnchorStyles.Right | System.Windows.Forms.AnchorStyles.Bottom);
                }

                // 成分毎の「カーブエディタで開く」ボタンを追加する前の境界を取得しておく。
                if (!initialBounds_.HasValue)
                {
                    initialBounds_ = Bounds;
                }

                // すべてのコントロールを内包する境界。
                // 座標はクラス外部で配置されるので、サイズだけを更新。
                var bounds = ComponentCurveEditorOpeningButtons.Select(x => x.Bounds).Concat(Enumerable.Repeat(initialBounds_.Value, 1)).Aggregate(System.Drawing.Rectangle.Union);
                Size = bounds.Size;

                // アンカーを戻す。
                for (int i = 0; i < Controls.Count; i++)
                {
                    Controls[i].Anchor = oColorEditPanelAnchors[i];
                }

                showButtons = true;
            }
            while (false);

            if (showButtons)
            {
                // 成分毎の「カーブエディタで開く」ボタンを追加する。
                Controls.AddRange(ComponentCurveEditorOpeningButtons.Where(x => !Controls.Contains(x)).ToArray());
            }
            else
            {
                foreach (var button in ComponentCurveEditorOpeningButtons)
                {
                    Controls.Remove(button);
                }

                if (initialBounds_.HasValue)
                {
                    // 座標はクラス外部で配置されるので、サイズだけを更新。
                    Size = initialBounds_.Value.Size;
                }
            }
        }
    }
}
