﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Windows.Forms;
using System.Linq;
using System.Diagnostics;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using System.IO;
using App.Utility;
using System.Collections.Generic;
using nw.g3d.iflib;
using System.Text;

namespace App.PropertyEdit
{
    public partial class MaterialPreviewPage : MaterialPropertyPage
    {
        public MaterialPreviewPage() :
            base(PropertyPageID.MaterialPreview)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_material_property_window_preview_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new MaterialPreviewPage();
        }

        protected override void InitializeFormInternal()
        {
            // シェーダー送信状態更新用にイベントを登録
            Action<Document, bool> attachedChanged = (x, y) =>
            {
                if (IsHandleCreated)
                {
                    BeginInvoke((Action)(() =>
                    {
                        if (Visible && (x is Model || x is SeparateMaterial) && Targets.Active is Material && ActiveTarget.OwnerDocument == x)
                        {
                            UpdateShaderState();
                        }
                    }));
                }
            };
            Model.IsAttachedChanged += attachedChanged;
            Disposed += (s, e) => { Model.IsAttachedChanged -= attachedChanged; };
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            var savedItem = DocumentManager.ProjectDocument.GetSavedMaterialItem(ActiveTarget);
            cbxOptimize.IsModified = savedItem.OptimizeShader != ActiveTarget.OptimizeShader;
            cbxOptimize.Checked = ActiveTarget.OptimizeShader;

            cbxOptimizeOnReload.Enabled = ActiveTarget.OptimizeShader;
            cbxOptimizeOnReload.IsModified = savedItem.OptimizeShaderOnReload != ActiveTarget.OptimizeShaderOnReload;
            cbxOptimizeOnReload.Checked = ActiveTarget.OptimizeShaderOnReload;

            UpdateShaderState();
        }

        // シェーダー状態の表示の更新
        private void UpdateShaderState()
        {
            if (ActiveTarget.Referrers.All(x => !x.IsAttached))
            {
                tbxShaderState.Text = res.Strings.MaterialPreview_ModelNotSent;
            }
            else if (ActiveTarget.ShaderSent)
            {
                if (ActiveTarget.OptimizeShaderSent)
                {
                    if (ActiveTarget.ModifiedShaderSent)
                    {
                        tbxShaderState.Text = res.Strings.MaterialPreview_OptimizedEdittedShaderSent;
                    }
                    else
                    {
                        tbxShaderState.Text = res.Strings.MaterialPreview_OptimizedShaderSent;
                    }
                }
                else
                {
                    Debug.Assert(ActiveTarget.ModifiedShaderSent);
                    tbxShaderState.Text = res.Strings.MaterialPreview_EdittedShaderSent;
                }
            }
            else
            {
                bool stateTextSet = false;
                var shaderDefinition = ActiveTarget.MaterialShaderAssign.ShaderDefinition;
                if (shaderDefinition != null)
                {
                    if (shaderDefinition.FailedToBinarize)
                    {
                        tbxShaderState.Text = res.Strings.MaterialPreview_ShaderConvertError;
                        stateTextSet = true;
                    }
                    else
                    {
                        // アタッチシェーダー
                        var shadingModelName = ActiveTarget.MaterialShaderAssign.ShaderName;
                        if (shaderDefinition.IsAttached &&
                            shaderDefinition.updatedAttachedShadingModel.Contains(shadingModelName))
                        {
                            tbxShaderState.Text = res.Strings.MaterialPreview_AttachedShaderSent;
                            stateTextSet = true;
                        }
                    }
                }

                if (!stateTextSet)
                {
                    tbxShaderState.Text = res.Strings.MaterialPreview_ShaderNotSent;
                }
            }
        }

        public static bool IsModified(Material activeTarget)
        {
            if (activeTarget == null)
            {
                return false;
            }
            var savedItem = DocumentManager.ProjectDocument.GetSavedMaterialItem(activeTarget);
            return savedItem.OptimizeShader != activeTarget.OptimizeShader ||
                savedItem.OptimizeShaderOnReload != activeTarget.OptimizeShaderOnReload;
        }

        private void cbxOptimizeShader_CheckedChanged(object sender, EventArgs e)
        {
            bool optimize = (bool)((UICheckBox)sender).Checked;

            TheApp.CommandManager.Execute(CreateEditCommand_OptimizeShader(Targets, optimize, true));
        }

        private void cbxOptimizeOnReload_CheckedChanged(object sender, EventArgs e)
        {
            bool optimizeOnReload = (bool)((UICheckBox)sender).Checked;
            TheApp.CommandManager.Execute(CreateEditCommand_OptimizeShaderOnReload(Targets, optimizeOnReload, true));
        }

        private void btnOptimize_Click(object sender, EventArgs e)
        {
            foreach (var model in Targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct())
            {
                Viewer.LoadOptimizedShaderArchive.Send(model);
            }
        }

        #region コマンド

        static private GroupEditCommand CreateEditCommand_OptimizeShader(GuiObjectGroup targets, bool optimize, bool reload)
        {
            var materials = targets.GetObjects(GuiObjectID.Material).OfType<Material>().ToArray();
            var optimizes = materials.Select(x => optimize).ToArray();
            return
                new GeneralGroupValueEditCommand<int>(
                    new GuiObjectGroup(DocumentManager.ProjectDocument),
                    GuiObjectID.Project,
                    0,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        for (int i = 0; i < materials.Length; i++)
                        {
                            var material = materials[i];
                            var tmp = material.OptimizeShader;
                            material.OptimizeShader = optimizes[i];
                            optimizes[i] = tmp;
                        }
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        if (reload)
                        {
                            foreach (var model in materials.SelectMany(x => x.Referrers).Distinct())
                            {
                                Viewer.LoadOrReloadModel.Send(model);
                            }
                        }
                    }
                );
        }

        static private GroupEditCommand CreateEditCommand_OptimizeShaderOnReload(GuiObjectGroup targets, bool optimize, bool reload)
        {
            var materials = targets.GetObjects(GuiObjectID.Material).OfType<Material>().ToArray();
            var optimizes = materials.Select(x => optimize).ToArray();
            return
                new GeneralGroupValueEditCommand<int>(
                    new GuiObjectGroup(DocumentManager.ProjectDocument),
                    GuiObjectID.Project,
                    0,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        for (int i = 0; i < materials.Length; i++)
                        {
                            var material = materials[i];
                            var tmp = material.OptimizeShaderOnReload;
                            material.OptimizeShaderOnReload = optimizes[i];
                            optimizes[i] = tmp;
                        }
                    },
                    postEditDelegate: (editTargets, data) =>
                        {
                            if (reload)
                            {
                                foreach (var model in materials.Where(x => DocumentManager.OptimizeShader && x.OptimizeShader && x.OptimizeShaderOnReload)
                                    .SelectMany(x => x.Referrers).Distinct())
                                {
                                    Viewer.LoadOrReloadModel.Send(model);
                                }
                            }
                        }
                );
        }
        #endregion

        #region コピー＆ペースト
        private class CopyData
        {
            public bool OptimizeShader { get; set; }
            public bool OptimizeShaderOnReload { get; set; }
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(Material target)
        {
            return
                new CopyData()
                {
                    OptimizeShader = target.OptimizeShader,
                    OptimizeShaderOnReload = target.OptimizeShaderOnReload,
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            var commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;
                commandSet.Add(CreateEditCommand_OptimizeShader(targets, copyData.OptimizeShader, reload:false));
                commandSet.Add(CreateEditCommand_OptimizeShaderOnReload(targets, copyData.OptimizeShaderOnReload, reload:false));
                commandSet.OnPostEdit += (s, e) =>
                    {
                        foreach (var model in targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct())
                        {
                            Viewer.LoadOrReloadModel.Send(model);
                        }
                    };
            }
            return commandSet.Execute();
        }
        #endregion
    }
}
