﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Data;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public class LightAnimationCurveTreeNodeInfo : IAnimationCurve
    {
        public bool IsEditable { get; set; }
        public AnimationDocument.NonEditableKind NonEditableKind { get; set; }
        public object NonEditableKindDisplayAux { get; set; }
        public bool IsColorCurve
        {
            get
            {
                switch (targetType_)
                {
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color0_r:
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color0_g:
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color0_b:
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color1_r:
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color1_g:
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color1_b:
                        return true;
                    default:
                        return false;
                }
            }
        }

        public int ColorComponentIndex
        {
            get
            {
                switch (targetType_)
                {
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color0_r:
                        return 0;
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color0_g:
                        return 1;
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color0_b:
                        return 2;
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color1_r:
                        return 0;
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color1_g:
                        return 1;
                    case nw.g3d.nw4f_3dif.light_anim_target_targetType.color1_b:
                        return 2;
                    default:
                        return -1;
                }
            }
        }

        public AnimTarget GetAnimTarget(GuiObject editTarget)
        {
            Debug.Assert(editTarget is LightAnimation);
            var animation = (LightAnimation)editTarget;
            return animation.GetTarget(targetType_);
        }

        public void UpdateIsModified(GuiObject editTarget)
        {
            var animation = (LightAnimation)editTarget;
            var animTarget = GetAnimTarget(animation);
            var savedTarget = animation.savedLightAnimTargets.FirstOrDefault(x => x.targetType == targetType_);
            animation.UpdateIsModifiedAnimTarget(
                (LightAnimationTarget)animTarget,
                savedTarget);
        }

        public AnimTarget CreateAnimTarget(GuiObject editTarget)
        {
            Debug.Assert(false);
            return null;
        }

        public AnimTarget CreateTemporaryTarget(GuiObject editTarget)
        {
            var source = (LightAnimationTarget)GetAnimTarget(editTarget);
            return new LightAnimationTarget()
            {
                targetType = source.targetType,
                CurveInterpolationType = targetType_ == light_anim_target_targetType.enable ? InterpolationType.Step: InterpolationType.Hermite,
            };
        }

        public void SetAnimTarget(GuiObject editTarget, AnimTarget animTarget)
        {
            var anims = ((LightAnimation)editTarget).LightAnimTargets;
            var index = anims.FindIndex(x => x.targetType == targetType_);
            Debug.Assert(index >= 0);
            anims[index] = (LightAnimationTarget)animTarget;
        }

        public float GetDefaultValue(GuiObject editTarget)
        {
            return GetAnimTarget(editTarget).GetBaseValue();
        }


        public void IntermediateFileIndices(LightAnimation target, out int cameraIndex, out int curveIndex, out int curveComponentIndex)
        {
            curveComponentIndex = (int)targetType_;
            cameraIndex = target.Owner.LightAnims.IndexOf(target);
            curveIndex = 0;
            foreach (var animTarget in target.LightAnimTargets.Where(x => x.ExportType == CurveExportType.Constant || x.ExportType == CurveExportType.Curve))
            {
                if (animTarget.targetType == targetType_)
                {
                    return;
                }

                curveIndex++;
            }

            curveIndex = -1;
        }

        private readonly LightAnimation	target_;
        private readonly light_anim_target_targetType targetType_;

        public LightAnimationCurveTreeNodeInfo(LightAnimation target, light_anim_target_targetType targetType)
        {
            //Debug.Assert(this is IHasAnimTarget);
            target_			= target;
            targetType_	= targetType;
        }

        public List<KeyFrame> KeyFrames
        {
            get
            {
                return
                    (paramAnim != null) ?
                        paramAnim.KeyFrames :
                        new List<KeyFrame>();
            }
        }

        public LightAnimationTarget paramAnim { get { return target_.GetTarget(targetType_); } }
        public InterpolationType CurveInterpolationType
        {
            get
            {
                return paramAnim != null ? paramAnim.CurveInterpolationType : InterpolationType.Linear;
            }

            set
            {
                ;   // 無視
            }
        }

        public string ParentName{		get { return string.Empty;		} }
        public string Name
        {
            get
            {
                switch (targetType_)
                {
                    case light_anim_target_targetType.enable: return "enable";
                    case light_anim_target_targetType.position_x: return "position";
                    case light_anim_target_targetType.position_y: return "position";
                    case light_anim_target_targetType.position_z: return "position";
                    case light_anim_target_targetType.direction_x: return "direction";
                    case light_anim_target_targetType.direction_y: return "direction";
                    case light_anim_target_targetType.direction_z: return "direction";
                    case light_anim_target_targetType.aim_x: return "aim";
                    case light_anim_target_targetType.aim_y: return "aim";
                    case light_anim_target_targetType.aim_z: return "aim";
                    case light_anim_target_targetType.dist_attn_start: return "dist_attn";
                    case light_anim_target_targetType.dist_attn_end: return "dist_attn";
                    case light_anim_target_targetType.angle_attn_start: return "angle_attn";
                    case light_anim_target_targetType.angle_attn_end: return "angle_attn";
                    case light_anim_target_targetType.color0_r: return "color0";
                    case light_anim_target_targetType.color0_g: return "color0";
                    case light_anim_target_targetType.color0_b: return "color0";
                    case light_anim_target_targetType.color1_r: return "color1";
                    case light_anim_target_targetType.color1_g: return "color1";
                    case light_anim_target_targetType.color1_b: return "color1";
                }
                throw new NotImplementedException();
            }
        }
        public string FullPath
        {
            get
            {
                switch(targetType_)
                {
                    case light_anim_target_targetType.enable:			return target_.Name + " / enable";
                    case light_anim_target_targetType.position_x:		return target_.Name + " / position / X";
                    case light_anim_target_targetType.position_y:		return target_.Name + " / position / Y";
                    case light_anim_target_targetType.position_z:		return target_.Name + " / position / Z";
                    case light_anim_target_targetType.direction_x:		return target_.Name + " / direction / X";
                    case light_anim_target_targetType.direction_y:		return target_.Name + " / direction / Y";
                    case light_anim_target_targetType.direction_z:		return target_.Name + " / direction / Z";
                    case light_anim_target_targetType.aim_x:			return target_.Name + " / aim / X";
                    case light_anim_target_targetType.aim_y:			return target_.Name + " / aim / Y";
                    case light_anim_target_targetType.aim_z:			return target_.Name + " / aim / Z";
                    case light_anim_target_targetType.dist_attn_start:	return target_.Name + " / dist_attn / start";
                    case light_anim_target_targetType.dist_attn_end:	return target_.Name + " / dist_attn / end";
                    case light_anim_target_targetType.angle_attn_start:	return target_.Name + " / angle_attn / start";
                    case light_anim_target_targetType.angle_attn_end:	return target_.Name + " / angle_attn / end";
                    case light_anim_target_targetType.color0_r:			return target_.Name + " / color0 / R";
                    case light_anim_target_targetType.color0_g:			return target_.Name + " / color0 / G";
                    case light_anim_target_targetType.color0_b:			return target_.Name + " / color0 / B";
                    case light_anim_target_targetType.color1_r:			return target_.Name + " / color1 / R";
                    case light_anim_target_targetType.color1_g:			return target_.Name + " / color1 / G";
                    case light_anim_target_targetType.color1_b:			return target_.Name + " / color1 / B";
                }

                Debug.Assert(false);
                return string.Empty;
            }
        }

        public int ComponentIndex{		get { return (int)targetType_;	} }
        public bool IsRotate
        {
            get
            {
                return
                    (targetType_ == light_anim_target_targetType.angle_attn_start) ||
                    (targetType_ == light_anim_target_targetType.angle_attn_end);
            }
        }
        public float? MinClampValue{	get { return MinFitValue; } }
        public float? MaxClampValue{	get { return MaxFitValue; } }
        public float? MinFitValue{
            get {
                switch (targetType_)
                {
                    case light_anim_target_targetType.enable:
                    case light_anim_target_targetType.color0_r:
                    case light_anim_target_targetType.color0_g:
                    case light_anim_target_targetType.color0_b:
                    case light_anim_target_targetType.color1_r:
                    case light_anim_target_targetType.color1_g:
                    case light_anim_target_targetType.color1_b:
                        return 0;
                }
                return null;
            }
        }

        public float? MaxFitValue{
            get {
                switch (targetType_)
                {
                    case light_anim_target_targetType.enable:
                    case light_anim_target_targetType.color0_r:
                    case light_anim_target_targetType.color0_g:
                    case light_anim_target_targetType.color0_b:
                    case light_anim_target_targetType.color1_r:
                    case light_anim_target_targetType.color1_g:
                    case light_anim_target_targetType.color1_b:
                        return 1;
                }
                return null;
            }
        }
        public PrimitiveTypeKind CurvePrimitiveType
        {
            get {
                return targetType_ == light_anim_target_targetType.enable ?
                    PrimitiveTypeKind.Bool:
                    PrimitiveTypeKind.Float;
            }
        }

        public Color CurveColor
        {
            get
            {
                switch (paramAnim.targetType)
                {
                    case light_anim_target_targetType.color0_r:
                    case light_anim_target_targetType.color1_r:
                        return CurveView.MakeRGBAColorFromIndex(0);
                    case light_anim_target_targetType.color0_g:
                    case light_anim_target_targetType.color1_g:
                        return CurveView.MakeRGBAColorFromIndex(1);
                    case light_anim_target_targetType.color0_b:
                    case light_anim_target_targetType.color1_b:
                        return CurveView.MakeRGBAColorFromIndex(2);
                }

                return CurveView.MakeColorFromString(target_.Name + ParentName + Name);
            }
        }

        public bool IsSame(IAnimationCurve curve)
        {
            var lightCurve = (LightAnimationCurveTreeNodeInfo)curve;
            return lightCurve.target_ == target_ &&
                lightCurve.targetType_ == targetType_;
        }
    }
}
