﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

using App.Command;
using App.Controls;
using App.Data;
using App.Utility;

using ConfigCommon;

using nw.g3d.nw4f_3dif;

using Viewer;

namespace App.PropertyEdit
{
    public partial class CameraAnimationGeneralPage : CameraAnimationPropertyPage
    {
        public CameraAnimationGeneralPage() :
            base(PropertyPageID.CameraAnimationGeneral)
        {
            InitializeComponent();

            // 位置
            InitCameraAnimFloatEditPanel(fepTranslateX, camera_anim_target_targetType.position_x);
            InitCameraAnimFloatEditPanel(fepTranslateY, camera_anim_target_targetType.position_y);
            InitCameraAnimFloatEditPanel(fepTranslateZ, camera_anim_target_targetType.position_z);

            // 回転
            InitCameraAnimFloatEditPanel(fepRotateX, camera_anim_target_targetType.rotate_x);
            InitCameraAnimFloatEditPanel(fepRotateY, camera_anim_target_targetType.rotate_y);
            InitCameraAnimFloatEditPanel(fepRotateZ, camera_anim_target_targetType.rotate_z);

            // 注視点
            InitCameraAnimFloatEditPanel(fepAimX, camera_anim_target_targetType.aim_x);
            InitCameraAnimFloatEditPanel(fepAimY, camera_anim_target_targetType.aim_y);
            InitCameraAnimFloatEditPanel(fepAimZ, camera_anim_target_targetType.aim_z);
            InitCameraAnimFloatEditPanel(fepTwist, camera_anim_target_targetType.twist);

            // その他
            InitCameraAnimFloatEditPanel(fepAspect, camera_anim_target_targetType.aspect);
            InitCameraAnimFloatEditPanel(fepNear, camera_anim_target_targetType.near);
            InitCameraAnimFloatEditPanel(fepFar, camera_anim_target_targetType.far);
            InitCameraAnimFloatEditPanel(fepFov, camera_anim_target_targetType.persp_fovy);
            InitCameraAnimFloatEditPanel(fepHeight, camera_anim_target_targetType.ortho_height);
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_animation_property_window_general_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new CameraAnimationGeneralPage();
        }

        protected override void InitializeFormInternal()
        {
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            var target = ActiveTarget;

            // 全般
            lblFrameCount.IsModified = target.IsValueChanged(x => x.frame_count);
            iepFrameSize.Value = target.Data.frame_count;

            cbxLoop.IsModified = target.IsValueChanged(x => x.loop);
            cbxLoop.Checked = target.Data.loop;

            // モード
            lblRotateMode.IsModified = target.IsValueChanged(x => x.rotate_mode);
            rbtAim.Checked = target.Data.rotate_mode == camera_anim_rotate_modeType.aim;
            rbtEuler.Checked = target.Data.rotate_mode == camera_anim_rotate_modeType.euler_zxy;

            lblProjectionMode.IsModified = target.IsValueChanged(x => x.projection_mode);
            rbtOrtho.Checked = target.Data.projection_mode == camera_anim_projection_modeType.ortho;
            rbtPersp.Checked = target.Data.projection_mode == camera_anim_projection_modeType.persp;

            // 位置
            UpdateCameraAnimFloatEditPanel(fepTranslateX, lblTranslateX, camera_anim_target_targetType.position_x);
            UpdateCameraAnimFloatEditPanel(fepTranslateY, lblTranslateY, camera_anim_target_targetType.position_y);
            UpdateCameraAnimFloatEditPanel(fepTranslateZ, lblTranslateZ, camera_anim_target_targetType.position_z);

            // 回転
            var enableRotate = target.Data.rotate_mode == camera_anim_rotate_modeType.euler_zxy;
            UpdateCameraAnimFloatEditPanel(fepRotateX, lblRotateX, camera_anim_target_targetType.rotate_x, enableRotate);
            UpdateCameraAnimFloatEditPanel(fepRotateY, lblRotateY, camera_anim_target_targetType.rotate_y, enableRotate);
            UpdateCameraAnimFloatEditPanel(fepRotateZ, lblRotateZ, camera_anim_target_targetType.rotate_z, enableRotate);
            pnlRotate.Visible = enableRotate;

            // 注視点
            var enableAim = target.Data.rotate_mode == camera_anim_rotate_modeType.aim;
            UpdateCameraAnimFloatEditPanel(fepAimX, lblAimX, camera_anim_target_targetType.aim_x, enableAim);
            UpdateCameraAnimFloatEditPanel(fepAimY, lblAimY, camera_anim_target_targetType.aim_y, enableAim);
            UpdateCameraAnimFloatEditPanel(fepAimZ, lblAimZ, camera_anim_target_targetType.aim_z, enableAim);
            UpdateCameraAnimFloatEditPanel(fepTwist, lblTwist, camera_anim_target_targetType.twist, enableAim);
            pnlAim.Visible = enableAim;

            // その他
            var isPerspective = target.Data.projection_mode == camera_anim_projection_modeType.persp;
            UpdateCameraAnimFloatEditPanel(fepAspect, lblAspect, camera_anim_target_targetType.aspect);
            UpdateCameraAnimFloatEditPanel(fepNear, lblNear, camera_anim_target_targetType.near);
            UpdateCameraAnimFloatEditPanel(fepFar, lblFar, camera_anim_target_targetType.far);
            UpdateCameraAnimFloatEditPanel(fepFov, lblFov, camera_anim_target_targetType.persp_fovy, isPerspective);
            UpdateCameraAnimFloatEditPanel(fepHeight, lblHeight, camera_anim_target_targetType.ortho_height, !isPerspective);
            pnlFov.Visible = isPerspective;
            pnlHeight.Visible = !isPerspective;
        }

        private void InitCameraAnimFloatEditPanel(FloatEditPanel floatEditPanel, camera_anim_target_targetType targetType)
        {
            // tagに編集前の値を入れて保持する
            floatEditPanel.Tag = null;

            floatEditPanel.SequentialValueChanged += (sender, args) =>
                {
                    var animationCurve = new CameraAnimationCurveTreeNodeInfo(ActiveTarget, targetType);
                    FloatEditValueChanged(floatEditPanel, targetType, args);
                };
        }

        private void FloatEditValueChanged(FloatEditPanel floatEditPanel, camera_anim_target_targetType targetType, SequentialValueChangedEventArgs args)
        {
            var animationCurve = new CameraAnimationCurveTreeNodeInfo(ActiveTarget, targetType);
            var value = floatEditPanel.Value;
            if (!(floatEditPanel.Tag is float))
            {
                floatEditPanel.Tag = animationCurve.paramAnim.GetBaseValue();
            }

            if (args.Changing && animationCurve.HasKeyFrame())
            {
                animationCurve.KeyFrames.ForEach(x => x.Value = value);
                var activeTarget = ActiveTarget;

                SendAnimationCurve(activeTarget, animationCurve);
            }
            else
            {

                var oldValue = (float)floatEditPanel.Tag;
                floatEditPanel.Tag = null;
                animationCurve.KeyFrames.ForEach(x => x.Value = oldValue);

                var commandSet = CreateEditCommand_ConstantAnimation(new GuiObjectGroup(ActiveTarget), targetType, value);
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        internal static void SendAnimationCurve(GuiObject activeTarget, IAnimationCurve animationCurve)
        {
            TransactionMessage.Send(true);
            {
                var document = (AnimationDocument)activeTarget.OwnerDocument;
                AnimTarget animTarget;
                int parentIndex;
                int curveIndex;
                int curveComponentIndex;
                int curveIndexInBinary;
                var sendCurve = CurveEditorPanel.CurveIndex(
                    activeTarget,
                    animationCurve,
                    out parentIndex,
                    out curveIndex,
                    out curveComponentIndex,
                    out curveIndexInBinary,
                    out animTarget);
                if (animTarget != null && sendCurve)
                {
                    animTarget.IsFileOutputable = AnimationCurveEditCommand.MakeIsFileOutputable(activeTarget, animationCurve);

                    var exportType = animTarget.ExportType;
                    switch (exportType)
                    {
                        case CurveExportType.Curve:
                        case CurveExportType.Constant:
                            EditCurve.Send(
                                document,
                                (uint)parentIndex,
                                (uint)curveIndex,
                                (uint)curveComponentIndex,
                                animationCurve.IsRotate,
                                animTarget,
                                curveIndexInBinary,
                                activeTarget.ObjectID,
                                animationCurve.CurvePrimitiveType,
                                false,
                                exportType);
                            break;
                    }
                }
            }
            TransactionMessage.Send(false);
        }

        private void UpdateCameraAnimFloatEditPanel(FloatEditPanel floatEditPanel, UIModifiedMarkLabel uilabel, camera_anim_target_targetType targetType, bool enable = true)
        {
            var animTarget = ActiveTarget.GetTarget(targetType);
            UpdateFloatEditPanel(floatEditPanel, uilabel, animTarget, enable);
        }

        internal static void UpdateFloatEditPanel(
            FloatEditPanel floatEditPanel,
            UIModifiedMarkLabel uilabel,
            AnimTarget animTarget,
            bool enable)
        {
            float baseValue;
            enable = enable && animTarget != null
                     && (animTarget.KeyFrames.Count == 0 || animTarget.KeyFrames.IsConstantCurve(out baseValue));
            floatEditPanel.Enabled = enable;
            uilabel.Enabled = enable;
            floatEditPanel.Value = enable ? animTarget.GetBaseValue() : 0.0f;
            uilabel.IsModified = animTarget != null && animTarget.IsModified;
        }

        public static bool IsModified(CameraAnimation activeTarget)
        {
            if (activeTarget == null)
            {
                return false;
            }
            var targetTypes = Enum.GetValues(typeof(camera_anim_target_targetType)).OfType<camera_anim_target_targetType>();
            var animModified = targetTypes.Select(activeTarget.GetTarget).Any(x => x != null && x.IsModified);

            return animModified ||
                activeTarget.IsValueChanged(x => x.frame_count) ||
                activeTarget.IsValueChanged(x => x.loop) ||
                activeTarget.IsValueChanged(x => x.rotate_mode) ||
                activeTarget.IsValueChanged(x => x.projection_mode);
        }

        #region コマンド
        private static EditCommandSet CreateEditCommand_ConstantAnimation(
            GuiObjectGroup targets,
            camera_anim_target_targetType targetType,
            float value)
        {
            var commandSet = new EditCommandSet();
            var active = targets.Active as CameraAnimation;
            if (active == null)
            {
                return commandSet;
            }
            var animCurve = new CameraAnimationCurveTreeNodeInfo(active, targetType);

            var newKeys = new List<KeyFrame>();
            if (animCurve.KeyFrames.Any())
            {
                newKeys.AddRange(ObjectUtility.Clone(animCurve.KeyFrames));
                newKeys.ForEach(x => x.Value = value);
            }
            else
            {
                var key = new KeyFrame { Frame = 0, Value = value };
                newKeys.Add(key);
            }
            commandSet.Add(AnimationCurveEditCommand.Create(targets, GuiObjectID.CameraAnimation, animCurve, newKeys));

            commandSet.Add(
                new LazyCommand(() => AnimationCurveEditCommand.CreateQuantizeAnalyseAllCommand(targets.Active, false)));
            commandSet.OnPostEdit += (s, e) =>
            {
                if (active.OwnerDocument is AnimationDocument)
                {
                    LoadOrReloadAnimation.Send((AnimationDocument)active.OwnerDocument);
                }
            };
            return commandSet;
        }

        private static GroupEditCommand CreateEditCommand_rotate_mode(GuiObjectGroup targets, camera_anim_rotate_modeType FrameCount)
        {
            return
                new GeneralGroupValueEditCommand<camera_anim_rotate_modeType>(
                    targets,
                    GuiObjectID.CameraAnimation,
                    FrameCount,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as CameraAnimation;

                        swap = anim.Data.rotate_mode;
                        anim.Data.rotate_mode = (camera_anim_rotate_modeType)data;

                        //  Viewerへ再転送
                        LoadOrReloadAnimation.Send(anim.Owner);
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                        // 現状複数選択は無いので、こちらでは送らない。
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_projection_mode(GuiObjectGroup targets, camera_anim_projection_modeType FrameCount)
        {
            return
                new GeneralGroupValueEditCommand<camera_anim_projection_modeType>(
                    targets,
                    GuiObjectID.CameraAnimation,
                    FrameCount,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as CameraAnimation;

                        swap = anim.Data.projection_mode;
                        anim.Data.projection_mode = (camera_anim_projection_modeType)data;

                        //  Viewerへ再転送
                        LoadOrReloadAnimation.Send(anim.Owner);
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                        // 現状複数選択は無いので、こちらでは送らない。
                    }
                );
        }

        public static GroupEditCommand CreateEditCommand_frame_count(GuiObjectGroup targets, int FrameCount)
        {
            return
                new GeneralGroupValueEditCommand<int>(
                    targets,
                    GuiObjectID.CameraAnimation,
                    FrameCount,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as CameraAnimation;

                        swap = anim.Data.frame_count;
                        anim.Data.frame_count = (int)data;
                        AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);

                        //  Viewerへ再転送
                        LoadOrReloadAnimation.Send(anim.Owner);
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                            // 現状複数選択は無いので、こちらでは送らない。
                    }
                );
        }

        public static GroupEditCommand CreateEditCommand_loop(GuiObjectGroup targets, bool value)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    GuiObjectID.CameraAnimation,
                    value,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as CameraAnimation;

                        swap = anim.Data.loop;
                        anim.Data.loop = (bool)data;

                        //  Viewerへ転送
                        LoadOrReloadAnimation.Send(anim.Owner);
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                            // 現状複数選択は無いので、こちらでは送らない。
                    }
                );
        }
        #endregion

        #region コピー＆ペースト
        private class CopyData
        {
            // アニメーション情報
            public int frame_count { get; set; }
            public bool loop { get; set; }

            // カメラ全般情報
            public camera_anim_rotate_modeType rotateMode { get; set; }
            public camera_anim_projection_modeType projectionMode { get; set; }

            // カメラアニメーション固定値
            public Dictionary<camera_anim_target_targetType, float> constantAnimTargets = new Dictionary<camera_anim_target_targetType, float>();
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(CameraAnimation target)
        {
            var data = new CopyData()
                {
                    rotateMode = target.Data.rotate_mode,
                    projectionMode = target.Data.projection_mode,
                    frame_count					= target.Data.frame_count,
                    loop						= target.Data.loop,
                };

            // カメラアニメーションの固定値だけをコピー
            foreach (var animTarget in target.CameraAnimTargets.Where(animTarget => animTarget != null))
            {
                float baseValue;
                if (animTarget.KeyFrames.Count == 0 || animTarget.KeyFrames.IsConstantCurve(out baseValue))
                {
                    baseValue = animTarget.GetBaseValue();
                    data.constantAnimTargets.Add(animTarget.targetType, baseValue);
                }
            }

            return data;
        }

        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            return CanPaste(ActiveTarget, copiedObjectInfo, copiedObject);
        }

        public static bool CanPaste(CameraAnimation activeTarget, object copiedObjectInfo, object copiedObject)
        {
            return true;
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            EditCommandSet commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(CreateEditCommand_frame_count(targets, copyData.frame_count));
                commandSet.Add(CreateEditCommand_loop(targets, copyData.loop));
                commandSet.Add(CreateEditCommand_projection_mode(targets, copyData.projectionMode));
                commandSet.Add(CreateEditCommand_rotate_mode(targets, copyData.rotateMode));

                // カメラアニメーションの固定値をペースト
                var camAnim = targets.Active as CameraAnimation;
                if (camAnim != null)
                {
                    foreach (var constAnim in copyData.constantAnimTargets)
                    {
                        var animTarget = camAnim.GetTarget(constAnim.Key);
                        if (animTarget != null)
                        {
                            float basevalue;
                            // ペースト先にアニメーションが設定されているときはペーストしない
                            if (animTarget.KeyFrames.Count == 0 || animTarget.KeyFrames.IsConstantCurve(out basevalue))
                            {
                                commandSet.Add(CreateEditCommand_ConstantAnimation(new GuiObjectGroup(camAnim), constAnim.Key, constAnim.Value));
                            }
                        }
                    }
                }
            }

            return commandSet.Execute();
        }
        #endregion

        #region イベント
        private void rbtAimOrEuler_RadioChecked(object sender, EventArgs e)
        {
            var rotateMode = rbtAim.Checked
                                ? camera_anim_rotate_modeType.aim
                                : camera_anim_rotate_modeType.euler_zxy;


            var commandSet = new EditCommandSet();
            commandSet.Add(CreateEditCommand_rotate_mode(Targets, rotateMode));
            TheApp.CommandManager.Add(commandSet.Execute());
        }

        private void rbtOrthoOrPersp_RadioChecked(object sender, EventArgs e)
        {
            var projMode = rbtOrtho.Checked
                               ? camera_anim_projection_modeType.ortho
                               : camera_anim_projection_modeType.persp;
            var commandSet = new EditCommandSet();
            commandSet.Add(CreateEditCommand_projection_mode(Targets, projMode));
            TheApp.CommandManager.Add(commandSet.Execute());
        }

        private void iepFrameSize_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (e.Changing)
            {
                ;
            }
            else
            {
                TheApp.CommandManager.Execute(CreateEditCommand_frame_count(Targets, iepFrameSize.Value));
            }
        }

        private void cbxLoop_CheckedChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_loop(Targets, cbxLoop.Checked));
        }
        #endregion //イベント
    }
}
