﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Threading;
using System.Windows.Forms;
using App.ConfigData;
using App.Data;
using App.res;
using App.Utility;
using System.Diagnostics;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using App.Bridge;
using App.Controls;
using TeamConfig;

namespace App
{
    public class UserCommandUtil
    {
        public static void ExecuteUserCommand(IEnumerable<IntermediateFileDocument> documents, string userCommandPath, FileIo.UserCommand command)
        {
            var temporary = command.Temporary;
            var args = command.Args;

            using (new WaitCursor())
            using (var tempDir = TemporaryFileUtility.MakeDisposableDirectoryName())
            {
                try
                {
                    var groups = from document in documents
                                 group document by new { document.Name, document.ObjectID };

                    var paths = new List<Tuple<IntermediateFileDocument, string>>();

                    if (temporary)
                    {
                        int index = 0;
                        var saver = new DocumentSaver();
                        foreach (var group in groups)
                        {
                            var items = group.ToArray();
                            foreach (var item in items)
                            {

                                string path;
                                if (items.Length > 1)
                                {
                                    path = Path.Combine(tempDir.Path, index.ToString());
                                    index++;
                                    Directory.CreateDirectory(path);
                                }
                                else
                                {
                                    path = tempDir.Path;
                                }

                                var filePath = Path.Combine(path, item.FileName);
                                saver.WriteDocument(item, filePath, false, true);
                                paths.Add(new Tuple<IntermediateFileDocument, string>(item, filePath));
                            }
                        }
                    }
                    else
                    {
                        foreach (var group in groups)
                        {
                            foreach (var item in group)
                            {
                                if (!string.IsNullOrEmpty(item.FilePath))
                                {
                                    paths.Add(new Tuple<IntermediateFileDocument, string>(item, item.FilePath));
                                }
                            }
                        }
                    }

                    var argFile = Path.Combine(tempDir.Path, "args.txt");
                    //using (var writer = new StreamWriter(argFile, false, Encoding.Default))
                    using (var writer = new StreamWriter(argFile, false, new UTF8Encoding(false)))
                    {
                        foreach (var path in paths.Select(x => x.Item2))
                        {
                            writer.WriteLine(Path.GetFullPath(path));
                        }
                    }

                    // Args が有れば展開する
                    if (!String.IsNullOrEmpty(args))
                    {
                        if (args.Contains("%"))
                        {
                            args = Environment.ExpandEnvironmentVariables(args);
                        }

                        if (args.Contains("$(ArgFile)"))
                        {
                            args = args.Replace("$(ArgFile)", argFile);
                        }

                        if (args.Contains("$(Models)"))
                        {
                            args = args.Replace("$(Models)", String.Join(" ", documents.OfType<Model>().Select(x => x.Name)));
                        }

                        if (args.Contains("$(Materials)"))
                        {
                            var materials = App.AppContext.SelectedTarget.GetObjects(GuiObjectID.Material);//.OfType<Material>();
                            // モデルが1つ選択されているときはそのモデルのマテリアルだけ出力
                            args = args.Replace("$(Materials)",
                                String.Join(" ", materials.Where(x => documents.Contains(x.OwnerDocument)).Select(x => x.Name)));
                        }

                        if (args.Contains("$(Bones)"))
                        {
                            var bones = App.AppContext.SelectedTarget.GetObjects(GuiObjectID.Bone);
                            // モデルが1つ選択されているときはそのモデルのボーンだけ出力
                            args = args.Replace("$(Bones)",
                                String.Join(" ", bones.Where(x => documents.Contains(x.OwnerDocument)).Select(x => x.Name)));
                        }

                        if (args.Contains("$(Shapes)"))
                        {
                            var shapes = App.AppContext.SelectedTarget.GetObjects(GuiObjectID.Shape);
                            // モデルが1つ選択されているときはそのモデルのシェイプだけ出力
                            args = args.Replace("$(Shapes)",
                                String.Join(" ", shapes.Where(x => documents.Contains(x.OwnerDocument)).Select(x => x.Name)));
                        }
                    }
                    else
                    {
                        args = argFile;
                    }


                    var info = new ProcessStartInfo(userCommandPath, args)
                    {
                        CreateNoWindow = true,
                        UseShellExecute = false,
                        RedirectStandardError = true,
                        RedirectStandardOutput = true,
                    };
#if true
                    var exitcode = UserCommandLog.Execute(info);
                    if (exitcode != 0)
                    {
                        var commandName = Path.GetFileName(userCommandPath);
                        UIMessageBox.Warning(string.Format(res.Strings.UserCommand_ExitCode, userCommandPath, exitcode));
                    }

                    // command.Temporary が false のときは、command.Reload の設定に関わらずリロードする仕様だが、
                    // この場合には DocumentManager.Reload() が呼ばれるため、ここでのリロードを除外している模様。
                    if (exitcode == 0 && command.Reload && temporary)
                    {
                        // SIGLO-62402 対応で、ユーザーコマンド実行時のリロードは「アンロード/ロード」コマンドを実行するようになったため
                        // UserCommandUtil.Reload() の forceUnloadAndLoad 引数は常に true に。
                        // 対応前の forceUnloadAndLoad は command.ForceReconnection || command.ForceUnloadAndLoad。
                        // DocumentManager.Reload() では「アンロード/ロード」が行われているため、SIGLO-62402 の対応はここだけでよい。
                        // ※モデルアタッチ時に「アンロード/ロード」を行うとアタッチが継続されなくなるので
                        // モデルアタッチ時には forceUnloadAndLoad は評価せず、「アンロード/ロード」ではなく「リロード」を行うようにしている。
                        // BridgeManager.ExecuteReload() を参照。
                        Reload(paths, true);
                    }
#else
                    try
                    {
                        process = Process.Start(info);
                        Debug.Assert(process != null, "process != null");
                        process.OutputDataReceived += (s, a) => DebugConsole.WriteLineIf(!string.IsNullOrEmpty(a.Data), a.Data);
                        process.ErrorDataReceived  += (s, a) => DebugConsole.WriteLineIf(!string.IsNullOrEmpty(a.Data), a.Data);
                        process.BeginErrorReadLine();
                        process.BeginOutputReadLine();

                        while (!process.WaitForExit(10000)) { }

                    }
                    catch (Exception)
                    {
                        if (process != null)
                        {
                            try
                            {
                                // プロセスが走っているかもしれないので止める
                                process.Kill();
                            }
                            catch
                            {
                                // 何もしない
                                DebugConsole.WriteLine("Failed to Kill process");
                            }
                        }
                    }

                    if (process.ExitCode != 0)
                    {
                        var commandName = Path.GetFileName(userCommandPath);
                        UIMessageBox.Warning(string.Format(res.Strings.UserCommand_ExitCode, userCommandPath, process.ExitCode));
                    }

                    if (process.ExitCode == 0 && command.Reload && temporary)
                    {
                        Reload(paths, command.ForceReconnection || command.ForceUnloadAndLoad);
                    }
#endif
                }
                catch (Exception exception)
                {
                    UIMessageBox.Warning(string.Format(res.Strings.UserCommand_Exception, exception.ToString()));
                }
            }
        }


        /*
         * ユーザーコマンドのショートカットを処理
         */
        public static bool ProcessShortCutKey(Keys keyData)
        {
            // テンキーの数字の場合はフルキーの数字に変換して比較
            var keyCode = keyData & Keys.KeyCode;
            var newKey = keyData;
            if (keyCode >= Keys.NumPad0 && keyCode <= Keys.NumPad9)
            {
                newKey = (keyCode - Keys.NumPad0 + Keys.D0) | (keyData & Keys.Modifiers);
            }


            foreach (var com in ApplicationConfig.FileIo.UserCommands.Where(x => x.Enable && x.ShortCutKeyBind != Keys.None))
            {
                if (com.Commands.Count() != 1 || com.ShortCutKeyBind != newKey) continue;

                var prog = com.Commands.FirstOrDefault();
                var document = GetSelectedDocument(com.FileTreeShortCut, com.Name);
                if (document == null) return true;

                if (com.FileIDs.Any() && !com.FileIDs.Contains(document.ObjectID))
                {
                    MessageLog.Write(MessageLog.LogType.Warning, string.Format(Strings.UserCommandUtil_ProcessShortCutKey_NotSuitableDocumentSelected, com.Name, document.FileName));
                    return true;
                }

                ExecuteUserCommand( new[]{ document }, prog, com);
                return true;
            }

            return false;
        }

        /// <summary>
        /// 現在選択中のドキュメントを取得する。
        /// 複数選択時はエラーメッセージを出しnullを返す
        /// </summary>
        /// <param name="isFileTree"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        public static IntermediateFileDocument GetSelectedDocument(bool isFileTree, string name)
        {
            IntermediateFileDocument document = null;
            if (isFileTree)
            {
                document = App.AppContext.CurrentSelectedObject(false) as IntermediateFileDocument;
                if (document == null)
                {
                    MessageLog.Write(MessageLog.LogType.Warning,
                        string.Format(Strings.UserCommandUtil_ProcessShortCutKey_DocmentNotSelected, name));
                    return null;
                }
            }
            else
            {
                var docobjgroup = App.AppContext.SelectedTarget.Objects.GroupBy(x => x.OwnerDocument).ToArray();
                if (docobjgroup.Count() == 1)
                {
                    document = docobjgroup.First().Key as IntermediateFileDocument;
                }
                else if (docobjgroup.Count() > 1)
                {
                    MessageLog.Write(MessageLog.LogType.Warning,
                        string.Format(Strings.UserCommandUtil_ProcessShortCutKey_MultiDocumentsSelected, name));
                    return null;
                }

                if (document == null)
                {
                    MessageLog.Write(MessageLog.LogType.Warning,
                        string.Format(Strings.UserCommandUtil_ProcessShortCutKey_ObjectNotSelected, name));
                    return null;
                }
            }
            return document;
        }

        private static void Reload(IEnumerable<Tuple<IntermediateFileDocument, string>> paths, bool forceUnloadAndLoad)
        {
            var newAndOlds = new List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>>>();
            foreach (var item in paths)
            {
                var doc = item.Item1;
                var streams = new List<G3dStream>();
                var nw4F_3Dif = nw.g3d.iflib.IfReadUtility.Read(streams, item.Item2, DocumentManager.XsdBasePath);
                if (nw4F_3Dif.Item.GetType() !=doc.nw4f_3difItem.GetType())
                {
                    throw new Exception(res.Strings.Bridge_ExtConflict);
                }
                newAndOlds.Add(new Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>>(item.Item1, nw4F_3Dif, streams));

                doc.file_info = nw4F_3Dif.file_info;
            }

            BridgeManager.ExecuteReload(newAndOlds, forceUnloadAndLoad);
        }
    }
}
