﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using App.Command;
using App.Controls;
using App.Utility;
using App.res;
using nw.g3d.iflib;
using Viewer;
using System.IO;

namespace App.Data
{
    public class ModelMerger
    {
        /// <summary>
        /// モデルファイルのマージ。
        /// </summary>
        public void MergeFile(Model dstModel, MergerUtility.MergeMode mergeMode)
        {
            string[] fileNames;
            if (DialogUtility.ExecuteOpenFileDialog(out fileNames, dstModel.ObjectID, false))
            {
                using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                {
                    Model mergedModel = null;
                    try
                    {
                        using (new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
                        {
                            mergedModel = MergeFile(dstModel, fileNames[0], mergeMode);
                        }
                    }
                    catch (Exception e)
                    {
                        UIMessageBox.Error(Strings.Merge_Error + "\n" + e.Message);
                        mergedModel = null;
                    }

                    if (mergedModel != null)
                    {
                        var commandSet = new EditCommandSet();

                        using (var vdsb = new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
                        {
                            // 編集コマンドで結果を戻す
                            commandSet.Add(DocumentManager.CreateDocumentReplaceCommand(dstModel, mergedModel).Execute());
                            foreach (var pathWithName in DocumentManager.MakeReferenceTextureFilename(mergedModel))
                            {
                                var texturePath = Path.GetFullPath(pathWithName.path);
                                var texture = DocumentManager.Textures.FirstOrDefault(x => string.Compare(texturePath, x.FilePath, true) == 0);
                                if (texture != null && !mergedModel.ReferenceTexturePaths.ContainsKey(texture.Name))
                                {
                                    mergedModel.ReferenceTexturePaths[texture.Name] = texturePath;
                                }
                            }
                            commandSet.Add(PropertyEdit.MaterialSamplerPage.CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(mergedModel), ConfigCommon.GuiObjectID.Model).Execute());
                            commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(mergedModel.AllAnimations).Execute());
                        }
                        var oldModel = dstModel;
                        var newModel = mergedModel;
                        EventHandler postEdit = (s, e) =>
                            {
                                Viewer.Close.Send(oldModel);
                                Viewer.LoadOrReloadModel.Send(newModel);
                                newModel.SendBindAnimations();
                                var tmp = oldModel;
                                oldModel = newModel;
                                newModel = tmp;
                            };
                        using (new Viewer.ViewerDrawSuppressBlock())
                        {
                            postEdit(null, null);
                        }
                        commandSet.Reverse();
                        commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                        var commandSet2 = new EditCommandSet();
                        commandSet2.OnPostEdit += postEdit;
                        commandSet2.Add(commandSet);
                        TheApp.CommandManager.Add(commandSet2);
                    }
                }
            }
        }

        public Model MergeFile(Model dstModel, string srcModelPath, MergerUtility.MergeMode mergeMode)
        {
            using (var vdsb = new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
            {
                // マージ元モデルを読み込む
                var srcModel = MergerUtility.LoadDocument<Model>(srcModelPath);

                var notShaderMaterials = srcModel.Materials.Where(x => !ShaderAssignUtility.IsMaterialShader(x)).ToArray();
                if (notShaderMaterials.Any())
                {
                    using (var dialog = new OkCancelTextBoxDialog())
                    {
                        dialog.Text = res.Strings.MergeModel_NotMaterialShaderText;
                        dialog.lblDescription.Text = res.Strings.MergeModel_NotMaterialShaderDescription;
                        foreach (var material in notShaderMaterials)
                        {
                            dialog.AddLine(string.Format(res.Strings.MergeModel_NotMaterialShaderItem,
                                srcModel.Name, material.Name,
                                material.MaterialShaderAssign.ShaderName,
                                material.MaterialShaderAssign.ShaderDefinitionFileName));
                        }

                        var result = dialog.ShowDialog();
                        if (result == DialogResult.Cancel)
                        {
                            return null;
                        }
                    }
                }

                // マージダイアログを開いて変換情報を作る
                IfModelMerger.IfModelMergerInfo info = null;

                if(mergeMode != MergerUtility.MergeMode.Auto)
                {
                    using(var dialog = new ModelMergeDialog(dstModel, srcModel))
                    {
                        if ((mergeMode == MergerUtility.MergeMode.Manual) || (dialog.IsSame == false))
                        {
                            if (dialog.ShowDialog() != DialogResult.OK)
                            {
                                return null;
                            }
                        }

                        info = new IfModelMerger.IfModelMergerInfo()
                        {
                            // TODO: render_set は廃止されました。
                            //IsMergeRenderSet		= dialog.IsMergeRenderSet,
                            IsMergeShape			= dialog.IsMergeShape,
                            IsMergeVertex			= dialog.IsMergeVertex,
                            IsMergeScaleEnable		= dialog.IsMergeScaleEnable,
                            BoneNamePairTable		= dialog.BoneNamePairTable,
                            MaterialNamePairTable	= dialog.MaterialNamePairTable
                        };
                    }
                }

                // 対象のクローンを作る
                var clonedDstModel = MergerUtility.CloneDocument(dstModel);

                // マージする
                {
                    var dstStreams = clonedDstModel.BinaryStreams.ToList();
                    var srcStreams = srcModel.BinaryStreams.ToList();

                    var context = new IfModelMergeContext(
                        clonedDstModel.Data, dstStreams,
                        srcModel.Data,       srcStreams
                    );

                    if (context.Setup(true))
                    {
                        MessageBox.Show(Strings.Merge_StructChanged);
                    }

                    if(info != null)
                    {
                        IfModelMerger.Merge(context, info);
                    }
                    else
                    {
                        IfModelMerger.Merge(context);
                    }

                    // 最適化ログの追加
                    MergerUtility.AddOptimizeLog(clonedDstModel, clonedDstModel.Data, srcModelPath);
                }

                clonedDstModel.Initialize();

                // TODO:応急処置：浅いコピーになっている部分があるので、一度作りなおす
                clonedDstModel = MergerUtility.CloneDocument(clonedDstModel);

                // Animations, AnimationSets もコピーする
                clonedDstModel.DefaultAnimationSet.Animations = ObjectUtility.Clone(dstModel.DefaultAnimationSet.Animations);

#if false
                clonedDstModel.AnimationSets = ObjectUtility.Clone(dstModel.AnimationSets);
#else
                // シリアライザブルでないので、最低限の情報のみをクローンする
                clonedDstModel.AnimationSets = new List<AnimationSet>();
                foreach (var animSet in dstModel.AnimationSets)
                {
                    clonedDstModel.AnimationSets.Add(
                        new AnimationSet(false)
                        {
                            Name = ObjectUtility.Clone(animSet.Name),
                            Animations = ObjectUtility.Clone(animSet.Animations)
                        }
                    );
                }

                clonedDstModel.PreviewInfo = ObjectUtility.Clone(dstModel.PreviewInfo);

                // いったん設定されるが、ドキュメントのパス決定後にも調整される
                clonedDstModel.ReferenceTexturePaths = ObjectUtility.Clone(dstModel.ReferenceTexturePaths);

#endif

                // マテリアルのシェーダー割り当ての修正
                var materials = (from material in clonedDstModel.Materials
                                 where !ShaderAssignUtility.IsConsistentWithDefinition(material)
                                 select material).ToArray();
                if (materials.Any())
                {
                    ShaderAssignUtility.ExecuteFixParameters(materials, reload: false);
                }

                foreach (var mtl in clonedDstModel.Materials)
                {
                    var dstMtl = dstModel.Materials.FirstOrDefault(x => x.Name == mtl.Name);
                    if (dstMtl != null)
                    {
                        mtl.OptimizeShader = dstMtl.OptimizeShader;
                        mtl.OptimizeShaderOnReload = dstMtl.OptimizeShaderOnReload;
                    }
                }

                return clonedDstModel;
            }
        }
    }
}
