﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Windows.Forms;
using App.Controls;
using nw.g3d.iflib;
using App.Utility;

namespace App.Data
{
    public class TexturePatternAnimationMerger
    {
        public TexturePatternAnimation MergeFile(TexturePatternAnimation dstAnim, string srcAnimPath, MergerUtility.MergeMode mergeMode)
        {
            mergeMode = MergerUtility.MergeMode.Auto;

            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
            {
                // マージ元アニメーションを読み込む
                var srcAnim = MergerUtility.LoadDocument<TexturePatternAnimation>(srcAnimPath);

                // マージダイアログを開いて変換情報を作る
                IfTexPatternAnimMerger.IfTexPatternAnimMergerInfo info = null;

                if(mergeMode != MergerUtility.MergeMode.Auto)
                {
                    using(var dialog = new TexturePatternAnimationMergeDialog(dstAnim, srcAnim))
                    {
                        if ((mergeMode == MergerUtility.MergeMode.Manual) || (dialog.IsSame == false))
                        {
                            if (dialog.ShowDialog() != DialogResult.OK)
                            {
                                return null;
                            }
                        }

                        info = new IfTexPatternAnimMerger.IfTexPatternAnimMergerInfo()
                        {
                            MaterialNamePairTable	= dialog.MaterialNamePairTable,
                            SamplerNamePairTable	= dialog.SamplerNamePairTable,
                        };
                    }
                }

                // 対象のクローンを作る
                var clonedDstAnim = MergerUtility.CloneDocument(dstAnim);

                // マージする
                {
                    Debug.Assert(clonedDstAnim is TexturePatternAnimation);
                    Debug.Assert(srcAnim is TexturePatternAnimation);

                    var dstStreams = clonedDstAnim.BinaryStreams;
                    var srcStreams = srcAnim.BinaryStreams;

                    try
                    {
                        if (info != null)
                        {
                            IfTexPatternAnimMerger.Merge(
                                (clonedDstAnim as TexturePatternAnimation).Data, dstStreams,
                                (srcAnim       as TexturePatternAnimation).Data, srcStreams,
                                info
                            );
                        }
                        else
                        {
                            IfTexPatternAnimMerger.Merge(
                                (clonedDstAnim as TexturePatternAnimation).Data, dstStreams,
                                (srcAnim       as TexturePatternAnimation).Data, srcStreams
                            );
                        }
                    }
                    catch(Exception exception)
                    {
                        UIMessageBox.Error(res.Strings.Error_Merge + "\n" + exception.Message, srcAnim.FileName);
                        return null;
                    }

                    // 最適化ログの追加
                    MergerUtility.AddOptimizeLog(clonedDstAnim, clonedDstAnim.Data, srcAnimPath);
                }

                clonedDstAnim.ReferenceTexturePaths = ObjectUtility.Clone(dstAnim.ReferenceTexturePaths);
                clonedDstAnim.Initialize();

                return clonedDstAnim;
            }
        }
    }
}
