﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using App.Command;
using App.Controls;
using App.Utility;
using App.res;
using ConfigCommon;
using System.IO;

namespace App.Data
{
    public class AnimationMerger
    {
        /// <summary>
        /// アニメーションファイルのマージ。
        /// </summary>
        public void MergeFile(AnimationDocument targetAnim, MergerUtility.MergeMode mergeMode)
        {
            string[] fileNames;
            if (DialogUtility.ExecuteOpenFileDialog(out fileNames, targetAnim.ObjectID, false))
            {
                using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                {
                    AnimationDocument mergedAnimation = null;
                    try
                    {
                        using (new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
                        {
                            mergedAnimation = MergeFile(targetAnim, fileNames[0], mergeMode);
                        }
                    }
                    catch (Exception e)
                    {
                        string message = Strings.Merge_Error;
                        if (!string.IsNullOrEmpty(e.Message))
                        {
                            message += e.Message;
                        }
                        UIMessageBox.Error(message);
                        mergedAnimation = null;
                    }

                    if (mergedAnimation != null)
                    {
                        var commandSet = new EditCommandSet();
                        using (var vdsb = new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
                        {
                            // 編集コマンドで結果を戻す
                            commandSet.Add(DocumentManager.CreateDocumentReplaceCommand(targetAnim, mergedAnimation).Execute());

                            // 参照設定を更新する
                            var mergedTexturePatternAnimation = mergedAnimation as TexturePatternAnimation;
                            if (mergedTexturePatternAnimation != null)
                            {

                                foreach (var pathWithName in DocumentManager.MakeReferenceTextureFilename(mergedTexturePatternAnimation))
                                {
                                    var texturePath = Path.GetFullPath(pathWithName.path);
                                    var texture = DocumentManager.Textures.FirstOrDefault(x => string.Compare(texturePath, x.FilePath, true) == 0);
                                    if (texture != null && !mergedTexturePatternAnimation.ReferenceTexturePaths.ContainsKey(texture.Name))
                                    {
                                        mergedTexturePatternAnimation.ReferenceTexturePaths[texture.Name] = texturePath;
                                    }
                                }

                                commandSet.Add(PropertyEdit.TexturePatternAnimationPatternPage.CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(mergedTexturePatternAnimation)));
                            }

                            var bindCommand = mergedAnimation.CreateUpdateBindCommand();
                            if (bindCommand != null)
                            {
                                commandSet.Add(bindCommand.Execute());
                            }
                        }
                        commandSet.Reverse();
                        var oldAniamtion = targetAnim;
                        var newAnimation = mergedAnimation;
                        EventHandler postEdit = (s, e) =>
                            {
                                Viewer.Close.Send(oldAniamtion);
                                Viewer.LoadOrReloadAnimation.Send(newAnimation);
                                foreach (var model in DocumentManager.Models)
                                {
                                    model.SendBindAnimations();
                                }
                                var tmp = oldAniamtion;
                                oldAniamtion = newAnimation;
                                newAnimation = tmp;
                            };
                        using (new Viewer.ViewerDrawSuppressBlock())
                        {
                            postEdit(null, null);
                        }
                        commandSet.Reverse();
                        commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                        var commandSet2 = new EditCommandSet();
                        commandSet2.OnPostEdit += postEdit;
                        commandSet2.Add(commandSet);
                        TheApp.CommandManager.Add(commandSet2);
                    }
                }
            }
        }

        public AnimationDocument MergeFile(AnimationDocument dstAnim, string srcAnimPath, MergerUtility.MergeMode mergeMode)
        {
            AnimationDocument mergedDoc = null;
            switch(dstAnim.ObjectID)
            {
                case GuiObjectID.SkeletalAnimation:				mergedDoc = (new SkeletalAnimationMerger()			).MergeFile(dstAnim as SkeletalAnimation,			srcAnimPath, mergeMode); break;
//				case GuiObjectID.ShapeAnimation:				mergedDoc = (new ShapeAnimationMerger()				).MergeFile(dstAnim as ShapeAnimation,				srcAnimPath, mergeMode); break;		// シェイプはマージしない
                case GuiObjectID.BoneVisibilityAnimation:		mergedDoc = (new BoneVisibilityAnimationMerger()		).MergeFile(dstAnim as BoneVisibilityAnimation,		srcAnimPath, mergeMode); break;
                case GuiObjectID.TexturePatternAnimation:		mergedDoc = (new TexturePatternAnimationMerger()		).MergeFile(dstAnim as TexturePatternAnimation,		srcAnimPath, mergeMode); break;
                case GuiObjectID.MaterialVisibilityAnimation:	mergedDoc = (new MaterialVisibilityAnimationMerger()	).MergeFile(dstAnim as MaterialVisibilityAnimation,	srcAnimPath, mergeMode); break;
                case GuiObjectID.ShaderParameterAnimation:		mergedDoc = (new ShaderParameterAnimationMerger()	).MergeFile(dstAnim as ShaderParameterAnimation,	srcAnimPath, mergeMode); break;
                case GuiObjectID.ColorAnimation:				mergedDoc = (new ShaderParameterAnimationMerger()	).MergeFile(dstAnim as ShaderParameterAnimation,	srcAnimPath, mergeMode); break;
                case GuiObjectID.TextureSrtAnimation:			mergedDoc = (new ShaderParameterAnimationMerger()	).MergeFile(dstAnim as ShaderParameterAnimation,	srcAnimPath, mergeMode); break;
                case GuiObjectID.SceneAnimation:				mergedDoc = (new SceneAnimationMerger()				).MergeFile(dstAnim as SceneAnimation,				srcAnimPath, mergeMode); break;
                default: throw new NotImplementedException();
            }

            if (mergedDoc != null && dstAnim.PauseFrames.Any())
            {
                mergedDoc.Pause.PauseFrames = dstAnim.Pause.PauseFrames;
            }
            return mergedDoc;
        }
    }
}
