﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Runtime.Remoting.Messaging;
using System.Text;
using System.Windows.Forms;
using System.Xml.Serialization;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System.Text.RegularExpressions;
using App.PropertyEdit;
using nw.g3d.iflib.nw3de;
using Viewer;

namespace App.Data
{
    public static class DocumentManager
    {
        private static readonly ModelContainer modelContainer = new ModelContainer();
        private static readonly SeparateMaterialContainer separateMaterialContainer = new SeparateMaterialContainer();
        private static readonly TextureContainer textureContainer = new TextureContainer();
        private static readonly AnimationContainer animationContainer = new AnimationContainer();
        private static readonly ShaderDefinitionContainer shaderDefinitionContainer = new ShaderDefinitionContainer();

        // スキーマのベースパス
        public static string XsdBasePath
        {
            get
            {
                // 新 3dtools名対応
                var path = Path.Combine(Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"), @"..\3dTools\3dIntermediateFileXsd\");

                if (!Directory.Exists(path))
                {
                    path = Path.Combine(Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"), @"..\G3dTool\g3dxsd\");
                }

                // 新構成対応
                if (!Directory.Exists(path))
                {
                    path = Path.Combine(Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"), @"..\3dTools\g3dxsd\");
                }

                return Directory.Exists(path) ? path : null;
            }
        }

        // スタートアップフォルダ
        public static string StartupFolderPath
        {
            get
            {
                if (startupFolderPath_ == null)
                {
                    startupFolderPath_ = System.Environment.ExpandEnvironmentVariables(@"%NW4F_3DEDITOR_ROOT%\Startup");
                }

                return startupFolderPath_;
            }
        }
        private static string startupFolderPath_ = null;

        // スタートアップフォルダの作成
        public static void CreateStartupFolderIfNotExists()
        {
            try
            {
                if (!Directory.Exists(StartupFolderPath))
                {
                    Directory.CreateDirectory(StartupFolderPath);
                }
            }
            catch
            {
                // 何もしない
            }
        }

        #region ドキュメント
        public static ProjectDocument ProjectDocument = new ProjectDocument() { DefaultProject = true };
        public static IEnumerable<Document> Documents
        {
            get { return Enumerable.Repeat(ProjectDocument, 1).Concat(DocumentsWithoutProject); }
        }

        public static IEnumerable<Document> DocumentsWithoutProject
        {
            get
            {
                return Models
                    .Concat<Document>(Textures)
                    .Concat<Document>(Animations)
                    .Concat<Document>(ShaderDefinitions)
                    .Concat<Document>(SeparateMaterials);
            }
        }

        public static IEnumerable<Model> BindSortedModels
        {
            get
            {
                var modelDepth = new List<Tuple<Model,int>>();
                foreach (var model in modelContainer.Documents)
                {
                    if (!model.PreviewInfo.IsBind)
                    {
                        modelDepth.Add(new Tuple<Model, int>(model, 0));
                    }
                    else
                    {
                        var depth = 0;
                        var parent = modelContainer.Documents.FirstOrDefault(x => x.Name == model.PreviewInfo.BindModelName);
                        while (parent != null)
                        {
                            depth++;
                            parent = modelContainer.Documents.FirstOrDefault(x => x.Name == parent.PreviewInfo.BindModelName);
                        }
                        modelDepth.Add(new Tuple<Model, int>(model, depth));
                    }
                }
                modelDepth.Sort((tuple1, tuple2) => tuple1.Item2.CompareTo(tuple2.Item2));
                return modelDepth.Select(x => x.Item1);
            }

        }


        public static int ModelCount { get { return modelContainer.Count; } }
        public static IEnumerable<Model> Models { get { return modelContainer.Documents; } }

        public static int SeparateMaterialCount { get { return separateMaterialContainer.Count; } }
        public static IEnumerable<SeparateMaterial> SeparateMaterials { get { return separateMaterialContainer.Documents; } }

        public static int TextureCount { get { return textureContainer.Count; } }
        public static IEnumerable<Texture> Textures { get { return textureContainer.Documents; } }

        public static int AnimationCount { get { return animationContainer.Count; } }
        public static IEnumerable<AnimationDocument> Animations { get { return animationContainer.Documents; } }

        public static IEnumerable<AnimationDocument> SceneAnimations { get { return Animations.Where(x => x.ObjectID == GuiObjectID.SceneAnimation); } }

        public static int ShaderDefinitionCount { get { return shaderDefinitionContainer.Count; } }
        public static IEnumerable<ShaderDefinition> ShaderDefinitions { get { return shaderDefinitionContainer.Documents; } }

        public static int MaterialCount { get { return Models.Sum(x => x.Materials.Count) + SeparateMaterials.Sum(x => x.Materials.Count); } }
        public static IEnumerable<Material> Materials { get { return Models.SelectMany(x => x.Materials).Concat(SeparateMaterials.SelectMany(x => x.Materials)); } }

        public static int BoneCount { get { return Models.Sum(x => x.Bones.Count); } }
        public static IEnumerable<Bone> Bones { get { return Models.SelectMany(x => x.Bones); } }

        public static int ShapeCount { get { return Models.Sum(x => x.Shapes.Count); } }
        public static IEnumerable<Shape> Shapes { get { return Models.SelectMany(x => x.Shapes); } }

        public static IEnumerable<AnimationSet> HasModelAnimations { get { return Models.SelectMany(x => x.AnimationSetsWithDefault); } }

        public static IEnumerable<Document> EditTargets
        {
            get
            {
                return Models.Concat<Document>(Animations);
            }
        }

        public static IEnumerable<Document> EditModelTargets
        {
            get
            {
                return Models;
            }
        }

        public static IEnumerable<GuiObject> Objects(GuiObjectID id)
        {
            switch (id)
            {
                case GuiObjectID.Model:
                    return Models;
                case GuiObjectID.Texture:
                    return Textures;
                case GuiObjectID.SkeletalAnimation:
                case GuiObjectID.ShapeAnimation:
                case GuiObjectID.MaterialAnimation:
                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                case GuiObjectID.MaterialVisibilityAnimation:
                case GuiObjectID.BoneVisibilityAnimation:
                case GuiObjectID.TexturePatternAnimation:
                case GuiObjectID.SceneAnimation:
                    return Animations.Where(x => x.ObjectID == id);
                case GuiObjectID.ShaderDefinition:
                    return ShaderDefinitions;
                case GuiObjectID.Material:
                    return Materials;
                case GuiObjectID.Bone:
                    return Bones;
                case GuiObjectID.SeparateMaterial:
                    return SeparateMaterials;
            }
            return Enumerable.Empty<GuiObject>();
        }

        public static IEnumerable<AnimationDocument> GetAnimations(IEnumerable<AnimationSetItem> fileNames)
        {
            // fileNames の順番で返す
            foreach(var fileName in fileNames)
            {
                var anim = Animations.FirstOrDefault(x =>
                    x.FileName == fileName.Name && x.FileLocation.Equals(fileName.Directory, StringComparison.OrdinalIgnoreCase));
                if (anim != null)
                {
                    yield return anim;
                }
            }
        }

        public static AnimationDocument GetAnimation(string fileName, string directory)
        {
            return Animations.FirstOrDefault(x => x.FileName == fileName && x.FileLocation.Equals(directory, StringComparison.OrdinalIgnoreCase));
        }

        #endregion

        #region モデル
        private static void RemoveDocument(Document document)
        {
            if (document is Model)
            {
                RemoveModel((Model)document);
            }
            else if (document is Texture)
            {
                RemoveTexture((Texture)document);
            }
            else if (document is AnimationDocument)
            {
                RemoveAnimation((AnimationDocument)document);
            }
            else if (document is ShaderDefinition)
            {
                RemoveShaderDefinition((ShaderDefinition)document);
            }
            else if (document is SeparateMaterial)
            {
                RemoveSeparateMaterial((SeparateMaterial)document);
            }
        }

        /// <summary>
        /// 参照されているドキュメントも含めて閉じる
        /// </summary>
        public static bool RemoveDocumentWithChildren(Document document, AnimationSet owner, bool removeChildren, bool isPassDialog)
        {
            return RemoveDocuments(Enumerable.Repeat(document, 1), owner, removeChildren, isPassDialog);
        }

        public static List<Document> GetAllSaveTargetDocuments()
        {
            var saveDocuments = new List<Document>();
            {
                foreach (var document in DocumentManager.Documents)
                {
                    // 保存確認対象でないプロジェクトを弾く
                    if (document is ProjectDocument)
                    {
                        if ((document as ProjectDocument).DefaultProject == false)
                        {
                            saveDocuments.Add(document);
                        }
                    }
                    else
                        if (document is ShaderDefinition)
                        {
                            // 保存対象でない
                            ;
                        }
                        else
                            // 編集されていない
                            if ((document.IsModifiedObject == false) && (document.ContentsModified == false))
                            {
                                // 保存対象でない
                                ;
                            }
                            else if (document is Texture && (document as Texture).IsTemporary)
                            {
                                // 保存対象でない
                                ;
                            }
                            else
                            {
                                // 保存対象を追加する
                                saveDocuments.Add(document);
                            }
                }
            }
            return saveDocuments;
        }

        /// <summary>
        /// 参照されているドキュメントを閉じる
        /// owner はツリービューのアニメーションの親ノードを想定
        /// </summary>
        public static bool RemoveDocuments(IEnumerable<Document> documents, AnimationSet owner, bool removeChildren, bool isPassDialog)
        {
            Debug.Assert(!documents.Any(x => x.ObjectID == GuiObjectID.Project));
            List<Document> closingDocuments = new List<Document>();

            var orderedDocuments = documents.OrderBy(x => ObjectIDUtility.CloseOrder[x.ObjectID]).ToArray();

            using(
                var option =
                    new UIMessageBox.Option(){
                        IsVisibleSameOpForAll = orderedDocuments.Count() >= 2,
                        Result_IsSameOpForAll = false
                    }
            )
            {
                DialogResult? resultForAll = null;

                // アニメーションのバインドの確認
                foreach (var document in orderedDocuments)
                {
                    DialogResult? result = null;

                    if (document is AnimationDocument)
                    {
                        if (document is SceneAnimation)
                        {
                            if (AllSceneAnimationSets.Any(x => x != owner && x.Animations.Contains(new AnimationSetItem(document.FileName, document.FileLocation))))
                            {
                                if (resultForAll.HasValue)
                                {
                                    result = resultForAll.Value;
                                }
                                else
                                {
                                    result = orderedDocuments.Length == 1 ?
                                        (App.AppContext.NotificationHandler.MessageBoxYesNo(MessageContext.IO_CloseSceneAnimationBoundToAnObject, res.Strings.IO_CloseSceneAnimationBoundToAnObject, document.FileName) ? DialogResult.Yes : DialogResult.No) :
                                        App.AppContext.NotificationHandler.MessageBoxYesNoCancel(MessageContext.IO_CloseSceneAnimationBoundToAnObject, res.Strings.IO_CloseSceneAnimationBoundToAnObject, document.FileName);

                                    if (option.Result_IsSameOpForAll)
                                    {
                                        Debug.Assert(result.HasValue);
                                        resultForAll = result;
                                    }
                                }

                                if (result == DialogResult.Cancel)
                                {
                                    return false;
                                }
                                else if (result == DialogResult.No)
                                {
                                    continue;
                                }
                            }
                        }
                        else
                        {
                            // 親が閉じられない場合はバインドの確認を行う
                            bool checkBinding = Models.Except(closingDocuments.OfType<Model>()).Any(x => x.AllAnimations.Contains(document));

                            if (checkBinding && (isPassDialog == false))
                            {
                                if (resultForAll.HasValue)
                                {
                                    result = resultForAll.Value;
                                }
                                else
                                {
                                    if (HasModelAnimations.First(x =>
                                        x.Animations.Contains(new AnimationSetItem(document.FileName, document.FileLocation))) !=
                                        HasModelAnimations.First(y => y.Animations.Contains(new AnimationSetItem(document.FileName, document.FileLocation))))
                                    {
                                        result =
                                            orderedDocuments.Length == 1 ?
                                                (App.AppContext.NotificationHandler.MessageBoxYesNo(MessageContext.IO_CloseAnimationBoundToObjects, Strings.IO_CloseAnimationBoundToObjects, document.FileName) ? DialogResult.Yes : DialogResult.No) :
                                                App.AppContext.NotificationHandler.MessageBoxYesNoCancel(MessageContext.IO_CloseAnimationBoundToObjects, Strings.IO_CloseAnimationBoundToObjects, document.FileName);
                                    }
                                    else
                                    {
                                        result =
                                            orderedDocuments.Length == 1 ?
                                                (App.AppContext.NotificationHandler.MessageBoxYesNo(MessageContext.IO_CloseAnimationBoundToAnObject, Strings.IO_CloseAnimationBoundToAnObject, document.FileName) ? DialogResult.Yes : DialogResult.No) :
                                                App.AppContext.NotificationHandler.MessageBoxYesNoCancel(MessageContext.IO_CloseAnimationBoundToAnObject, Strings.IO_CloseAnimationBoundToAnObject, document.FileName);
                                    }

                                    if (option.Result_IsSameOpForAll)
                                    {
                                        Debug.Assert(result.HasValue);
                                        resultForAll = result;
                                    }
                                }

                                if (result == DialogResult.Cancel)
                                {
                                    return false;
                                }
                                else if (result == DialogResult.No)
                                {
                                    continue;
                                }
                            }
                        }
                    }
                    else
                    {
                        if (DocumentsWithoutProject.Except(closingDocuments).Any(x => x.ReferenceDocuments.Contains(document)))
                        {
                            if (resultForAll.HasValue)
                            {
                                result = resultForAll.Value;
                            }
                            else
                            {
                                result =
                                    orderedDocuments.Length == 1 ?
                                        (App.AppContext.NotificationHandler.MessageBoxYesNo(MessageContext.IO_FileIsReferenced, Strings.IO_FileIsReferenced, document.FileName) ? DialogResult.Yes : DialogResult.No) :
                                        App.AppContext.NotificationHandler.MessageBoxYesNoCancel(MessageContext.IO_FileIsReferenced, res.Strings.IO_FileIsReferenced, document.FileName);

                                if (option.Result_IsSameOpForAll)
                                {
                                    Debug.Assert(result.HasValue);
                                    resultForAll = result;
                                }
                            }

                            if (result == DialogResult.Cancel)
                            {
                                return false;
                            }
                            else if (result == DialogResult.No)
                            {
                                continue;
                            }
                        }
                    }

                    closingDocuments.Add(document);
                }
            }

            // 子供を削除対象に追加
            if (removeChildren)
            {
                for (int i = 0; i < closingDocuments.Count; i++)
                {
                    closingDocuments.AddRange(closingDocuments[i].ReferenceDocuments.
                        Where(x => !closingDocuments.Contains(x) &&
                            Documents.Except(closingDocuments).All(y => !y.ReferenceDocuments.Contains(x)) &&
                            x.ObjectID != GuiObjectID.ShaderDefinition));
                }
            }

            // 保存確認
            var commandSet = new EditCommandSet();
            var ordered = closingDocuments.OrderBy(x => ObjectIDUtility.CloseOrder[x.ObjectID]);
            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {
                using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                {

                    if (CheckAndSaveModified(ordered, true, commandSet))
                    {
                        // アニメーションを削除します
                        var animations = ordered.OfType<AnimationDocument>().ToArray();
                        if (animations.Any())
                        {
                            foreach (var model in Models)
                            {
                                foreach (var animationSet in model.AnimationSetsWithDefault)
                                {
                                    var remain = animationSet.Animations.Except(animations.Select(x => new AnimationSetItem(x.FileName, x.FileLocation))).ToArray();
                                    if (remain.Length != animationSet.Animations.Count)
                                    {
                                        commandSet.Add(CreateAnimationsEditCommand(animationSet, remain, true).Execute());
                                    }
                                }
                            }

                            // シーンから削除
                            var sceneAnimationSets = DocumentManager.SceneAnimationSets.Concat(Enumerable.Repeat(DocumentManager.DefaultSceneAnimationSet, 1));
                            foreach (var animationSet in sceneAnimationSets)
                            {
                                var remain = animationSet.Animations.Except(animations.Select(x => new AnimationSetItem(x.FileName, x.FileLocation))).ToArray();
                                if (remain.Length != animationSet.Animations.Count)
                                {
                                    commandSet.Add(CreateAnimationsEditCommand(animationSet, remain, true).Execute());
                                }
                            }
                        }

                        if (ordered.Any())
                        {
                            if (ApplicationConfig.UserSetting.EditHistory.ClearAfterFileClosed)
                            {
                                TheApp.CommandManager.ClearNext();
                            }
                            commandSet.Add(CreateAddOrRemoveDocumentCommand(ordered, false).Execute());
                            var paths = ordered.Select(x => Tuple.Create(x, x.FilePath)).Where(x => !string.IsNullOrEmpty(x.Item2)).ToArray();
                            commandSet.canUndo += () => PrePostIO.ExecutePreOpenUndoRedo(paths, true);
                        }
                    }

                    if (commandSet.CommandCount > 0)
                    {
                        var editCommandSet = CreateRemoveTextureReferencePaths();
                        if (editCommandSet.CommandCount > 0)
                        {
                            commandSet.Add(editCommandSet.Execute());
                        }
                    }
                }

                if (commandSet.CommandCount > 0)
                {
                    // バインドの更新
                    // TODO: 絞り込む
                    var bindCommand = CreateAnimationUpdateBindCommand(Animations);
                    if (bindCommand.CommandCount > 0)
                    {
                        commandSet.Add(bindCommand.Execute());
                    }
                }
            }

            if (commandSet.CommandCount > 0)
            {
                commandSet.Reverse();
                TheApp.CommandManager.Add(commandSet);
            }

            // 閉じられるファイルすべてをまとめてクローズ後コマンドに渡す
            DocumentManager.ExecutePostCloseCommand(closingDocuments);

            return true;
        }

        /// <summary>
        /// ドキュメントをまとめてクローズ後コマンドに渡します。
        /// </summary>
        public static void ExecutePostCloseCommand(List<Document> documents)
        {
            List<string> files = new List<string>();
            foreach (var document in documents)
            {
                if (!string.IsNullOrEmpty(document.FilePath))
                {
                    files.Add(document.FilePath);
                }
            }

            if (files.Any())
            {
                PrePostIO.ExecutePostClose(files.ToArray());
            }
        }

        /// <summary>
        /// 参照されているかどうかを調べます。
        /// </summary>
        public static bool IsReferenced(Document source, Document target)
        {
            if (source.ReferenceDocuments.Contains(target))
            {
                return true;
            }

            foreach (var document in source.ReferenceDocuments)
            {
                if (IsReferenced(document, target))
                {
                    return true;
                }
            }

            return false;
        }

        private static void AddModel(Model model)
        {
            Debug.Assert(model != null);
            AddModels(new[] { model });
        }

        private static void RemoveModel(Model model)
        {
            RemoveModels(new[] { model });
        }

        private static void AddModels(IEnumerable<Model> models)
        {
            Debug.Assert(models != null);

            if (models.Any())
            {
                modelContainer.Add(models);
            }

            // ビューアへ転送
            if (Viewer.Manager.Instance.IsConnected == true)
            {
                // Editへは、Modelの参照データ(テクスチャ)を一緒にバイナリ化しないといけないので、
                // ここで送ると、参照データが読み込まれていない可能性がある。
                // なので、HioUtilityに、送信データを保持させる。
                foreach (var model in models)
                {
                    Viewer.HioUtility.AddViewerDrawSuppressBlockModel(model);

                    //Viewer.LoadModel.Send(model);
                    Viewer.ViewerUtility.SendAnimationSet(model);
                }
            }
        }

        private static void RemoveModels(IEnumerable<Model> models)
        {
            Debug.Assert(models != null);

            if (models.Any())
            {
                modelContainer.Remove(models);
            }

            // ビューアへ転送
            foreach (var model in models)
            {
                Viewer.Close.Send(model);
            }
        }

        /// <summary>
        /// 送信済みのモデル（通信スレッド用）
        /// </summary>
        public static HashSet<Model> PreviewingModels = new HashSet<Model>();

        /// <summary>
        /// 送信済みのテクスチャ（通信スレッド用）
        /// </summary>
        public static Dictionary<Texture, Tuple<byte[], List<G3dStreamCachedComparer>>> PreviewingTextures = new Dictionary<Texture, Tuple<byte[], List<G3dStreamCachedComparer>>>();

        /// <summary>
        /// テクスチャにバインド済みのモデルとテクスチャパターンアニメーション（通信スレッド用）
        /// </summary>
        public static Dictionary<Texture, HashSet<GuiObject>> TextureBindings = new Dictionary<Texture,HashSet<GuiObject>>();

        /// <summary>
        /// モデルとテクスチャパターンアニメーションにバインド済みのテクスチャ（通信スレッド用）
        /// </summary>
        public static Dictionary<GuiObject, List<Texture>> BindingTextures = new Dictionary<GuiObject, List<Texture>>();
        #endregion

        #region テクスチャ
        private static void AddTexture(Texture texture)
        {
            Debug.Assert(texture != null);
            AddTextures(new[] { texture });
        }

        private static void RemoveTexture(Texture texture)
        {
            RemoveTextures(new[] { texture });
        }

        private static void AddTextures(IEnumerable<Texture> textures)
        {
            Debug.Assert(textures != null);

            if (textures.Any())
            {
                textureContainer.Add(textures);
            }

            // 参照切れを解決するテクスチャの追加や、
            // テクスチャを閉じる処理の Undo のテクスチャロードを行う。
            // テクスチャのロード処理はモデルやアニメーションのロードによって行われる。
            // そのため、ここでのテクスチャ読み込みは HioLoaded 済みのモデルやアニメーションに対して行えばよい。
            // HioLoaded が済んでいなければ、モデルやアニメーションのロード時にテクスチャもロードされるため、ここでロードが行われる必要はない。
            // HioLoaded が済んでいるときにもここでロードを行うようにすると、
            // ビューアー接続状態でプロジェクトファイルを開いたときに、モデルのバイナライズと転送がテクスチャ毎に行われてしまう。
            if (Viewer.Manager.Instance.IsConnected == true)
            {
                // 再転送するドキュメント
                // 参照切れテクスチャの解決も行う。
                var loadedModels = Models.Where(x => x.HioLoaded).ToArray();
                var loadedAnimations = Animations.Where(x => x.HioLoaded).ToArray();
                var reloadModels = textures.SelectMany(x => loadedModels.Where(y => y.ReferenceDocuments.Contains(x) || y.UnresolvedTextures().Contains(x.Name))).Distinct().ToArray();
                var reloadAnimations = textures.SelectMany(
                    x => loadedAnimations.Where(
                        y => y.ReferenceDocuments.Contains(x) ||
                        ((y is TexturePatternAnimation) && ((TexturePatternAnimation)y).UnresolvedTextures().Contains(x.Name)) ||
                        ((y is MaterialAnimation) && ((MaterialAnimation)y).UnresolvedTextures().Contains(x.Name)))).Distinct().ToArray();

                // Editへは、Modelの参照データ(テクスチャ)を一緒にバイナリ化しないといけないので、
                // ここで送ると、参照データが読み込まれていない可能性がある。
                // なので、HioUtilityに、送信データを保持させる。
                foreach (var model in reloadModels)
                {
                    Viewer.HioUtility.AddViewerDrawSuppressBlockModel(model);

                    //Viewer.LoadModel.Send(model);
                    Viewer.ViewerUtility.SendAnimationSet(model);
                }

                foreach (var animation in reloadAnimations)
                {
                    // Editへは、Animationの参照データ(テクスチャ)を一緒にバイナリ化しないといけないので、
                    // ここで送ると、参照データが読み込まれていない可能性がある。
                    // なので、HioUtilityに、送信データを保持させる。
                    //Viewer.LoadAnimation.Send(animation);
                    Viewer.HioUtility.AddViewerDrawSuppressBlockAnim(animation);
                }
            }
        }

        private static void RemoveTextures(IEnumerable<Texture> textures)
        {
            Debug.Assert(textures != null);

            if (textures.Any())
            {
                textureContainer.Remove(textures);
            }

            // ビューアへ転送
            foreach (var texture in textures)
            {
                Viewer.Close.Send(texture);
            }
        }

        /// <summary>
        /// ソートされたテクスチャリスト。
        /// </summary>
        public static IEnumerable<Texture> SortedTextures
        {
            get
            {
                return textureContainer.Documents.OrderBy(x => x.Name).OrderBy(x => x.FileName).ThenBy(x => { int index; IndexInSameNames.TryGetValue(x, out index); return index; });
            }
        }

        #endregion

        #region アニメーション
        private static void AddAnimation(AnimationDocument animation)
        {
            Debug.Assert(animation != null);
            AddAnimations(new[] { animation });
        }

        private static void RemoveAnimation(AnimationDocument animation)
        {
            RemoveAnimations(new[] { animation });
        }

        private static void AddAnimations(IEnumerable<AnimationDocument> animations)
        {
            Debug.Assert(animations != null);
            animationContainer.Add(animations);

            foreach (var anim in animations)
            {
                ShaderParameterAnimation spAnim = anim as ShaderParameterAnimation;
                if (spAnim != null)
                {
                    spAnim.ShowTextureSRTModeCurveDeleteMessage();
                }
            }

            // ビューアへ転送
            if (Viewer.Manager.Instance.IsConnected == true)
            {
                foreach (var animation in animations)
                {
                    // Editへは、Animationの参照データ(テクスチャ)を一緒にバイナリ化しないといけないので、
                    // ここで送ると、参照データが読み込まれていない可能性がある。
                    // なので、HioUtilityに、送信データを保持させる。
                    //Viewer.LoadAnimation.Send(animation);
                    Viewer.HioUtility.AddViewerDrawSuppressBlockAnim(animation);
                }
            }
        }

        private static void RemoveAnimations(IEnumerable<AnimationDocument> animations)
        {
            Debug.Assert(animations != null);
            animationContainer.Remove(animations);

            // ビューアへ転送
            foreach (var animation in animations)
            {
                Viewer.Close.Send(animation);
            }
        }
        #endregion

        #region シーンアニメーションセット
        public static List<AnimationSet> SceneAnimationSets = new List<AnimationSet>();

        public static List<Tuple<string, SceneAnimation>> PreviewingSceneAnimations = new List<Tuple<string, SceneAnimation>>();
        private static AnimationSet previewSceneAnimSet_ = null;

        public static AnimationSet DefaultSceneAnimationSet =
            new AnimationSet(true)
            {
                Name = res.Strings.FileTreeView_DefaultAnimationSet,
                IsDefaultAnimationSet = true,
            };

        public static IEnumerable<AnimationSet> SceneAnimationSetsWithDefault
        {
            get
            {
                return Enumerable.Repeat(DefaultSceneAnimationSet, 1).Concat(SceneAnimationSets);
            }
        }

        static DocumentManager()
        {
            DefaultSceneAnimationSet.UpdateSavedData();
            DefaultSceneAnimationSet.SetNotModified();
            previewSceneAnimSet_ = DefaultSceneAnimationSet;
            App.AppContext.DocumentAddedOrRemoved += AppContext_DocumentAddedOrRemoved;
        }

        static void AppContext_DocumentAddedOrRemoved(object sender, IEnumerable<Document> added, IEnumerable<Document> removed, Dictionary<GuiObject, GuiObject> swaped, IEnumerable<Document> reloaded)
        {
            // マテリアルの変更フラグを更新する
            var shaderDefinitions = new HashSet<string>();
            foreach (var item in added.Concat(removed).Concat(reloaded).OfType<ShaderDefinition>())
            {
                shaderDefinitions.Add(item.Name);
            }

            foreach (var item in swaped)
            {
                if (item.Key is ShaderDefinition)
                {
                    shaderDefinitions.Add(item.Key.Name);
                }

                if (item.Value is ShaderDefinition)
                {
                    shaderDefinitions.Add(item.Value.Name);
                }
            }

            foreach (var material in Materials)
            {
                if (shaderDefinitions.Contains(
                    material.MaterialShaderAssign.ShaderDefinitionFileName))
                {
                    material.SetMaybeModified();
                }
            }
        }

        public static IEnumerable<AnimationSet> AllSceneAnimationSets
        {
            get
            {
                return SceneAnimationSets.Concat(Enumerable.Repeat(DefaultSceneAnimationSet, 1));
            }
        }

        public static AnimationSet PreviewSceneAnimSet
        {
            get
            {
                return previewSceneAnimSet_;
            }
            set
            {
                previewSceneAnimSet_ = value;
                PreviewSceneAnimSetUpdated();
            }
        }

        public static void PreviewSceneAnimSetUpdated()
        {
            DocumentManager.ProjectDocument.SetMaybeModified();
            Viewer.ViewerUtility.SendPreviewSceneAnimSet();
            App.AppContext.NotifyPropertyChanged(null, new PreviewAnimSetUpdatedArg());
        }

        public static void SendBindSceneAnimations()
        {
            using (var vdsb = new Viewer.ViewerDrawSuppressBlock(null))
            {
                Viewer.HioUtility.AddViewerDrawSuppressBlockBindSceneAnim();
            }
        }

        #endregion
        #region ShaderDefinition
        private static void AddShaderDefinition(ShaderDefinition shaderDefinition)
        {
            shaderDefinitionContainer.Add(shaderDefinition);

            Viewer.LoadOptimizedShaderArchive.OnShaderDefinitionChanged(shaderDefinition, shaderDefinition.shadingModelModifiedCounts.Keys.ToArray());
        }

        private static void RemoveShaderDefinition(ShaderDefinition shaderDefinition)
        {
            shaderDefinitionContainer.Remove(shaderDefinition);

            // ビューアへ転送
            Viewer.Close.Send(shaderDefinition);

            Viewer.LoadOptimizedShaderArchive.OnShaderDefinitionChanged(shaderDefinition, shaderDefinition.shadingModelModifiedCounts.Keys.ToArray());
        }
        #endregion

        #region マテリアル専用ドキュメント
        private static void AddSeparateMaterial(SeparateMaterial separateMaterial)
        {
            separateMaterialContainer.Add(separateMaterial);
        }

        private static void RemoveSeparateMaterial(SeparateMaterial separateMaterial)
        {
            separateMaterialContainer.Remove(separateMaterial);
        }
        #endregion

        #region IO
        // startupLoadedFile スタートアップで読み込まれたファイルも閉じるか？
        public static void CloseAll(EditCommandSet commandSet, bool startupLoadedFile, bool executePreOpen)
        {
            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {
                using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                {
                    // スタートアップで読み込まれたファイルは対象外にする
                    var closeDocuments = startupLoadedFile ? DocumentsWithoutProject :
                                                             DocumentsWithoutProject.Where(x => x.OpenedFromStartUp == false);

                    if (executePreOpen)
                    {
                        var paths = closeDocuments.Select(x => Tuple.Create(x, x.FilePath)).Where(x => !string.IsNullOrEmpty(x.Item2)).ToArray();
                        if (paths.Any())
                        {
                            commandSet.canUndo += () =>
                            {
                                return PrePostIO.ExecutePreOpenUndoRedo(paths, true);
                            };
                        }
                    }

                    // 閉じられるファイルすべてをまとめてクローズ後コマンドに渡す
                    DocumentManager.ExecutePostCloseCommand(closeDocuments.ToList());

                    if (closeDocuments.Any())
                    {
                        if (ApplicationConfig.UserSetting.EditHistory.ClearAfterFileClosed)
                        {
                            TheApp.CommandManager.ClearNext();
                        }

                        commandSet.Add(DocumentManager.CreateAddOrRemoveDocumentCommand(closeDocuments, false).Execute());

                        commandSet.Add(CreateRemoveTextureReferencePaths().Execute());
                    }

                    if (startupLoadedFile || !ProjectDocument.OpenedFromStartUp)
                    {
                        commandSet.Add(CreateProjectChangeCommand(new ProjectDocument() { DefaultProject = true }, true).Execute());
                    }

                    var args = new DocumentContentsChangedArgs(ProjectDocument, null);
                    App.AppContext.NotifyPropertyChanged(null, args);

                    // 全て閉じるメッセージ
                    //Viewer.CloseAll.Send();
                }
            }
        }

        public static EditCommandSet CreateRemoveTextureReferencePaths()
        {
            var commandSet = new EditCommandSet();
            var texturePaths = new HashSet<Tuple<string, string>>();
            foreach (var texture in DocumentManager.Textures)
            {
                texturePaths.Add(new Tuple<string, string>(texture.Name, texture.FilePath.ToLower()));
            }

            foreach (var model in DocumentManager.Models)
            {
                var texNames = model.Materials.SelectMany(x => x.sampler_array.sampler).Select(x => x.tex_name).Distinct();
                var referenceTextures = texNames.Where(x => model.ReferenceTexturePaths.ContainsKey(x) &&
                    texturePaths.Contains(new Tuple<string, string>(x, model.ReferenceTexturePaths[x].ToLower())))
                    .ToDictionary(x => x, x => model.ReferenceTexturePaths[x]);

                if (referenceTextures.Count != model.ReferenceTexturePaths.Count)
                {
                    commandSet.Add(new GeneralGroupReferenceEditCommand<Dictionary<string, string>>(
                        new GuiObjectGroup(model),
                        GuiObjectID.Model,
                        Enumerable.Repeat(referenceTextures, 1),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            var model0 = (Model)target;
                            swap = model0.ReferenceTexturePaths;
                            model0.ReferenceTexturePaths = (Dictionary<string, string>)data;
                        }));

                }
            }

            foreach (var anim in DocumentManager.Animations.OfType<TexturePatternAnimation>())
            {
                var texNames = anim.TexPatterns.Select(x => x.tex_name).Distinct();
                var referenceTextures = texNames.Where(x => anim.ReferenceTexturePaths.ContainsKey(x) &&
                    texturePaths.Contains(new Tuple<string, string>(x, anim.ReferenceTexturePaths[x].ToLower())))
                    .ToDictionary(x => x, x => anim.ReferenceTexturePaths[x]);

                if (referenceTextures.Count != anim.ReferenceTexturePaths.Count)
                {
                    commandSet.Add(new GeneralGroupReferenceEditCommand<Dictionary<string, string>>(
                        new GuiObjectGroup(anim),
                        GuiObjectID.TexturePatternAnimation,
                        Enumerable.Repeat(referenceTextures, 1),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            var anim0 = (TexturePatternAnimation)target;
                            swap = anim0.ReferenceTexturePaths;
                            anim0.ReferenceTexturePaths = (Dictionary<string, string>)data;
                        }));
                }
            }

            return commandSet;
        }

        /// <summary>
        /// 変更されたファイルを保存します。プロジェクトが書き換わることがあります。
        /// </summary>
        public static bool CheckAndSaveModified(IEnumerable<Document> documents, bool includeProject = true, EditCommandSet commandSet = null)
        {
            // デフォルト状態のプロジェクトと、ShaderDefinitionを除く変更されたドキュメント
            var modified = documents.Where(
                x => x.ContentsModified &&
                    (x != DocumentManager.ProjectDocument || !DocumentManager.ProjectDocument.DefaultProject) &&
                    (x.ObjectID != GuiObjectID.ShaderDefinition) &&
                    (includeProject || x.ObjectID != GuiObjectID.Project)).ToArray();

            if (!modified.Any()) return true;

            IEnumerable<Document> closeDoc;
            if (!App.AppContext.NotificationHandler.DocumentsCloseDialog(modified, out closeDoc))
            {
                return false;
            }

            if (closeDoc != null)
            {
                (new DocumentSaver()).SaveDocuments(closeDoc, commandSet);
            }
            return true;
        }

        public static void LoadFromProject(string projectPath, bool startup, bool reloadFromMenu, bool executePreCommand, EditCommandSet commandSet)
        {
            var startupFiles = new HashSet<string>(FilterExsitingFiles(StartUpPaths).Select(x => Path.GetFullPath(x.path)));

            using (var reloadModelSuppressBlock = new Viewer.HioUtility.ReloadModelSuppressBlock())
            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {
                using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                {
                    // 既存ファイルの変更をチェックします。
                    if (!CheckAndSaveModified(Documents, includeProject: !reloadFromMenu))
                    {
                        return;
                    }

                    Project project = null;

                    bool stack = false;
                    if (commandSet == null)
                    {
                        commandSet = new EditCommandSet();
                        stack = true;
                    }

                    // プロジェクトをデシリアライズします。
                    try
                    {
                        using (var watch = new DebugStopWatch("Load " + projectPath))
                        using (FileStream fileStream = new FileStream(projectPath, FileMode.Open, FileAccess.Read))
                        {
                            project = (Project)(new XmlSerializer(typeof(Project))).Deserialize(fileStream);
                        }
                    }
                    catch (Exception exception)
                    {
                        App.AppContext.NotificationHandler.MessageBoxError(App.res.Strings.IO_Load_Failed + "\n{0}\n{1}", projectPath, exception.Message);
                        return;
                    }

                    var preOpenDocs = new List<Document>();
                    var excludePreOpenDocList = new List<Document>();
                    var excludePreOpenPaths = new HashSet<string>();
                    if (executePreCommand)
                    {
                        if (!PrePostIO.ExecutePreOpen(new[] { projectPath }))
                        {
                            // 失敗時は何もしない
                            return;
                        }

                        // ここで preOpenDocs に newProject を追加したいが、この時点では未作成なため作成後に追加する。
                    }
                    else
                    {
                        foreach (var doc in DocumentManager.Documents.Where(x => !string.IsNullOrEmpty(x.FilePath)))
                        {
                            excludePreOpenDocList.Add(doc);
                            excludePreOpenPaths.Add(Path.GetFullPath(doc.FilePath).ToLower());
                        }
                    }

                    // 既存ファイルを閉じます。
                    CloseAll(commandSet, true, executePreCommand);

                    // プロジェクト名を変更します。
                    var newProject = new ProjectDocument();
                    newProject.OpenedFromStartUp = startup;
                    newProject.Name = Path.GetFileNameWithoutExtension(projectPath);
                    newProject.FileDotExt = Path.GetExtension(projectPath);
                    newProject.SetFileLocation(Path.GetDirectoryName(projectPath), null);
                    newProject.FileInfo = project.FileInfo;
                    newProject.LoadToolData(project.tool_data);
                    newProject.MakeComment(project.comment);
                    commandSet.Add(CreateProjectChangeCommand(newProject, true).Execute());
                    newProject.UpdateSavedData();
                    newProject.SetNotModified();
                    newProject.SetContentsNotModified();
                    newProject.savedProject = project;
                    newProject.SetMaybeModified();

                    // 上部での PrePostIO.ExecutePreOpen() 実行時に追加できなかった newProject をここで追加する。
                    if (executePreCommand)
                    {
                        // 念のため明示的に先頭に挿入しておく。
                        preOpenDocs.Insert(0, newProject);
                    }

                    // バージョンの更新
                    if (project.Version != ProjectDocument.ProjectVersion)
                    {
                        App.AppContext.NotificationHandler.WriteMessageLog(MessageLog.LogType.Information,
                            string.Format(App.res.Strings.IO_ProjectUpdated, newProject.FilePath, ProjectDocument.ProjectVersion));
                    }

                    TheApp.MainFrame.FileAdded(newProject);

                    // ファイルを開きます。
                    var openedDocuments = new List<Document>();

                    var paths = new List<PathWithName>();
                    List<Error> errors0 = new List<Error>();
                    foreach (var item in project.Items)
                    {
                        PathWithName path;
                        // スタートアップフォルダ相対
                        if (item.Relative == Project.RelativeType.StartUpFolder)
                        {
                            var fullPath = Path.GetFullPath(Path.Combine(StartupFolderPath, item.Path));
                            item.Directory = Path.GetDirectoryName(fullPath);
                            item.FileName = Path.GetFileName(fullPath);
                            path = new PathWithName(Path.Combine(StartupFolderPath, item.Path), null);
                        }
                        // プロジェクト相対、SearchPath相対
                        else
                        {
                            string basePath = ProjectDocument.FileLocation;
                            if (item.Base != null)
                            {
                                var configItem = ConfigData.ApplicationConfig.GetSearchPaths().FirstOrDefault(x => x.Name == item.Base);
                                if (configItem == null)
                                {
                                    errors0.Add(new Error(item.Path, ErrorType.BasePathNotExists, string.Format(res.Strings.IO_BasePathNotFound, item.Base)));
                                    continue;
                                }
                                basePath = configItem.path;
                            }
                            else if (item.BasePath != null)
                            {
                                basePath = Environment.ExpandEnvironmentVariables(item.BasePath);
                            }
                            var fullPath = Path.GetFullPath(Path.Combine(basePath, item.Path));
                            item.Directory = Path.GetDirectoryName(fullPath);
                            item.FileName = Path.GetFileName(fullPath);
                            path = new PathWithName(Path.Combine(basePath, item.Path), item.Base, item.BasePath);
                        }

                        string correctPath;
                        if (PathUtility.ExistsFileName(path.path, out correctPath))
                        {
                            path.path = correctPath;
                            paths.Add(path);
                        }
                        else
                        {
                            if (correctPath != null)
                            {
                                var name0 = Path.GetFileNameWithoutExtension(correctPath);
                                var name1 = Path.GetFileNameWithoutExtension(path.path);
                                errors0.Add(new Error(path.path, ErrorType.FileNotExists, string.Format(
                                    res.Strings.IO_Load_FileNotExistsStrictly, name0, name1)));
                            }
                            else
                            {
                                errors0.Add(new Error(path.path, ErrorType.FileNotExists, res.Strings.IO_Load_FileNotExists));
                            }
                        }
                    }

                    // Directory の設定
                    foreach (var animationItem in project.Scene.AnimationItems.Concat(project.Scene.AnimationSetItems.SelectMany(x => x.AnimationItems))
                        .Concat(project.Items.SelectMany(x => x.AnimationItems.Concat(x.AnimationSetItems.SelectMany(y => y.AnimationItems)))))
                    {
                        if (animationItem.ItemIndex == -1)
                        {
                            var item = project.Items.FirstOrDefault(x => Path.GetFileName(x.Path) == animationItem.Name);
                            Debug.Assert(item != null);
                            animationItem.Directory = item.Directory;
                        }
                        else
                        {
                            animationItem.Directory = project.Items[animationItem.ItemIndex].Directory;
                        }
                    }

                    List<Error> errors1;
                    var openedDocuments1 = new List<Document>();
                    using (var cursor = new WaitCursor())
                    using (var watch = new DebugStopWatch(""))
                    {
                        ExecuteLoadFromFileCommand(
                            openedDocuments1,
                            paths,
                            out errors1,
                            startup,
                            commandSet,
                            true,
                            excludePreOpenPaths);

                        openedDocuments.AddRange(openedDocuments1);

                        watch.SetMessage(string.Format("Load {0} Files", openedDocuments1.Count));
                    }

                    // リターゲットの設定
                    foreach (var item in project.Items.Where(x => !string.IsNullOrEmpty(x.RetargetingHostModelName)))
                    {
                        var animation = Animations.FirstOrDefault(x => x.FileLocation == item.Directory && x.FileName == item.FileName);
                        if (animation != null)
                        {
                            animation.RetargetingHostModelName = item.RetargetingHostModelName;
                        }
                    }

                    // アニメーションのバインド
                    foreach (var item in project.Items.Where(x => x.AnimationItems.Any() || x.AnimationSetItems.Any() || x.PreviewItem != null || x.MaterialItems.Any()))
                    {
                        string name = Path.GetFileNameWithoutExtension(item.Path);
                        var model = Models.FirstOrDefault(x => x.Name == name);
                        if (model != null)
                        {
                            foreach (var animationItem in item.AnimationItems)
                            {
                                var animation = openedDocuments.
                                    FirstOrDefault(x => x.FileName == animationItem.Name && x.FileLocation == animationItem.Directory)
                                    as AnimationDocument;

                                if (animation != null)
                                {
                                    model.DefaultAnimationSet.Animations.Add(new AnimationSetItem(animation.FileName, animation.FileLocation));
                                    if (!animationItem.Preview)
                                    {
                                        animation.Pause.InvisibleBinds.Add(new KeyValuePair<object, string>(model.ModelId, model.DefaultAnimationSet.Name));
                                    }
                                }
                            }

                            if (item.PreviewDefaultAnimationSet)
                            {
                                model.PreviewAnimSet = model.DefaultAnimationSet;
                            }
                            else
                            {
                                model.PreviewAnimSet = null;
                            }

                            model.DefaultAnimationSet.UpdateSavedData();
                            model.DefaultAnimationSet.SetNotModified();
                            model.DefaultAnimationSet.SetMaybeModified();

                            foreach (var animationSetItem in item.AnimationSetItems)
                            {
                                var animationSet = new AnimationSet(false)
                                {
                                    Name = animationSetItem.Name,
                                };

                                foreach (var animationItem in animationSetItem.AnimationItems)
                                {
                                    var animation = openedDocuments.FirstOrDefault(x => x.FileName == animationItem.Name && x.FileLocation == animationItem.Directory)
                                        as AnimationDocument;

                                    if (animation != null)
                                    {
                                        animationSet.Animations.Add(new AnimationSetItem(animation.FileName, animation.FileLocation));
                                        if (!animationItem.Preview)
                                        {
                                            animation.Pause.InvisibleBinds.Add(new KeyValuePair<object, string>(model.ModelId, animationSet.Name));
                                        }
                                    }
                                }

                                model.AnimationSets.Add(animationSet);
                                if (animationSetItem.Preview == true)
                                {
                                    model.PreviewAnimSet = animationSet;
                                }

                                animationSet.UpdateSavedData();
                                animationSet.SetNotModified();
                                animationSet.SetMaybeModified();
                            }

                            foreach (var materialItem in item.MaterialItems)
                            {
                                var material = model.Materials.FirstOrDefault(x => x.Name == materialItem.Name);
                                if (material != null)
                                {
                                    material.OptimizeShader = materialItem.OptimizeShader;
                                    material.OptimizeShaderOnReload = materialItem.OptimizeShaderOnReload;
                                }
                            }

                            if (item.PreviewItem != null)
                            {
                                item.PreviewItem.Load(model.PreviewInfo);
                            }

                            if (model.DefaultAnimationSet.Animations.Any())
                            {
                                commandSet.Add(model.CreateUpdatePreviewAnimationCommand().Execute());
                            }
                        }
                    }

                    List<Error> errors2 = new List<Error>();
                    using (var cursor = new WaitCursor())
                    using (var watch = new DebugStopWatch(""))
                    {
                        // テクスチャの読み込みは、アニメーションのバインド完了後に行う必要がある。
                        // 追加参照パスがモデル側に設定されている場合には、バインド完了後でなければ
                        // アニメーション内のテクスチャパス解決に追加参照パスを参照できないため。

                        List<Document> openedDocuments2 = new List<Document>();

                        var referencingDocuments = new Dictionary<Document, PathWithName[]>();
                        using (var wait = new WaitCursor())
                        using (var watch2 = new DebugStopWatch("Load Files 2"))
                        {
                            // 参照の列挙
                            var pathWithNames = Enumerable.Empty<PathWithName>();
                            foreach (var openedDocument in openedDocuments1)
                            {
                                var textures = MakeReferenceTextureFilename(openedDocument).ToArray();
                                if (textures.Length > 0)
                                {
                                    pathWithNames = pathWithNames.Concat(textures);
                                    referencingDocuments[openedDocument] = textures;
                                }

                                pathWithNames = pathWithNames.Concat(MakeReferenceShaderDefinitionFilename(openedDocument));
                            }

                            ExecuteLoadFromFileCommand(openedDocuments2,
                                pathWithNames.Distinct(),
                                out errors2,
                                startup,
                                commandSet,
                                true,
                                excludePreOpenPaths);

                            watch2.SetMessage(string.Format("Load {0} Files", openedDocuments2.Count));
                        }

                        foreach (var model in openedDocuments.OfType<IReferTexture>())
                        {
                            PathWithName[] pathWithNames;
                            if (referencingDocuments.TryGetValue((Document)model, out pathWithNames))
                            {
                                foreach (var pathWithName in pathWithNames)
                                {
                                    var texturePath = Path.GetFullPath(pathWithName.path);
                                    var texture = DocumentManager.Textures.FirstOrDefault(x => string.Compare(texturePath, x.FilePath, true) == 0);

                                    if (texture != null)
                                    {
                                        model.ReferenceTexturePaths[texture.Name] = texturePath;
                                    }
                                }
                            }
                        }
                        openedDocuments.AddRange(openedDocuments2);

                        preOpenDocs.AddRange(openedDocuments.Where(x => !excludePreOpenPaths.Contains(Path.GetFullPath(x.FilePath).ToLower())));
                        if (preOpenDocs.Any())
                        {
                            var paths1 = preOpenDocs.Select(x => Tuple.Create(x, x.FilePath)).ToArray();
                            commandSet.canRedo += () => PrePostIO.ExecutePreOpenUndoRedo(paths1, false);
                        }

                        if (!executePreCommand)
                        {
                            var openedPaths = new HashSet<string>(Enumerable.Repeat(newProject, 1).Concat(openedDocuments).Select(x => Path.GetFullPath(x.FilePath).ToLower()));
                            var paths1 = excludePreOpenDocList.Select(x => Tuple.Create(x, x.FilePath)).Where(x => !openedPaths.Contains(Path.GetFullPath(x.Item2).ToLower())).ToArray();
                            if (paths1.Any())
                            {
                                commandSet.canUndo += () => PrePostIO.ExecutePreOpenUndoRedo(paths1, true);
                            }
                        }


                        watch.SetMessage(string.Format("Load {0} Files in {1}", openedDocuments.Count, ProjectDocument.FileName));

                        // スタートアップパス内のファイルであれば、スタートアップで読み込まれたことにする
                        // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1324
                        foreach (var doc in openedDocuments.Where(x => startupFiles.Contains(x.FilePath)))
                        {
                            doc.OpenedFromStartUp = true;
                        }
                    }

                    // シーンの設定
                    var newSceneAnimationSets = project.Scene.AnimationSetItems.Select(x => new AnimationSet(true)
                    {
                        Name = x.Name,
                        Animations = x.AnimationItems.Where(y => openedDocuments.Any(z => z.FileName == y.Name && z.FileLocation == y.Directory))
                            .Select(y => new AnimationSetItem(y.Name, y.Directory)).ToList(),
                    }).ToList();
                    foreach (var animSetItem in project.Scene.AnimationSetItems.Select((x, i) => Tuple.Create(x, newSceneAnimationSets[i])))
                    {
                        foreach (var animItem in animSetItem.Item1.AnimationItems)
                        {
                            var sceneAnim = GetAnimation(animItem.Name, animItem.Directory) as SceneAnimation;
                            if (sceneAnim == null)
                            {
                                continue;
                            }

                            // シーンアニメーションの表示・非表示を設定。
                            if (animItem.Preview)
                            {
                                sceneAnim.InvisibleBinds.Remove(animSetItem.Item2);
                            }
                            else
                            {
                                sceneAnim.InvisibleBinds.Add(animSetItem.Item2);
                            }
                        }
                    }
                    foreach (var animationSet in newSceneAnimationSets)
                    {
                        animationSet.UpdateSavedData();
                        animationSet.SetNotModified();
                        animationSet.SetMaybeModified();
                    }
                    commandSet.Add(CreateSceneAnimationSetEditCommand(newSceneAnimationSets).Execute());
                    var newSceneAnimations = project.Scene.AnimationItems.Where(y => openedDocuments.Any(z => z.FileName == y.Name && z.FileLocation == y.Directory))
                        .Select(y => new AnimationSetItem(y.Name, y.Directory)).ToArray();
                    var newDefaultSceneAnimation = new AnimationSet(true)
                    {
                        Name = res.Strings.FileTreeView_DefaultAnimationSet,
                        IsDefaultAnimationSet = true,
                        Animations = newSceneAnimations.ToList()
                    };
                    foreach (var animItem in project.Scene.AnimationItems)
                    {
                        var sceneAnim = GetAnimation(animItem.Name, animItem.Directory) as SceneAnimation;
                        if (sceneAnim == null)
                        {
                            continue;
                        }

                        // シーンアニメーションの表示・非表示を設定。
                        if (animItem.Preview)
                        {
                            sceneAnim.InvisibleBinds.Remove(newDefaultSceneAnimation);
                        }
                        else
                        {
                            sceneAnim.InvisibleBinds.Add(newDefaultSceneAnimation);
                        }
                    }

                    newDefaultSceneAnimation.UpdateSavedData();
                    newDefaultSceneAnimation.SetNotModified();
                    newDefaultSceneAnimation.SetMaybeModified();
                    commandSet.Add(CreateDefaultSceneAnimationSetEditCommand(newDefaultSceneAnimation).Execute());

                    var validationErrors = openedDocuments.OfType<ShaderDefinition>().SelectMany(x => ValidateShaderDefinition(x)).ToArray();
                    if (validationErrors.Any())
                    {
                        ShowErrorDialog(MessageContext.IO_InvalidItems, validationErrors, string.Format(res.Strings.IO_InvalidItems));
                    }

                    // シーンアニメーションセットのプレビュー状態
                    var previewAnimSet = project.Scene.AnimationSetItems.FirstOrDefault(x => x.Preview);
                    DocumentManager.PreviewSceneAnimSet = previewAnimSet != null ? newSceneAnimationSets.FirstOrDefault(x => x.Name == previewAnimSet.Name) :
                        project.Scene.PreviewDefaultAnimationSet ? newDefaultSceneAnimation :
                        null;

                    // スケルタルアニメーションのバインドチェック
                    {
                        var skeletalAnims = openedDocuments.OfType<SkeletalAnimation>().ToArray();
                        var skeletalAnimError = (from animation in skeletalAnims
                                                 from model in DocumentManager.Models
                                                 where model.AnimationSetsWithDefault.Any(x => x.Animations.Contains(new AnimationSetItem(animation.FileName, animation.FileLocation)))
                                                 let messages = animation.CheckBind(model).ToArray()
                                                 where messages.Any()
                                                 select new SkeletalAnimBindDialog.ErrorInfo()
                                                 {
                                                     animation = animation,
                                                     model = model,
                                                     messages = messages
                                                 }).ToArray();

                        if (skeletalAnimError.Any())
                        {
                            App.AppContext.NotificationHandler.SkeletalAnimBindDialog(skeletalAnimError, false);
                        }
                    }

                    // 読み込み時のサンプラの修正は行わない
#if false
                    // サンプラの修正
                    if (ApplicationConfig.Preset.FollowDccSamplerNameRule)
                    {
                        var errorSamplers = openedDocuments.OfType<Model>().SelectMany(x => x.Materials).Select(x => GetSamplerErrors(x).ToArray()).Where(x => x.Length > 0).ToArray();
                        if (errorSamplers.Any())
                        {
                            bool modify = ShowErrorSamplerDialog(errorSamplers);
                            if (modify)
                            {
                                foreach (var error in errorSamplers.SelectMany(x => x).Where(x => !string.IsNullOrEmpty(x.correctedName)))
                                {
                                    commandSet.Add(ExecuteFixSamplerNameCommand(error));
                                }
                            }
                        }
                    }
#endif
                    // ライトのターゲット割り当ての修正
                    {
                        var sceneAnims = openedDocuments.OfType<SceneAnimation>().ToArray();
                        var lightAnims = from sceneAnim in sceneAnims
                                         from lightAnim in sceneAnim.LightAnims select lightAnim;

                        // プリセットに設定されていないターゲットを持つライトアニメーションを列挙
                        var errorLightAnims = new List<LightAnimation>();
                        foreach (var lightAnim in lightAnims)
                        {
                            var lightAnimTargets = lightAnim.LightAnimTargets;
                            var lightAnimTargetTypes = LightAnimation.GetTargetTypes(lightAnim.Data.type);

                            bool added = false;

                            // プリセットに設定されていないターゲットを探す
                            foreach (var lightAnimTarget in lightAnimTargets)
                            {
                                if (lightAnimTargetTypes.Contains(lightAnimTarget.targetType) == false)
                                {
                                    if (lightAnimTarget.KeyFrames.Count > 0)
                                    {
                                        errorLightAnims.Add(lightAnim);
                                        added = true;
                                        break;
                                    }
                                }
                            }

                            // プリセットに設定されていて割り当てられていないターゲットを探す
                            if (!added)
                            {
                                foreach (var lightAnimTargetType in lightAnimTargetTypes)
                                {
                                    var lightAnimTarget = lightAnimTargets.FirstOrDefault(x => x.targetType == lightAnimTargetType);
                                    if (lightAnimTarget.KeyFrames.Count == 0)
                                    {
                                        errorLightAnims.Add(lightAnim);
                                        added = true;
                                        break;
                                    }
                                }
                            }
                        }

                        // 不正なライトアニメーションを修正する
                        if (errorLightAnims.Any())
                        {
                            AddLightAnimationFixCommand(commandSet, errorLightAnims.ToArray());
                        }
                    }

                    // マテリアルのシェーダー割り当ての修正
                    {
                        // シェイプに紐づいていない fmt の頂点属性割り当てを行うために、未割り当て属性も扱う。
                        foreach (var material in DocumentManager.Materials.Where(x => x.OwnerDocument is SeparateMaterial))
                        {
                            material.MaterialShaderAssign.AddNotAssignedAttributes();
                        }

                        var shaderDefinitionNames = openedDocuments.Where(x => x.ObjectID == GuiObjectID.ShaderDefinition).Select(x => x.Name).ToArray();
                        var models = openedDocuments.Where(x => x.ObjectID == GuiObjectID.Model).ToArray();

                        // シェーダー側
                        var materials = (from material in DocumentManager.Materials
                                         where shaderDefinitionNames.Any(x => x == material.MaterialShaderAssign.ShaderDefinitionFileName) || models.Any(x => material.Referrers.Contains(x))
                                         let errorType = ShaderAssignUtility.IsConsistentWithDefinitionDetail(material)
                                         select new { material, errorType }).ToArray();
                        var notMaterialShaderMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.NotMaterialShader).Select(x => x.material).ToArray();
                        if (notMaterialShaderMaterials.Any())
                        {
                            ShowNotMaterialShaderDialog(notMaterialShaderMaterials);
                        }

                        var errorMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.Critical).Select(x => x.material).ToArray();
                        if (errorMaterials.Any())
                        {
                            AddMaterialFixCommand(commandSet, errorMaterials);
                        }
                        var disorderedMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.Unimportant).Select(x => x.material).ToArray();
                        if (disorderedMaterials.Any())
                        {
                            commandSet.Add(ShaderAssignUtility.ExecuteFixParameters(disorderedMaterials, false, reload: false));
                        }
                    }

                    // シェーダーパラメーターアニメーションの修正
                    {
                        var list = new List<ShaderAssignUtility.MaterialAnimationError>();
                        foreach (Document document in openedDocuments)
                        {
                            var paramAnim = document as IHasShaderParameterAnimation;
                            if (paramAnim != null)
                            {
                                foreach (var error in ShaderAssignUtility.CheckAnimation(paramAnim))
                                {
                                    list.Add(error);
                                }
                            }
                        }
                        var errorMaterialAnims = list.ToArray();

                        if (errorMaterialAnims.Any())
                        {
                            var animErrors = App.AppContext.NotificationHandler.FixShaderParamAnimDialog(errorMaterialAnims);
                            if (animErrors != null)
                            {
                                commandSet.Add(ShaderAssignUtility.CreateFixShaderParamAnim(animErrors).Execute());
                            }
                        }
                    }

                    // スケルタルアニメーションの修正
                    {
                        var oldFormatAnims = (from anim in openedDocuments.OfType<SkeletalAnimation>()
                                              where RetargetingHostModelInToolData(anim)
                                              select anim).ToArray();

                        if (oldFormatAnims.Any())
                        {
                            App.AppContext.NotificationHandler.OkListBoxDialog(
                                MessageContext.IO_SkeletalAnimation_MoveRetagetingHostModelToProject,
                                Strings.IO_SkeletalAnimation_MoveRetagetingHostModelToProjectText,
                                Strings.IO_SkeletalAnimation_MoveRetagetingHostModelToProject,
                                oldFormatAnims.Select(x => x.FileName).ToArray());

                            foreach (var oldFormatAnim in oldFormatAnims)
                            {
                                commandSet.Add(CreateFixRetargetingHostModelInToolDataCommand(oldFormatAnim).Execute());
                            }
                        }
                    }

                    // バインドの修正
                    // TODO: 条件を緩くする
                    if (commandSet.CommandCount > 0)
                    {
                        var command = CreateAnimationUpdateBindCommand(DocumentManager.Animations).Execute();
                        if (command.IsValid())
                        {
                            commandSet.Add(command);
                        }
                    }

                    var errors = errors0.Concat(errors1).Concat(errors2.Where(x => x.errorType != ErrorType.AlreadyOpened));

                    // プロジェクト読み込み時は変更しない。
                    if (errors.Any())
                    {
                        var command = CreateEditCommand_IO();
                        commandSet.Add(command.Execute());

                        // block.Release() 通知によって App.AppContext.DocumentAddedOrRemoved イベントが発生する。
                        // このイベント内では MainFrame.SendPreviewInfo() が呼び出されるが、
                        // Viewer.LoadOrReloadModel.Send() や Viewer.LoadOrReloadAnimation.Send() の実行は vdsb.Dispose() で行われるので
                        // block.Release() の呼び出しは vdsb.Dispose() の後に行わなければならない。
                        // ここでの block.Release() は行わず、using による block.Dispose() に任せる。
                        // block.Release();

                        if (errors.Any())
                        {
                            ShowErrorDialog(MessageContext.IO_Load_Failed, errors, string.Format(res.Strings.IO_FileInProjectOpenFailed, ProjectDocument.Name));
                        }
                    }
                    else
                    {
                        ProjectDocument.FirstModifiedCommand = null;
                        var args = new DocumentContentsChangedArgs(ProjectDocument, null);
                        App.AppContext.NotifyPropertyChanged(null, args);
                    }

                    if (stack)
                    {
                        commandSet.Reverse();
                        TheApp.CommandManager.Add(commandSet);
                    }
                }
            }
        }

        private static bool RetargetingHostModelInToolData(SkeletalAnimation anim)
        {
            var key = "RetargetingHostModel";
            var item = anim.UserDataArray.Data.FirstOrDefault(x => x.Name == key) as App.Data.UserDataArray.UserDataString;
            if (item != null && item.Item.Count == 1)
            {
                return true;
            }
            return false;
        }

        private static EditCommandSet CreateFixRetargetingHostModelInToolDataCommand(SkeletalAnimation anim)
        {
            var key = "RetargetingHostModel";
            var userDataArray = ObjectUtility.Clone(anim.UserDataArray);
            var item = userDataArray.Data.FirstOrDefault(x => x.Name == key) as App.Data.UserDataArray.UserDataString;
            Debug.Assert(item != null);
            Debug.Assert(item.Item.Count == 1);
            var commandSet = new EditCommandSet();
            commandSet.Add(SkeletalAnimationPreviewPage.CreateSetRetargetingHostModelCommand(Enumerable.Repeat(anim, 1), item.Item[0]));
            userDataArray.Data.Remove(item);
            commandSet.Add(App.PropertyEdit.UserDataPage.CreateEditCommand_UserDataArray(
                new GuiObjectGroup(anim), userDataArray, GuiObjectID.SkeletalAnimation));
            return commandSet;
        }

        private static EditCommand ExecuteFixSamplerNameCommand(SamplerError error)
        {
            var targetGroup = new GuiObjectGroup(error.material);
            var commandSet = new EditCommandSet();
            var samplerName = error.sampler.name;
            var index = Array.IndexOf(error.material.sampler_array.sampler, error.sampler);//error.sampler.sampler_index;
            commandSet.Add(new GeneralGroupReferenceEditCommand<string>(
                targetGroup,
                GuiObjectID.Material,
                Enumerable.Repeat(error.correctedName, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var materail = target as Material;
                    var name = (string)data;
                    var sampler = materail.sampler_array.sampler[index];
                    swap = sampler.name;
                    sampler.name = name;
                }).Execute());

            // サンプラ割り当ての書き換え
            foreach (var samplerAssign in error.material.MaterialShaderAssign.SamplerAssigns)
            {
                if (samplerAssign.sampler_name == samplerName)
                {
                    commandSet.Add(App.PropertyEdit.ShaderParamControls.ShaderParamControlGroup.CreateEditCommand_material_shader_assign_sampler_name(
                        targetGroup,
                        samplerAssign.id,
                        error.correctedName,
                        false).Execute());
                }
            }

            commandSet.Reverse();
            return commandSet;
        }

        private static Dictionary<string, string> samplerHintToName = new Dictionary<string, string>()
        {
            {"albedo", "_a" },
            {"opacity", "_o" },
            {"emission", "_e" },
            {"normal", "_n" },
            {"tangent", "_t" },
            {"specular", "_s" },
            {"reflection", "_r" },
        };

        private static Regex regexSamplerHint = new Regex(@"(albedo|opacity|emission|normal|tangent|specular|reflection)([0-9]+)");
        private static Regex regexSamplerName = new Regex(@"_(a|o|e|n|t|s|r)([0-9]+)");
        private static IEnumerable<SamplerError> GetSamplerErrors(Material material)
        {
            var samplerNames = new HashSet<string>(material.sampler_array.sampler.Select(x => x.name));
            var hintCounts = new Dictionary<string, int>();
            foreach (var hint in material.sampler_array.sampler.Select(x => x.hint))
            {
                if (!hintCounts.ContainsKey(hint))
                {
                    hintCounts[hint] = 0;
                }
                hintCounts[hint]++;
            }
            foreach (var sampler in material.sampler_array.sampler)
            {
                SamplerError.ErrorType type = SamplerError.ErrorType.OK;
                string correctedName = null;
                if (!string.IsNullOrEmpty(sampler.hint))
                {
                    var match = regexSamplerHint.Match(sampler.hint);
                    int postFix;
                    if (match.Success &&
                        match.Groups.Count == 3 &&
                        match.Length == sampler.hint.Length &&
                        int.TryParse(match.Groups[2].Value, out postFix) &&
                        0<=postFix && postFix<=254)
                    {
                        var name = samplerHintToName[match.Groups[1].Value] + match.Groups[2].Value;
                        if (sampler.name != name)
                        {
                            if (!samplerNames.Contains(name) && hintCounts[sampler.hint] == 1)
                            {
                                correctedName = name;
                            }
                            type = SamplerError.ErrorType.NotFollowDccRule;
                        }

                        if (type == SamplerError.ErrorType.OK && hintCounts[sampler.hint] != 1)
                        {
                            type = SamplerError.ErrorType.Duplicated;
                        }
                    }
                    else
                    {
                        // ヒントが非定型のときは何もしない
                        //type = SamplerError.ErrorType.NotFollowDccRule;
                    }
                }
                else if (!string.IsNullOrEmpty(sampler.name))
                {
                    var match = regexSamplerName.Match(sampler.name);
                    int postFix;
                    if (match.Success &&
                        match.Groups.Count == 3 &&
                        match.Length == sampler.name.Length &&
                        int.TryParse(match.Groups[2].Value, out postFix) &&
                        0 <= postFix && postFix <= 254)
                    {
                        type = SamplerError.ErrorType.NotFollowDccRule;
                    }
                }

                if (type != SamplerError.ErrorType.OK)
                {
                    yield return new SamplerError()
                    {
                        material = material,
                        sampler = sampler,
                        correctedName = correctedName,
                        type = type,
                    };
                }
            }
        }

/*
        private static bool ShowErrorSamplerDialog(SamplerError[][] samplerErrors)
        {
            using (var dialog = new SamplerErrorDialog(samplerErrors))
            {
                var result = dialog.ShowDialog();
                if (result == DialogResult.OK && dialog.CanFix)
                {
                    return true;
                }
            }

            return false;
        }
*/

        public class SamplerError
        {
            public Material material;
            public samplerType sampler;
            public string correctedName;
            public ErrorType type;

            public enum ErrorType
            {
                OK,
                Duplicated,
                NotFollowDccRule,
            }
        }

        private static void ShowNotMaterialShaderDialog(Material[] materials)
        {
            App.AppContext.NotificationHandler.OkListBoxDialog(
                MessageContext.IO_NotMaterialShaderAssined,
                Strings.IO_NotMaterialShaderAssignedText,
                Strings.IO_NotMaterialShaderAssignedDescription,
                materials.Select(material => string.Format(Strings.IO_NotMaterialShaderAssignedItem, material.OwnerDocument.Name, material.Name, material.MaterialShaderAssign.ShaderName, material.MaterialShaderAssign.ShaderDefinitionFileName))
                );
        }

        private static IEnumerable<Error> ValidateShaderDefinition(ShaderDefinition shaderDefinition)
        {
            foreach (var shading_model in shaderDefinition.Data.shading_model_array.GetItems())
            {
                // RenderInfo の範囲チェック
                foreach (var slot in shading_model.render_info_slot_array.GetItems())
                {
                    switch (IfRenderInfoUtility.IsValidRenderInfoSlot(slot))
                    {
                        case IfRenderInfoUtility.RenderInfoSlotErrorType.OK:
                            break;
                        case IfRenderInfoUtility.RenderInfoSlotErrorType.ChoiceCannotParse:
                        case IfRenderInfoUtility.RenderInfoSlotErrorType.InvalidRange:
                        case IfRenderInfoUtility.RenderInfoSlotErrorType.DuplicateChoice:
                        case IfRenderInfoUtility.RenderInfoSlotErrorType.DuplicateAlias:
                            yield return new Error(
                                shaderDefinition.FilePath,
                                ErrorType.Other,
                                string.Format(res.Strings.RenderInfo_InvalidChoice, shading_model.name, slot.type.ToString(), slot.name, slot.choice));
                            break;
                        case IfRenderInfoUtility.RenderInfoSlotErrorType.DefaultCannotParse:
                            yield return new Error(
                                shaderDefinition.FilePath,
                                ErrorType.Other,
                                string.Format(res.Strings.RenderInfo_InvalidDefault, shading_model.name, slot.type.ToString(), slot.name, slot.@default));
                            break;
                        case IfRenderInfoUtility.RenderInfoSlotErrorType.InvalidDefaultCount:
                            yield return new Error(
                                shaderDefinition.FilePath,
                                ErrorType.Other,
                                string.Format(res.Strings.RenderInfo_InvalidDefaultCount, shading_model.name, slot.name, slot.@default, slot.count));
                            break;
                        case IfRenderInfoUtility.RenderInfoSlotErrorType.DefaultNotInChoice:
                            yield return new Error(
                                shaderDefinition.FilePath,
                                ErrorType.Other,
                                string.Format(res.Strings.RenderInfo_DefaultOutOfRange, shading_model.name, slot.name, slot.@default, slot.choice));
                            break;
                        default:
                            break;
                    }
                }

                // オプション変数の範囲チェック
                foreach (var option in shading_model.Options())
                {
                    switch (IfShaderOptionUtility.ValidateOptionVar(option))
                    {
                        case IfShaderOptionUtility.OptionVarErrorType.OK:
                            break;
                        case IfShaderOptionUtility.OptionVarErrorType.ChoiceCannotParse:
                        case IfShaderOptionUtility.OptionVarErrorType.InvalidRange:
                        case IfShaderOptionUtility.OptionVarErrorType.DuplicateChoice:
                        case IfShaderOptionUtility.OptionVarErrorType.DuplicateAlias:
                            yield return new Error(
                                shaderDefinition.FilePath,
                                ErrorType.Other,
                                string.Format(res.Strings.OptionVar_InvalidChoice, shading_model.name, option.id, option.choice));
                            break;
                        case IfShaderOptionUtility.OptionVarErrorType.DefaultCannotParse:
                            yield return new Error(
                                shaderDefinition.FilePath,
                                ErrorType.Other,
                                string.Format(res.Strings.OptionVar_InvalidDefault, shading_model.name, option.id, option.@default));
                            break;
                        case IfShaderOptionUtility.OptionVarErrorType.DefaultNotInChoice:
                            yield return new Error(
                                shaderDefinition.FilePath,
                                ErrorType.Other,
                                string.Format(res.Strings.OptionVar_DefaultOutOfRange, shading_model.name, option.id, option.@default, option.choice));
                            break;
                        default:
                            break;
                    }
                }

                // グループのチェック
                var groups = new HashSet<string>(shading_model.Groups().Select(x => x.name).Distinct());
                foreach (var option in shading_model.Options())
                {
                    if (option.ui_group != null && !groups.Contains(option.ui_group.group_name))
                    {
                        yield return new Error(
                            shaderDefinition.FilePath,
                            ErrorType.Other,
                            string.Format(res.Strings.InvalidGroup_Option, shading_model.name, option.id, option.ui_group.group_name));
                    }
                }

                foreach (var sampler in shading_model.Samplers())
                {
                    if (sampler.ui_group != null && !groups.Contains(sampler.ui_group.group_name))
                    {
                        yield return new Error(
                            shaderDefinition.FilePath,
                            ErrorType.Other,
                            string.Format(res.Strings.InvalidGroup_Sampler, shading_model.name, sampler.id, sampler.ui_group.group_name));
                    }
                }

                foreach (var uniform in shading_model.Uniforms())
                {
                    if (uniform.ui_group != null && !groups.Contains(uniform.ui_group.group_name))
                    {
                        yield return new Error(
                            shaderDefinition.FilePath,
                            ErrorType.Other,
                            string.Format(res.Strings.InvalidGroup_Uniform, shading_model.name, uniform.id, uniform.ui_group.group_name));
                    }
                }

                foreach (var info in shading_model.RenderInfoSlots())
                {
                    if (info.ui_group != null && !groups.Contains(info.ui_group.group_name))
                    {
                        yield return new Error(
                            shaderDefinition.FilePath,
                            ErrorType.Other,
                            string.Format(res.Strings.InvalidGroup_RenderInfo, shading_model.name, info.name, info.ui_group.group_name));
                    }
                }

                // ページのチェック
                var pages = shading_model.Pages().Select(x => x.name).ToArray();
                foreach (var group in shading_model.Groups())
                {
                    if (!string.IsNullOrEmpty(group.page_name) && !pages.Contains(group.page_name))
                    {
                        yield return new Error(
                            shaderDefinition.FilePath,
                            ErrorType.Other,
                            string.Format(res.Strings.InvalidPage_Group, shading_model.name, group.name, group.page_name));
                    }
                }
            }
        }

        // 存在するパスを列挙する
        private static IEnumerable<PathWithName> FilterExsitingFiles(IEnumerable<PathWithName> paths, bool ignoreProject = false)
        {
            foreach (var path in paths)
            {
                if (Directory.Exists(path.path))
                {
                    ObjectIDUtility.UnsupportedType unsupportedType;

                    // プロジェクトファイル以外のファイルを列挙
                    var files = Directory.GetFiles(path.path, "*.*", SearchOption.AllDirectories)
                        .Where(x => ObjectIDUtility.IsSupportExtension(Path.GetExtension(x), out unsupportedType))
                        .Where(x => !ignoreProject || ObjectIDUtility.ExtToId(Path.GetExtension(x)) != GuiObjectID.Project);
                    foreach (var file in files)
                    {
                        yield return new PathWithName(file, path.baseName);
                    }
                }
                else if (File.Exists(path.path))
                {
                    yield return path;
                }
            }
        }

        public static GroupEditCommand CreateReloadEditCommand(IEnumerable<Document> reloadDocuments)
        {

            var targets = new GuiObjectGroup(reloadDocuments);
            return
                 new GeneralGroupValueEditCommand<int>(
                    targets,
                    GuiObjectID.Project,
                    0,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        if (target is Document)
                        {
                            var d = target as Document;
                            Reload(d.FilePath, d, false);
                        }
                    }
                );
        }


        private static void LoadFromFileOrDirectory(IEnumerable<PathWithName> paths, AnimationSet animationOwner = null, bool isStartup = false, Model animationOwnerModel = null, bool executePreOpenCommand = true)
        {
            LoadFromFile(FilterExsitingFiles(paths, true), animationOwner, isStartup, isUpdateRecentlyUsedFiles: true, animationOwnerModel:animationOwnerModel, executePreOpenCommand:executePreOpenCommand);
        }

        public static void LoadFromFileOrDirectory(IEnumerable<string> paths, AnimationSet animationOwner = null, Model animationOwnerModel = null, bool executePreOpenCommand = true)
        {
            LoadFromFileOrDirectory(paths.Select(x => new PathWithName(x, null)), animationOwner, animationOwnerModel:animationOwnerModel, executePreOpenCommand:executePreOpenCommand);
        }

        /// <summary>
        /// ファイルをロードします。
        /// </summary>
        /// <param name="fileNames">ファイルパスと基準パス名</param>
        /// <param name="animationOwner">アニメーションを指定する場合のアニメーション</param>
        /// <param name="isStartup">スタートアップ時の起動かどうか</param>
        /// <param name="commandSet">実行したコマンドをスタックします。null の場合はUndo スタックに積みます。</param>
        /// <param name="isBindAnimation"></param>
        /// <param name="RelativePaths"></param>
        /// <param name="isUpdateRecentlyUsedFiles"></param>
        public static void LoadFromFile(
            IEnumerable<PathWithName>	fileNames,
            AnimationSet				animationOwner				= null, // アニメーションをバインドするアニメーションセット
            bool						isStartup					= false,
            EditCommandSet				commandSet					= null,
            bool						isBindAnimation				= true,
            bool						isUpdateRecentlyUsedFiles	= false,
            bool						isShowErrorDialog			= true,
            bool                        reloadFromMenu              = false,
            Model                       animationOwnerModel = null, // アニメーションをバインドするモデル: animationOwner が優先される
            bool executePreOpenCommand = true,
            bool executePreOpenRedoCommand = true,
            bool resolveTextureReference = true)
        {
            {
                var projects = fileNames.Where(x => Path.GetExtension(x.path).ToLower() == ProjectDocument.DefaultDotExt);
                if (projects.Any())
                {
                    if (fileNames.Count() > 1)
                    {
                        App.AppContext.NotificationHandler.MessageBoxInformation(Strings.IO_OpenProjectAlone);
                        return;
                    }

                    LoadFromProject(fileNames.First().path, isStartup, reloadFromMenu, executePreOpenCommand, commandSet);
                    return;
                }
            }

            if (XsdBasePath == null)
            {
                App.AppContext.NotificationHandler.MessageBoxInformation(Strings.IO_XsdNotFound);
            }

            List<Error> errors;
            List<Error> errors3;
            var bindError = new List<Tuple<AnimationDocument, Model>>();
            var openedDocuments1 = new List<Document>();
            var openedDocuments2 = new List<Document>();
            var openedDocuments3 = new List<Document>();

            //var reloadDocuments = new List<Document>();

            bool stack;
            if (commandSet != null)
            {
                stack = false;
            }
            else
            {
                stack = true;
                commandSet = new EditCommandSet();
            }

            using (var reloadModelSuppressBlock = new Viewer.HioUtility.ReloadModelSuppressBlock())
            using (var propertyChangedSuppressBlock = new App.AppContext.PropertyChangedSuppressBlock())
            {
                using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                {
                    using (var wait = new WaitCursor())
                    {
                        // 既に開かれているファイルをあらかじめ調べる
                        var sameFiles = new List<String>();
                        foreach (var name in fileNames)
                        {
                            bool isSame;
                            if (ExistsFileName(name.path, out isSame) && isSame)
                            {
                                sameFiles.Add(name.path);
                            }
                        }

                        // 既に開かれているファイルを再読込するかダイアログを出す
                        if (sameFiles.Any())
                        {
                            var samedocs = DocumentsWithoutProject.Where(x => sameFiles.Contains(x.FilePath)).ToList();
                            var updateddocs = samedocs.Where(d => Math.Abs((File.GetLastWriteTime(d.FilePath) - d.LastOpenedWriteTime).TotalMilliseconds) > 100.0).ToArray();
                            var skipeddocs = samedocs.Where(x => !updateddocs.Contains(x)).ToList();

                            // 更新されたファイル
                            if (updateddocs.Any())
                            {
                                var doReload = ApplicationConfig.UserSetting.IO.AutoReloadUpdatedOnLoad;

                                // ダイアログ
                                if (!doReload)
                                {
                                    doReload =
                                        App.AppContext.NotificationHandler.SameFileReload(updateddocs);
                                }

                                if (doReload)
                                {
                                    foreach (var doc in updateddocs)
                                    {
                                        var comset = Reload(doc.FilePath, doc, false, true, true);
                                        if (comset != null)
                                        {
                                            commandSet.Add(comset);
                                        }
                                    }
                                }
                            }
                            // 読み込みをスキップされたファイル
                            if (skipeddocs.Any())
                            {
                                App.AppContext.NotificationHandler.OkListBoxDialog(
                                    MessageContext.DocumentManager_LoadFromFile_SkipReload,
                                    Strings.DocumentManager_LoadFromFile_SkipReloadTitle,
                                    Strings.DocumentManager_LoadFromFile_SkipReloadText,
                                    skipeddocs.Select(doc => doc.FilePath)
                                    );
                            }

                            // 既に開かれているファイルはリストから外す
                            fileNames = fileNames.Where(x => sameFiles.All(y => x.path != y));
                        }

                        using (var watch = new DebugStopWatch("Load Files"))
                        {
                            ExecuteLoadFromFileCommand(openedDocuments1, fileNames, out errors, isStartup, commandSet, executePreOpenCommand);
                            watch.SetMessage(string.Format("Load {0} Files", openedDocuments1.Count));
                        }

                        if (isUpdateRecentlyUsedFiles)
                        {
                            foreach (var document in openedDocuments1)
                            {
                                TheApp.MainFrame.FileAdded(document);
                            }
                        }

                        // マテリアル参照で使われる親マテリアルをロードする
                        // 再帰的に継承されるため、全てロードするまで繰り返す
                        var models = openedDocuments1.OfType<Model>().ToList();
                        using (var watch = new DebugStopWatch("Load Parent Material Model Files"))
                        {
                            while (models.Any())
                            {
                                var parentModelFileNames = new List<PathWithName>();
                                foreach (var model in models)
                                {
                                    var parentMaterials = model.Materials.Where(mtl => mtl.MaterialReference?.ParentMaterials?.Any() == true)
                                        .SelectMany(mtl => mtl.MaterialReference?.ParentMaterials).Where(x => x != null);
                                    foreach (var parentMaterial in parentMaterials)
                                    {
                                        var modelPath = parentMaterial.Path;
                                        if (string.IsNullOrEmpty(modelPath)) continue;

                                        // まず子モデルからの相対で探し、見つからなければ親マテリアルパスから探す
                                        var parentModelFullPath = IfApplyBaseMaterialUtility.GetParentModelFullPath(modelPath, model.FilePath);

                                        if (!File.Exists(parentModelFullPath))
                                        {
                                            var fromParentMaterialPath = Material.SearchParentModelFromParentMaterialPaths(modelPath);
                                            if (!string.IsNullOrEmpty(fromParentMaterialPath))
                                            {
                                                parentModelFullPath = fromParentMaterialPath;
                                            }
                                        }

                                        if (string.IsNullOrEmpty(parentModelFullPath))
                                        {
                                            parentModelFullPath = modelPath;
                                        }

                                        bool isSame;
                                        if (ExistsFileName(parentModelFullPath, out isSame) && isSame) continue;

                                        if (parentModelFileNames.All(
                                            x => !x.path.Equals(parentModelFullPath, StringComparison.OrdinalIgnoreCase)))
                                        {
                                            parentModelFileNames.Add(new PathWithName(parentModelFullPath));
                                        }
                                    }
                                }

                                models.Clear();
                                if (parentModelFileNames.Any())
                                {
                                    var documents = new List<Document>();
                                    List<Error> err;
                                    ExecuteLoadFromFileCommand(documents, parentModelFileNames, out err, isStartup, commandSet, executePreOpenCommand);
                                    errors.AddRange(err);
                                    openedDocuments1.AddRange(documents);
                                    //var parentModels = documents.OfType<Model>().ToArray();
                                    models.AddRange(documents.OfType<Model>());
                                    foreach (var parentModel in models)
                                    {
                                        parentModel.PreviewInfo.Visible = false;
                                        parentModel.PreviewInfo.ShowInObjView = false;
                                    }
                                }
                            }
                        }
                    }

                    var validationErrors = openedDocuments1.OfType<ShaderDefinition>().SelectMany(x => ValidateShaderDefinition(x)).ToArray();
                    if (validationErrors.Any())
                    {
                        ShowErrorDialog(MessageContext.IO_InvalidItems, validationErrors, string.Format(res.Strings.IO_InvalidItems));
                    }

                    var referencingDocuments = new Dictionary<Document, PathWithName[]>();
                    using (var wait = new WaitCursor())
                    using (var watch = new DebugStopWatch("Load Files 2"))
                    {
                        // 参照の列挙
                        var pathWithNames = Enumerable.Empty<PathWithName>();
                        foreach (var openedDocument in openedDocuments1)
                        {
                            var textures = MakeReferenceTextureFilename(openedDocument).ToArray();
                            if (textures.Length > 0)
                            {
                                pathWithNames = pathWithNames.Concat(textures);
                                referencingDocuments[openedDocument] = textures;
                            }

                            pathWithNames = pathWithNames.Concat(MakeReferenceShaderDefinitionFilename(openedDocument));
                        }

                        ExecuteLoadFromFileCommand(openedDocuments3,
                            pathWithNames.Distinct(),
                            out errors3,
                            isStartup,
                            commandSet,
                            true);

                        watch.SetMessage(string.Format("Load {0} Files", openedDocuments3.Count));
                    }

                    foreach (var irefer in openedDocuments1.OfType<IReferTexture>())
                    {
                        PathWithName[] pathWithNames;
                        if (referencingDocuments.TryGetValue((Document)irefer, out pathWithNames))
                        {
                            foreach (var pathWithName in pathWithNames)
                            {
                                var texturePath = Path.GetFullPath(pathWithName.path);
                                var texture = DocumentManager.Textures.FirstOrDefault(x => string.Compare(texturePath, x.FilePath, true) == 0);

                                if (texture != null)
                                {
                                    irefer.ReferenceTexturePaths[texture.Name] = texturePath;
                                }
                            }
                        }
                    }



                    var openedDocuments = openedDocuments1.Concat(openedDocuments3);

                    // テクスチャ参照の解決
                    if (resolveTextureReference)
                    {
                        var openedTextures = new HashSet<string>(openedDocuments.OfType<Texture>().Select(x => x.Name));
                        var unresolvedModels = new Dictionary<string, List<Model>>();
                        foreach (var model in DocumentManager.Models)
                        {
                            // パスが決まらないと解決できないはず(やりようによってはできるが...)
                            if (string.IsNullOrEmpty(model.FileLocation))
                            {
                                continue;
                            }

                            foreach (var texture in model.UnresolvedTextures())
                            {
                                if (openedTextures.Contains(texture))
                                {
                                    List<Model> models;
                                    if (!unresolvedModels.TryGetValue(texture, out models))
                                    {
                                        models = new List<Model>();
                                        unresolvedModels[texture] = models;
                                    }
                                    models.Add(model);
                                }
                            }
                        }

                        var unresolvedMaterialDocs = new Dictionary<string, List<SeparateMaterial>>();
                        foreach (var materialDoc in DocumentManager.SeparateMaterials)
                        {
                            // パスが決まらないと解決できないはず(やりようによってはできるが...)
                            if (string.IsNullOrEmpty(materialDoc.FileLocation))
                            {
                                continue;
                            }

                            foreach (var texture in materialDoc.UnresolvedTextures())
                            {
                                if (openedTextures.Contains(texture))
                                {
                                    List<SeparateMaterial> materialDocs;
                                    if (!unresolvedMaterialDocs.TryGetValue(texture, out materialDocs))
                                    {
                                        materialDocs = new List<SeparateMaterial>();
                                        unresolvedMaterialDocs[texture] = materialDocs;
                                    }
                                    materialDocs.Add(materialDoc);
                                }
                            }
                        }

                        var unresolvedAnims = new Dictionary<string, List<TexturePatternAnimation>>();
                        foreach (var anim in DocumentManager.Animations.OfType<TexturePatternAnimation>())
                        {
                            // パスが決まらないと解決できないはず(やりようによってはできるが...)
                            if (string.IsNullOrEmpty(anim.FileLocation))
                            {
                                continue;
                            }

                            foreach (var texture in anim.UnresolvedTextures())
                            {
                                if (openedTextures.Contains(texture))
                                {
                                    List<TexturePatternAnimation> anims;
                                    if (!unresolvedAnims.TryGetValue(texture, out anims))
                                    {
                                        anims = new List<TexturePatternAnimation>();
                                        unresolvedAnims[texture] = anims;
                                    }
                                    anims.Add(anim);
                                }
                            }
                        }

                        // アニメーションに一つだけ結びつける
                        var ownerModels = new Dictionary<Tuple<string,string>, Model>();
                        foreach (var model in DocumentManager.Models)
                        {
                            if (!string.IsNullOrEmpty(model.FileLocation))
                            {
                                foreach (var animSet in model.AnimationSetsWithDefault)
                                {
                                    foreach (var anim in animSet.Animations)
                                    {
                                        var ext = Path.GetExtension(anim.Name);
                                        if (ObjectIDUtility.ExtToId(ext) == GuiObjectID.TexturePatternAnimation)
                                        {
                                            var tuple = new Tuple<string, string>(anim.Name, anim.Directory.ToLower());
                                            if (!ownerModels.ContainsKey(tuple))
                                            {
                                                ownerModels[tuple] = model;
                                            }
                                        }
                                    }
                                }
                            }
                        }

                        foreach (var openedDocument in openedDocuments.Where(x => x.ObjectID == GuiObjectID.Texture))
                        {
                            List<Model> models;
                            if (unresolvedModels.TryGetValue(openedDocument.Name, out models))
                            {
                                foreach (var model in models)
                                {
                                    var pathWithName = MakeReferenceTextureFilenames(
                                            model.FileLocation,
                                            model,
                                            model.BaseName,
                                            openedDocument.Name).FirstOrDefault();

                                        if (pathWithName != null &&
                                            string.Compare(pathWithName.path, openedDocument.FilePath, true) == 0)
                                        {
                                            commandSet.Add(App.PropertyEdit.MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(
                                                new GuiObjectGroup(model),
                                                GuiObjectID.Model,
                                                openedDocument.Name,
                                                openedDocument.FilePath).Execute());
                                        }
                                }
                            }

                            List<SeparateMaterial> materialDocs;
                            if (unresolvedMaterialDocs.TryGetValue(openedDocument.Name, out materialDocs))
                            {
                                foreach (var materialDoc in materialDocs)
                                {
                                    var pathWithName = MakeReferenceTextureFilenames(
                                            materialDoc.FileLocation,
                                            materialDoc,
                                            materialDoc.BaseName,
                                            openedDocument.Name).FirstOrDefault();

                                    if (pathWithName != null &&
                                        string.Compare(pathWithName.path, openedDocument.FilePath, true) == 0)
                                    {
                                        commandSet.Add(App.PropertyEdit.MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(
                                            new GuiObjectGroup(materialDoc),
                                            GuiObjectID.SeparateMaterial,
                                            openedDocument.Name,
                                            openedDocument.FilePath).Execute());
                                    }
                                }
                            }

                            List<TexturePatternAnimation> anims;
                            if (unresolvedAnims.TryGetValue(openedDocument.Name, out anims))
                            {
                                foreach (var anim in anims)
                                {
                                    Debug.Assert(anim.FileLocation != null);

                                    var tuple = new Tuple<string, string>(anim.Name, anim.FileLocation.ToLower());

                                    Model model;
                                    ownerModels.TryGetValue(tuple, out model);
                                    var pathWithName = MakeReferenceTextureFilenames(
                                            anim.FileLocation,
                                            model,
                                            anim.BaseName,
                                            openedDocument.Name).FirstOrDefault();

                                    if (pathWithName != null &&
                                        string.Compare(pathWithName.path, openedDocument.FilePath, true) == 0)
                                    {
                                        commandSet.Add(App.PropertyEdit.TexturePatternAnimationPatternPage.CreateEditCommand_ReferenceTexturePath(
                                            new GuiObjectGroup(anim),
                                            openedDocument.Name,
                                            openedDocument.FilePath).Execute());
                                    }
                                }
                            }

                            // 注意: 参照が解決されたモデルやアニメーションの転送していない。
                            // 必要なら実装する
                        }
                    }

                    // アニメーションのバインド設定
                    if (isBindAnimation)
                    {
                        var boundDocuments = new List<AnimationDocument>();
                        var ownerModelAnims = new Dictionary<Model, List<AnimationDocument>>();
                        var ownerAnimSetAnims = new Dictionary<AnimationSet, List<AnimationDocument>>();
                        var forceOpenAnimeSetFolder = (Control.ModifierKeys & Keys.Control) != 0;
                        var sceneAnimationsWithoutOwner = new List<SceneAnimation>();

                        foreach (AnimationDocument animation in openedDocuments1.Where(x => x is AnimationDocument))
                        {
                            AnimationSet owner = null;
                            Model ownerModel = null;
                            if (animationOwner != null && (animation.ObjectID == GuiObjectID.SceneAnimation) == (animationOwner is AnimationSet && ((AnimationSet)animationOwner).IsSceneAnimationSet))
                            {
                                owner = animationOwner;
                            }

                            if (owner == null && animation.ObjectID != GuiObjectID.SceneAnimation)
                            {
                                // 現時点ではモデルのみ探索
                                // SearchModelForAnimations() の block 引数には null を渡す。
                                // null 以外を渡すと、SelectTargetModel() で block.Release() が行われることがある。
                                // バインド前の Model.SendEditBoneBind() と Model.SendEditModelLayout() の呼び出しを回避するには
                                // null を渡して block.Release() を回避しなければならない。
                                // 詳細は SelectTargetModel() を参照。
                                bool applyAll = false;
                                ownerModel = animationOwnerModel ?? SearchModelForAnimations(animation, null, out applyAll);
                                if (applyAll)
                                {
                                    animationOwnerModel = ownerModel;
                                }
                            }

                            boundDocuments.Add(animation);

                            if (owner is AnimationSet)
                            {
                                var ownerAnimSet = owner as AnimationSet;
                                Debug.Assert(ownerAnimSet != null);

                                if (ownerAnimSetAnims.ContainsKey(ownerAnimSet) == false)
                                {
                                    ownerAnimSetAnims[ownerAnimSet] = new List<AnimationDocument>();
                                }

                                ownerAnimSetAnims[ownerAnimSet].Add(animation);
                            }
                            else
                                if (ownerModel is Model)
                                {
                                    Debug.Assert(ownerModel != null);

                                    if (ownerModelAnims.ContainsKey(ownerModel) == false)
                                    {
                                        ownerModelAnims[ownerModel] = new List<AnimationDocument>();
                                    }

                                    ownerModelAnims[ownerModel].Add(animation);
                                }
                                else
                                    if (owner == null)
                                    {
                                        // オーナーなし
                                        if (animation is SceneAnimation)
                                        {
                                            sceneAnimationsWithoutOwner.Add((SceneAnimation)animation);
                                        }
                                    }
                                    else
                                    {
                                        Debug.Assert(false);
                                    }
                        }

                        // アニメーションセットに対しての編集コマンドを実行する
                        // ownerAnimSetAnims
                        {
                            var group = from tuple in ownerAnimSetAnims
                                        let model = Models.FirstOrDefault(x => x.AnimationSetsWithDefault.Contains(tuple.Key))
                                        group tuple by model;
                            foreach (var modelItems in group)
                            {
                                var ownerModel = modelItems.Key;
                                foreach (var ownerAnim in modelItems)
                                {
                                    var bindAnim = ownerAnim.Value.Where(x => CanAddAnimation(x, ownerModel))
                                        .GroupBy(x => x.FileName.ToLower()).Select(x => x.First());
                                    var ngAnim = ownerAnim.Value.Where(x => !bindAnim.Contains(x));

                                    foreach (var ng in ngAnim)
                                    {
                                        bindError.Add(new Tuple<AnimationDocument, Model>(ng, ownerModel));
                                    }

                                    commandSet.Add(
                                        CreateAnimationsEditCommand(
                                            ownerAnim.Key,
                                            ownerAnim.Key.Animations.Concat(
                                                bindAnim.Select(x => new AnimationSetItem(x.FileName, x.FileLocation))).ToArray(),
                                            true
                                        ).Execute()
                                    );
                                }
                            }
                        }

                        // シーンに対するアニメーションセットの生成
                        if (ApplicationConfig.Setting.MainFrame.IsAutoAnimationBindMode)
                        {
                            var newAnimSets = new List<AnimationSet>();
                            // アニメーションごとに同名のアニメーションセットに含める
                            foreach (var anim in sceneAnimationsWithoutOwner)
                            {
                                var item = new AnimationSetItem(anim.FileName, anim.FileLocation);
                                var existedAnimSet = SceneAnimationSets.FirstOrDefault(x => String.Compare(x.Name, anim.Name, StringComparison.OrdinalIgnoreCase) == 0);
                                if (existedAnimSet != null)
                                {
                                    // 既に指定名のアニメーションセットがあればそれを使う
                                    commandSet.Add(
                                        CreateAnimationsEditCommand(
                                            existedAnimSet,
                                            existedAnimSet.Animations.Concat(Enumerable.Repeat(item, 1)).ToArray(),
                                            true
                                        ).Execute()
                                    );
                                    existedAnimSet.SuppressOpenTreeNode = !forceOpenAnimeSetFolder;
                                }
                                else
                                {
                                    // 新規作成して追加する
                                    // アニメーションセットを閉じた状態で作成する
                                    var newAnimSet = new AnimationSet(true)
                                    {
                                        Name = anim.Name,
                                        SuppressOpenTreeNode = !forceOpenAnimeSetFolder
                                    };

                                    newAnimSet.Animations.Add(item);
                                    newAnimSets.Add(newAnimSet);
                                }
                            }

                            if (newAnimSets.Any())
                            {
                                commandSet.Add(CreateSceneAnimationSetEditCommand(SceneAnimationSets.Concat(newAnimSets).ToList()).Execute());
                            }
                        }

                        // モデルに対しての編集コマンドを実行する
                        // ownerModelAnims
                        {
                            foreach (var ownerAnim in ownerModelAnims)
                            {
                                var ownerModel = ownerAnim.Key;

                                if (ApplicationConfig.Setting.MainFrame.IsAutoAnimationBindMode)
                                {
                                    var newAnimSets = new List<AnimationSet>();
                                    {
                                        foreach (var animSetName in ownerAnim.Value.Select(x => x.Name).GroupBy(x => x.ToLower()).Select(x => x.First()))
                                        {
                                            var sameNameAnims = ownerAnim.Value.Where(x => String.Compare(x.Name, animSetName, StringComparison.OrdinalIgnoreCase)==0)
                                                .Select(x => new { anim = x, item = new AnimationSetItem(x.FileName, x.FileLocation) }).ToArray();
                                            var bindAnim = sameNameAnims.Where(x => CanAddAnimation(x.anim, ownerModel))
                                                .GroupBy(x => x.anim.FileName.ToLower()).Select(x => x.First());
                                            var ngAnim = sameNameAnims.Where(x => !bindAnim.Contains(x));

                                            foreach (var ng in ngAnim)
                                            {
                                                bindError.Add(new Tuple<AnimationDocument, Model>(ng.anim, ownerModel));
                                            }

                                            var existedAnimSet = ownerModel.AnimationSets.FirstOrDefault(x => String.Compare(x.Name, animSetName, StringComparison.OrdinalIgnoreCase) == 0);
                                            if (existedAnimSet != null)
                                            {
                                                // 既に指定名のアニメーションセットがあればそれを使う
                                                commandSet.Add(
                                                    CreateAnimationsEditCommand(
                                                        existedAnimSet,
                                                        existedAnimSet.Animations.Concat(
                                                            bindAnim.Select(x => x.item)).ToArray(),
                                                        true
                                                    ).Execute()
                                                );
                                                existedAnimSet.SuppressOpenTreeNode = !forceOpenAnimeSetFolder;
                                            }
                                            else
                                            {
                                                // 新規作成して追加する
                                                // アニメーションセットを閉じた状態で作成する
                                                var newAnimSet = new AnimationSet(false)
                                                {
                                                    Name = animSetName,
                                                    SuppressOpenTreeNode = !forceOpenAnimeSetFolder
                                                };

                                                newAnimSet.Animations.AddRange(bindAnim.Select(x => x.item));
                                                newAnimSets.Add(newAnimSet);
                                            }
                                        }
                                    }

                                    if (newAnimSets.Any())
                                    {
                                        commandSet.Add(CreateAnimationSetEditCommand(ownerModel, ownerModel.AnimationSets.Concat(newAnimSets).ToArray()).Execute());
                                    }
                                }
                                else
                                {
                                    var bindAnim = ownerAnim.Value.Where(x => CanAddAnimation(x, ownerModel))
                                        .GroupBy(x => x.FileName.ToLower()).Select(x => x.First());
                                    var ngAnim = ownerAnim.Value.Where(x => !bindAnim.Contains(x));

                                    foreach (var ng in ngAnim)
                                    {
                                        bindError.Add(new Tuple<AnimationDocument, Model>(ng, ownerModel));
                                    }
                                    commandSet.Add(
                                        CreateAnimationsEditCommand(
                                            ownerModel.DefaultAnimationSet,
                                            ownerModel.DefaultAnimationSet.Animations.Concat(
                                                bindAnim.Select(x => new AnimationSetItem(x.FileName, x.FileLocation))).ToArray(),
                                            true
                                        ).Execute()
                                    );
                                }
                            }
                        }

                        if (boundDocuments.Any())
                        {
                            commandSet.Add(CreateAnimationUpdateBindCommand(boundDocuments).Execute());

                            // hint がついた ShaderParameterAnim に対して初回読み込み時(shader_param_mat_anim_array が存在しない場合)に
                            // オリジナルを適用相当の処理を行います。
                            foreach (var doc in boundDocuments)
                            {
                                IHasShaderParameterAnimation shaderParamAnim = doc as IHasShaderParameterAnimation;
                                if (shaderParamAnim != null &&
                                    !shaderParamAnim.HasAnyShaderParamCurves())
                                {
                                    shaderParamAnim.ApplyOriginalAnimation(commandSet, false);
                                }
                            }
                        }
                    }

                    // 読み込み時のサンプラの修正は行わない
#if false
                    // サンプラの修正
                    if (ApplicationConfig.Preset.FollowDccSamplerNameRule)
                    {
                        var errorSamplers = openedDocuments.OfType<Model>().SelectMany(x => x.Materials).Select(x => GetSamplerErrors(x).ToArray()).Where(x => x.Length > 0).ToArray();
                        if (errorSamplers.Any())
                        {
                            bool modify = ShowErrorSamplerDialog(errorSamplers);
                            if (modify)
                            {
                                foreach (var error in errorSamplers.SelectMany(x => x).Where(x => !string.IsNullOrEmpty(x.correctedName)))
                                {
                                    commandSet.Add(ExecuteFixSamplerNameCommand(error));
                                }
                            }
                        }
                    }
#endif
                    // ライトのターゲット割り当ての修正
                    {
                        var sceneAnims = openedDocuments.OfType<SceneAnimation>().ToArray();
                        var lightAnims = from sceneAnim in sceneAnims
                                         from lightAnim in sceneAnim.LightAnims select lightAnim;

                        // プリセットに設定されていないターゲットを持つライトアニメーションを列挙
                        var errorLightAnims = new List<LightAnimation>();
                        foreach (var lightAnim in lightAnims)
                        {
                            var lightAnimTargets = lightAnim.LightAnimTargets;
                            var lightAnimTargetTypes = LightAnimation.GetTargetTypes(lightAnim.Data.type);

                            bool added = false;

                            // プリセットに設定されていないターゲットを探す
                            foreach (var lightAnimTarget in lightAnimTargets)
                            {
                                if (lightAnimTargetTypes.Contains(lightAnimTarget.targetType) == false)
                                {
                                    if (lightAnimTarget.KeyFrames.Count > 0)
                                    {
                                        errorLightAnims.Add(lightAnim);
                                        added = true;
                                        break;
                                    }
                                }
                            }

                            // プリセットに設定されていて割り当てられていないターゲットを探す
                            if (!added)
                            {
                                foreach (var lightAnimTargetType in lightAnimTargetTypes)
                                {
                                    var lightAnimTarget = lightAnimTargets.FirstOrDefault(x => x.targetType == lightAnimTargetType);
                                    if (lightAnimTarget.KeyFrames.Count == 0)
                                    {
                                        errorLightAnims.Add(lightAnim);
                                        added = true;
                                        break;
                                    }
                                }
                            }
                        }

                        // 不正なライトアニメーションを修正する
                        if (errorLightAnims.Any())
                        {
                            AddLightAnimationFixCommand(commandSet, errorLightAnims.ToArray());
                        }
                    }

                    // マテリアルのシェーダー割り当ての修正
                    {
                        // シェイプに紐づいていない fmt の頂点属性割り当てを行うために、未割り当て属性も扱う。
                        foreach (var material in DocumentManager.Materials.Where(x => x.OwnerDocument is SeparateMaterial))
                        {
                            material.MaterialShaderAssign.AddNotAssignedAttributes();
                        }

                        var shaderDefinitionNames = openedDocuments.Where(x => x.ObjectID == GuiObjectID.ShaderDefinition).Select(x => x.Name).ToArray();
                        var models = openedDocuments.Where(x => x.ObjectID == GuiObjectID.Model).ToArray();
                        var materialDocs = openedDocuments.Where(x => x.ObjectID == GuiObjectID.SeparateMaterial).ToArray();

                        // シェーダー側
                        var materials = (from material in DocumentManager.Materials
                                         where shaderDefinitionNames.Any(x => x == material.MaterialShaderAssign.ShaderDefinitionFileName) || models.Any(x => material.Referrers.Contains(x)) || materialDocs.Contains(material.OwnerDocument)
                                         let errorType = ShaderAssignUtility.IsConsistentWithDefinitionDetail(material)
                                         select new { material, errorType }).ToArray();

                        var notMaterialShaderMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.NotMaterialShader).Select(x => x.material).ToArray();
                        if (notMaterialShaderMaterials.Any())
                        {
                            ShowNotMaterialShaderDialog(notMaterialShaderMaterials);
                        }

                        var errorMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.Critical).Select(x => x.material).ToArray();
                        if (errorMaterials.Any())
                        {
                            AddMaterialFixCommand(commandSet, errorMaterials);
                        }
                        var disorderedMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.Unimportant).Select(x => x.material).ToArray();
                        if (disorderedMaterials.Any())
                        {
                            // reload は常に true にするべきだと思うが、false にしていた意図が不明なのでコンバイナーを持っている場合のみ true にしておく。
                            // 本関数で fsdb を読み込んだ時、下記 2 例の順に操作されると、シェーダーが送られないのでは？
                            // A. ビューアー接続 -> fmdb 読み込み -> fsdb 読み込み
                            // B. fmdb 読み込み -> ビューアー接続 -> fsdb 読み込み
                            var needsReload = disorderedMaterials.Any(x => x.GetCombiners().Any());
                            commandSet.Add(ShaderAssignUtility.ExecuteFixParameters(disorderedMaterials, false, reload: needsReload));
                        }
                    }

                    // スケルタルアニメーションのバインドチェック
                    // モデル再読み込み時のチェックはここでは行わない。
                    {
                        var skeletalAnims = openedDocuments.OfType<SkeletalAnimation>().ToArray();
                        var skeletalAnimError = (from animation in skeletalAnims
                                                 from model in DocumentManager.Models
                                                 where model.AnimationSetsWithDefault.Any(x => x.Animations.Contains(new AnimationSetItem(animation.FileName, animation.FileLocation)))
                                                 let messages = animation.CheckBind(model).ToArray()
                                                 where messages.Any()
                                                 select new SkeletalAnimBindDialog.ErrorInfo()
                                                 {
                                                     animation = animation,
                                                     model = model,
                                                     messages = messages
                                                 }).ToArray();

                        if (skeletalAnimError.Any())
                        {
                            App.AppContext.NotificationHandler.SkeletalAnimBindDialog(skeletalAnimError, false);
                        }
                    }

                    // シェーダーパラメーターアニメーションの修正
                    {
                        var errorMaterialAnims = (from paramAnim in openedDocuments.OfType<IHasShaderParameterAnimation>()
                                                  from error in ShaderAssignUtility.CheckAnimation(paramAnim)
                                                  select error).ToArray();

                        if (errorMaterialAnims.Any())
                        {
//                            App.AppContext.NotificationHandler.FixShaderParamAnimDialog(errorMaterialAnims, commandSet);
                            var animErrors = App.AppContext.NotificationHandler.FixShaderParamAnimDialog(errorMaterialAnims);
                            if (animErrors != null)
                            {
                                commandSet.Add(ShaderAssignUtility.CreateFixShaderParamAnim(animErrors).Execute());
                            }
                        }
                    }

                    // スケルタルアニメーションの修正
                    {
                        var oldFormatAnims = (from anim in openedDocuments.OfType<SkeletalAnimation>()
                                              where RetargetingHostModelInToolData(anim)
                                              select anim).ToArray();
                        if (oldFormatAnims.Any())
                        {
                            App.AppContext.NotificationHandler.OkListBoxDialog(
                                MessageContext.IO_SkeletalAnimation_MoveRetagetingHostModelToProject,
                                Strings.IO_SkeletalAnimation_MoveRetagetingHostModelToProjectText,
                                Strings.IO_SkeletalAnimation_MoveRetagetingHostModelToProject,
                                oldFormatAnims.Select(x => x.FileName).ToArray());

                            foreach (var oldFormatAnim in oldFormatAnims)
                            {
                                commandSet.Add(CreateFixRetargetingHostModelInToolDataCommand(oldFormatAnim).Execute());
                            }
                        }
                    }

                    // コマンドスタックに追加
                    if (commandSet.CommandCount > 0 && stack)
                    {
                        commandSet.Reverse();
                        TheApp.CommandManager.Add(commandSet);
                    }
                }
            }

            if (executePreOpenRedoCommand)
            {
                var paths = executePreOpenCommand ?
                    openedDocuments1.Concat(openedDocuments3).Select(x => Tuple.Create(x, x.FilePath)).ToArray() :
                    openedDocuments3.Select(x => Tuple.Create(x, x.FilePath)).ToArray();
                if (paths.Any())
                {
                    commandSet.canRedo += () => PrePostIO.ExecutePreOpenUndoRedo(paths, false);
                }
            }

            if (isShowErrorDialog)
            {
                // 以下エラーメッセージ
                ShowErrorDialog(MessageContext.IO_Load_Failed, errors.Concat(errors3.Where(x => x.errorType != ErrorType.AlreadyOpened)), res.Strings.IO_Load_Failed);

                if (bindError.Any())
                {
                    App.AppContext.NotificationHandler.OkListBoxDialog(
                        MessageContext.IO_FailedToBindDialog,
                        Strings.IO_FailedToBindDialogText,
                        Strings.IO_FailedToBindDialogDescrioption,
                        bindError.Select(item => string.Format(Strings.IO_FailedToBindDialogMessage, item.Item2.Name, item.Item1.FileName))
                        );
                }
            }
        }

        public static bool CanAddAnimation(AnimationDocument animation, Model model)
        {
            if (model == null)
            {
                return true;
            }

            return model.AnimationSetsWithDefault.SelectMany(x => x.Animations).All(x =>
            {
                var ext = Path.GetExtension(x.Name);
                var id = ObjectIDUtility.ExtToId(ext);
                var name = Path.GetFileNameWithoutExtension(x.Name);
                return !(id == animation.ObjectID && String.Compare(name, animation.Name,StringComparison.OrdinalIgnoreCase)==0);
            });
        }

        /// <summary>
        /// ファイルを開きます。
        /// </summary>
        /// <param name="fileNames">開く対象</param>
        /// <param name="animationOwner">アニメーションのバインド先</param>
        /// <param name="showErrorDialog"></param>
        public static void LoadFromFile(IEnumerable<string> fileNames, AnimationSet animationOwner = null, bool showErrorDialog = true, EditCommandSet commandSet = null, Model animationOwnerModel = null, bool executePreOpenCommand = true)
        {
            LoadFromFile(fileNames.Select(x => new PathWithName(x, null)), animationOwner, isUpdateRecentlyUsedFiles: true, isShowErrorDialog:showErrorDialog, commandSet:commandSet, animationOwnerModel:animationOwnerModel, executePreOpenCommand:executePreOpenCommand);
        }

        /// <summary>
        /// ファイルを再読み込みします。
        /// </summary>
        public static EditCommandSet Reload(string filePath, Document document, bool ShowUpdateDialog, bool fromMenu = false, bool returnCommanSet = false)
        {
            DebugConsole.WriteLine("Reload");

            if (!DocumentManager.Documents.Contains(document))
            {
                // 不具合ではないかもしれないけど念のため
                // 不要ならアサートを削除
                Debug.Assert(false);

                document = DocumentManager.Documents.FirstOrDefault(x => x.FilePath == filePath);
                if (document != null)
                {
                    // リロードする機会を待つ間に差し替え
                    ShowUpdateDialog = true;
                }
                else
                {
                    // リロードする機会を待つ間に閉じられた
                    return null;
                }
            }

            if (document.ContentsModified)
            {
                if (fromMenu)
                {
                    if (!App.AppContext.NotificationHandler.MessageBoxYesNo(MessageContext.IO_ReloadFromMenuModified, Strings.IO_ReloadFromMenuModified, document.FileName))
                    {
                        return null;
                    }
                }
                else
                {
                    if (!App.AppContext.NotificationHandler.MessageBoxYesNo(MessageContext.IO_ReloadModified, Strings.IO_ReloadModified, document.FileName))
                    {
                        return null;
                    }
                }
            }
            else if (ShowUpdateDialog)
            {
                if (!App.AppContext.NotificationHandler.MessageBoxYesNo(MessageContext.IO_Reload, Strings.IO_Reload, document.FileName))
                {
                    return null;
                }
            }

            if (!fromMenu)
            {
                App.AppContext.NotificationHandler.WriteMessageLog(MessageLog.LogType.Information, string.Format(Strings.Document_ReloadOnFileUpdating, filePath));
            }

            var commandSet2 = new EditCommandSet();
            Document newDocument = null;
            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {

                // 再転送対象
                var reloadModels = new Model[0];
                var reloadAnimations = new AnimationDocument[0];

                // シェーダファイル再読み込み時に再バインドするアニメーション。
                var rebindAnimations = new AnimationDocument[0];

                // 最低限必要なリロードをcommandSet2 の OnPostEdit で行う
                var commandSet = new EditCommandSet();
                if (!(document is ProjectDocument))
                {
                    commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                }

                var vdsb = document is ProjectDocument ?
                    new Viewer.ViewerDrawSuppressBlock():
                    new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);
                using (vdsb)
                {
                    if (document is Texture)
                    {
                        // 再転送するドキュメント
                        // vdsb.Dispose() から DiscardAllMessages が実行されることで
                        // AddTextures() での再転送がキャンセルされるため、ここで再転送が必要。
                        // つまり、ここでの転送と AddTextures() での転送が二重に行われることはない。
                        reloadModels = DocumentManager.Models.Where(x => x.ReferenceDocuments.Contains(document)).ToArray();
                        reloadAnimations = DocumentManager.Animations.Where(x => x.ReferenceDocuments.Contains(document)).ToArray();
                    }
                    else if (document is ShaderDefinition && !((ShaderDefinition)document).IsAttached)
                    {
                        // 再転送するドキュメント
                        reloadModels = DocumentManager.Models.Where(x => x.ReferenceDocuments.Contains(document)).ToArray();

                        // シェーダの再読み込み時に再バインドするアニメーション。
                        rebindAnimations =
                            (from x in DocumentManager.Animations
                             from y in reloadModels
                             where y.ContainsAnimation(x)
                             select x).Distinct().ToArray();
                    }
                    else
                    {
                    }

                    if (ConfigData.ApplicationConfig.UserSetting.EditHistory.ClearAfterFileReloaded && !fromMenu)
                    {
                        TheApp.CommandManager.ClearNext();
                    }

                    // リロードだけでプロジェクトに変更通知がいかないようにする変更しないように
                    int originalDocumentCount = Documents.Count();
                    commandSet.Add(CreateAddOrRemoveDocumentCommand(Enumerable.Repeat(document, 1), false).Execute());
                    LoadFromFile(
                        Enumerable.Repeat(new PathWithName(document.FilePath, document.BaseName, document.BasePath), 1),
                        null,
                        document.OpenedFromStartUp, commandSet,
                        false,
                        isUpdateRecentlyUsedFiles: false,
                        reloadFromMenu:fromMenu,
                        executePreOpenCommand:false);

                    if (document is Model)
                    {
                        var original = (Model)document;
                        string name = Path.GetFileNameWithoutExtension(filePath);
                        var model = Models.FirstOrDefault(x => x.Name == name);
                        if (model != null)
                        {
                            newDocument = model;

                            // id の引き継ぎ
                            model.ModelId = original.ModelId;

                            // プレビュー情報の引き継ぎ
                            model.PreviewInfo = original.PreviewInfo;

                            // LOD プレビュー情報の引継ぎ
                            // ここで設定した LOD レベルは LoadOrReloadModel() で送られる。
                            if (model.PreviewLodLevel != original.PreviewLodLevel)
                            {
                                var itemCount = model.Shapes.Any() ? model.Shapes.Max(x => x.VertexCountInMesh.Length) : 0;
                                if (original.PreviewLodLevel < itemCount)
                                {
                                    model.PreviewLodLevel = original.PreviewLodLevel;
                                }
                            }

                            // 頂点量子化の引継ぎ
                            // ここで設定した値は LoadOrReloadModel() で送られる。
                            model.DisableVertexQuantize = original.DisableVertexQuantize;

                            // シェーダー最適化情報の引き継ぎ
                            model.lastMaterialOptimizeData = original.lastMaterialOptimizeData;
                            model.MaterialOptimizeDataCache = original.MaterialOptimizeDataCache;
                            model.pcFirstOptimizedData = original.pcFirstOptimizedData;
                            model.deviceFirstOptimizedData = original.deviceFirstOptimizedData;
                            model.OptimizedLogFolders = original.OptimizedLogFolders;

                            // マテリアルのプレビュー情報の引き継ぎ
                            foreach (var material in model.Materials)
                            {
                                var originalMaterial = original.Materials.FirstOrDefault(x => x.Name == material.Name);
                                if (originalMaterial != null)
                                {
                                    material.OptimizeShader = originalMaterial.OptimizeShader;
                                    material.OptimizeShaderOnReload = originalMaterial.OptimizeShaderOnReload;
                                }
                            }

                            // アニメーションの引き継ぎ
                            commandSet.Add(CreateCopyAnimationSetCommand(model, original).Execute());
                            if (model.AllAnimations.Any())
                            {
                                commandSet.Add(CreateAnimationUpdateBindCommand(model.AllAnimations).Execute());
                            }

                            // スケルタルアニメーションのバインドチェック
                            var skeletalAnims = model.AllAnimations.OfType<SkeletalAnimation>().ToArray();
                            var skeletalAnimError = (from animation in skeletalAnims
                                                     let messages = animation.CheckBind(model).ToArray()
                                                     where messages.Any()
                                                     select new SkeletalAnimBindDialog.ErrorInfo()
                                                     {
                                                         animation = animation,
                                                         model = model,
                                                         messages = messages
                                                     }).ToArray();

                            if (skeletalAnimError.Any())
                            {
                                App.AppContext.NotificationHandler.SkeletalAnimBindDialog(skeletalAnimError, false);
                            }
                        }
                    }
                    else if (document is AnimationDocument)
                    {
                        var original = (AnimationDocument)document;
                        AnimationDocument animation = Animations.FirstOrDefault(x => x.FileName == document.FileName &&
                            x.FileLocation == document.FileLocation &&
                            x.ObjectID == document.ObjectID);
                        if (animation != null)
                        {
                            newDocument = animation;
                            animation.Pause = ((AnimationDocument)document).Pause;
                            commandSet.Add(CreateAnimationUpdateBindCommand(Enumerable.Repeat((AnimationDocument)animation, 1)).Execute());

                            // アニメーションが空ならオリジナルアニメーションを適用
                            IHasShaderParameterAnimation shaderParamAnim = animation as IHasShaderParameterAnimation;
                            if (shaderParamAnim != null &&
                                !shaderParamAnim.HasAnyShaderParamCurves())
                            {
                                shaderParamAnim.ApplyOriginalAnimation(commandSet, false);
                            }

                            // リターゲット情報の引継ぎ
                            if (animation is SkeletalAnimation)
                            {
                                ((SkeletalAnimation)newDocument).RetargetingHostModelName = ((SkeletalAnimation)original).RetargetingHostModelName;
                            }
                        }
                    }
                    else if (Documents.Count() != originalDocumentCount)
                    {
                        commandSet.Add(CreateEditCommand_IO());
                    }

                    // アニメーションを再バインドする。
                    foreach (var animation in rebindAnimations)
                    {
                        commandSet.Add(CreateAnimationUpdateBindCommand(Enumerable.Repeat((AnimationDocument)animation, 1)).Execute());

                        // アニメーションが空ならオリジナルアニメーションを適用
                        IHasShaderParameterAnimation shaderParamAnim = animation as IHasShaderParameterAnimation;
                        if (shaderParamAnim != null &&
                            !shaderParamAnim.HasAnyShaderParamCurves())
                        {
                            shaderParamAnim.ApplyOriginalAnimation(commandSet, false);
                        }
                    }

                    commandSet.Add(CreateRemoveTextureReferencePaths().Execute());
                }

                if (reloadModels.Any() || reloadAnimations.Any() || document is Model || document is AnimationDocument || document is ShaderDefinition)
                {
                    // 再転送
                    var oldDocument = document;
                    EventHandler postEdit = (s, e) =>
                    {
                        Viewer.SuspendUnloadTexture.Send(true);
                        if (oldDocument is Model || oldDocument is AnimationDocument || oldDocument is ShaderDefinition)
                        {
                            Viewer.Close.Send(oldDocument);
                        }
                        if (newDocument is Model)
                        {
                            Viewer.LoadOrReloadModel.Send((Model)newDocument);
                        }
                        else if (newDocument is AnimationDocument)
                        {
                            Viewer.LoadOrReloadAnimation.Send((AnimationDocument)newDocument);
                        }
                        var tmp = newDocument;
                        newDocument = oldDocument;
                        oldDocument = tmp;

                        foreach (var model in reloadModels)
                        {
                            Viewer.LoadOrReloadModel.Send(model);
                        }

                        foreach (var animation in reloadAnimations)
                        {
                            Viewer.LoadOrReloadAnimation.Send(animation);
                        }

                        // モデルのアニメーションをバインド
                        foreach (var model in DocumentManager.Documents.OfType<Model>())
                        {
                            model.SendBindAnimations();
                        }

                        // シーンアニメーションをバインド
                        SendBindSceneAnimations();

                        Viewer.SuspendUnloadTexture.Send(false);
                    };

                    {
                        commandSet2.OnPostEdit += postEdit;
                    }

                    // アニメーションをバインドするためのブロック
                    using (new Viewer.ViewerDrawSuppressBlock())
                    {
                        postEdit(null, null);
                    }
                }

                commandSet.Reverse();
                commandSet2.Add(commandSet);
                if (!returnCommanSet)
                {
                    TheApp.CommandManager.Add(commandSet2);
                }
            }

            return commandSet2;
        }

        private static void ShowErrorDialog(MessageContext context, IEnumerable<Error> files, string message)
        {
            if (files.Any())
            {
                Func<string, string> fileName = x =>
                {
                    try
                    {
                        return Path.GetFileName(x);
                    }
                    catch
                    {
                        return "";
                    }
                };
                Func<string, string> fullPath = x =>
                {
                    try
                    {
                        if (Path.IsPathRooted(x))
                        {
                            return Path.GetFullPath(x);
                        }
                    }
                    catch
                    {
                    }
                    return x;
                };

                var errors = files.Select(x => new Tuple<string, string, string>(x.message, fileName(x.filePath), fullPath(x.filePath)));

                App.AppContext.NotificationHandler.FileErrorDialog(context, errors, message);
            }
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 対象モデルの選択。
        /// </summary>
        public static Model SelectTargetModel(AnimationDocument animation, App.AppContext.PropertyChangedSuppressBlock block, ref bool applyAll)
        {
            if (ModelCount == 0) { return null; }

            // モデルが一つしかロードされていなければそれ
            if (ModelCount == 1) { return Models.First(); }

            // 元々ここで通知していたが、ファイルオープン時にこの関数が実行されると
            // 通知によって App.AppContext.DocumentAddedOrRemoved イベントが発生する。
            // このイベント内では Model.SendEditBoneBind() と Model.SendEditModelLayout() を呼び出しており
            // アニメーションをバインドしていない状態での呼び出しとなってしまうため不適切。
            // ここで通知していた理由が不明なので、通知も残しておき、block が null であれば通知しないようにしておく。
            if (block != null)
            {
                block.Release();
            }

            // ダイアログで選択する
            return App.AppContext.NotificationHandler.ModelSelectDialog(Models, animation, Strings.DocumentManager_SelectTargetModel_ApplyAllAnim, ref applyAll);
        }

        private static Model SearchModelForAnimations(AnimationDocument animation, App.AppContext.PropertyChangedSuppressBlock block, out bool applyAll)
        {
            Model ownerModel = null;
            applyAll = false;

            if (ModelCount == 1)
            {
                ownerModel = Models.First();
            }
            else
            {
                // ターゲットモデルをパスから推測する
                List<Model> candidates = new List<Model>();
                {
                    string animationDirectory = Path.GetDirectoryName(animation.FilePath);
                    animationDirectory += Path.DirectorySeparatorChar;
                    foreach (Model model in Models)
                    {
                        if (string.IsNullOrEmpty(model.FilePath))
                        {
                            continue;
                        }

                        // モデルがアニメーションの上位ディレクトリにあれば候補に追加
                        // 上位ディレクトリに拡張する場合はモデルディレクトリにセパレータを付加する
                        string modelDirectory = Path.GetDirectoryName(model.FilePath);
                        modelDirectory += Path.DirectorySeparatorChar;
                        if (animationDirectory.StartsWith(modelDirectory))
                        {
                            candidates.Add(model);
                        }
                    }
                }

                if (candidates.Count == 1)
                {
                    // 候補モデルがひとつなら確定
                    ownerModel = candidates[0];
                }
                else if (candidates.Count > 1)
                {
                    // 候補モデルが複数ならファイル名の前方一致が必要
                    ownerModel = SearchModel(animation.FilePath, candidates);
                }

                if (ownerModel == null)
                {
                    // 候補モデルが無ければ読み込んだモデルからファイル名の前方一致を探す
                    ownerModel = SearchModel(animation.FilePath, Models) ?? SelectTargetModel(animation, block, ref applyAll);
                }
            }

            return ownerModel;
        }

        /// <summary>
        /// パスによるモデル検索
        /// </summary>
        private static Model SearchModel(string animationPath, IEnumerable<Model> models)
        {
            Model targetModel = null;
            string animationName = Path.GetFileNameWithoutExtension(animationPath);
            foreach (Model model in models)
            {
                string modelName = model.Name;
                if (animationName.StartsWith(modelName))
                {
                    // 一番長いモデル名を選択する
                    if ((targetModel == null) ||
                        (modelName.Length > targetModel.Name.Length))
                    {
                        targetModel = model;
                    }
                }
            }
            return targetModel;
        }

        public static bool ExistsFileName(string fileName, out bool sameFile)
        {
            sameFile = false;
            if (string.IsNullOrEmpty(fileName))
            {
                return false;
            }
            var name = Path.GetFileNameWithoutExtension(fileName);
            var ext = Path.GetExtension(fileName);
            var id = ObjectIDUtility.ExtToId(ext);
            if (id == null)
            {
                if (ext == null || ObjectIDUtility.IsTgaExtension(ext)== false) return false;
                id = GuiObjectID.Texture;
            }

            if (Objects(id.Value).OfType<IntermediateFileDocument>().Any(x => !string.IsNullOrEmpty(x.FilePath) &&
                Path.GetFullPath(x.FilePath).Equals(Path.GetFullPath(fileName), StringComparison.OrdinalIgnoreCase)))
            {
                sameFile = true;
                return true;
            }

            if (!ObjectIDUtility.CanOpenSameNameFile(ext) && Objects(id.Value).Any(x => string.Compare(x.Name, name, true) == 0))
            {
                return true;
            }

            return false;
        }

        public class Error
        {
            public Error(string path, ErrorType type, string message)
            {
                filePath = path;
                errorType = type;
                this.message = message;
            }
            public string filePath;
            public ErrorType errorType;
            public string message;
        }

        public enum ErrorType
        {
            AlreadyOpened,
            FileNotExists,
            BasePathNotExists,
            SerializeFailed,
            NotSupportExtension,
            UnkownError,
            PreCommandFailed,
            Other,
        }

        /// <summary>
        /// ファイルを開きます。
        /// </summary>
        /// <param name="LoadedDocuments">この関数中で開かれたファイル</param>
        /// <param name="fileNames">開くファイル</param>
        /// <param name="errors">読み込みエラー</param>
        /// <param name="startup">スタートアップかどうか</param>
        /// <param name="commandSet"></param>
        public static void ExecuteLoadFromFileCommand(
            List<Document> LoadedDocuments,
            //
            IEnumerable<PathWithName> fileNames,
            out List<Error> errors,
            bool startup,
            EditCommandSet commandSet,
            bool executePreCommand,
            HashSet<string> excludePreCommandPaths = null
        )
        {
            // エラーチェック
            errors = new List<Error>();
            PathWithName[] names = fileNames.ToArray();
            var namesGroup = names.GroupBy(
                x =>
                {
                    var ext = Path.GetExtension(x.path);
                    if (!string.IsNullOrEmpty(ext)) ext = ext.ToLower();
                    var id = ObjectIDUtility.IsTgaExtension(ext) ? GuiObjectID.Texture : ObjectIDUtility.ExtToId(Path.GetExtension(ext));
                    var idstr = ObjectIDUtility.CanOpenSameNameFile(ext) ? Path.GetFullPath(x.path).ToLower() : Path.GetFileNameWithoutExtension(x.path).ToLower();
                    return new Tuple<string, GuiObjectID?>(idstr, id);
                }).ToArray();

            Document[] documents = Enumerable.Repeat<Document>(null, namesGroup.Length).ToArray();
            List<Error>[] errorLists = documents.Select(x => new List<Error>()).ToArray();

            var preCommandPaths = new HashSet<string>();
            bool preCommandResult = false;
            {
                if (executePreCommand)
                {
                    var paths = new List<string>();
                    foreach (var group in namesGroup)
                    {
                        if (group.Key.Item2 == null)
                        {
                            continue;
                        }
                        var first = group.First();
                        if (group.Key.Item2.HasValue && group.Key.Item2.Value == GuiObjectID.Texture)
                        {
                            first = group.OrderBy(x => ObjectIDUtility.IsTgaExtension(Path.GetExtension(x.path)) ? 1 : 0).First();
                        }
                        if (excludePreCommandPaths != null && excludePreCommandPaths.Contains(Path.GetFullPath(first.path).ToLower()))
                        {
                            continue;
                        }
                        bool sameFile;
                        if (!ExistsFileName(first.path, out sameFile))
                        {
                            paths.Add(first.path);
                            preCommandPaths.Add(first.path);
                        }
                    }

                    if (paths.Any())
                    {
                        preCommandResult = PrePostIO.ExecutePreOpen(paths.ToArray());
                    }
                }
            }

            var updatedFileNames = new string[namesGroup.Length];

            var preCommandLock = new object();
            var actions = new List<Action>();
            G3dParallel.For(0, namesGroup.Length, i =>
                {
                    Document document = null;
                    var group = namesGroup[i].ToArray();
                    if (namesGroup[i].Key.Item2 == GuiObjectID.Texture)
                    {
                        group = namesGroup[i].OrderBy(x => ObjectIDUtility.IsTgaExtension(Path.GetExtension(x.path)) ? 1 : 0).ToArray();
                    }

                    foreach (var pathWithName in group)
                    {
                        Error error = null;
                        if (document != null)
                        {
                            error = new Error(pathWithName.path, ErrorType.AlreadyOpened, res.Strings.IO_Load_AlreadyOpened);
                        }
                        else
                        {
                            if (executePreCommand)
                            {
                                lock (preCommandLock)
                                {
                                    if (preCommandPaths.Contains(pathWithName.path))
                                    {
                                        if (!preCommandResult)
                                        {
                                            error = new Error(pathWithName.path, ErrorType.PreCommandFailed, res.Strings.IO_PreOpenCommandFailedError);
                                        }
                                    }
                                    else
                                    {
                                        bool sameFile;
                                        if (!ExistsFileName(pathWithName.path, out sameFile))
                                        {
                                            Action action;
                                            if ((excludePreCommandPaths == null || !excludePreCommandPaths.Contains(Path.GetFullPath(pathWithName.path).ToLower())) && !PrePostIO.ExecutePreOpen(new[] { pathWithName.path }, out action))
                                            {
                                                error = new Error(pathWithName.path, ErrorType.PreCommandFailed, res.Strings.IO_PreOpenCommandFailedError);
                                                actions.Add(action);
                                            }
                                        }
                                    }
                                }
                            }

                            if (error == null)
                            {
                                var name = pathWithName;
                                Read(pathWithName, out document, out error, true, () => updatedFileNames[i] = name.path);
                                if (error != null)
                                {
                                    document = null;
                                }
                            }
                        }
                        if (error != null)
                        {
                            errorLists[i].Add(error);
                        }
                    }

                    documents[i] = document;
                }
            );

            // PreOpenCommand 失敗通知
            foreach (var action in actions)
            {
                action();
            }

            // アップデートされたファイルをログバーにメッセージとして表示する
            foreach (var updatedFileName in updatedFileNames.Where(x => x != null))
            {
                var normalizePathString = updatedFileName;
                PathUtility.NormalizePathString(updatedFileName, ref normalizePathString);

                App.AppContext.NotificationHandler.WriteMessageLog(MessageLog.LogType.Information, string.Format(res.Strings.IfUpdated, normalizePathString));
            }

            foreach (var document in documents.Where(x => x != null))
            {
                document.OpenedFromStartUp = startup;
                LoadedDocuments.Add(document);

                // セーブデータを格納
                DocumentManager.AddSaveDocument(document);
            }

            if (documents.Any(x => x != null))
            {
                commandSet.Add(CreateAddOrRemoveDocumentCommand(documents.Where(x => x != null), true).Execute());
            }

            errors.AddRange(errorLists.SelectMany(x => x));
        }

        public static bool Read(PathWithName filePath, out Document document)
        {
            Error error = null;

            Read(filePath, out document, out error, false);

            if (error != null)
            {
                document = null;
                return false;
            }
            else
            {
                return true;
            }
        }

        /// <summary>
        /// ファイルを読み込みます。
        /// </summary>
        /// <remarks>
        /// 並列実行可能です。
        /// </remarks>
        private static void Read(PathWithName filePath, out Document document, out Error error, bool isCheckAlreadyOpened, Action onIfUpdate = null)
        {
            document = null;
            error = null;

            var fileName = filePath.path;
            var isTgaFile = false;

            // 対応している拡張子か？
            ObjectIDUtility.UnsupportedType unsupportedType;

            var ext = Path.GetExtension(fileName);
            if (fileName == null)
            {
                return;
            }

            ext= ext.ToLower();
            if (ObjectIDUtility.IsTgaExtension(ext))
            {
                isTgaFile = true;
            }
            else if (ObjectIDUtility.IsSupportExtension(ext, out unsupportedType) == false)
            {
                switch(unsupportedType)
                {
                    case ObjectIDUtility.UnsupportedType.ShaderDefinitionTextExtension:
                    {
                        error = new Error(fileName, ErrorType.NotSupportExtension, res.Strings.IO_Load_NotSupportedExtension_ShaderDefinitionText);
                        break;
                    }

                    default:
                    {
                        error = new Error(fileName, ErrorType.NotSupportExtension, res.Strings.IO_Load_NotSupportedExtension);
                        break;
                    }
                }
                return;
            }

            // 存在確認
            bool sameFile;
            if (isCheckAlreadyOpened && ExistsFileName(fileName, out sameFile))
            {
                error = new Error(fileName,
                    ErrorType.AlreadyOpened,
                    sameFile? res.Strings.IO_Load_SameFile: res.Strings.IO_Load_AlreadyOpened);
                return;
            }

            if (!File.Exists(fileName))
            {
                error = new Error(fileName, ErrorType.FileNotExists, res.Strings.IO_Load_FileNotExists);
                return;
            }

            using (var watch = new DebugStopWatch(string.Format("Load {0}", fileName)))
            {
                var isIfUpdated = false;
                nw4f_3difType nw4f_3dif = null;

                if (ObjectIDUtility.IsTgaExtension(ext))
                {
                    using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                    using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
                    {
                        lock (Texture.TexConvertLock)
                        {
                            byte[] convertedDataArray = null;
                            {
                                // 一旦ファイルにする
                                using (var dstTempFileName = TemporaryFileUtility.MakeDisposableFileName(".current" + G3dPath.TextureBinaryExtension))
                                {
                                    // ファイルとフォーマットの指定
                                    {
                                        TexcvtrManager.Clear();

                                        var isSuccess = TexcvtrManager.ReadInputFile(
                                            new[] { fileName, null },
                                            new string[] {
                                                null
        							        }
                                        );

                                        if (!isSuccess)
                                        {
                                            var err = TexcvtrManager.GetError();
                                            TexcvtrManager.Clear();
                                            throw new Exception(err);
                                        }
                                    }

                                    // コンバートする
                                    var convertedData = IntPtr.Zero;
                                    int convertedDataSize = 0;
                                    {
                                        bool isSuccess = TexcvtrManager.ConvertToData(ref convertedData, ref convertedDataSize);
                                        if (!isSuccess)
                                        {
                                            var err = TexcvtrManager.GetError();
                                            TexcvtrManager.Clear();
                                            throw new Exception(err);
                                        }
                                    }

                                    // マネージドなところに持ってくる
                                    convertedDataArray = new byte[convertedDataSize];
                                    Marshal.Copy(convertedData, convertedDataArray, 0, convertedDataSize);

                                    TexcvtrManager.Clear();
                                }
                            }

                            // document を作成する
                            var streamArray = new List<G3dStream>();
                            try
                            {
                                nw4f_3dif = IfBinaryReadUtility.Read(streamArray, convertedDataArray, XsdBasePath, out isIfUpdated);
                            }
                            catch (Exception e)
                            {
                                var builder = new StringBuilder();
                                builder.Append(res.Strings.IO_Load_Error);

                                // 内部例外をさかのぼる
                                var ex = e;
                                while (ex != null)
                                {
#if DEBUG
                                    builder.Append(ex.ToString());
#else
                                    if (!string.IsNullOrEmpty(ex.Message))
                                    {
                                        builder.Append(ex.Message);
                                    }
#endif
                                    ex = ex.InnerException;
                                }

                                error = new Error(fileName, ErrorType.SerializeFailed, builder.ToString());
                                return;
                            }
                            if (nw4f_3dif.Item is textureType)
                            {
                                document = new Texture(nw4f_3dif.Item as textureType, streamArray);
                                document.IsTemporary = isTgaFile;
                            }

                        }
                    }
                }

                // 中間ファイル。 TGAは読み込み済み
                else if (document == null)
                {
                    List<G3dStream> streamArray = null;
                    try
                    {
                        if (G3dPath.IsStreamBinaryPath(fileName))
                        {
                            streamArray = new List<G3dStream>();
                            nw4f_3dif = IfBinaryReadUtility.Read(streamArray, fileName, XsdBasePath, out isIfUpdated);
                        }
                        else
                        {
                            nw4f_3dif = IfReadUtility.Read(fileName, XsdBasePath, out isIfUpdated);
                        }

                        if (isIfUpdated)
                        {
                            if (onIfUpdate != null)
                            {
                                onIfUpdate();
                            }
                            else
                            {
                                // デフォルト動作として、ログバーに情報を出力する
                                var normalizePathString = fileName;
                                PathUtility.NormalizePathString(fileName, ref normalizePathString);

                                App.AppContext.NotificationHandler.WriteMessageLog(MessageLog.LogType.Information, string.Format(res.Strings.IfUpdated, normalizePathString));
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        var builder = new StringBuilder();
                        builder.Append(res.Strings.IO_Load_Error);

                        // 内部例外をさかのぼる
                        Exception ex = e;
                        while (ex != null)
                        {
#if DEBUG
                            builder.Append(ex.ToString());
#else
                            if (!string.IsNullOrEmpty(ex.Message))
                            {
                                builder.Append(ex.Message);
                            }
#endif
                            ex = ex.InnerException;
                        }

                        error = new Error(fileName, ErrorType.SerializeFailed, builder.ToString());
                        return;
                    }

                    document = SetupDocument(nw4f_3dif, streamArray, fileName, isTgaFile);

                    var intermediateFile = document as IntermediateFileDocument;
                    if (intermediateFile != null)
                    {
                        intermediateFile.file_info = nw4f_3dif.file_info;
                    }
                }

                if (document == null)
                {
                    error = new Error(fileName, ErrorType.UnkownError, res.Strings.IO_Load_Error);
                    return;
                }

                try
                {
                    document.Name = Path.GetFileNameWithoutExtension(fileName);
                    document.FileDotExt = Path.GetExtension(fileName);
                    document.SetFileLocation(Path.GetDirectoryName(fileName), filePath.baseName, filePath.basePath);
                    document.UpdateSavedData();
                    document.SetContentsNotModified();
                    document.IsIfUpdated = isIfUpdated;

                    watch.SetMessage(string.Format("Load {0}", document.FilePath));
                }
                catch (Exception)
                {
                    error = new Error(fileName, ErrorType.UnkownError, res.Strings.IO_Load_Error);
                    return;
                }
            }
        }

        public static Document SetupDocument(nw4f_3difType nw4f_3dif, List<G3dStream> streamArray, string fileName,
            bool isTgaFile = false)
        {
            Document document = null;
            if (nw4f_3dif.Item is modelType)
            {
                document = new Model(nw4f_3dif.Item as modelType, streamArray);
            }
            else if (nw4f_3dif.Item is textureType)
            {
                document = new Texture(nw4f_3dif.Item as textureType, streamArray);
                document.IsTemporary = isTgaFile;
                //document.OriginalFileName = string.IsNullOrEmpty(sourceFileName) ? string.Empty : sourceFileName;
            }
            else if (nw4f_3dif.Item is skeletal_animType)
            {
                document = new SkeletalAnimation(nw4f_3dif.Item as skeletal_animType, streamArray);
            }
            else if (nw4f_3dif.Item is shape_animType)
            {
                document = new ShapeAnimation(nw4f_3dif.Item as shape_animType, streamArray);
            }
            else if (nw4f_3dif.Item is material_animType)
            {
                document = new MaterialAnimation(nw4f_3dif.Item as material_animType, streamArray);
            }
            else if (nw4f_3dif.Item is shader_param_animType)
            {
                document = G3dPath.IsColorAnimPath(fileName)
                    ? new ColorAnimation(nw4f_3dif.Item as shader_param_animType, streamArray)
                    : G3dPath.IsTexSrtAnimPath(fileName)
                        ? new TextureSrtAnimation(nw4f_3dif.Item as shader_param_animType, streamArray)
                        : G3dPath.IsShaderParamAnimPath(fileName)
                            ? new ShaderParameterAnimation(nw4f_3dif.Item as shader_param_animType, streamArray)
                            : null;
                Debug.Assert(document != null);
            }
            else if (nw4f_3dif.Item is mat_visibility_animType)
            {
                document = new MaterialVisibilityAnimation(nw4f_3dif.Item as mat_visibility_animType, streamArray);
            }
            else if (nw4f_3dif.Item is bone_visibility_animType)
            {
                document = new BoneVisibilityAnimation(nw4f_3dif.Item as bone_visibility_animType, streamArray);
            }
            else if (nw4f_3dif.Item is tex_pattern_animType)
            {
                document = new TexturePatternAnimation(nw4f_3dif.Item as tex_pattern_animType, streamArray);
            }
            else if (nw4f_3dif.Item is scene_animType)
            {
                document = new SceneAnimation(nw4f_3dif.Item as scene_animType, streamArray);
            }
            else if (nw4f_3dif.Item is shader_definitionType)
            {
                document = new ShaderDefinition((shader_definitionType)nw4f_3dif.Item, streamArray);
            }
            else if (nw4f_3dif.Item is materialType)
            {
                document = new SeparateMaterial((materialType)nw4f_3dif.Item, streamArray);
            }
            else
            {
                Debug.Assert(false);
            }
            return document;
        }

        public static IntermediateFileDocument Construct(string filePath, nw4f_3difType nw4f_3dif, List<G3dStream> streamArray)
        {
            IntermediateFileDocument document = null;
            if (nw4f_3dif.Item is modelType)
            {
                document = new Model(nw4f_3dif.Item as modelType, streamArray);
            }
            else if (nw4f_3dif.Item is textureType)
            {
                document = new Texture(nw4f_3dif.Item as textureType, streamArray);
            }
            else if (nw4f_3dif.Item is skeletal_animType)
            {
                document = new SkeletalAnimation(nw4f_3dif.Item as skeletal_animType, streamArray);
            }
            else if (nw4f_3dif.Item is shape_animType)
            {
                document = new ShapeAnimation(nw4f_3dif.Item as shape_animType, streamArray);
            }
            else if (nw4f_3dif.Item is shader_param_animType)
            {
                document = G3dPath.IsColorAnimPath(filePath) ? new ColorAnimation(nw4f_3dif.Item as shader_param_animType, streamArray) :
                            G3dPath.IsTexSrtAnimPath(filePath) ? new TextureSrtAnimation(nw4f_3dif.Item as shader_param_animType, streamArray) :
                            G3dPath.IsShaderParamAnimPath(filePath) ? new ShaderParameterAnimation(nw4f_3dif.Item as shader_param_animType, streamArray) :
                                                                          null;
                Debug.Assert(document != null);
            }
            else if (nw4f_3dif.Item is mat_visibility_animType)
            {
                document = new MaterialVisibilityAnimation(nw4f_3dif.Item as mat_visibility_animType, streamArray);
            }
            else if (nw4f_3dif.Item is bone_visibility_animType)
            {
                document = new BoneVisibilityAnimation(nw4f_3dif.Item as bone_visibility_animType, streamArray);
            }
            else if (nw4f_3dif.Item is tex_pattern_animType)
            {
                document = new TexturePatternAnimation(nw4f_3dif.Item as tex_pattern_animType, streamArray);
            }
            else if (nw4f_3dif.Item is scene_animType)
            {
                document = new SceneAnimation(nw4f_3dif.Item as scene_animType, streamArray);
            }
            else if (nw4f_3dif.Item is shader_definitionType)
            {
                document = new ShaderDefinition((shader_definitionType)nw4f_3dif.Item, streamArray);
            }
            else if (nw4f_3dif.Item is material_animType)
            {
                document = new MaterialAnimation(nw4f_3dif.Item as material_animType, streamArray);
            }

            if (document != null)
            {
                document.file_info = nw4f_3dif.file_info;
                document.Name = Path.GetFileNameWithoutExtension(filePath);
                document.FileDotExt = Path.GetExtension(filePath);

                document.UpdateSavedData();
                document.SetContentsNotModified();
            }

            return document;
        }
        private static void AddDocument(Document document)
        {
            switch (document.ObjectID)
            {
                case GuiObjectID.Model:
                    AddModel(document as Model);
                    break;

                case GuiObjectID.Texture:
                    AddTexture(document as Texture);
                    break;

                case GuiObjectID.SkeletalAnimation:
                case GuiObjectID.ShapeAnimation:
                case GuiObjectID.MaterialAnimation:
                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                case GuiObjectID.MaterialVisibilityAnimation:
                case GuiObjectID.BoneVisibilityAnimation:
                case GuiObjectID.TexturePatternAnimation:
                case GuiObjectID.SceneAnimation:
                    AddAnimation(document as AnimationDocument);
                    break;

                case GuiObjectID.SeparateMaterial:
                    AddSeparateMaterial(document as SeparateMaterial);
                    break;

                case GuiObjectID.ShaderDefinition:
                    AddShaderDefinition(document as ShaderDefinition);
                    break;
            }
        }

        public static IEnumerable<PathWithName> MakeReferenceShaderDefinitionFilename(Document document)
        {
            var pathWithNames = Enumerable.Empty<PathWithName>();
            if (document.ObjectID == GuiObjectID.Model)
            {
                var model = (Model)document;
                if (model.Data.material_array != null)
                {
                    foreach (var material in model.Materials)
                    {
                        if (!string.IsNullOrEmpty(material.MaterialShaderAssign.ShaderDefinitionFileName))
                        {
                            if (ShaderDefinitions.Any(x => x.Name == material.MaterialShaderAssign.ShaderDefinitionFileName) == false)
                            {
                                pathWithNames = pathWithNames.Concat(MakeReferenceShaderFilenames(model.FileLocation, model, model.BaseName,
                                                                                       material.MaterialShaderAssign.ShaderDefinitionFileName));
                            }
                        }
                    }
                }
            }

            return pathWithNames.Distinct();
        }


        /// <summary>
        /// オープンするファイルの候補
        /// </summary>
        /// <param name="documents"></param>
        /// <param name="specificFolder"></param>
        /// <returns></returns>
        public static IEnumerable<PathWithName> MakeReferenceTextureFilename(
            Document document)
        {
            var pathWithNames = Enumerable.Empty<PathWithName>();
            switch (document.ObjectID)
            {
                case GuiObjectID.Model:
                    {
                        var model = document as Model;

                        if (model.Data.material_array != null)
                        {
                            foreach (var material in model.Materials)
                            {
                                if (material.sampler_array != null)
                                {
                                    foreach (var sampler in material.sampler_array.sampler)
                                    {
                                        pathWithNames = pathWithNames.Concat(MakeReferenceTextureFilenames(
                                            document.FileLocation,
                                            model,
                                            document.BaseName,
                                            sampler.tex_name).Take(1));
                                    }
                                }
                            }
                        }
                        break;
                    }

                case GuiObjectID.TexturePatternAnimation:
                    {
                        var texPatternAnim = document as TexturePatternAnimation;

                        Model ownerModel = Models.FirstOrDefault(x => x.ContainsAnimation(texPatternAnim));

                        string ownerLocation = null;
                        string ownerBaseName = null;
                        if (ownerModel != null)
                        {
                            ownerLocation = ownerModel.FileLocation;
                            ownerBaseName = ownerModel.BaseName;
                        }

                        foreach (var texPatterns in texPatternAnim.TexPatterns)
                        {
                            pathWithNames = pathWithNames.Concat(MakeReferenceTextureFilenames(
                                document.FileLocation,
                                texPatternAnim,
                                document.BaseName,
                                texPatterns.tex_name,
                                ownerLocation,
                                ownerModel,
                                ownerBaseName).Take(1));
                        }

                        break;
                    }
                case GuiObjectID.MaterialAnimation:
                    {
                        var texPatternAnim = document as MaterialAnimation;

                        Model ownerModel = Models.FirstOrDefault(x => x.ContainsAnimation(texPatternAnim));

                        string ownerLocation = null;
                        string ownerBaseName = null;
                        if (ownerModel != null)
                        {
                            ownerLocation = ownerModel.FileLocation;
                            ownerBaseName = ownerModel.BaseName;
                        }

                        foreach (var texPatterns in texPatternAnim.TexPatterns)
                        {
                            pathWithNames = pathWithNames.Concat(MakeReferenceTextureFilenames(
                                document.FileLocation,
                                texPatternAnim,
                                document.BaseName,
                                texPatterns.tex_name,
                                ownerLocation,
                                ownerModel,
                                ownerBaseName).Take(1));
                        }

                        break;
                    }
                case GuiObjectID.SeparateMaterial:
                    {
                        var materialDoc = document as SeparateMaterial;

                        if (materialDoc.Data?.sampler_array?.sampler != null)
                        {
                            foreach (var sampler in materialDoc.Data.sampler_array.sampler)
                            {
                                pathWithNames = pathWithNames.Concat(MakeReferenceTextureFilenames(
                                    document.FileLocation,
                                    materialDoc,
                                    document.BaseName,
                                    sampler.tex_name).Take(1));
                            }
                        }
                        break;
                    }
            }

            return pathWithNames.Distinct(new PathWithNameComparer());
        }

        public static IEnumerable<PathWithName> MakeReferenceTextureFilenames(
            string fileLocation,
            IntermediateFileDocument document,
            string baseName,
            string textureName,
            string ownerLocation = null,
            Model ownerModel = null,
            string ownerBaseName = null,
            Dictionary<string, string> additionalLowerCaseFilePaths = null)
        {
            var textures = new List<PathWithName>();
            {
                string[] fileNames = { textureName + ".ftxb", textureName + ".ftxa" };
                IEnumerable<PathWithName> folders;

                // 参照元ファイルが含まれるフォルダ
                folders = Enumerable.Repeat(new PathWithName(fileLocation, baseName), 1);

                // 参照元の中間ファイルに指定された追加参照パスのフォルダ
                if (document != null)
                {
                    folders = folders.Concat(ExtractSearchPaths(fileLocation, document.SearchPaths, baseName));
                }

                // バインドされたモデルに対して指定された追加参照パスのフォルダ
                if (!string.IsNullOrEmpty(ownerLocation))
                {
                    folders = folders.Concat(SearchPathsInModel(ownerLocation, ownerModel, ownerBaseName));
                }

                // 追加参照パスで指定されたフォルダやサブフォルダ
                folders = folders.Concat(SearchPathsInConfig(fileLocation, baseName));

                // textures フォルダ
                folders = folders.Concat(
                    Enumerable.Repeat(new PathWithName(Path.Combine(fileLocation, Const.TexturesDirectoryName), baseName), 1));

                // バインドされたモデルに対する textures フォルダ
                if (!string.IsNullOrEmpty(ownerLocation))
                {
                    folders = folders.Concat(
                        Enumerable.Repeat(new PathWithName(Path.Combine(ownerLocation, Const.TexturesDirectoryName), ownerBaseName), 1));
                }

                foreach (var folder in folders)
                {
                    foreach (var fileName in fileNames)
                    {
                        string path = Path.Combine(folder.path, fileName);
                        string correctPath;
                        if (PathUtility.ExistsFileName(path, out correctPath))
                        {
                            yield return new PathWithName(correctPath, folder.baseName, folder.basePath);
                        }
                        else if (additionalLowerCaseFilePaths != null)
                        {
                            string name;
                            if (additionalLowerCaseFilePaths.TryGetValue(Path.GetFullPath(path).ToLower(), out name))
                            {
                                // 名前は完全一位する必要がある。
                                if (name == textureName)
                                {
                                    yield return new PathWithName(Path.GetFullPath(path));
                                }
                            }
                        }
                    }
                }
            }
        }

        private static IEnumerable<PathWithName> MakeReferenceShaderFilenames(string fileLocation, Model model, string baseName, string shaderName)
        {
            var shaderDefinitions = new List<PathWithName>();
            {
                string[] fileNames = { shaderName + ".fsdb"};
                IEnumerable<PathWithName> folders;

                // 参照元ファイルが含まれるフォルダ
                folders = Enumerable.Repeat(new PathWithName(fileLocation, baseName), 1);

                // 参照元の中間ファイルに対して指定された追加参照パスのフォルダ
                if (model != null)
                {
                    folders = folders.Concat(SearchPathsInModel(fileLocation, model, baseName));
                }

                // 追加参照パスで指定されたフォルダやサブフォルダ
                folders = folders.Concat(SearchPathsInConfig(fileLocation, baseName));

                foreach (var folder in folders)
                {
                    foreach (var fileName in fileNames)
                    {
                        string path = Path.Combine(folder.path, fileName);
                        string correctPath;
                        if (PathUtility.ExistsFileName(path, out correctPath))
                        {
                            yield return new PathWithName(correctPath, folder.baseName, folder.basePath);
                        }
                    }
                }
            }
        }

        #endregion

        public static GroupEditCommand CreateEditCommand_IO()
        {
            GuiObjectGroup targets = new GuiObjectGroup(ProjectDocument);
            return
                 new GeneralGroupValueEditCommand<int>(
                    targets,
                    GuiObjectID.Project,
                    0,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        // 何もしません
                    }
                );
        }

        public class PathWithName
        {
            public PathWithName(string path, string baseName = null, string basePath = null)
            {
                this.path = path;
                this.baseName = baseName;
                this.basePath = basePath;
            }

            // パス
            public string path;

            // コンフィグの追加参照パス名
            public string baseName;

            // モデルの起点(環境変数未展開で)
            public string basePath;
        }

        public class PathWithNameComparer : IEqualityComparer<PathWithName>
        {
            public bool Equals(PathWithName a, PathWithName b)
            {
                return a.path == b.path && a.baseName == b.baseName && a.basePath == b.basePath;
            }

            public int GetHashCode(PathWithName a)
            {
                // path だけで十分そう
                return a.path.GetHashCode();
            }
        }

        private static IEnumerable<PathWithName> SearchPathsInConfig(string fileLocation, string baseName0)
        {
            foreach (var searchPath in ConfigData.ApplicationConfig.GetSearchPaths())
            {
                string rootPath = searchPath.path;
                string baseName = baseName0;
                if (!string.IsNullOrEmpty(fileLocation))
                {
                    if (Path.IsPathRooted(rootPath))
                    {
                        baseName = searchPath.Name;
                    }
                    rootPath = Path.Combine(fileLocation, rootPath);
                }
                yield return new PathWithName(rootPath, baseName);
                if (searchPath.Recursive && Directory.Exists(rootPath))
                {
                    List<string> paths = null;
                    try
                    {
                        paths = new List<string>(Directory.GetDirectories(rootPath, "*", SearchOption.AllDirectories));
                    }
                    catch (Exception)
                    {
                    }

                    if (paths != null)
                    {
                        foreach (var p in paths)
                        {
                            yield return new PathWithName(p, baseName);
                        }
                    }
                }
            }
        }

        private static IEnumerable<PathWithName> SearchPathsInModel(string fileLocation, Model model, string baseName)
        {
            return ExtractSearchPaths(fileLocation, model.SearchPaths, baseName);
        }

        private static IEnumerable<PathWithName> ExtractSearchPaths(string fileLocation, IEnumerable<nw3de_SearchPath.SearchPath> searchPaths, string baseName)
        {
            foreach (var searchPath in searchPaths)
            {
                var rootPath = Environment.ExpandEnvironmentVariables(searchPath.Path);
                string basePath = null;
                try
                {
                    bool expanded = rootPath != searchPath.Path;
                    bool isRooted = Path.IsPathRooted(rootPath);
                    if (!isRooted)
                    {
                        rootPath = Path.Combine(fileLocation, rootPath);
                    }
                    else if (expanded)
                    {
                        basePath = searchPath.Path;
                    }
                }
                catch
                {
                    continue;
                }
                yield return new PathWithName(rootPath, baseName, basePath);
                if (searchPath.Recursive && Directory.Exists(rootPath))
                {
                    string[] paths = null;
                    try
                    {
                        paths = Directory.GetDirectories(rootPath, "*", SearchOption.AllDirectories);
                    }
                    catch (Exception)
                    {
                    }

                    if (paths != null)
                    {
                        // サブフォルダ以下に同名のファイルが複数存在する場合、
                        // パスを大文字・小文字の区別なしの文字列順でソートした先頭が適用される
                        Array.Sort(paths, StringComparer.OrdinalIgnoreCase);

                        foreach (var p in paths)
                        {
                            yield return new PathWithName(p, baseName, basePath);
                        }
                    }
                }
            }
        }

        private static IEnumerable<PathWithName> StartUpPaths
        {
            get
            {
                // チーム設定から
                var files = FilterExsitingFiles(ApplicationConfig.FileIo.StartUpPaths.Select(x => new PathWithName(x.FullPath())), true).ToList();

                // 個人用設定から
                files.AddRange(FilterExsitingFiles(ApplicationConfig.UserSetting.StartUpPaths.Select(x => new PathWithName(x.FullPath())), true));

                // デフォルトフォルダ
                files.AddRange(FilterExsitingFiles(Enumerable.Repeat(new PathWithName(StartupFolderPath), 1), false));

                return files;
            }
        }

        /// <summary>
        /// スタートアップ時にファイルを開きます
        /// </summary>
        public static void LoadFromStartup()
        {
            if (StartUpPaths.Any())
            {
                LoadFromFileOrDirectory(StartUpPaths, null, true);
            }
        }

        /// <summary>
        /// スタートアップ時にコマンドラインから指定されたファイルを開きます
        /// </summary>
        public static void LoadFromCommandLine()
        {
            if (CommandLineOptionManager.Files.Any())
            {
                LoadFromFileOrDirectory(CommandLineOptionManager.Files);
            }
        }

        /// <summary>
        /// アニメーションアンバインドするコマンド
        /// </summary>
        public static void UnbindAnimation(AnimationDocument animation, AnimationSet hasAnimation)
        {
            TheApp.CommandManager.Add(CreateUnbindAnimation(animation, hasAnimation, true));
        }

        public static ICommand CreateUnbindAnimation(AnimationDocument animation, AnimationSet hasAnimation, bool isUpdateBind)
        {
            var animations = hasAnimation.Animations.Where(x => x != new AnimationSetItem(animation.FileName, animation.FileLocation)).ToArray();
            var commandSet = new EditCommandSet();
            commandSet.Add(CreateAnimationsEditCommand(hasAnimation, animations, false).Execute());

            if (isUpdateBind)
            {
                commandSet.Add(CreateAnimationUpdateBindCommand(Enumerable.Repeat(animation, 1)).Execute());
            }

            commandSet.Reverse();

            return commandSet;
        }

        public static void UnbindSceneAnimation(AnimationDocument animation, AnimationSet hasAnimation)
        {
            var animations = hasAnimation.Animations.Where(x => x != new AnimationSetItem(animation.FileName, animation.FileLocation)).ToArray();
            var commandSet = new EditCommandSet();
            commandSet.Add(CreateAnimationsEditCommand(hasAnimation, animations, false).Execute());
            commandSet.Add(CreateAnimationUpdateBindCommand(Enumerable.Repeat(animation, 1)).Execute());
            commandSet.Reverse();
            TheApp.CommandManager.Add(commandSet);
        }

        public static GroupEditCommand CreateAnimationsEditCommand(AnimationSet hasAnimation, AnimationSetItem[] animations, bool isAutoBind)
        {
            return new GeneralGroupReferenceEditCommand<Tuple<AnimationSet, AnimationSetItem[]>>(
                new GuiObjectGroup(ProjectDocument),
                GuiObjectID.Project,
                Enumerable.Repeat(new Tuple<AnimationSet, AnimationSetItem[]>(hasAnimation, animations), 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var tuple = data as Tuple<AnimationSet, AnimationSetItem[]>;
                    swap = new Tuple<AnimationSet, AnimationSetItem[]>(tuple.Item1, tuple.Item1.Animations.ToArray());

                    tuple.Item1.Animations.Clear();
                    tuple.Item1.Animations.AddRange(tuple.Item2);

                    if (tuple.Item1 is AnimationSet)
                    {
                        ((AnimationSet)tuple.Item1).SetMaybeModified();
                    }

                    {
                        // アニメーションセットに追加される場合に、Viewerへ送信する。
                        if (tuple.Item1 is AnimationSet)
                        {
                            var animationSet = (AnimationSet)tuple.Item1;
                            if (animationSet.IsSceneAnimationSet)
                            {
                                if (DocumentManager.PreviewSceneAnimSet == animationSet)
                                {
                                    DocumentManager.PreviewSceneAnimSetUpdated();
                                }
                            }
                            else
                            {
                                foreach (Model model in DocumentManager.Models)
                                {
                                    if (model.PreviewAnimSet == animationSet)
                                    {
                                        model.PreviewAnimSetUpdated();
                                    }
                                }
                            }
                        }
                        /*
                        else if (tuple.Item1 is Model)
                        {
                            // プレビュー用アニメーションセットを再設定する。
                            Model targetModel = (Model)tuple.Item1;
                            if (targetModel.PreviewAnimSet == targetModel.DefaultAnimationSet)
                            {
                                targetModel.PreviewAnimSetUpdated();
                            }
                        }*/
                    }
                },
                createEventArgsDelegate: (x, y) => { return new DocumentAnimationBindArg(x, y, isAutoBind); });
        }

        public static GroupEditCommand CreateCopyAnimationSetCommand(Model model, Model oldModel)
        {
            return new GeneralGroupReferenceEditCommand<Tuple<Model, Model>>(
                DummyObject.TheDummyObjectGroup,
                GuiObjectID.DummyObject,
                Enumerable.Repeat(new Tuple<Model, Model>(model, oldModel), 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var tuple = data as Tuple<Model, Model>;
                    swap = new Tuple<Model, Model>(tuple.Item2, tuple.Item1);
                    tuple.Item1.AnimationSets = tuple.Item2.AnimationSets;
                    tuple.Item1.DefaultAnimationSet = tuple.Item2.DefaultAnimationSet;
                    tuple.Item1.PreviewAnimSet = tuple.Item2.PreviewAnimSet;
                });
        }

        public static void AddAnimationSet(Model model, AnimationSet animationSet)
        {
            var animationSets = model.AnimationSets.Concat(Enumerable.Repeat(animationSet, 1));
            TheApp.CommandManager.Execute(CreateAnimationSetEditCommand(model, animationSets.ToArray()));
        }

        public static GroupEditCommand CreateAnimationSetEditCommand(Model model, AnimationSet[] sets)
        {
            AnimationSet previewAnimationSet = null;
            bool updatePreviewAnimationSet = model.PreviewAnimSet != null && model.PreviewAnimSet != model.DefaultAnimationSet && !sets.Contains(model.PreviewAnimSet);
            return new GeneralGroupReferenceEditCommand<Tuple<Model, AnimationSet[]>>(
                new GuiObjectGroup(ProjectDocument),
                GuiObjectID.Project,
                Enumerable.Repeat(new Tuple<Model, AnimationSet[]>(model, sets), 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var tuple = data as Tuple<Model, AnimationSet[]>;
                    swap = new Tuple<Model, AnimationSet[]>(tuple.Item1, tuple.Item1.AnimationSets.ToArray());
                    tuple.Item1.AnimationSets.Clear();
                    tuple.Item1.AnimationSets.AddRange(tuple.Item2);

                    foreach (var animationSet in tuple.Item2)
                    {
                        animationSet.SetMaybeModified();
                    }

                    if (updatePreviewAnimationSet)
                    {
                        var tmp = tuple.Item1.PreviewAnimSet;
                        tuple.Item1.PreviewAnimSet = previewAnimationSet;
                        previewAnimationSet = tmp;
                    }
                    else
                    {
                        tuple.Item1.PreviewAnimSetUpdated();
                    }
                });
        }

        public static GroupEditCommand CreatePreviewAnimationSetEditCommand(Model model, string animationSetName)
        {
            AnimationSet animationSet = model.AnimationSetsWithDefault.FirstOrDefault(x => x.Name == animationSetName);
            return new GeneralGroupReferenceEditCommand<object>(
                new GuiObjectGroup(ProjectDocument),
                GuiObjectID.Project,
                Enumerable.Repeat(new object(), 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var tmp = model.PreviewAnimSet;
                    model.PreviewAnimSet = animationSet;
                    animationSet = tmp;
                }
            );
        }

        public static void AddSceneAnimationSet(AnimationSet animationSet)
        {
            var animationSets = DocumentManager.SceneAnimationSets.Concat(Enumerable.Repeat(animationSet, 1)).ToList();
            TheApp.CommandManager.Execute(CreateSceneAnimationSetEditCommand(animationSets));
        }

        public static GroupEditCommand CreateSceneAnimationSetEditCommand(List<AnimationSet> sets)
        {
            AnimationSet previewAnimationSet = null;
            bool updatePreviewAnimationSet = DocumentManager.PreviewSceneAnimSet != null &&
                DocumentManager.PreviewSceneAnimSet != DocumentManager.DefaultSceneAnimationSet &&
                !sets.Contains(DocumentManager.PreviewSceneAnimSet);
            return new GeneralGroupReferenceEditCommand<List<AnimationSet>>(
                new GuiObjectGroup(ProjectDocument),
                GuiObjectID.Project,
                Enumerable.Repeat(sets, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    swap = SceneAnimationSets;
                    SceneAnimationSets = (List<AnimationSet>)data;

                    foreach (var animationSet in SceneAnimationSets)
                    {
                        animationSet.SetMaybeModified();
                    }

                    // プレビュー用アニメーションセットを再設定する。
                    if (updatePreviewAnimationSet)
                    {
                        var tmp = DocumentManager.PreviewSceneAnimSet;
                        DocumentManager.PreviewSceneAnimSet = previewAnimationSet;
                        previewAnimationSet = tmp;
                    }
                    else
                    {
                        DocumentManager.PreviewSceneAnimSetUpdated();
                    }
                });
        }

        public static GroupEditCommand CreateDefaultSceneAnimationSetEditCommand(AnimationSet animationSet)
        {
            return new GeneralGroupReferenceEditCommand<AnimationSet>(
                new GuiObjectGroup(ProjectDocument),
                GuiObjectID.Project,
                Enumerable.Repeat(animationSet, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    swap = DefaultSceneAnimationSet;
                    DefaultSceneAnimationSet = (AnimationSet)data;

                    if (((AnimationSet)swap) == DocumentManager.PreviewSceneAnimSet)
                    {
                        DocumentManager.PreviewSceneAnimSet = DefaultSceneAnimationSet;
                    }
                    else
                    {
                        DocumentManager.PreviewSceneAnimSetUpdated();
                    }
                });
        }

        /// <summary>
        /// リネームします。コマンドターゲットはプロジェクト!
        /// </summary>
        public static GroupEditCommand CreateProjectGuiObjectRenameEditCommand(GuiObject obj, string name)
        {
            return new GeneralGroupReferenceEditCommand<Tuple<GuiObject, string>>(
                new GuiObjectGroup(ProjectDocument),
                GuiObjectID.Project,
                Enumerable.Repeat(new Tuple<GuiObject, string>(obj, name), 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var tuple = data as Tuple<GuiObject, string>;
                    swap = new Tuple<GuiObject, string>(tuple.Item1, tuple.Item1.Name);
                    tuple.Item1.Name = tuple.Item2;
                },
                createEventArgsDelegate: (x,y) => (new ObjectRenamedArg(x,y,obj))
            );
        }

        private class DocumentRenameEditCommandInfo
        {
            public string Name{ get; set; }
            public string FileDotExt{ get; set; }
            public string FileLocation{ get; set; }
            public string BaseName{ get; set; }
            public string BasePath { get; set; }
        }

        public static GroupEditCommand CreateDocumentRenameEditCommand(Document targetDoc, string name, string fileDotExt)
        {
            var info = new DocumentRenameEditCommandInfo()
                        {
                            Name			= name,
                            FileDotExt		= fileDotExt,
                            FileLocation	= string.Empty,
                            BaseName		= string.Empty
                        };

            List<Model> models = DocumentManager.Models.Where(x => x.AllAnimations.Contains(targetDoc)).ToList();

            return new GeneralGroupReferenceEditCommand<DocumentRenameEditCommandInfo>(
                new GuiObjectGroup(targetDoc),
                targetDoc.ObjectID,
                Enumerable.Repeat(info, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var document = (Document)target;
                    string docFileName = document.FileName;

                    var i = data as DocumentRenameEditCommandInfo;

                    swap = new DocumentRenameEditCommandInfo()
                            {
                                Name			= document.Name,
                                FileDotExt		= document.FileDotExt,
                                FileLocation	= document.FileLocation,
                                BaseName		= document.BaseName,
                                BasePath = document.BasePath,
                            };

                    document.Name			= i.Name;
                    document.FileDotExt		= i.FileDotExt;
                    document.FileLocation	= i.FileLocation;
                    document.BaseName		= i.BaseName;
                    document.BasePath = i.BasePath;

                    // Viewerへ転送
                    Viewer.ViewerUtility.RenameDocument(docFileName, document, models);
                    TheApp.MainFrame.SendCurrentFrame();
                }
            );
        }

        public static GroupEditCommand CreateSceneAnimationObjectRenameEditCommand(SceneAnimation targetDoc, GuiObject obj, string name)
        {
            Debug.Assert(obj is ISceneAnimationObject);

            return new GeneralGroupReferenceEditCommand<string>(
                new GuiObjectGroup(targetDoc),
                targetDoc.ObjectID,
                Enumerable.Repeat(name, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    swap = obj.Name;
                    obj.Name = (string)data;

                    obj.SetMaybeModified();

                    // Viewerへ転送
                    Viewer.LoadOrReloadAnimation.Send(targetDoc);
                    TheApp.MainFrame.SendCurrentFrame();
                }
            );
        }

        public static GroupEditCommand CreateAnimationSetRenameEditCommand(AnimationSet targetDoc, string name)
        {
            return new GeneralGroupReferenceEditCommand<string>(
                new GuiObjectGroup(targetDoc),
                targetDoc.ObjectID,
                Enumerable.Repeat(name, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var animSet = target as AnimationSet;

                    swap			= ObjectUtility.Clone(animSet.Name);
                    animSet.Name	= (string)data;

                    // 変更状態を更新
                    ProjectDocument.SetMaybeModified();

                    // 一つしか扱わないはずなのと、新旧の名前が必要なので、こちらでビューアに転送します。
                    foreach (var model in DocumentManager.Models.Where(x => x.AnimationSets.Contains(animSet)))
                    {
                        if (model.PreviewAnimSet == animSet)
                        {
                            //model.PreviewAnimSetUpdated();
                            //Viewer.ViewerUtility.SendAnimationSet(model);
                        }
//						int model_id = model.FileName.GetHashCode();
//!!						Viewer.RenameAnimationSet.Send((uint)model_id, (string)swap, (string)data, 0xFFFFFFFF, 0xFFFFFFFF);
                    }

                    UpdatePreviewAnimations();

                    TheApp.MainFrame.SendCurrentFrame();
                }
            );
        }

        /// <summary>
        /// パスの変更
        /// </summary>
        public static EditCommand CreatePathChangeCommand(Document document, string path)
        {
            return CreatePathChangeCommand(document, Path.GetDirectoryName(path), Path.GetFileNameWithoutExtension(path), Path.GetExtension(path), null, null);
        }

        /// <summary>
        /// パスの変更
        /// </summary>
        public static EditCommand CreatePathChangeCommand(Document document, string location, string name, string dotExt, string baseName, string basePath)
        {
            var commandSet = new EditCommandSet();

            //List<Model> models = DocumentManager.Models.Where(x => x.AllAnimations.Contains(document)).ToList();

            var t = new Tuple<string, string, string, string, string>(location, name, dotExt, baseName, basePath);
            commandSet.Add(new GeneralGroupReferenceEditCommand<Tuple<string, string, string, string, string>>(
                new GuiObjectGroup(document),
                null,
                Enumerable.Repeat(t, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var doc = target as Document;
                    var tuple = data as Tuple<string, string, string, string, string>;
                    swap = new Tuple<string, string, string, string, string>(doc.FileLocation, doc.Name, doc.FileDotExt, doc.BaseName, doc.BasePath);
                    doc.FileLocation = tuple.Item1;
                    doc.Name = tuple.Item2;
                    doc.FileDotExt = tuple.Item3;
                    doc.BaseName = tuple.Item4;
                    doc.BasePath = tuple.Item5;
                }));

            commandSet.Add(new GeneralGroupReferenceEditCommand<object>(
                new GuiObjectGroup(DocumentManager.ProjectDocument),
                GuiObjectID.Project,
                Enumerable.Repeat<object>(null, 1),
                delegate(ref GuiObject target, ref object data, ref object swap) {
                },
                createEventArgsDelegate: (x, y) => (new ObjectRenamedArg(x, y, document))));
            return commandSet;
        }

        /// <summary>
        /// ファイル情報変更コマンド、コマンド以外でも変更を許しているので、注意。Undo/Redo で違う情報になることがある。
        /// </summary>
        public static EditCommand CreateFileInfoEditCommand(ProjectDocument document, FileInfo fileInfo)
        {
            return new GeneralGroupReferenceEditCommand<FileInfo>(
                new GuiObjectGroup(document),
                null,
                Enumerable.Repeat(fileInfo, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var doc = target as ProjectDocument;
                    swap = doc.FileInfo;
                    doc.FileInfo = (FileInfo)data;
                });
        }

        /// <summary>
        /// ファイル情報変更コマンド、コマンド以外でも変更を許しているので、注意。Undo/Redo で違う情報になることがある。
        /// </summary>
        public static EditCommand CreateFileInfoEditCommand(IntermediateFileDocument document, file_infoType fileInfo)
        {
            return new GeneralGroupReferenceEditCommand<file_infoType>(
                new GuiObjectGroup(document),
                null,
                Enumerable.Repeat(fileInfo, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var doc = target as IntermediateFileDocument;
                    swap = ObjectUtility.Clone(doc.file_info);
                    doc.file_info = (file_infoType)data;
                });
        }

        public static GroupEditCommand CreateAddOrRemoveDocumentCommand(IEnumerable<Document> objs, bool add)
        {
            bool initial = add;
            Document[] documents = objs.ToArray();
            IEnumerable<Document> ordered = null;
            return new GeneralGroupReferenceEditCommand<object>(
                new GuiObjectGroup(ProjectDocument),
                GuiObjectID.Project,
                Enumerable.Repeat<object>(null, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    ordered = initial == add ? documents : documents.Reverse();
                    if (add)
                    {
                        foreach (var obj in ordered)
                        {
                            AddDocument(obj);
                        }

                        // セーブデータとの差分を検出して変更マークを付ける
                        DetectDocumentDifference();
                    }
                    else
                    {
                        foreach (var obj in ordered)
                        {
                            RemoveDocument(obj);
                        }

                        // テクスチャのビットマップを消去
                        foreach (var texture in ordered.Where(x => x.ObjectID == GuiObjectID.Texture).OfType<Texture>())
                        {
                            texture.ClearCache();
                        }
                    }
                    add = !add;
                },
                createEventArgsDelegate: delegate(GuiObject target, EditCommand command)
                {
                    if (!add)
                    {
                        return new DocumentAddedOrRemovedArg(target, command, ordered, Enumerable.Empty<Document>());
                    }
                    else
                    {
                        return new DocumentAddedOrRemovedArg(target, command, Enumerable.Empty<Document>(), ordered);
                    }
                }
            );
        }

        public static GroupEditCommand CreateCameraAnimationsEditCommand(SceneAnimation owner, List<CameraAnimation> cameraAnimations)
        {
            return new GeneralGroupReferenceEditCommand<List<CameraAnimation>>(
                new GuiObjectGroup(owner),
                GuiObjectID.SceneAnimation,
                Enumerable.Repeat<List<CameraAnimation>>(cameraAnimations, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var scene = (SceneAnimation)target;
                    swap = scene.CameraAnims;
                    scene.CameraAnims = (List<CameraAnimation>)data;

                    // Viewerへ転送
                    Viewer.LoadOrReloadAnimation.Send(owner);

                    AnimationDocument.NotifyFrameCountChanged(owner, EventArgs.Empty);
                },
                createEventArgsDelegate: (x, y) => { return new SceneAnimationContentArg(x, y); }
            );
        }

        public static GroupEditCommand CreateLightAnimationsEditCommand(SceneAnimation owner, List<LightAnimation> lightAnimations)
        {
            return new GeneralGroupReferenceEditCommand<List<LightAnimation>>(
                new GuiObjectGroup(owner),
                GuiObjectID.SceneAnimation,
                Enumerable.Repeat<List<LightAnimation>>(lightAnimations, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var scene = (SceneAnimation)target;
                    swap = scene.LightAnims;
                    scene.LightAnims = (List<LightAnimation>)data;

                    // Viewerへ転送
                    Viewer.LoadOrReloadAnimation.Send(owner);

                    AnimationDocument.NotifyFrameCountChanged(owner, EventArgs.Empty);
                },
                createEventArgsDelegate: (x, y) => { return new SceneAnimationContentArg(x, y); }
            );
        }

        public static GroupEditCommand CreateFogAnimationsEditCommand(SceneAnimation owner, List<FogAnimation> fogAnimations)
        {
            return new GeneralGroupReferenceEditCommand<List<FogAnimation>>(
                new GuiObjectGroup(owner),
                GuiObjectID.SceneAnimation,
                Enumerable.Repeat<List<FogAnimation>>(fogAnimations, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var scene = (SceneAnimation)target;
                    swap = scene.FogAnims;
                    scene.FogAnims = (List<FogAnimation>)data;

                    // Viewerへ転送
                    Viewer.LoadOrReloadAnimation.Send(owner);

                    AnimationDocument.NotifyFrameCountChanged(owner, EventArgs.Empty);
                },
                createEventArgsDelegate: (x, y) => { return new SceneAnimationContentArg(x, y); }
            );
        }

        public static EditCommand CreateProjectChangeCommand(ProjectDocument newProject, bool clearScene)
        {
            ProjectDocument removed = null;
            List<AnimationSet> animationSets = new List<AnimationSet>();
            List<AnimationSetItem> animations = new List<AnimationSetItem>();
            AnimationSet preview = DocumentManager.DefaultSceneAnimationSet;
            return new GeneralGroupReferenceEditCommand<ProjectDocument>(
                DummyObject.TheDummyObjectGroup, // ダミー
                GuiObjectID.DummyObject,
                Enumerable.Repeat(newProject, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    swap = removed = ProjectDocument;
                    ProjectDocument = (ProjectDocument)data;
                    if (clearScene)
                    {
                        var temp0 = DocumentManager.SceneAnimationSets;
                        DocumentManager.SceneAnimationSets = animationSets;
                        animationSets = temp0;

                        var temp1 = DocumentManager.DefaultSceneAnimationSet.Animations;
                        DocumentManager.DefaultSceneAnimationSet.Animations = animations;
                        animations = temp1;

                        var temp2 = DocumentManager.PreviewSceneAnimSet;
                        DocumentManager.PreviewSceneAnimSet = preview;
                        preview = temp2;
                    }

                    var animationSetObjects = DocumentManager.Models.SelectMany(x => x.AnimationSetsWithDefault).Concat(
                        DocumentManager.AllSceneAnimationSets);
                    foreach (var animationSet in animationSetObjects)
                    {
                        animationSet.SetMaybeModified();
                    }


                },
                createEventArgsDelegate: delegate(GuiObject target, EditCommand command)
                {
                    return new DocumentAddedOrRemovedArg(target, command, Enumerable.Repeat(ProjectDocument, 1), Enumerable.Repeat(removed, 1));
                });
        }

        // ドキュメント入れ替えコマンドを作る
        public static EditCommand CreateDocumentReplaceCommand(Document dstDoc, Document srcDoc, Action PostEditAction = null)
        {
            Debug.Assert(
                (dstDoc is Model) ||
                (dstDoc is Texture) ||
                (dstDoc is AnimationDocument) ||
                (dstDoc is ShaderDefinition)
            );

            Debug.Assert(
                (srcDoc is Model) ||
                (srcDoc is Texture) ||
                (srcDoc is AnimationDocument) ||
                (srcDoc is ShaderDefinition)
            );

            Debug.Assert(dstDoc.GetType() == srcDoc.GetType());

            return
                (dstDoc is Model  )				? modelContainer.CreateDocumentReplaceCommand(				dstDoc as Model,				srcDoc as Model, false, PostEditAction) :
                (dstDoc is SeparateMaterial)    ? separateMaterialContainer.CreateDocumentReplaceCommand(dstDoc as SeparateMaterial, srcDoc as SeparateMaterial, false, PostEditAction) :
                (dstDoc is Texture)				? textureContainer.CreateDocumentReplaceCommand(			dstDoc as Texture,				srcDoc as Texture, false, PostEditAction) :
                (dstDoc is AnimationDocument)	? animationContainer.CreateDocumentReplaceCommand(			dstDoc as AnimationDocument,	srcDoc as AnimationDocument, false, PostEditAction) :
                (dstDoc is ShaderDefinition) ? shaderDefinitionContainer.CreateDocumentReplaceCommand(dstDoc as ShaderDefinition, srcDoc as ShaderDefinition, false, PostEditAction) :
                                                  null;
        }

        public static EditCommandSet CreateAnimationUpdateBindCommand(IEnumerable<AnimationDocument> animations)
        {
            EditCommandSet commandSet = new EditCommandSet();
            foreach (var animation in animations)
            {
                var command = animation.CreateUpdateBindCommand();
                if (command != null)
                {
                    var editCommandSet = command as EditCommandSet;
                    if (editCommandSet?.CommandCount == 0)
                    {
                        continue;
                    }
                    commandSet.Add(command);
                }
            }

            return commandSet;
        }

        public static void ReplaceShaderDefinition(ShaderDefinition shaderDef, string replaceShaderDefFilePath, string fullPath, byte[] fsdFile, bool isNotReplaceConfigPath, bool validateShaderDefinition)
        {
            List<G3dStream> streams = new List<G3dStream>();
            var nw4f_3dif = IfBinaryReadUtility.Read(streams, fsdFile, null);
            var newDoc = new ShaderDefinition((shader_definitionType)nw4f_3dif.Item, streams);

            // ファイルロケーションを元ファイルのものに変更します。
            newDoc.Name = shaderDef.Name;
            newDoc.FileDotExt = shaderDef.FileDotExt;
            newDoc.SetFileLocation(shaderDef.FileLocation);

            // ランタイム連携用の値もコピーする。
            ShaderDefinition newDef = newDoc as ShaderDefinition;
            string sourceName = Path.GetFileName(fullPath);
            if (newDef != null)
            {
                newDef.SourceFileNames = shaderDef.SourceFileNames;
                newDef.SetModifiedShadingModel(shaderDef, sourceName);
                newDef.CopyBinaryPaths(shaderDef, sourceName);
            }

            string[] modifiedShadingModelNames = newDef.ShadingModelIndices(sourceName).Select(x => newDef.Definitions[x].Name).ToArray();

            // コマンドセット生成
            var commandSet = new EditCommandSet();
            {
                Action action;
                {
                    ShaderDefinition oldDef = shaderDef;
                    ShaderDefinition def = (ShaderDefinition)newDoc;
                    bool first = true;
                    action = () =>
                    {
                        (new DocumentSaver()).WriteDocument(def, replaceShaderDefFilePath, false);

                        def.ResFileKey = oldDef.ResFileKey;
                        def.IsAttached = oldDef.IsAttached;
                        def.ShaderArchiveKey = oldDef.ShaderArchiveKey;
                        def.updatedAttachedShadingModel = oldDef.updatedAttachedShadingModel;

                        if (def.IsAttached == true && def.ShaderArchiveKey != 0 && !def.FailedToBinarize)
                        {
                            lock (def)
                            {
                                if (!Viewer.Manager.IsDeviceTargeted && !def.CompileTested)
                                {
                                    // コンパイルテストを行う
                                    byte[] result;
                                    if (def.FailedToBinarize || ShaderDifinitionUtility.RebuildShaderDefinition(def, out result, true, Connecter.Platform) == null)
                                    {
                                        def.FailedToBinarize = true;
                                    }

                                    def.CompileTested = true;
                                }
                            }
                        }

                        Viewer.LoadOptimizedShaderArchive.OnShaderDefinitionChanged(def, modifiedShadingModelNames);

                        // ビューアへ転送
                        def.SendEditShadingModels(Path.GetFileName(fullPath));

                        // 監視の切り替え
                        if (first)
                        {
                            // 再度ファイル監視を行う。
                            ShaderDifinitionUtility.CreateGLSLFilePathList(def);
                            first = false;
                        }
                        else
                        {
                            ShaderDifinitionUtility.RefreshGLSLFileFromFsd(def);
                        }

                        var tmp = oldDef;
                        oldDef = def;
                        def = tmp;


                    };
                }

                if (isNotReplaceConfigPath == false)
                {
                    newDoc.Data.shader_definition_info.config_path = shaderDef.Data.shader_definition_info.config_path;
                }

                // ドキュメント入れ替えコマンド生成
                commandSet.Add(DocumentManager.CreateDocumentReplaceCommand(shaderDef, newDoc, action).Execute());

                // 描画情報とオプション変数のチェック
                if (validateShaderDefinition)
                {
                    var validationErrors = ValidateShaderDefinition(newDef).ToArray();
                    if (validationErrors.Any())
                    {
                        ShowErrorDialog(MessageContext.ShaderConvert_InvalidItems, validationErrors, string.Format(res.Strings.ShaderConvert_InvalidItems));
                    }
                }

                // マテリアルのシェーダー割り当ての修正
                {
                    List<string> shaderDefinitionNames = new List<string>();
                    shaderDefinitionNames.Add(newDoc.Name);
                    //					var shaderDefinitionNames = openedDocuments.Where(x => x.ObjectID == GuiObjectID.ShaderDefinition).Select(x => x.Name).ToArray();
                    var models = modelContainer.Documents;

                    // シェーダー側
                    var materials = (from material in DocumentManager.Materials
                                     where shaderDefinitionNames.Any(x => x == material.MaterialShaderAssign.ShaderDefinitionFileName) || models.Any(x => material.Referrers.Contains(x))
                                     let errorType = ShaderAssignUtility.IsConsistentWithDefinitionDetail(material)
                                     select new { material, errorType }).ToArray();

                    var notMaterialShaderMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.NotMaterialShader).Select(x => x.material).ToArray();
                    if (notMaterialShaderMaterials.Any())
                    {
                        ShowNotMaterialShaderDialog(notMaterialShaderMaterials);
                    }

                    var errorMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.Critical).Select(x => x.material).ToArray();
                    if (errorMaterials.Any())
                    {
                        AddMaterialFixCommand(commandSet, errorMaterials);
                    }
                    var disorderedMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.Unimportant).Select(x => x.material).ToArray();
                    if (disorderedMaterials.Any())
                    {
                        commandSet.Add(ShaderAssignUtility.ExecuteFixParameters(disorderedMaterials, false, reload:true));
                    }

                    // アニメーションの3DEditor内部のバインド設定
                    var targetAnims = materials.SelectMany(x => x.material.Referrers).Distinct().SelectMany(x => x.AllAnimations).Distinct().ToArray();
                    if (targetAnims.Any())
                    {
                        commandSet.Add(CreateAnimationUpdateBindCommand(targetAnims).Execute());
                    }
                }
            }

            // コマンドスタックに追加
            if (commandSet.CommandCount > 0)
            {
                commandSet.Reverse();
                TheApp.CommandManager.Add(commandSet);
            }
        }

        /// <summary>
        /// ライトアニメーションのターゲット割り当てを修正するコマンドを追加します。
        /// </summary>
        /// <param name="commandSet">コマンドの追加先</param>
        /// <param name="errorLightAnims">修正するライトアニメーション</param>
        public static void AddLightAnimationFixCommand(EditCommandSet commandSet, LightAnimation[] errorLightAnims)
        {
            Debug.Assert(errorLightAnims.Any());

            DebugConsole.WriteLine("private static void AddLightAnimationFixCommand(LightAnimation[] errorLightAnims)");

            LightAnimation[] fixLightAnims = null;
            LightAnimation[] notFixLightAnims = null;
            {
                // ダイアログで選択
                if (ApplicationConfig.UserSetting.IO.AutoFixLightAnimation == false)
                {
                    App.AppContext.NotificationHandler.FixLightAnimAssignDialog(errorLightAnims, out fixLightAnims, out notFixLightAnims);
                }
                // 自動
                else
                {
                    fixLightAnims = errorLightAnims;
                    notFixLightAnims = new LightAnimation[0];

                    MessageLogFixLightAnimation(errorLightAnims, App.AppContext.NotificationHandler);
                }
            }
            Debug.Assert(fixLightAnims != null);
            Debug.Assert(notFixLightAnims != null);

            commandSet.Add(LightAnimationAssignUtility.ExecuteFixLightAnimations(fixLightAnims).Execute());
        }

        /// <summary>
        /// ライトアニメーションの自動修正情報をメッセージログに表示します。
        /// </summary>
        /// <param name="errorLightAnims">ライトアニメーション</param>
        /// <param name="handler"></param>
        public static void MessageLogFixLightAnimation(LightAnimation[] errorLightAnims, INotificationHandler handler)
        {
            var lightAnimInconsistentInfos = errorLightAnims.Select(
                lightAnim =>
                {
                    int count;
                    var message = FixLightAnimationAssignDialog.InconsistentMessage(lightAnim, out count, FixLightAnimationAssignDialog.InconsistentMessageType.Result);

                    return
                        new FixLightAnimationAssignDialog.LightAnimationInconsistentInfo()
                        {
                            LightAnimationName			= lightAnim.Name,
                            LightAnimationOwnerName		= lightAnim.Owner.Name,
                            LightAnimationTypeName	    = lightAnim.Data.type,
                            Count						= count,
                            Message						= message
                        };
                }
            ).ToArray();		// ライトアニメーションへの参照を残さないように実体化しておく

            // メッセージログに情報を表示
            var lightNames = errorLightAnims.Select(x => x.Name).Distinct();  // x.Owner.Name
            var logMessage = string.Format(res.Strings.LogMessage_LightAnimationAutoFixed, string.Join(", ", lightNames));

            handler.WriteMessageLog(
                MessageLog.LogType.Information,
                logMessage,
                () =>
                {
                    // メッセージログのダブルクリックで呼ばれるので、ErrorHandlerでは扱わない
                    using (var dialog = new FixLightAnimationAssignDialog(lightAnimInconsistentInfos))
                    {
                        dialog.ShowDialog(TheApp.MainFrame);
                    }
                }
            );
        }

        public static void AddMaterialFixCommand(EditCommandSet commandSet, Material[] errorMaterials)
        {
            Debug.Assert(errorMaterials.Any());

            DebugConsole.WriteLine("private static void AddMaterialFixCommand(EditCommandSet commandSet, Material[] errorMaterials)");

            Material[] fixMaterials = null;
            Material[] notFixMaterials = null;
            {
                // ダイアログで選択
                if (ApplicationConfig.UserSetting.IO.AutoFix == false)
                {
                    App.AppContext.NotificationHandler.FixShaderAssignDialog(errorMaterials, out fixMaterials, out notFixMaterials);
                }
                // 自動
                else
                {
                    fixMaterials = errorMaterials;
                    notFixMaterials = new Material[0];

                    MessageLogFixMaterial(errorMaterials, App.AppContext.NotificationHandler);
                }
            }
            Debug.Assert(fixMaterials != null);
            Debug.Assert(notFixMaterials != null);

            commandSet.Add(ShaderAssignUtility.ExecuteFixParameters(fixMaterials, reload:false));
            commandSet.Add(ShaderAssignUtility.AddNotAssignedItems(notFixMaterials));
        }

        public static void MessageLogFixMaterial(Material[] errorMaterials, INotificationHandler handler)
        {
            var materialInconsistentInfos = errorMaterials.Select(
                material =>
                {
                    var counts = new List<int>();
                    var messages = new List<string>();
                    foreach (var model in material.Referrers)
                    {
                        int count;
                        messages.Add(ShaderAssignUtility.InconsistentMessage(model, material, out count, IfShaderAssignUtility.InconsistentMessageType.Result));
                        counts.Add(count);
                    }

                    return
                        new FixShaderAssignDialog.MaterialInconsistentInfo()
                        {
                            MaterialName				= material.Name,
                            MaterialOwnerName			= string.Join(", ", material.Referrers.Select(x => x.Name)),
                            ShaderDefinitionFileName	= material.MaterialShaderAssign.ShaderDefinitionFileName,
                            ShaderName					= material.MaterialShaderAssign.ShaderName,
                            Count						= string.Join(", ", counts.Select(x => x.ToString())),
                            Message						= string.Join(", ", messages)
                        };
                }
            ).ToArray();		// マテリアルへの参照を残さないように実体化しておく

            // メッセージログに情報を表示
            var modelNames = errorMaterials.Select(x => x.OwnerDocument.Name).Distinct();
            var logMessage = string.Format(res.Strings.LogMessage_MaterialAutoFixed, string.Join(", ", modelNames));

            handler.WriteMessageLog(
                MessageLog.LogType.Information,
                logMessage,
                () =>
                {
                    // メッセージログのダブルクリックで呼ばれるので、ErrorHandlerでは扱わない
                    using (var dialog = new FixShaderAssignDialog(materialInconsistentInfos))
                    {
                        dialog.ShowDialog(TheApp.MainFrame);
                    }
                }
            );
        }

        public static List<TexturePatternAnimation> GetInvalidTexPatAnimation()
        {
            List<TexturePatternAnimation> list = new List<TexturePatternAnimation>();
            IEnumerable<AnimationDocument> anims = Animations.Where(x => x is TexturePatternAnimation);
            foreach (var anim in anims)
            {
                TexturePatternAnimation texAnim = anim as TexturePatternAnimation;
                if (texAnim.IsValueOutOfRange())
                {
                    list.Add(texAnim);
                }
            }

            return list;
        }

        public static bool OptimizeShader = false;

        // シェーダー自動生成の設定
        public enum AutomaticShaderRegenerationID
        {
            Enabled,    // 有効
            Disabled,   // 無効
            MaterialSetting, // マテリアルの設定を使用する
        }

        // 作業中に随時シェーダの更新する必要が無い場合、一括で有効・無効を設定する
        public static AutomaticShaderRegenerationID AutomaticShaderRegeneration = AutomaticShaderRegenerationID.MaterialSetting;
        public static bool GetMaterialOptimizeShaderOnReload(Material material)
        {
            switch (AutomaticShaderRegeneration)
            {
                case AutomaticShaderRegenerationID.Enabled:
                    return true; // 一括で有効
                case AutomaticShaderRegenerationID.Disabled:
                    return false; // 一括で無効
                default:
                    break;
            }

            // 本来のマテリアルの設定を使用する
            return material.OptimizeShaderOnReload;
        }

        #region 同名アニメーション処理
        public static Dictionary<string, int> TextureNameCount = new Dictionary<string, int>();
        public static Dictionary<string, int> AnimationOrTextureFileNameCount = new Dictionary<string, int>();

        // 1-based index
        public static Dictionary<GuiObject, int> IndexInSameNames = new Dictionary<GuiObject, int>();
        public static void UpdateIndexInSameNames()
        {
            AnimationOrTextureFileNameCount = new Dictionary<string, int>();
            TextureNameCount = new Dictionary<string, int>();
            IndexInSameNames = new Dictionary<GuiObject, int>();
            foreach (var animation in DocumentManager.Animations.OfType<Document>().Concat(DocumentManager.Textures)
                .OrderBy(x => x.FilePath))
            {
                int count = 0;
                AnimationOrTextureFileNameCount.TryGetValue(animation.FileName.ToLower(), out count);
                count++;
                IndexInSameNames[animation] = count;
                AnimationOrTextureFileNameCount[animation.FileName.ToLower()] = count;
            }

            foreach (var texture in DocumentManager.Textures)
            {
                int num;
                TextureNameCount.TryGetValue(texture.Name, out num);
                TextureNameCount[texture.Name] = num + 1;
            }
        }

        public static string GetSameNameIndexText(Document document, bool path)
        {
            if (!(document is AnimationDocument) && !(document is Texture))
            {
                return string.Empty;
            }

            if (!AnimationOrTextureFileNameCount.ContainsKey(document.FileName.ToLower()))
            {
                // ツリービュー更新中など、古いアイテムに対してに呼ばれる場合がある。
                return "";
            }
            if (!IndexInSameNames.ContainsKey(document))
            {
                // ツリービュー更新中など、古いアイテムに対してに呼ばれる場合がある。
                return "";
            }

            int count = AnimationOrTextureFileNameCount[document.FileName.ToLower()];
            if (count > 1)
            {
                var text = string.Format("{0}/{1}", IndexInSameNames[document], count);
                if (path)
                {
                    text = " " + text;
                    if (!string.IsNullOrEmpty(document.FilePath))
                    {
                        text += " [" + document.FilePath + "]";
                    }
                }

                return text;
            }
            else
            {
                return string.Empty;
            }
        }
        #endregion

        public static void UpdatePreviewAnimations()
        {
            //var modelIds = new HashSet<object>(Models.Select(x => x.ModelId));

            // ないものを削除
            foreach (var animation in Animations)
            {
                foreach (var id in animation.PreviewingAnimationSets.Keys.Except(Models.Select(x => x.ModelId)).ToArray())
                {
                    animation.ClearPreviewAnimationSet(id);
                }
            }

            // あるものを追加
            foreach (var model in Models)
            {
                if (model.PreviewAnimSet != null)
                {
                    foreach (var animation in DocumentManager.GetAnimations(model.PreviewAnimSet.Animations))
                    {
                        animation.SetPreviewAnimationSet(model.ModelId, model.PreviewAnimSet.Name);
                    }
                }
            }
        }

        // モデルの読み込み順
        public static Model[] OrderedModels = new Model[0];

        // テクスチャの読み込み順
        public static Texture[] OrderedTextures = new Texture[0];

        // マテリアル専用ドキュメントの読み込み順
        public static SeparateMaterial[] OrderedSeparateMaterials = new SeparateMaterial[0];

        public static void UpdateModelsAndTexturesOrder(Dictionary<GuiObject, GuiObject> swapped)
        {
            for (int i = 0; i < OrderedModels.Length; i++)
            {
                GuiObject obj;
                if (swapped.TryGetValue(OrderedModels[i], out obj))
                {
                    OrderedModels[i] = (Model)obj;
                }
            }

            var oldModels = OrderedModels.Intersect(DocumentManager.Models).ToArray();
            OrderedModels = oldModels.Concat(DocumentManager.Models.Except(oldModels)).ToArray();

            for (int i = 0; i < OrderedTextures.Length; i++)
            {
                GuiObject obj;
                if (swapped.TryGetValue(OrderedTextures[i], out obj))
                {
                    OrderedTextures[i] = (Texture)obj;
                }
            }

            var oldTextures = OrderedTextures.Intersect(DocumentManager.Textures).ToArray();
            OrderedTextures = oldTextures.Concat(DocumentManager.Textures.Except(oldTextures)).ToArray();

            for (int i = 0; i < OrderedSeparateMaterials.Length; i++)
            {
                GuiObject obj;
                if (swapped.TryGetValue(OrderedSeparateMaterials[i], out obj))
                {
                    OrderedSeparateMaterials[i] = (SeparateMaterial)obj;
                }
            }

            var oldSeparateMaterials = OrderedSeparateMaterials.Intersect(DocumentManager.SeparateMaterials).ToArray();
            OrderedSeparateMaterials = oldSeparateMaterials.Concat(DocumentManager.SeparateMaterials.Except(oldSeparateMaterials)).ToArray();
        }

        // 比較用のセーブデータ
        // undo スタックから消えて復元できなくなった Document のメモリが解放されるように弱参照で扱う
        private static Dictionary<string, WeakReference<Document>> saveDocuments = new Dictionary<string, WeakReference<Document>>();

        public static void AddSaveDocument(Document document)
        {
            if (!string.IsNullOrEmpty(document.FilePath))
            {
                saveDocuments[document.FilePath] = new WeakReference<Document>(document);
            }
        }

        public static Document GetSaveDocument(string name)
        {
            if (!string.IsNullOrEmpty(name) && saveDocuments.ContainsKey(name))
            {
                Document document;
                if (saveDocuments[name].TryGetTarget(out document))
                {
                    return document;
                }
            }

            return null;
        }

        // セーブデータとの差分を検出して変更マークを付ける
        // Document が持つセーブデータは、必ずしも最新のセーブデータとは一致しないので、
        // DocumentManager が管理している最新のセーブデータで更新してから比較を行う
        public static void DetectDocumentDifference()
        {
            // モデル
            foreach (var model in DocumentManager.Models)
            {
                model.SetMaybeModified();

                var saveDocument = DocumentManager.GetSaveDocument(model.FilePath);
                if (saveDocument != null)
                {
                    model.CopySavedData((Model)saveDocument);
                }
            }

            // マテリアル（モデル内でセーブデータは更新済み）
            foreach (var material in DocumentManager.Materials)
            {
                var ownerDocument = DocumentManager.GetSaveDocument(material.OwnerDocument.FilePath);
                if (ownerDocument != null && (ownerDocument as IMaterialOwner)?.Materials?.Where(x => x.Name == material.Name).Count() <= 0)
                {
                    material.OwnerDocument.SavedContents.Remove(material);
                }
                else
                {
                    material.SetMaybeModified();
                    material.OwnerDocument.SavedContents.Add(material);
                }
            }

            // シェイプ（モデル内でセーブデータは更新済み）
            foreach (var shape in DocumentManager.Shapes)
            {
                Model ownerModel = (Model)DocumentManager.GetSaveDocument(shape.Owner.FilePath);
                if (ownerModel != null && ownerModel.Shapes.Where(x => x.Name == shape.Name).Count() <= 0)
                {
                    shape.Owner.SavedContents.Remove(shape);
                }
                else
                {
                    shape.SetMaybeModified();
                    shape.Owner.SavedContents.Add(shape);
                }
            }

            // ボーン（モデル内でセーブデータは更新済み）
            foreach (var bone in DocumentManager.Bones)
            {
                Model ownerModel = (Model)DocumentManager.GetSaveDocument(bone.Owner.FilePath);
                if (ownerModel != null && ownerModel.Bones.Where(x => x.Name == bone.Name).Count() <= 0)
                {
                    bone.Owner.SavedContents.Remove(bone);
                }
                else
                {
                    bone.SetMaybeModified();
                    bone.Owner.SavedContents.Add(bone);
                }
            }

            // テクスチャ
            foreach (var texture in DocumentManager.Textures)
            {
                texture.SetMaybeModified();

                var saveDocument = DocumentManager.GetSaveDocument(texture.FilePath);
                if (saveDocument != null)
                {
                    texture.CopySavedData((Texture)saveDocument);
                }
            }

            // シェーダ定義
            foreach (var shaderDefinition in DocumentManager.ShaderDefinitions)
            {
                shaderDefinition.SetMaybeModified();

                var saveDocument = DocumentManager.GetSaveDocument(shaderDefinition.FilePath);
                if (saveDocument != null)
                {
                    shaderDefinition.CopySavedData((ShaderDefinition)saveDocument);
                }
            }

            // アニメーション
            foreach (var animation in DocumentManager.Animations)
            {
                animation.SetMaybeModified();

                var saveDocument = DocumentManager.GetSaveDocument(animation.FilePath);
                if (saveDocument != null)
                {
                    if (animation is SkeletalAnimation)
                    {
                        ((SkeletalAnimation)animation).CopySavedData((SkeletalAnimation)saveDocument);
                    }
                    else if (animation is ShapeAnimation)
                    {
                        ((ShapeAnimation)animation).CopySavedData((ShapeAnimation)saveDocument);
                    }
                    else if (animation is MaterialAnimation)
                    {
                        ((MaterialAnimation)animation).CopySavedData((MaterialAnimation)saveDocument);
                    }
                    else if (animation is ShaderParameterAnimation)
                    {
                        ((ShaderParameterAnimation)animation).CopySavedData((ShaderParameterAnimation)saveDocument);
                    }
                    else if (animation is ColorAnimation)
                    {
                        ((ColorAnimation)animation).CopySavedData((ColorAnimation)saveDocument);
                    }
                    else if (animation is TextureSrtAnimation)
                    {
                        ((TextureSrtAnimation)animation).CopySavedData((TextureSrtAnimation)saveDocument);
                    }
                    else if (animation is MaterialVisibilityAnimation)
                    {
                        ((MaterialVisibilityAnimation)animation).CopySavedData((MaterialVisibilityAnimation)saveDocument);
                    }
                    else if (animation is BoneVisibilityAnimation)
                    {
                        ((BoneVisibilityAnimation)animation).CopySavedData((BoneVisibilityAnimation)saveDocument);
                    }
                    else if (animation is TexturePatternAnimation)
                    {
                        ((TexturePatternAnimation)animation).CopySavedData((TexturePatternAnimation)saveDocument);
                    }
                    else if (animation is SceneAnimation)
                    {
                        ((SceneAnimation)animation).CopySavedData((SceneAnimation)saveDocument);
                    }
                }
            }
        }
    }
}
