﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms.VisualStyles;
using App.PropertyEdit;
using App.res;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public class Texture : IntermediateFileDocument, IHasUserData, NintendoWare.G3d.Edit.IEditTextureTarget
    {
        public textureType Data{ get; set; }

        public override object nw4f_3difItem
        {
            get { return Data; }
        }
        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public override process_log_arrayType process_log_array
        {
            get { return Data.process_log_array; }
        }

        // 一次元テクスチャかどうか？
        public bool Is1d
        {
            get
            {
                return
                    (Data.texture_info.dimension == texture_info_dimensionType.Item1d) ||
                    (Data.texture_info.dimension == texture_info_dimensionType.Item1d_array);
            }
        }

        // 配列テクスチャかどうか？
        public bool IsArray
        {
            get
            {
                return
                    (Data.texture_info.dimension == texture_info_dimensionType.Item1d_array) ||
                    (Data.texture_info.dimension == texture_info_dimensionType.Item2d_array) ||
                    (Data.texture_info.dimension == texture_info_dimensionType.cube_array);
            }
        }

        // キューブテクスチャかどうか？
        public bool IsCube
        {
            get
            {
                return
                    (Data.texture_info.dimension == texture_info_dimensionType.cube) ||
                    (Data.texture_info.dimension == texture_info_dimensionType.cube_array);
            }
        }

        public int DepthCount
        {
            get
            {
                return Data.texture_info.depth;
            }
        }
        public static object TexConvertLock = new object();

        public object LoadedLock = new object();
        public bool Loaded;

        public override bool IsFileOutputable
        {
            get { return IsTemporary == false; }
        }

        public override string FileNotOutputErrorMessage
        {
            get
            {
                Debug.Assert(IsFileOutputable == false);
                return Strings.Texture_FileNotOutputErrorMessage;
            }
        }

        public override bool Editable
        {
            get { return !IsTemporary; }
        }

        public Texture(textureType texture, List<G3dStream> streams)
            : base(GuiObjectID.Texture, streams)
        {
            Data = texture;
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            MipmapMinSize = new Size(1, 1);
            ConverterName = "";
            IsOriginalImage = false;

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, BinaryStreams);
            //Data.original_image_array.Items
            //Data.texture_info.stream_index

            Action add = delegate
            {
                lock (LoadedLock)
                {
                    Loaded = true;
                }
            };
            DocumentAdd += add;
            DocumentSwapAdd += d=>add();
            Action remove = delegate {
                lock (LoadedLock)
                {
                    Loaded = false;
                }
            };
            DocumentRemove += remove;
            DocumentSwapRemove += remove;
        }


        #region Bridge
        // モデルデータ差し替え用
        public class SwapData
        {
            public ToolData ToolData;
            public string Comment;
            public RgbaColor? EditColor;
            public UserDataArray userData;
            public TextureGeneralPage.TextureEditData textureData;
        }

        public SwapData CreateSwapData(textureType data, List<G3dStream> streams)
        {
            var toolData = new ToolData();
            toolData.Load(data.tool_data);
            var textureData = new TextureGeneralPage.TextureEditData();
            textureData.streams = streams;
            textureData.texture = data;
            return new SwapData()
            {
                ToolData = toolData,
                Comment = ToComment(data.comment),
                EditColor = ToColor(data.comment),
                userData = HasUserDataExtensions.MakeUserDataArray(data.user_data_array, streams),
                textureData = textureData,
            };
        }

        public SwapData GetSwapData()
        {
            var textureData = new TextureGeneralPage.TextureEditData();
            textureData.texture = Data;
            textureData.streams = BinaryStreams;
            textureData.WriteToFile(); // 一時ファイルに書き込み
            return new SwapData()
            {
                textureData = textureData,
                Comment = Comment,
                EditColor = EditColor,
                userData = UserDataArray,
            };
        }

        public SwapData Swap(SwapData data, bool copyHistory)
        {
            var current = GetSwapData();
            Comment = data.Comment;
            EditColor = data.EditColor;
            UserDataArray = data.userData;
            Data = data.textureData.texture;
            BinaryStreams = data.textureData.streams;
            data.textureData.Dispose(); // 一時ファイルを破棄

            if (copyHistory)
            {
                UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);
                TextureStreamChanged = !IsSameStreamToSavedStream(BinaryStreams[Data.texture_info.stream_index]);
            }

            return current;
        }
        #endregion

        public bool HasColor { get { return true; } }
        public bool HasAlpha { get; private set; }

        public object creatingImagesLock_ = new object();

        private Bitmap		ColorThumbnail_;
        private Bitmap		AlphaThumbnail_;
        private Bitmap      ColorLargeThumbnail_;
        private Bitmap      AlphaLargeThumbnail_;

        // コンバータの名前をテクスチャごとに設定しておく
        public string ConverterName;
        public bool IsOriginalImage { get; set; }

        public int OrigianlImageCount => Data?.original_image_array?.original_image?.Length ?? 0;

        public int MipLevel
        {
            get { return Data.texture_info.mip_level; }
        }

        public Size Size
        {
            get
            {
                return Data == null ? new Size(0, 0) : new Size(Data.texture_info.width, Data.texture_info.height);
            }
        }

        private texture_info_quantize_typeType QuantizeType
        {
            get { return Data.texture_info.quantize_type; }
        }

        public void ClearCache()
        {
            lock(creatingImagesLock_)
            {
                ColorThumbnail_ = null;
                AlphaThumbnail_ = null;
                ColorLargeThumbnail_ = null;
                AlphaLargeThumbnail_ = null;
            }
        }

        private static void BitmapDispose(Bitmap bitmap)
        {
            try
            {
                bitmap.Dispose();
            }
            catch (Exception e)
            {
                DebugConsole.WriteLine("ClearCacheException {0}", e.Message);
            }
        }

        protected override void OnChangeBinaryStreams()
        {
            ClearCache();
        }


        // 最大サムネイルサイズ
        public const int ThumbnailSize = 32;
        public const int ThumbnailLargeSize = 256;

        public void PrepareThumbnails()
        {
            var hasThumbs = true;
            if (HasColor)
            {
                if (ColorLargeThumbnail_ == null || ColorThumbnail_ == null)
                {
                    hasThumbs = false;
                }
            }

            if (HasAlpha)
            {
                if (AlphaLargeThumbnail_ == null || AlphaThumbnail_ == null)
                {
                    hasThumbs = false;
                }
            }

            if (hasThumbs) return;

            var colorImages = new Bitmap[] { };
            var alphaImages = new Bitmap[] { };
            CreateImages(ref colorImages, ref alphaImages);

            lock (creatingImagesLock_)
            {
                int depth = Data.texture_info.depth;
                if (Data.texture_info.dimension == texture_info_dimensionType.cube ||
                    Data.texture_info.dimension == texture_info_dimensionType.cube_array)
                {
                    depth /= 6;
                }
                int sqrtDepth = (int)Math.Ceiling(Math.Sqrt(depth));
                if (HasColor)
                {
                    if (ColorLargeThumbnail_ == null)
                    {
                        ColorLargeThumbnail_ = CreateThumbnailImage(colorImages[0], sqrtDepth*ThumbnailLargeSize, sqrtDepth);
                    }
                    if (ColorThumbnail_ == null)
                    {
                        // LargeThumbnailから生成する
                        ColorThumbnail_ = CreateThumbnailImage(ColorLargeThumbnail_, ThumbnailSize, 1);
                    }
                }
                if (HasAlpha)
                {
                    if (AlphaLargeThumbnail_ == null)
                    {
                        AlphaLargeThumbnail_ = CreateThumbnailImage(alphaImages[0], sqrtDepth*ThumbnailLargeSize, sqrtDepth);
                    }
                    if (AlphaThumbnail_ == null)
                    {
                        // LargeThumbnailから生成する
                        AlphaThumbnail_ = CreateThumbnailImage(AlphaLargeThumbnail_, ThumbnailSize, 1);
                    }
                }
            }
        }

        // カラーサムネイル用イメージ
        private Bitmap ColorThumbnail
        {
            get
            {
                PrepareThumbnails();
                return ColorThumbnail_;
            }
        }


        // カラーLargeサムネイル用イメージ
        private Bitmap ColorLargeThumbnail
        {
            get
            {
                PrepareThumbnails();
                return ColorLargeThumbnail_;
            }
        }

        // アルファサムネイル用イメージ
        private Bitmap AlphaThumbnail
        {
            get
            {
                PrepareThumbnails();
                return AlphaThumbnail_;
            }
        }

        // アルファLargeサムネイル用イメージ
        private Bitmap AlphaLargeThumbnail
        {
            get
            {
                PrepareThumbnails();
                return AlphaLargeThumbnail_;
            }
        }

        public Bitmap GetColorThumbnailCopy()
        {
            DebugConsole.WriteLine("GetColorThumbnailCopy() {0}", Name);
            Bitmap bitmap = null;
            lock (creatingImagesLock_)
            {
#if true
                bitmap = new Bitmap(ColorThumbnail);
#else
                var org = ColorThumbnail;
                bitmap = org.Clone(new Rectangle(0,0,org.Width, org.Height), org.PixelFormat);

#endif
            }
            return bitmap;
        }

        public Bitmap GetAlphaThumbnailCopy()
        {
            if (AlphaThumbnail == null) return null;
            Bitmap bitmap = null;
            lock (creatingImagesLock_)
            {
#if true
                bitmap = new Bitmap(AlphaThumbnail);
#else
                var org = AlphaThumbnail;
                bitmap = org.Clone(new Rectangle(0, 0, org.Width, org.Height), org.PixelFormat);

#endif
            }
            return bitmap;
        }

        //public Bitmap GetColorLargeThumbnailCopy() { return new Bitmap(ColorLargeThumbnail); }
        //public Bitmap GetAlphaLargeThumbnailCopy() { return new Bitmap(AlphaLargeThumbnail); }

        // ミップマップ最小サイズ
        public Size MipmapMinSize { get; set; }

        public void CreateImages(ref Bitmap[] colorImages, ref Bitmap[] alphaImages)
        {
            var sourceImages = CreateSourceImages();

            CreateColorAlphaImages(ref sourceImages, ref colorImages, ref alphaImages);
        }

        /// <summary>
        /// サムネイルイメージ作成。
        /// </summary>
        private const int timeoutCount = 10;
        private static Bitmap CreateThumbnailImage(Bitmap srcImage, int thumbnailSize, int factor)
        {
            // TODO:他のスレッドで使用中だと 例外が起きる
            // ビットマップのインスタンスはスレッドセーフではない!
            for(int i = 0;i != timeoutCount;++ i)
            {
                try
                {
                    return CreateThumbnailImageCore(srcImage, thumbnailSize, factor);
                }
                catch(Exception e)
                {
                    Debug.Assert(false, e.Message);
                    Thread.Sleep(1 + i);
                }
            }

            DebugConsole.WriteLine("TIMEOUT -- Texture.CreateThumbnailImage()");

            // そのまま返す
            return srcImage;
        }

        private static Bitmap CreateThumbnailImageCore(Bitmap srcImage, int thumbnailSize, int factor)
        {
            // 規定値より大きい場合のみ
            if (srcImage.Width > thumbnailSize || srcImage.Height > thumbnailSize)
            {
                // サイズ
                Size thumbSize = new Size(thumbnailSize, thumbnailSize);
                if (srcImage.Width > srcImage.Height)
                {
                    thumbSize.Height = Convert.ToInt32(thumbnailSize * ((float)srcImage.Height / (float)srcImage.Width));
                    if (thumbSize.Height == 0) { thumbSize.Height = 1; }

                    // factor の倍数にする
                    if (thumbSize.Height % factor != 0)
                    {
                        thumbSize.Height += factor - (thumbSize.Height % factor);
                    }
                }
                else if (srcImage.Height > srcImage.Width)
                {
                    thumbSize.Width = Convert.ToInt32(thumbnailSize * ((float)srcImage.Width / (float)srcImage.Height));
                    if (thumbSize.Width == 0) { thumbSize.Width = 1; }

                    // factor の倍数にする
                    if (thumbSize.Width % factor != 0)
                    {
                        thumbSize.Width += factor - (thumbSize.Width % factor);
                    }
                }

                // サムネイル作成
                Image thumbImage = srcImage.GetThumbnailImage(
                    thumbSize.Width,
                    thumbSize.Height,
                    delegate() { return false; },
                    IntPtr.Zero
                );


                return new Bitmap(thumbImage);
            }

            // そのまま返す(オリジナルは消去される場合があるのでコピーを返す)
            return new Bitmap(srcImage);
        }

        /// <summary>
        /// カラーサムネイル描画。
        /// </summary>
        public void DrawColorThumbnail(Graphics g, Rectangle rect, bool enableState = true, bool drawBackground = true)
        {
            lock (creatingImagesLock_)
            {
                Bitmap thumbnail = null;
                if (rect.Width > ThumbnailSize || rect.Height > ThumbnailSize)
                {
                    thumbnail = ColorLargeThumbnail;
                }
                else
                {
                    thumbnail = ColorThumbnail;
                }

                if (thumbnail != null)
                {
                    BitmapUtility.DrawTextureImage(g, thumbnail, rect.Width, rect.Height, Is1d, rect.Top, rect.Left);
                }
                else
                {
                    DebugConsole.WriteLine("Fail -- Texture.DrawColorThumbnail()");
                }
            }
        }

        public void DrawDepthThumbnail(Graphics g, Rectangle rect, bool color, int depthIndex, bool enableState = true, bool drawBackground = true)
        {
            lock (creatingImagesLock_)
            {
                Bitmap thumbnail = color ? ColorLargeThumbnail: AlphaLargeThumbnail;
                if (thumbnail != null)
                {
                    var depth = Data.texture_info.depth;
                    if (Data.texture_info.dimension == texture_info_dimensionType.cube ||
                        Data.texture_info.dimension == texture_info_dimensionType.cube_array)
                    {
                        depth /= 6;
                    }
                    int sqrtDepth = (int)Math.Ceiling(Math.Sqrt(depth));
                    var width = thumbnail.Width / sqrtDepth;
                    var height = thumbnail.Height / sqrtDepth;
                    var x = (depthIndex % sqrtDepth) * width;
                    var y = (depthIndex / sqrtDepth) * height;

                    // TODO: 大きなサムネイルの一部を描画しているので深さを超えて滲む可能性がある。
                    BitmapUtility.DrawTextureSubImage(g, thumbnail, width, height, x, y, rect.Width, rect.Height, Is1d, rect.Top, rect.Left);
                }
            }
        }

        /// <summary>
        /// カラーサムネイル描画(縦横比を保持しない。グレイ対応)。
        /// </summary>
        public void DrawColorThumbnailFit(Graphics g, Rectangle rect, bool isGray = false)
        {
            lock (creatingImagesLock_)
            {
                Bitmap thumbnail = null;
                if (rect.Width > ThumbnailSize || rect.Height > ThumbnailSize)
                {
                    thumbnail = ColorLargeThumbnail;
                }
                else
                {
                    thumbnail = ColorThumbnail;
                }

                if (thumbnail != null)
                {
                    if (isGray)
                    {
                        GraphicsUtility.DrawGrayImageSafe(g, thumbnail, 0, 0, thumbnail.Width, thumbnail.Height, rect.X, rect.Y, rect.Width, rect.Height);
                    }
                    else
                    {
                        g.DrawImageSafe(thumbnail, rect.X, rect.Y, rect.Width, rect.Height);
                    }
                }
                else
                {
                    DebugConsole.WriteLine("Fail -- Texture.DrawColorThumbnailGray()");
                }
            }
        }

        /// <summary>
        /// アルファサムネイル描画。
        /// </summary>
        public void DrawAlphaThumbnail(Graphics g, Rectangle rect, bool enableState = true, bool drawBackground = true)
        {
            if (AlphaLargeThumbnail == null) return;

            lock (creatingImagesLock_)
            {
                Bitmap thumbnail = null;
                if (rect.Width > ThumbnailSize || rect.Height > ThumbnailSize)
                {
                    thumbnail = AlphaLargeThumbnail;
                }
                else
                {
                    thumbnail = AlphaThumbnail;
                }

                if (thumbnail != null)
                {
                    BitmapUtility.DrawTextureImage(g, thumbnail, rect.Width, rect.Height, Is1d, rect.Top, rect.Left);
                }
                else
                {
                    DebugConsole.WriteLine("Fail -- Texture.DrawAlphaThumbnail()");
                }
            }
        }

        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            lock (this)
            {
                UpdateData();

                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        // シリアライズ前処理
        private void UpdateData()
        {
            ToolData.Load(Data.tool_data);
            Data.comment = GetComment();

            // ユーザーデータのシリアライズ
            if (UserDataArray == null ||
                UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
            }
            else
            {

                Data.user_data_array = new user_data_arrayType();
                this.MakeSerializeData(Data.user_data_array, UserDataArray, BinaryStreams);
            }

            // ストリームのソート(や削除等)を行う。
            nw.g3d.iflib.StreamUtility.SortStream(Data, BinaryStreams);

            Data.stream_array = null;
        }

        // 注意:UIスレッド以外で実行されることがある
        private Bitmap[] CreateSourceImages()
        {
            Bitmap[] images;
            lock (creatingImagesLock_)
            {
                using (var dsw = new DebugStopWatch(string.Format("Texture.CreateSourceImages()[{0}]", FileName)))
                {
                    images = new Bitmap[MipLevel];
                    switch (Data.texture_info.dimension)
                    {
                        case texture_info_dimensionType.Item1d:
                        case texture_info_dimensionType.Item2d:
                        {
                            var bitmaps = TexUtilsProxy.ConvertTo1d2dBitmap(Data, BinaryStreams);
                            for (int i = 0; i != MipLevel; ++i)
                            {
                                images[i] = bitmaps[i];
                            }

                            break;
                        }

                        case texture_info_dimensionType.Item3d:
                        case texture_info_dimensionType.Item1d_array:
                        case texture_info_dimensionType.Item2d_array:
                        {
                            var bitmaps =
                                (Data.texture_info.dimension == texture_info_dimensionType.Item3d      ) ? TexUtilsProxy.ConvertTo3dBitmap     (Data, BinaryStreams) :
                                (Data.texture_info.dimension == texture_info_dimensionType.Item1d_array) ? TexUtilsProxy.ConvertTo1dArrayBitmap(Data, BinaryStreams) :
                                (Data.texture_info.dimension == texture_info_dimensionType.Item2d_array) ? TexUtilsProxy.ConvertTo2dArrayBitmap(Data, BinaryStreams) :
                                                                                                           null;

                            Debug.Assert(bitmaps != null);

                            int size = (int) Math.Ceiling(Math.Sqrt(Data.texture_info.depth));
                            int depthCount = Data.texture_info.depth;

                            int allWidth = size;
                            int allHeight = size;

                            // ミップマップ最小サイズを保存する
                            // 縦方向に並ぶ１次元テクスチャ最小サイズを制限する
                            MipmapMinSize = new Size(size, size);

                            for (int i = 0; i != MipLevel; ++ i)
                            {
                                int width = Math.Max(1, Data.texture_info.width >> i);
                                int height = Math.Max(1, Data.texture_info.height >> i);

                                images[i] = new Bitmap(width*allWidth, height*allHeight, PixelFormat.Format32bppArgb);
                                {
                                    using (var color = Graphics.FromImage(images[i]))
                                    {
                                        color.Clear(Color.Transparent);

                                        for (int j = 0; j != depthCount; ++ j)
                                        {
                                            int x = j%allWidth;
                                            int y = j/allHeight;

                                            color.DrawImage(bitmaps[j][i], width*x, height*y);
                                        }
                                    }
                                }

                                if (Data.texture_info.dimension == texture_info_dimensionType.Item3d)
                                {
                                    depthCount = Math.Max(depthCount >> 1, 1);
                                    allWidth = allHeight = Math.Max((int) Math.Ceiling(Math.Sqrt(depthCount)), 1);
                                }
                            }

                            break;
                        }

                        case texture_info_dimensionType.cube:
                        {
                            var bitmaps = TexUtilsProxy.ConvertToCubeBitmap(Data, BinaryStreams);

                            for (int i = 0; i != MipLevel; ++ i)
                            {
                                int width = Math.Max(1, Data.texture_info.width >> i);
                                int height = Math.Max(1, Data.texture_info.height >> i);

                                images[i] = new Bitmap(width*3, height*2, PixelFormat.Format32bppArgb);
                                {
                                    using (var color = Graphics.FromImage(images[i]))
                                    {
                                        color.Clear(Color.Transparent);
                                        color.DrawImage(bitmaps[0][i], width*0, height*0);
                                        color.DrawImage(bitmaps[1][i], width*0, height*1);
                                        color.DrawImage(bitmaps[2][i], width*1, height*0);
                                        color.DrawImage(bitmaps[3][i], width*1, height*1);
                                        color.DrawImage(bitmaps[4][i], width*2, height*0);
                                        color.DrawImage(bitmaps[5][i], width*2, height*1);
                                    }
                                }
                            }

                            break;
                        }

                        case texture_info_dimensionType.cube_array:
                        {
                            var bitmaps = TexUtilsProxy.ConvertToCubeArrayBitmap(Data, BinaryStreams);

                            int depthCount = Data.texture_info.depth/6;
                            int size = (int) Math.Ceiling(Math.Sqrt(depthCount));

                            int allWidth = size;
                            int allHeight = size;

                            for (int m = 0; m != MipLevel; ++ m)
                            {
                                int width = Math.Max(1, Data.texture_info.width >> m);
                                int height = Math.Max(1, Data.texture_info.height >> m);

                                images[m] = new Bitmap(width*3*allWidth, height*2*allHeight, PixelFormat.Format32bppArgb);
                                {
                                    using (var color = Graphics.FromImage(images[m]))
                                    {
                                        color.Clear(Color.Transparent);

                                        for (int d = 0; d != depthCount; ++ d)
                                        {
                                            int x = (d%size)*width*3;
                                            int y = (d/size)*height*2;

                                            color.DrawImage(bitmaps[d*6 + 0][m], x + width*0, y + height*0);
                                            color.DrawImage(bitmaps[d*6 + 1][m], x + width*0, y + height*1);
                                            color.DrawImage(bitmaps[d*6 + 2][m], x + width*1, y + height*0);
                                            color.DrawImage(bitmaps[d*6 + 3][m], x + width*1, y + height*1);
                                            color.DrawImage(bitmaps[d*6 + 4][m], x + width*2, y + height*0);
                                            color.DrawImage(bitmaps[d*6 + 5][m], x + width*2, y + height*1);
                                        }
                                    }
                                }
                            }

                            break;
                        }
                    }
                    // コンバータの名前をテクスチャごとに設定する
                    ConverterName = TexUtilsProxy.GetConverterName(Data);
                    IsOriginalImage = ConverterName.Contains("Original");
                }
            }

            return images;
        }



        // コンポーネントセレクタの色変換行列の列を求める
        private static float[] MakeCompSelMatrixCol_Color(texture_info_comp_selValue compSel, bool replaceGwithA)
        {
            var s = new float[5]{0.0f, 0.0f, 0.0f, 0.0f, 0.0f};
            switch(compSel)
            {
                case texture_info_comp_selValue.r:		s[0] = 1.0f;	break;
                case texture_info_comp_selValue.g:		s[1] = 1.0f;	break;
                case texture_info_comp_selValue.b:		s[2] = 1.0f;	break;
                case texture_info_comp_selValue.a:		s[3] = 1.0f;	break;
                case texture_info_comp_selValue.Item0:	s[4] = 0.0f;	break;
                case texture_info_comp_selValue.Item1:	s[4] = 1.0f;	break;
            }
            if (replaceGwithA)
            {
                var temp = s[1];
                s[1] = s[3];
                s[3] = temp;
            }
            return s;
        }


        private void CreateColorAlphaImages(ref Bitmap[] sourceImages, ref Bitmap[] colorImages, ref Bitmap[] alphaImages)
        {
            if (Data == null || Data.texture_info == null)
                return;

            var isHintNormal = Data.texture_info.hint == "normal";
            bool replaceGwithA = false;
            var isAstcFormat = (QuantizeType >= texture_info_quantize_typeType.unorm_astc_4x4 &&
                          QuantizeType <= texture_info_quantize_typeType.srgb_astc_12x12);
            if (IsOriginalImage)
            {
                if (isHintNormal)
                {
                    replaceGwithA = isAstcFormat;
                }
                else
                {
                    replaceGwithA = (QuantizeType == texture_info_quantize_typeType.unorm_4_4 ||
                                        QuantizeType == texture_info_quantize_typeType.unorm_8_8 ||
                                        QuantizeType == texture_info_quantize_typeType.unorm_bc5 ||
                                        QuantizeType == texture_info_quantize_typeType.unorm_eac_11_11);

                }
            }

            var compSelR = MakeCompSelMatrixCol_Color(Data.texture_info.comp_sel[0], replaceGwithA);
            var compSelG = MakeCompSelMatrixCol_Color(Data.texture_info.comp_sel[1], replaceGwithA);
            var compSelB = MakeCompSelMatrixCol_Color(Data.texture_info.comp_sel[2], replaceGwithA);
            var compSelA = MakeCompSelMatrixCol_Color(Data.texture_info.comp_sel[3], replaceGwithA);

            var colorMatrix = new ColorMatrix(
                new[]
                {
                    new[] {compSelR[0], compSelG[0], compSelB[0], 0.0f, 0.0f},
                    new[] {compSelR[1], compSelG[1], compSelB[1], 0.0f, 0.0f},
                    new[] {compSelR[2], compSelG[2], compSelB[2], 0.0f, 0.0f},
                    new[] {compSelR[3], compSelG[3], compSelB[3], 0.0f, 0.0f},
                    new[] {compSelR[4], compSelG[4], compSelB[4], 1.0f, 0.0f},
                }
                );

            var alphaMatrix = new ColorMatrix(
                new[]
                {
                    new[] {compSelA[0], compSelA[0], compSelA[0], 0.0f, 0.0f},
                    new[] {compSelA[1], compSelA[1], compSelA[1], 0.0f, 0.0f},
                    new[] {compSelA[2], compSelA[2], compSelA[2], 0.0f, 0.0f},
                    new[] {compSelA[3], compSelA[3], compSelA[3], 0.0f, 0.0f},
                    new[] {compSelA[4], compSelA[4], compSelA[4], 1.0f, 0.0f},
                }
                );

            var mipmapSize = sourceImages.Length;

            colorImages = new Bitmap[mipmapSize];
            alphaImages = new Bitmap[mipmapSize];

            for (int i = 0; i != mipmapSize; ++i)
            {
                var src = sourceImages[i];
                var srcW = src.Width;
                var srcH = src.Height;

                colorImages[i] = new Bitmap(srcW, srcH, PixelFormat.Format32bppArgb);
                alphaImages[i] = new Bitmap(srcW, srcH, PixelFormat.Format32bppArgb);

                {
                    using (var ia = new ImageAttributes())
                    using (var g = Graphics.FromImage(colorImages[i]))
                    {
                        ia.SetColorMatrix(colorMatrix);

                        g.DrawImage(
                            src,
                            new Rectangle(0, 0, srcW, srcH),
                            0, 0, srcW, srcH,
                            GraphicsUnit.Pixel,
                            ia
                            );
                    }
                }

                {
                    using (var ia = new ImageAttributes())
                    using (var g = Graphics.FromImage(alphaImages[i]))
                    {
                        ia.SetColorMatrix(alphaMatrix);

                        g.DrawImage(
                            src,
                            new Rectangle(0, 0, srcW, srcH),
                            0, 0, srcW, srcH,
                            GraphicsUnit.Pixel,
                            ia
                            );
                    }
                }

                if (isHintNormal)
                {
                    if (QuantizeType == texture_info_quantize_typeType.snorm_8_8 ||
                        QuantizeType == texture_info_quantize_typeType.sint_8_8)
                    {
                        var bitmap = colorImages[i];
                        var bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height),
                            ImageLockMode.ReadWrite, PixelFormat.Format32bppArgb);

                        unsafe
                        {
                            var startPtr = (UInt32*) bitmapData.Scan0;
                            var height = bitmap.Height;
                            var width = bitmap.Width;
                            var stride = bitmapData.Stride;

                            Parallel.For(0, height, i1 =>
                            {
                                var linePtr = startPtr + i1*stride/sizeof(UInt32);

                                Parallel.For(0, width, i2 =>
                                {
                                    var currentPtr = linePtr + i2;

                                    var colorPtr = (Byte*) currentPtr;

                                    var sr = colorPtr[2]/255.0f;
                                    var sg = colorPtr[1]/255.0f;
                                    var sb = Math.Sqrt(1.0f - Math.Min(sr*sr + sg*sg, 1.0f));

                                    colorPtr[2] = (byte) (128 + (int) (127.0f*sr));
                                    colorPtr[1] = (byte) (128 + (int) (127.0f*sg));
                                    colorPtr[0] = (byte) (128 + (int) (127.0f*sb));
                                });
                            });
                        }

                        bitmap.UnlockBits(bitmapData);
                    }
                    else if (QuantizeType == texture_info_quantize_typeType.snorm_bc5 || isAstcFormat)
                    {
                        var bitmap = colorImages[i];
                        var bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height),
                            ImageLockMode.ReadWrite, PixelFormat.Format32bppArgb);

                        unsafe
                        {
                            var startPtr = (UInt32*) bitmapData.Scan0;
                            var height = bitmap.Height;
                            var width = bitmap.Width;
                            var stride = bitmapData.Stride;

                            Parallel.For(0, height, i1 =>
                            {
                                var linePtr = startPtr + i1*stride/sizeof(UInt32);

                                Parallel.For(0, width, i2 =>
                                {
                                    var currentPtr = linePtr + i2;

                                    var colorPtr = (Byte*) currentPtr;

                                    var sr = Math.Min(Math.Max((((int) colorPtr[2]) - 128)/127.0f, -1.0f), +1.0f);
                                    var sg = Math.Min(Math.Max((((int) colorPtr[1]) - 128)/127.0f, -1.0f), +1.0f);
                                    var sb = Math.Sqrt(1.0f - Math.Min(sr*sr + sg*sg, 1.0f));

                                    colorPtr[0] = (byte) Math.Min(Math.Max(128 + (int) (127.0f*sb), 0), 255);
                                });
                            });
                        }

                        bitmap.UnlockBits(bitmapData);
                    }
                }

                // キューブマップであれば展開図にする
                if (Data.texture_info.dimension == texture_info_dimensionType.cube)
                {
                    colorImages[i] = MakeOpenedCubeImage(colorImages[i]);
                    alphaImages[i] = MakeOpenedCubeImage(alphaImages[i]);
                }
            }

            MakeHasAlpha(ref alphaImages);
        }

        private static Bitmap MakeOpenedCubeImage(Bitmap src)
        {
            int w = src.Width  / 3;
            int h = src.Height / 2;

            var image = new Bitmap(w * 4, h * 3, PixelFormat.Format32bppArgb);
            {
                using (var g = Graphics.FromImage(image))
                {
                    g.Clear(Color.Transparent);
                    g.DrawImage(src, w * 2, h * 1, new Rectangle(w * 0, h * 0, w, h), GraphicsUnit.Pixel);
                    g.DrawImage(src, w * 0, h * 1, new Rectangle(w * 0, h * 1, w, h), GraphicsUnit.Pixel);
                    g.DrawImage(src, w * 1, h * 0, new Rectangle(w * 1, h * 0, w, h), GraphicsUnit.Pixel);
                    g.DrawImage(src, w * 1, h * 2, new Rectangle(w * 1, h * 1, w, h), GraphicsUnit.Pixel);
                    g.DrawImage(src, w * 3, h * 1, new Rectangle(w * 2, h * 0, w, h), GraphicsUnit.Pixel);
                    g.DrawImage(src, w * 1, h * 1, new Rectangle(w * 2, h * 1, w, h), GraphicsUnit.Pixel);
                }
            }
            return image;
        }

        private static Bitmap MakeOpenedCubeArrayImage(Bitmap src, int depth)
        {
            depth /= 6;		// 一枚あたり単位に調整する

            int size = (int)Math.Ceiling(Math.Sqrt(depth));

            int w = src.Width  / 3 / size;
            int h = src.Height / 2 / size;

            var image = new Bitmap(w * 4 * size, h * 3 * size, PixelFormat.Format32bppArgb);
            {
                using (var g = Graphics.FromImage(image))
                {
                    g.Clear(Color.Transparent);

                    for(int d = 0;d != depth;++ d)
                    {
                        int sx = (d % size) * w * 3;
                        int sy = (d / size) * h * 2;
                        int dx = (d % size) * w * 4;
                        int dy = (d / size) * h * 3;

                        g.DrawImage(src, dx + w*2, dy + h*1, new Rectangle(sx + w*0, sy + h*0, w, h), GraphicsUnit.Pixel);
                        g.DrawImage(src, dx + w*0, dy + h*1, new Rectangle(sx + w*0, sy + h*1, w, h), GraphicsUnit.Pixel);
                        g.DrawImage(src, dx + w*1, dy + h*0, new Rectangle(sx + w*1, sy + h*0, w, h), GraphicsUnit.Pixel);
                        g.DrawImage(src, dx + w*1, dy + h*2, new Rectangle(sx + w*1, sy + h*1, w, h), GraphicsUnit.Pixel);
                        g.DrawImage(src, dx + w*1, dy + h*1, new Rectangle(sx + w*2, sy + h*0, w, h), GraphicsUnit.Pixel);
                        g.DrawImage(src, dx + w*3, dy + h*1, new Rectangle(sx + w*2, sy + h*1, w, h), GraphicsUnit.Pixel);
                    }
                }
            }

            return image;
        }

        private void MakeHasAlpha(ref Bitmap[] alphaImages)
        {
            if( (QuantizeType == texture_info_quantize_typeType.unorm_1_5_5_5) ||
                (QuantizeType == texture_info_quantize_typeType.unorm_5_5_5_1) ||
                (QuantizeType == texture_info_quantize_typeType.unorm_4_4_4_4) ||

                (QuantizeType == texture_info_quantize_typeType.unorm_2_10_10_10) ||

                (QuantizeType == texture_info_quantize_typeType.unorm_8_8_8_8) ||
                (QuantizeType == texture_info_quantize_typeType.uint_8_8_8_8) ||
                (QuantizeType == texture_info_quantize_typeType.snorm_8_8_8_8) ||
                (QuantizeType == texture_info_quantize_typeType.sint_8_8_8_8) ||
                (QuantizeType == texture_info_quantize_typeType.srgb_8_8_8_8) ||

                (QuantizeType == texture_info_quantize_typeType.unorm_10_10_10_2) ||
                (QuantizeType == texture_info_quantize_typeType.uint_10_10_10_2) ||
                (QuantizeType == texture_info_quantize_typeType.unorm_16_16_16_16) ||
                (QuantizeType == texture_info_quantize_typeType.uint_16_16_16_16) ||
                (QuantizeType == texture_info_quantize_typeType.snorm_16_16_16_16) ||
                (QuantizeType == texture_info_quantize_typeType.sint_16_16_16_16) ||
                (QuantizeType == texture_info_quantize_typeType.float_16_16_16_16) ||
                (QuantizeType == texture_info_quantize_typeType.uint_32_32_32_32) ||
                (QuantizeType == texture_info_quantize_typeType.sint_32_32_32_32) ||
                (QuantizeType == texture_info_quantize_typeType.float_32_32_32_32) ||

                (QuantizeType == texture_info_quantize_typeType.unorm_bc1) ||
                (QuantizeType == texture_info_quantize_typeType.srgb_bc1) ||
                (QuantizeType == texture_info_quantize_typeType.unorm_bc2) ||
                (QuantizeType == texture_info_quantize_typeType.srgb_bc2) ||
                (QuantizeType == texture_info_quantize_typeType.unorm_bc3) ||
                (QuantizeType == texture_info_quantize_typeType.srgb_bc3) ||

                (QuantizeType == texture_info_quantize_typeType.unorm_etc2_mask) ||
                (QuantizeType == texture_info_quantize_typeType.unorm_etc2_alpha) ||
                (QuantizeType == texture_info_quantize_typeType.srgb_etc2_mask) ||
                (QuantizeType == texture_info_quantize_typeType.srgb_etc2_alpha) ||
                (QuantizeType == texture_info_quantize_typeType.unorm_pvrtc1_alpha_2bpp) ||
                (QuantizeType == texture_info_quantize_typeType.unorm_pvrtc1_alpha_4bpp) ||
                (QuantizeType == texture_info_quantize_typeType.srgb_pvrtc1_alpha_2bpp) ||
                (QuantizeType == texture_info_quantize_typeType.srgb_pvrtc1_alpha_4bpp) ||
                (QuantizeType == texture_info_quantize_typeType.unorm_pvrtc2_alpha_2bpp) ||
                (QuantizeType == texture_info_quantize_typeType.unorm_pvrtc2_alpha_4bpp) ||
                (QuantizeType == texture_info_quantize_typeType.srgb_pvrtc2_alpha_2bpp) ||
                (QuantizeType == texture_info_quantize_typeType.srgb_pvrtc2_alpha_4bpp))
            {
                HasAlpha = true;
            }
            else
            {
                HasAlpha = Data.texture_info.comp_sel[3] != texture_info_comp_selValue.Item1;
            }
        }
        #region savedData
        private texture_infoType savedData;
        public G3dStream savedTextureStream;
        public bool TextureStreamChanged = false;
        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data.texture_info);
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;
            savedTextureStream = BinaryStreams[Data.texture_info.stream_index];
            TextureStreamChanged = false;
        }

        public void CopySavedData(Texture source)
        {
            CopyDocumentSavedData(source);
            savedData = source.savedData;
            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !source.SavedUserDataArray.IsSame(UserDataArray);
            savedTextureStream = source.savedTextureStream;
            TextureStreamChanged = !IsSameStreamToSavedStream(BinaryStreams[Data.texture_info.stream_index]);
        }

        public bool IsSameStreamToSavedStream(G3dStream rhs)
        {
            var lhs = savedTextureStream;
            if (lhs.type != rhs.type)
            {
                return false;
            }

            switch (lhs.type)
            {
                case stream_typeType.@byte:
                    return IsSameList(lhs.ByteData, rhs.ByteData);
                case stream_typeType.@float:
                    return IsSameList(lhs.FloatData, rhs.FloatData);
                case stream_typeType.@int:
                    return IsSameList(lhs.IntData, rhs.IntData);
                case stream_typeType.@string:
                case stream_typeType.wstring:
                    return lhs.StringData == rhs.StringData;
            }

            return false;
        }

        private bool IsSameList<T>(List<T> lhs, List<T> rhs) where T : struct
        {
            return lhs.Count == rhs.Count && lhs.SequenceEqual(rhs);
        }

        public bool IsValueChanged<T>(Func<texture_infoType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data.texture_info));
        }

        public bool IsValueChanged(Func<texture_infoType, string> select)
        {
            return select(savedData) != select(Data.texture_info);
        }

        public bool IsArrayChanged<T>(Func<texture_infoType, T[]> select) where T : struct
        {
            return !select(savedData).SequenceEqual(select(Data.texture_info));
        }

        public bool IsSequenceValueChanged<T>(Func<texture_infoType, IEnumerable<T>> select) where T : struct
        {
            return !select(savedData).SequenceEqual(select(Data.texture_info));
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !TextureGeneralPage.IsModified(this) &&
                !UserDataPage.IsModified(this) &&
                !IsImageModified();
        }

        #endregion

        public override void OnDocumentSwapAdd(Document old)
        {
            CopySavedData((Texture)old);
            base.OnDocumentSwapAdd(old);
        }

        // イメージに変更が無いか確認する
        private bool IsImageModified()
        {
            // ハッシュ値が違えば別のデータに差し替えられた。ハッシュ値が存在しない場合はストリームの確認が必要。
            if (string.IsNullOrEmpty(Data.texture_info.original_image_hash) ||
                string.IsNullOrEmpty(savedData.original_image_hash))
            {
                if (!IsSameStreamToSavedStream(BinaryStreams[Data.texture_info.stream_index]))
                {
                    return true;
                }
            }
            else
            {
                if (Data.texture_info.original_image_hash != savedData.original_image_hash)
                {
                    return true;
                }
            }

            return false;
        }
    }
}
