﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Linq;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public interface IHasUserData
    {
        UserDataArray UserDataArray { get; set; }

        // 差分用
        UserDataArray SavedUserDataArray { get; set; }

        // 差分の状態
        bool UserDataArrayChanged { get; set; }
    }

    // 拡張メソッド
    public static class HasUserDataExtensions
    {
        public static UserDataArray MakeUserDataArray(user_data_arrayType data, List<G3dStream> streams)
        {
            var userDataArray = new UserDataArray();
            if (data == null || streams == null)
            {
                return userDataArray;
            }

            foreach (var user_data in data.user_data)
            {
                if (user_data.Item is user_intType)
                {
                    user_intType Item = user_data.Item as user_intType;
                    int[] targetParamArray = G3dDataParser.ParseIntArray(Item.Value);
                    List<int> intItem = new List<int>();
                    if (targetParamArray != null)
                    {
                        intItem.AddRange(targetParamArray);
                    }
                    userDataArray.Add(user_data.name, intItem);
                }
                else if (user_data.Item is user_floatType)
                {
                    user_floatType Item = user_data.Item as user_floatType;
                    float[] targetParamArray = G3dDataParser.ParseFloatArray(Item.Value);
                    List<float> floatItem = new List<float>();
                    if (targetParamArray != null)
                    {
                        floatItem.AddRange(targetParamArray);
                    }
                    userDataArray.Add(user_data.name, floatItem);
                }
                else if (user_data.Item is user_stringType)
                {
                    user_stringType ArrayItem = user_data.Item as user_stringType;
                    if (ArrayItem == null)
                    {
                        continue;
                    }

                    List<string> stringList = new List<string>();
                    if (ArrayItem.@string != null)
                    {
                        foreach (var Item in ArrayItem.@string)
                        {
                            stringList.Add(Item);
                        }
                    }
                    userDataArray.Add(user_data.name, stringList, false);
                }
                else if (user_data.Item is user_wstringType)
                {
                    user_wstringType ArrayItem = user_data.Item as user_wstringType;
                    if (ArrayItem == null)
                    {
                        continue;
                    }

                    List<string> stringList = new List<string>();
                    if (ArrayItem.wstring != null)
                    {
                        foreach (var Item in ArrayItem.wstring)
                        {
                            stringList.Add(Item);
                        }
                    }
                    userDataArray.Add(user_data.name, stringList, true);
                }
                else if (user_data.Item is user_streamType)
                {
                    user_streamType Item = user_data.Item as user_streamType;
                    if (Item == null)
                    {
                        continue;
                    }

                    if (Item.stream_index < -1 || Item.stream_index >= streams.Count)
                    {
                        continue;
                    }

                    List<byte> byteItem = new List<byte>();

                    if (Item.stream_index == -1)
                    {
                        userDataArray.Add(user_data.name, byteItem, Item.stream_index);
                    }
                    else
                    {
                        G3dStream stream = streams[Item.stream_index];
                        if (stream == null)
                        {
                            continue;
                        }
                        byteItem.AddRange(stream.ByteData.ToArray());
                        userDataArray.Add(user_data.name, byteItem, Item.stream_index);
                    }
                }
            }
            return userDataArray;
        }
        public static void MakeUserData(this IHasUserData userDataObj, user_data_arrayType data, List<G3dStream> streams)
        {
            userDataObj.UserDataArray = MakeUserDataArray(data, streams);
        }

        private static G3dStream GetOriginalStream(UserDataArray.UserData userData, List<G3dStream> tempStreams)
        {
            G3dStream tempStream = null;
            if (userData is UserDataArray.UserDataStream)
            {
                UserDataArray.UserDataStream userDataStream = userData as UserDataArray.UserDataStream;
                if (tempStreams.Count > userDataStream.StreamIndex && userDataStream.StreamIndex != -1)
                {
                    tempStream = tempStreams[userDataStream.StreamIndex];
                    if (tempStream.type == stream_typeType.@byte &&
                       tempStream.column == 16)
                    {
                        if (tempStream.ByteData.Any())
                        {
                            tempStream.ByteData.Clear();
                        }
                    }
                    else
                    {
                        tempStream = null;
                    }
                }
            }
            return tempStream;
        }

        public static void MakeSerializeData(this IHasUserData userDataObj, user_data_arrayType user_data_array, UserDataArray userDataArray, List<G3dStream> tempStreams)
        {
            List<user_dataType> userDataList = new List<user_dataType>();
            int userDataIndex = 0;
            foreach (var userData in userDataArray.Data)
            {
                user_dataType tempUserData = new user_dataType();
                tempUserData.index = userDataIndex;
                userDataIndex++;
                tempUserData.name = userData.Name;
                if (userData is UserDataArray.UserDataInt)
                {
                    user_intType intItem = new user_intType();
                    UserDataArray.UserDataInt userIntData = userData as UserDataArray.UserDataInt;

                    intItem.count = userIntData.Item.Count;
                    intItem.Value = Utility.IfUtility.MakeArrayString(userIntData.Item);

                    tempUserData.Item = intItem;
                    tempUserData.name = userData.Name;
                }
                else if (userData is UserDataArray.UserDataFloat)
                {
                    user_floatType floatItem = new user_floatType();
                    UserDataArray.UserDataFloat userFloatData = userData as UserDataArray.UserDataFloat;

                    floatItem.count = userFloatData.Item.Count;
                    floatItem.Value = Utility.IfUtility.MakeArrayString(userFloatData.Item);

                    tempUserData.Item = floatItem;
                    tempUserData.name = userData.Name;
                }
                else if (userData is UserDataArray.UserDataString)
                {
                    user_stringType stringItem = new user_stringType();
                    UserDataArray.UserDataString userStringData = userData as UserDataArray.UserDataString;

                    stringItem.@string = userStringData.Item.ToArray();
                    stringItem.count = userStringData.Item.Count;

                    tempUserData.Item = stringItem;
                    tempUserData.name = userData.Name;
                }
                else if (userData is UserDataArray.UserDataWString)
                {
                    user_wstringType stringItem = new user_wstringType();
                    UserDataArray.UserDataWString userStringData = userData as UserDataArray.UserDataWString;

                    stringItem.wstring = userStringData.Item.ToArray();
                    stringItem.count = userStringData.Item.Count;

                    tempUserData.Item = stringItem;
                    tempUserData.name = userData.Name;
                }
                else if (userData is UserDataArray.UserDataStream)
                {
                    user_streamType streamItem = new user_streamType();
                    UserDataArray.UserDataStream userStreamData = userData as UserDataArray.UserDataStream;

                    streamItem.stream_index = userStreamData.StreamIndex;
                    streamItem.size = userStreamData.Item.Count;
                    if (streamItem.size == 0)
                    {
                        streamItem.stream_index = -1;
                        tempUserData.Item = streamItem;
                        tempUserData.name = userData.Name;
                        userDataList.Add(tempUserData);
                        continue;
                    }

                    G3dStream tempStream = GetOriginalStream(userStreamData, tempStreams);
                    bool orgFlg = false;
                    if (tempStream == null)
                    {
                        // 元のストリームは無かったので新規作成
                        tempStream = new G3dStream();
                        tempStream.type = stream_typeType.@byte;
                        tempStream.column = 16;
                        userStreamData.StreamIndex = tempStreams.Count();
                        streamItem.stream_index = userStreamData.StreamIndex;
                    }
                    else
                    {
                        streamItem.stream_index = userStreamData.StreamIndex;
                        orgFlg = true;
                    }

                    // データのコピー
                    tempStream.ByteData.AddRange(userStreamData.Item.ToArray());

                    if (!orgFlg)
                    {
                        tempStreams.Add(tempStream);
                    }
                    tempUserData.Item = streamItem;
                    tempUserData.name = userData.Name;
                }
                userDataList.Add(tempUserData);
            }

            user_data_array.length = userDataList.Count;
            user_data_array.user_data = userDataList.ToArray();
        }
    }
}
